/*==========================================================
 * Program : gsmts0338.c                   Project : smslink
 * Authors : Philippe Andersson.
 *           Franois Baligant <francois@euronet.be>.
 * Date    : 28/05/03
 * Version : 0.04b
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 * Comment : PDU Manipulation tool. Encodes and decodes GSM TS
 *           03.38 alphabet.
 *
 * Modification History :
 * - 0.01a (18/10/01) : Initial release.
 * - 0.02a (23/01/02) : Added support for default GSM alphabet.
 * ++++ Switch to Beta ++++
 * - 0.03b (29/01/02) : Added CLI option to set the alphabet to
 *   use. Added warning on trailing '@' after decoding.
 * - 0.04b (28/05/03) : Added handling of new udl param to
 *   packed8toascii7().
 *========================================================*/
#include <unistd.h>
#include <stdio.h>                         /* for fprintf */
#include <stdlib.h>                  /* for errno & stuff */
#include <errno.h>
#include <string.h>

#include "sms_serv.h"
#include "pdu.h"
#include "gsmts0338.h"

/*========================================================*/
/**********             LOCAL DEFINES              ********/
/*========================================================*/

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/
int debug = FALSE;
char *progname = NULL;
int default_alphabet = DEFAULT_ALPHABET;

/*========================================================*/
/**********           MAIN PROGRAM LOOP            ********/
/*========================================================*/
int main (int argc, char *argv[])
{
  extern char *optarg;
  extern int optind, optopt;
  int c;                /* option char returned by getopt */
  int errflag = 0;       /* error in command line options */
  int task = 0;
  char *errptr;
  char *string;
  char *message;
  char *csname = NULL;
  int tpudl = TP_UDL_UNKNOWN;

  /*---------------------------------------Store own name */
  progname = (char *) malloc ((strlen (argv[0]) + 1) * sizeof (char));
  if (! progname) {
    syserr("main(): can't malloc()");
  }
  strcpy (progname, argv[0]);
  l_basename (progname);
  
  /*--------------------------------------Initialisations */
  string = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  if (!string) {
    syserr("main(): can't malloc() string");
  }
  string[0] = '\0';
  message = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  if (!message) {
    syserr("main(): can't malloc() message");
  }
  message[0] = '\0';

  /*-------------------------Command line processing loop */
  while ((c = getopt (argc, argv, ":ghvd:e:l:z:")) != -1) {
    switch (c) {
      case 'g':                               /* debug ON */
        debug = TRUE;
        break;
	
      case 'd':                                 /* decode */
        if (strlen (optarg) > BIGBUFF) {
	  fprintf (stderr, "gsmts0338: -d parameter too long, max. is %d\n",
	          BIGBUFF);
	  errflag++;
	}
	else
	  strcpy (string, optarg);
	task = GSMTS_DECODE;
        break;

      case 'e':                                 /* encode */
        if (strlen (optarg) > BIGBUFF) {
	  fprintf (stderr, "gsmts0338: -e parameter too long, max. is %d\n",
	          BIGBUFF);
	  errflag++;
	}
	else
          strcpy (message, optarg);
	task = GSMTS_ENCODE;
        break;

      case 'l':
        if (optarg) {
          tpudl = strtol (optarg, &errptr, 10);
          if (*errptr != '\0') {
            fprintf (stderr, "Expected numeric value after -l (got: [%s])\n", optarg);
	    exit (-1);
          }
        }
        break;

      case 'z':
        if (optarg) {
          if ((default_alphabet = getcharsetID (optarg)) == -1) {
            fprintf (stderr, "Requested charset %s not supported\n", optarg);
	    exit (-1);
	  }
        }
        break;

      case 'v':                           /* version info */
	printf ("gsmts0338 ver. %s (%s), SMSLink PDU Manipulation Tool.\n", 
        	GSMTS_VERSION, GSMTS_DATE);
	printf ("(c) Les Ateliers du Heron, 1998-2001.\n");
        exit (0);
        break;
	
      case 'h':                             /* usage help */
	printf ("\ngsmts0338 ver. %s (%s), SMSLink PDU Manipulation Tool.\n", 
        	GSMTS_VERSION, GSMTS_DATE);
	printf ("(c) Les Ateliers du Heron, 1998-2001.\n");
	printf ("\nUsage:\n");
	printf ("gsmts0338 [options] (-d string | -e msg)\n");
	printf ("\nWhere: string  = GSM TS 03.38 string to decode\n");
	printf ("       msg     = message to encode to GSM TS 03.38\n");
	printf ("Other options:\n");
        printf ("-l<n>     specify encoded string length [def. = guess]\n");
        printf ("-z<s>     sets character set for GSMTS processing\n");
	printf ("-g        sets debugging ON [def. = OFF]\n");
	printf ("-v        prints version information and exits\n");
	printf ("-h        prints this usage information and exits\n");
        printf ("\n");
        printf ("Supported character sets:\n");
	listsupportedcharsets ();
        printf ("    (* indicates compiled-in default).\n");
        printf ("\n");
        exit (0);
        break;
	
      case ':':             /* parameter without argument */
        fprintf (stderr, "gsmts0338: Option -%c requires an argument.\n",
                optopt);
        errflag++;
        break;

      case '?':               /* unrecognized option char */
        fprintf (stderr, "gsmts0338: Unrecognized option: -%c.\n", optopt);
        errflag++;
        break;

      default:
        fprintf (stderr, "gsmts0338: getopt() returned char. code 0%o ??\n", c);
    }                                       /* switch (c) */
  }                                       /* while (c...) */
  /*----------------------Check for required parameter(s) */
  /* either -d or -e */
  if (!errflag) {
    if (task == 0) {
      fprintf (stderr, "gsmts0338: You must provide at least one of -d or -e.\n");
      errflag++;
    }
  }                                      /* if (!errflag) */

  /*-------------------Check for conflicting parameter(s) */
  /* -d with -e */
  if ((strlen (message) > 0) && (strlen (string) > 0)) {
      fprintf (stderr, "gsmts0338: Conflicting parameters -d and -e.\n");
      errflag++;
  }
  /*-----------------------On error, display usage & exit */
  if (errflag) {
    fprintf (stderr, "\n==> gsmts0338 -h for usage details.\n");
    exit (2);
  }
  /* DEBUG ...............................................*/
  if (debug) {
    fprintf (stderr, "Debugging mode is turned ON.\n");
    fprintf (stderr, "Those are the current parameters :\n");
    fprintf (stderr, "string   = [%s]\n", string);
    fprintf (stderr, "message  = [%s]\n", message);
    if (tpudl != TP_UDL_UNKNOWN) {
      fprintf (stderr, "<tpudl>  = [%d]\n", tpudl);
    }
    else {
      fprintf (stderr, "<tpudl>  = [guess] (default)\n");
    }
    fprintf (stderr, "charset  = [%d]\n", default_alphabet);
  }

  /* Log current charset */
  csname = getcharsetname (default_alphabet);
  if (default_alphabet == 0) {
    /* GSM default alphabet is not meant to be used on the server side */
    fprintf (stderr, "WARNING: charset set to GSM default alphabet.\n");
    fprintf (stderr, "         This may not have the intended result !\n");
  }
  else {
    printf ("Using charset %s\n", csname);
  }
  if (csname) {
    free (csname);
  }
  
  /*---------------------------Start real processing loop */
  switch (task) {
    case GSMTS_ENCODE :
      if (tpudl != TP_UDL_UNKNOWN) {
        printf ("Warning: setting <tpudl> when encoding is meaningless - ignored\n");
      }
      printf ("Original message: [%s] (%d char.)\n", message, strlen (message));
      ascii7topacked8 (message, string, default_alphabet);
      printf ("Encoded string: [%s] (%d char., <tpudl> = %d)\n", string, 
             strlen (string), strlen (message));
      break;
      
    case GSMTS_DECODE :
      printf ("Encoded string: [%s] (%d char.)\n", string, strlen (string));
      packed8toascii7 (string, message, tpudl, default_alphabet);
      printf ("Decoded message: [%s] (%d char.)\n", message, strlen (message));
      break;

    default :
      fprintf (stderr, "gsmts0338: Unsupported task #%d, internal error.\n", task);
      exit (1);
      break;
      
  }                                      /* switch (task) */


  /*------------------------------------------Conclusions */
  /* free what's need to be */
  free (string);
  free (message);

  /*------------------------------------------End program */
  exit (0);
}

/*==========================================================
 * EOF : gsmts0338.c
 *===================*/
