/*==========================================================
 * Program : dbaccess.c                    Project : smslink
 * Authors : Philippe Andersson.
 * Date    : 31/08/06
 * Version : 0.10b
 * Comment : MySQL credentials handling routines based on
 *           /etc/gsmdbtab file.
 *
 * Modification History :
 * - 0.01a (21/03/01) : Initial release.
 * - 0.02a (28/12/01) : Added the got_db_creds() function.
 * - 0.03a (30/12/01) : Added the got_any_creds() and
 *   validate_db_creds() functions.
 * ++++ Switched to Beta ++++
 * - 0.04b (07/08/02) : Corrected parameter order in the
 *   mysql_real_connect() call.
 * - 0.05b (05/09/02) : Added db_list_queue() and db_queue_delete()
 *   functions.
 * - 0.06b (20/06/03) : Added support for QMAXRUNS & qruns field.
 * - 0.07b (03/09/03) : Corrected bug where the outbox.stored
 *   column would get reset to NOW() on update. Thanks to Andrew
 *   Goldschmidt (<andrew@rsaweb.co.za>) for pointing it out.
 * - 0.08b (08/04/04) : Added "purge" command support functions.
 * - 0.09b (17/07/06) : Changed "schema" table name to "dbschema"
 *   as "schema" is a reserved word in MySQL v5.0. Contributed
 *   by Tony S. Sykes (<Tony.Sykes@BCPSoftware.com>).
 * - 0.10b (31/08/06) : Request 'stored' in formatted form to
 *   preserve column width (in db_list_queue()).
 *========================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>                     /* for FreeBSB */
#include <dial/modems.h>
#include <dial/mdmerrno.h>
#ifdef WITH_MYSQL
#  include <mysql/mysql.h>	  /* for mysql client api */
#endif

#include "sms_serv.h"

/*========================================================*/
/* For debugging purposes only - comment out for normal compile */
/* #define INCL_DEBUG_CODE */

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/
extern int debug;                    /* debug level/flags */
extern char *db_dbhost;        /* MySQL access parameters */
extern char *db_dbname;
extern char *db_username;
extern char *db_password;
extern char *dbaccessfile;

/*========================================================*/
/**********               FUNCTIONS                ********/
/*========================================================*/
int got_db_creds () {

  int creds = 0;
  
  if (db_dbhost && strlen (db_dbhost)) {
    creds++;
  }
  if (db_dbname && strlen (db_dbname)) {
    creds++;
  }
  if (db_username && strlen (db_username)) {
    creds++;
  }
  if (db_password) {
    /* password may be blank, but not NULL */
    creds++;
  }
  
  return (creds == 4);
}                                      /* got_db_creds () */
/*========================================================*/
int got_any_creds () {

  int got_any = FALSE;
  
  if (db_dbhost || db_dbname || db_username || db_password) {
    got_any = TRUE;
  }
  
  return (got_any);
}                                     /* got_any_creds () */
/*========================================================*/
int load_db_creds () {
	
  FILE *dbcredsfile;
  char *buffer;
  char *ptr, *pptr, *err;
  char *dbh, *dbn, *dbu, *dbp;
  int counter = 0;                 /* valid entries count */
  int result;
	
  /*------------------------------------- Initializations */
  /* If we already got all creds through CLI, leave now */
  if (got_db_creds ()) {
    /* all provided through CLI - no point in going any further */
    return (SUCCESS);
  }
  
  /* Otherwise, check the DB access file */
  if (file_exist (dbaccessfile)) {
    /* Open input file */
    if ((dbcredsfile = fopen (dbaccessfile, "r")) == NULL) {
      syslog ((FACILITY | LOG_WARNING), "can't open DB credentials file %s.",
              dbaccessfile);
      return (FAILURE);
    }

    buffer = mdmalloc (BUFFSIZE);
    if (!buffer) {
      mdmerrno = -EMDMEM;
      return (FAILURE);
    }

    dbh = mdmalloc (MINIBUFF);
    if (!dbh) {
      mdmerrno = -EMDMEM;
      return (FAILURE);
    }

    dbn = mdmalloc (MINIBUFF);
    if (!dbn) {
      mdmerrno = -EMDMEM;
      return (FAILURE);
    }

    dbu = mdmalloc (MINIBUFF);
    if (!dbu) {
      mdmerrno = -EMDMEM;
      return (FAILURE);
    }

    dbp = mdmalloc (MINIBUFF);
    if (!dbp) {
      mdmerrno = -EMDMEM;
      return (FAILURE);
    }

    /*------------------------------------ File upload loop */
    while (fgets (buffer, BUFFSIZE, dbcredsfile) != NULL) {
      buffer[strlen (buffer) - 1] = '\0';

      /* ignore blank lines and comments */
      if (buffer[0] == '#' || buffer[0] == '\0' || buffer[0] == '\n')
	continue;

      /* parse the line we read - silently ignore invalid ones */
      if (debug & DEBUG_DBACCESS) {
	fprintf (stderr, "now parsing : [%s]\n", buffer);
      }
      /*..................................hostname (char *) */
      if ((ptr = strchr (buffer, ':')) == NULL)
	continue;

      *ptr = '\0';

      if (strlen (buffer) > MINIBUFF)
	strncpy (dbh, buffer, MINIBUFF);
      else if (strlen (buffer) == 0)
	strcpy (dbh, "localhost");
      else
	strcpy (dbh, buffer);

      ptr++;

      /*.............................database name (char *) */
      if ((pptr = strchr (ptr, ':')) == NULL)
	continue;

      *pptr = '\0';

      if (strlen (ptr) > MINIBUFF)
	strncpy (dbn, ptr, MINIBUFF);
      else if (strlen (ptr) == 0)
	continue;
      else
	strcpy (dbn, ptr);

      ptr = pptr + 1;

      /*..................................username (char *) */
      if ((pptr = strchr (ptr, ':')) == NULL)
	continue;

      *pptr = '\0';

      if (strlen (ptr) > MINIBUFF)
	strncpy (dbu, ptr, MINIBUFF);
      else if (strlen (ptr) == 0)
	continue;
      else
	strcpy (dbu, ptr);

      ptr = pptr + 1;

      /*..................................password (char *) */
      if (strlen (ptr) > MINIBUFF)
	strncpy (dbp, ptr, MINIBUFF);
      else if (strlen (ptr) == 0)
	strcpy (dbp, "");
      else
	strcpy (dbp, ptr);

      counter++;

    }                                    /* while (fgets... */
    /* Close input file */
    if (fclose (dbcredsfile) != 0) {
      syserr ("load_db_creds(): can't fclose() DB creds file");
    }

    if (counter) {
      if (debug & DEBUG_DBACCESS) {
	fprintf (stderr, "DB credentials found in file %s:\n", dbaccessfile);
	fprintf (stderr, "- dbh = <%s>\n", dbh);
	fprintf (stderr, "- dbn = <%s>\n", dbn);
	fprintf (stderr, "- dbu = <%s>\n", dbu);
	fprintf (stderr, "- dbp = <%s>\n", dbp);
      }

      /* transfer credentials to global variables, w/o overriding CL options */
      if (! db_dbhost) {
	db_dbhost = (char *) malloc ((strlen (dbh) + 1) * sizeof (char));
	if (! db_dbhost)
          syserr ("load_db_creds(): can't malloc() db_dbhost");
	strcpy (db_dbhost, dbh);
      }
      if (! db_dbname) {
	db_dbname = (char *) malloc ((strlen (dbn) + 1) * sizeof (char));
	if (! db_dbname)
          syserr ("load_db_creds(): can't malloc() db_dbname");
	strcpy (db_dbname, dbn);
      }
      if (! db_username) {
	db_username = (char *) malloc ((strlen (dbu) + 1) * sizeof (char));
	if (! db_username)
          syserr ("load_db_creds(): can't malloc() db_username");
	strcpy (db_username, dbu);
      }
      if (! db_password) {
	db_password = (char *) malloc ((strlen (dbp) + 1) * sizeof (char));
	if (! db_password)
          syserr ("load_db_creds(): can't malloc() db_password");
	strcpy (db_password, dbp);
      }
      syslog ((FACILITY | LOG_NOTICE), "successfully loaded DB access credentials.");
    }
    else {
      syslog ((FACILITY | LOG_NOTICE), "no valid DB credentials found in file.");
    }
    mdmfree (buffer);
    mdmfree (dbh);
    mdmfree (dbn);
    mdmfree (dbu);
    mdmfree (dbp);
  }                     /* if (file_exist (dbaccessfile)) */
  
  /* set defaults if still missing */
  if (! db_dbhost) {
    db_dbhost = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
    if (! db_dbhost)
      syserr ("load_db_creds(): can't malloc() db_dbhost");
    strcpy (db_dbhost, "localhost");
  }
  if (! db_password) {
    db_password = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
    if (! db_password)
      syserr ("load_db_creds(): can't malloc() db_password");
    db_password[0] = '\0';
  }
  
  /* final test */
  if (got_db_creds ()) {
    result = SUCCESS;
  }
  else {
    result = FAILURE;
  }

  return (result);
}                                     /* load_db_creds () */
/*========================================================*/
int validate_db_creds ()
{
  int retval = SUCCESS;
  int sv;
  char *errptr;
#ifdef WITH_MYSQL
  char query[1000];
  MYSQL mysql;
  MYSQL_RES *result;
  MYSQL_ROW row;

  query[0] = '\0';

  mysql_init (&mysql);

  if (mysql_real_connect (&mysql, db_dbhost, db_username, db_password, db_dbname, 0, NULL, 0)) {
    syslog ((FACILITY | LOG_INFO), "Connected to mysql as %s@%s", db_username, db_dbname);
    /* Connected ok -- now check schema version */
    sprintf (query, "SELECT schema_version FROM dbschema WHERE client_app=\"sms_serv\"");

    /* Submit the query */
    if (mysql_query (&mysql, query)) {
      syslog ((FACILITY | LOG_ERR), "Mysql query error code: %s", mysql_error(&mysql));
      retval = FAILURE;
    }
    else {
      /* query was successfull */
      result = mysql_store_result (&mysql);
      if ((row = mysql_fetch_row (result)) != NULL) {
	sv = strtol (row[0], &errptr, 10);
        if (*errptr != '\0') {
          syslog ((FACILITY | LOG_ERR), "validate_db_creds(): cannot convert schema version <%s> to int.",
	         row[0]);
	  fprintf (stderr, "validate_db_creds(): cannot convert schema version <%s> to int.\n", row[0]);
	  retval = FAILURE;
	}
	else {
	  if (sv == SMS_SERV_EXP_SCHEMA) {
            syslog ((FACILITY | LOG_INFO), "Database schema version: %d.", sv);
            retval = SUCCESS;
	  }
	  else {
            syslog ((FACILITY | LOG_ERR), "validate_db_creds(): unsupported schema version, expected %d, got %d.",
	           SMS_SERV_EXP_SCHEMA, sv);
	    fprintf (stderr, "validate_db_creds(): unsupported schema version, expected %d, got %d.\n",
	            SMS_SERV_EXP_SCHEMA, sv);
	    retval = FAILURE;
	  }             /* if (sv == SMS_SERV_EXP_SCHEMA) */
	}                         /* if (*errptr != '\0') */
      }                   /* if ((row = mysql_fetch_row ( */
    }                           /* if (mysql_real_query ( */
    mysql_close(&mysql);
  }
  else {
    syslog ((FACILITY | LOG_ERR), "Can't connect to mysql DB %s. Error code: %s",
           db_dbname, mysql_error(&mysql));
    retval = FAILURE;
  }
  
#endif
  return (retval);
}                                 /* validate_db_creds () */
/*========================================================*/
int sms_queue_db ()
/*
 * Inserts a new record in the outbox table of the smsstore
 * database. Returns the (DB-assigned) ID on success, or
 * -1 in case of failure. [The ID 0 will never be used by
 * MySQL].
 */
{
  int msg_id = -1;
  extern struct symbols symbols;
  extern char *db_dbhost;
  extern char *db_dbname;
  extern char *db_username;
  extern char *db_password;
#ifdef WITH_MYSQL
  MYSQL mysql;
  char query[1000], *end;
  char cmode[5];
  char priority[5];

  query[0] = '\0';
  
  mysql_init (&mysql);

  if (mysql_real_connect (&mysql, db_dbhost, db_username, db_password, db_dbname, 0, NULL, 0)) {
    syslog ((FACILITY | LOG_INFO), "Connected to mysql as %s@%s", db_username, db_dbname);
    /* create INSERT sql statement */
    /* Notes: - id=0, stored omitted => automatically set by MySQL */
    /*        - status=0 means "not sent" */
    /*        - qruns omitted => defaults to 0 */
    end = (char *) l_stpcpy (query, "INSERT INTO outbox SET id=0, smsc=\'");
    end += mysql_escape_string (end, symbols.smsc, strlen(symbols.smsc));
    end = (char *) l_stpcpy (end, "\', togsm=\'");
    end += mysql_escape_string (end, symbols.destgsm, strlen(symbols.destgsm));
    end = (char *) l_stpcpy (end, "\', userid=\'");
    end += mysql_escape_string (end, symbols.user, strlen(symbols.user));
    end = (char *) l_stpcpy (end, "\', mode=");
    sprintf (cmode, "%d", symbols.mode); 
    end = (char *) l_stpcpy (end, cmode);
    end = (char *) l_stpcpy (end, ", priority=");
    sprintf (priority, "%d", symbols.priority); 
    end = (char *) l_stpcpy (end, priority);
    end = (char *) l_stpcpy (end, ", status=0, msgtext=\'");
    end += mysql_escape_string (end, symbols.message, strlen(symbols.message));
    end = (char *) l_stpcpy (end, "\'");

    if (debug & DEBUG_DBACCESS) {
      fprintf (stderr, "Now submitting following query:\n[%s].\n", query);
      if (debug & DEBUG_HEXDUMP) {
	fprintf (stderr, "Hexdump of the query string:\n");
	dump_string (query);
      }
    }

    /* Submit the query */
    if (mysql_real_query (&mysql, query, (unsigned int)(end - query))) {
      syslog ((FACILITY | LOG_ERR), "Mysql query error code: %s", mysql_error(&mysql));
    }
    else {
      /* query was successfull */
      msg_id = mysql_insert_id (&mysql);
    }
  }
  else {
    syslog ((FACILITY | LOG_ERR), "Can't connect to mysql DB %s. Error code: %s",
           db_dbname, mysql_error(&mysql));
  }                          /* if (mysql_real_connect... */

  mysql_close(&mysql);
#endif
  return (msg_id);
}                                      /* sms_queue_db () */
/*========================================================*/
int db_list_queue (int csfd)
{
  char *buffer;
  int qlen = 0;
  extern char *db_dbhost;      /* MySQL access parameters */
  extern char *db_dbname;
  extern char *db_username;
  extern char *db_password;
#ifdef WITH_MYSQL
  MYSQL mysql;
  MYSQL_RES *result;
  MYSQL_ROW row;
  char query[1000];
#endif                               /* #ifdef WITH_MYSQL */
  
  buffer = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (! buffer) {
    syserr ("db_list_queue(): can't malloc() buffer");
  }
  buffer[0] = '\0';
  
  sprintf (buffer, "ID       | User         | Destination    | Stored         | Len. | QR | P \r\n"
                   "---------+--------------+----------------+----------------+------+----+---\r\n");
  tellsock (csfd, buffer);
#ifdef WITH_MYSQL
  mysql_init (&mysql);
  query[0] = '\0';

  if (mysql_real_connect (&mysql, db_dbhost, db_username, db_password, db_dbname, 0, NULL, 0)) {
    /* build query */
    l_stpcpy (query, "SELECT id, userid, togsm, date_format(stored,'%Y%m%d%H%i%s'), "
             "msgtext, qruns, priority "
             "FROM outbox WHERE status=0 ORDER BY priority ASC, id ASC");
    /* submit it */
    if (mysql_query (&mysql, query)) {
      syslog ((FACILITY | LOG_ERR), "cannot query outbox. MySQL error code: %s",
             mysql_error(&mysql));
      sprintf (buffer, "Error when querying outbox. MySQL error code: %s",
             mysql_error(&mysql));
      tellsock (csfd, buffer);
    }
    else {
      result = mysql_store_result(&mysql);
      while ((row = mysql_fetch_row (result)) != NULL) {
        sprintf (buffer, "%8s | %-12s | %-14s | %-14s | %4d | %2s | %1s\r\n",
	        row[0], row[1], row[2], row[3], strlen (row[4]), row[5], row[6]);
        tellsock (csfd, buffer);
	qlen++;
      }                                  /* while ((row = */
      mysql_free_result(result);
    }
    mysql_close (&mysql);
    if (qlen) {
      sprintf (buffer, "---------+--------------+----------------+----------------+------+----+---\r\n");
      tellsock (csfd, buffer);
    }
  }
  else {
    syslog ((FACILITY | LOG_ERR), "Can't connect to mysql. Error code: %s",
           mysql_error(&mysql));
    sprintf (buffer, "Can't connect to the DB. MySQL error code: %s",
           mysql_error(&mysql));
    tellsock (csfd, buffer);
  }
#endif                               /* #ifdef WITH_MYSQL */

  free (buffer);
  return (qlen);
}                                     /* db_list_queue () */
/*========================================================*/
int db_queue_delete (int reqid)
/*
 * Removes the requested message ID from the queue. Returns
 * 0 on success, -1 on failure.
 */
{
  extern char *db_dbhost;      /* MySQL access parameters */
  extern char *db_dbname;
  extern char *db_username;
  extern char *db_password;
  int retval = 0;
#ifdef WITH_MYSQL
  MYSQL mysql;
  MYSQL_RES *result;
  MYSQL_ROW row;
  char query[1000];
#endif                               /* #ifdef WITH_MYSQL */

#ifdef WITH_MYSQL
  mysql_init (&mysql);
  query[0] = '\0';

  if (mysql_real_connect (&mysql, db_dbhost, db_username, db_password, db_dbname, 0, NULL, 0)) {
    /* build query */
    sprintf (query, "DELETE FROM outbox WHERE id=%d", reqid);
    if (debug & DEBUG_DBACCESS) {
      fprintf (stderr, "Now submitting following query:\n[%s].\n", query);
      if (debug & DEBUG_HEXDUMP) {
	fprintf (stderr, "Hexdump of the query string:\n");
	dump_string (query);
      }
    }
    /* submit it */
    if (mysql_query (&mysql, query)) {
      syslog ((FACILITY | LOG_ERR), "cannot delete requested item. MySQL error code: %s",
             mysql_error(&mysql));
      retval = -1;
    }
    mysql_close (&mysql);
  }
  else {
    syslog ((FACILITY | LOG_ERR), "Can't connect to mysql. Error code: %s",
           mysql_error(&mysql));
    retval = -1;
  }
#endif                               /* #ifdef WITH_MYSQL */
  return (retval);
}                                   /* db_queue_delete () */
/*========================================================*/
int db_purge_inbox ()
/*
 * Removes all "processed" entries from the inbox table. Returns
 * the number of deleted entries on success, -1 on failure.
 */
{
  extern char *db_dbhost;      /* MySQL access parameters */
  extern char *db_dbname;
  extern char *db_username;
  extern char *db_password;
  int retval = 0;
#ifdef WITH_MYSQL
  MYSQL mysql;
  MYSQL_RES *result;
  MYSQL_ROW row;
  char query[1000];
#endif                               /* #ifdef WITH_MYSQL */

#ifdef WITH_MYSQL
  mysql_init (&mysql);
  query[0] = '\0';

  if (mysql_real_connect (&mysql, db_dbhost, db_username, db_password, db_dbname, 0, NULL, 0)) {
    /* build query */
    sprintf (query, "DELETE FROM inbox WHERE processed='y'");
    if (debug & DEBUG_DBACCESS) {
      fprintf (stderr, "Now submitting following query:\n[%s].\n", query);
      if (debug & DEBUG_HEXDUMP) {
	fprintf (stderr, "Hexdump of the query string:\n");
	dump_string (query);
      }
    }
    /* submit it */
    if (mysql_query (&mysql, query)) {
      syslog ((FACILITY | LOG_ERR), "cannot delete requested item(s). MySQL error code: %s",
             mysql_error(&mysql));
      retval = -1;
    }
    else {
      /* success -- get the number of impacted rows */
      retval = (int) mysql_affected_rows (&mysql);
    }
    mysql_close (&mysql);
  }
  else {
    syslog ((FACILITY | LOG_ERR), "Can't connect to mysql. Error code: %s",
           mysql_error(&mysql));
    retval = -1;
  }
#endif                               /* #ifdef WITH_MYSQL */
  return (retval);
}                                    /* db_purge_inbox () */
/*========================================================*/
int db_purge_outbox (int mode)
/*
 * Removes all "sent" entries from the outbox table. Returns
 * the number of deleted entries on success, -1 on failure.
 */
{
  extern char *db_dbhost;      /* MySQL access parameters */
  extern char *db_dbname;
  extern char *db_username;
  extern char *db_password;
  int retval = 0;
#ifdef WITH_MYSQL
  MYSQL mysql;
  MYSQL_RES *result;
  MYSQL_ROW row;
  char query[1000];
#endif                               /* #ifdef WITH_MYSQL */

#ifdef WITH_MYSQL
  mysql_init (&mysql);
  query[0] = '\0';

  if (mysql_real_connect (&mysql, db_dbhost, db_username, db_password, db_dbname, 0, NULL, 0)) {
    /* build query */
    switch (mode) {
      case DB_OUT_PURGE_SENT: {
        sprintf (query, "DELETE FROM outbox WHERE status=1");
	break;
      }

      case DB_OUT_PURGE_FAIL: {
        sprintf (query, "DELETE FROM outbox WHERE status=2");
	break;
      }

      case DB_OUT_PURGE_ALL: {
        sprintf (query, "DELETE FROM outbox WHERE status=1 OR status=2");
	break;
      }
      
      default: {
	fprintf (stderr, "db_purge_outbox(): unknown mode (%d)\n", mode);
        syserr ("db_purge_outbox(): internal error");
	break;
      }

    }                                    /* switch (mode) */
    if (debug & DEBUG_DBACCESS) {
      fprintf (stderr, "Now submitting following query:\n[%s].\n", query);
      if (debug & DEBUG_HEXDUMP) {
	fprintf (stderr, "Hexdump of the query string:\n");
	dump_string (query);
      }
    }
    /* submit it */
    if (mysql_query (&mysql, query)) {
      syslog ((FACILITY | LOG_ERR), "cannot delete requested item(s). MySQL error code: %s",
             mysql_error(&mysql));
      retval = -1;
    }
    else {
      /* success -- get the number of impacted rows */
      retval = (int) mysql_affected_rows (&mysql);
    }
    mysql_close (&mysql);
  }
  else {
    syslog ((FACILITY | LOG_ERR), "Can't connect to mysql. Error code: %s",
           mysql_error(&mysql));
    retval = -1;
  }
#endif                               /* #ifdef WITH_MYSQL */
  return (retval);
}                                   /* db_purge_outbox () */
/*========================================================*/
int outbox_counter_op (char *id, int op)
/*
 * Gets, resets or increments a counter located in the "qruns"
 * field of the outbox table.
 * Returns the current counter value when successfull, -1 otherwise.
 */
{
  extern char *db_dbhost;      /* MySQL access parameters */
  extern char *db_dbname;
  extern char *db_username;
  extern char *db_password;
  int retval = 0;
  char *errptr;
#ifdef WITH_MYSQL
  MYSQL mysql;
  MYSQL_RES *result;
  MYSQL_ROW row;
  char query[1000];
#endif                               /* #ifdef WITH_MYSQL */

#ifdef WITH_MYSQL
  mysql_init (&mysql);
  query[0] = '\0';

  if (mysql_real_connect (&mysql, db_dbhost, db_username, db_password, db_dbname, 0, NULL, 0)) {

    switch (op) {
      case SMS_FC_RST: {
        sprintf (query, "UPDATE outbox SET qruns=0, stored=stored WHERE id=%s", id);
	if (mysql_query (&mysql, query)) {
          syslog ((FACILITY | LOG_ERR), "outbox_counter_op(): cannot reset counter on message %s. MySQL error code: %s",
	         id, mysql_error (&mysql));
	  fprintf (stderr, "outbox_counter_op(): cannot reset counter on message %s.\n", id);
	}
	break;
      }

      case SMS_FC_INC: {
        /* first, get the current value */
        sprintf (query, "SELECT qruns FROM outbox WHERE id=%s", id);
	if (mysql_query (&mysql, query)) {
          syslog ((FACILITY | LOG_ERR), "outbox_counter_op(): cannot get counter on message %s. MySQL error code: %s",
	         id, mysql_error (&mysql));
	  fprintf (stderr, "outbox_counter_op(): cannot get counter on message %s.\n", id);
	}
        result = mysql_store_result (&mysql);
	if ((row = mysql_fetch_row (result)) != NULL) {
	  retval = strtol (row[0], &errptr, 10);
          if (*errptr != '\0') {
            syslog ((FACILITY | LOG_ERR), "outbox_counter_op(): cannot convert counter <%s> on message %s to int.",
	           row[0], id);
	    fprintf (stderr, "outbox_counter_op(): cannot convert counter <%s> on message %s to int.\n", row[0],
	            id);
	    retval = -1;
	  }
	  else {
	    /* now increment and update */
	    retval++;
            sprintf (query, "UPDATE outbox SET qruns=%d, stored=stored WHERE id=%s", retval, id);
	    if (mysql_query (&mysql, query)) {
              syslog ((FACILITY | LOG_ERR), "outbox_counter_op(): cannot set counter on message %s. MySQL error code: %s",
	             id, mysql_error (&mysql));
	      fprintf (stderr, "outbox_counter_op(): cannot set counter on message %s.\n", id);
	    }
	  }
	}
	else {
          syslog ((FACILITY | LOG_ERR), "outbox_counter_op(): cannot get counter on message %s. MySQL error code: %s",
	         id, mysql_error (&mysql));
	  fprintf (stderr, "outbox_counter_op(): cannot get counter on message %s.\n", id);
	}
	break;
      }

      case SMS_FC_GET: {
        sprintf (query, "SELECT qruns FROM outbox WHERE id=%s", id);
	if (mysql_query (&mysql, query)) {
          syslog ((FACILITY | LOG_ERR), "outbox_counter_op(): cannot get counter on message %s. MySQL error code: %s",
	         id, mysql_error (&mysql));
	  fprintf (stderr, "outbox_counter_op(): cannot get counter on message %s.\n", id);
	}
        result = mysql_store_result (&mysql);
	if ((row = mysql_fetch_row (result)) != NULL) {
	  retval = strtol (row[0], &errptr, 10);
          if (*errptr != '\0') {
            syslog ((FACILITY | LOG_ERR), "outbox_counter_op(): cannot convert counter <%s> on message %s to int.",
	           row[0], id);
	    fprintf (stderr, "outbox_counter_op(): cannot convert counter <%s> on message %s to int.\n", row[0],
	            id);
	    retval = -1;
	  }
	}
	else {
          syslog ((FACILITY | LOG_ERR), "outbox_counter_op(): cannot get counter on message %s. MySQL error code: %s",
	         id, mysql_error (&mysql));
	  fprintf (stderr, "outbox_counter_op(): cannot get counter on message %s.\n", id);
	}
	break;
      }

      default: {
	fprintf (stderr, "sms_serv: outbox_counter_op(): unsupported op <%d>\n", op);
        retval = -1;
	break;
      }
    }                                        /* switch (op) */
    mysql_close (&mysql);
  }
  else {
    syslog ((FACILITY | LOG_ERR), "Can't connect to mysql. Error code: %s",
           mysql_error(&mysql));
    retval = -1;
  }                         /* if (mysql_real_connect (.. */
#endif                               /* #ifdef WITH_MYSQL */

  return (retval);
}                                 /* outbox_counter_op () */
/*========================================================*/

/*==========================================================
 * EOF : dbaccess.c
 *===================*/
