/*==========================================================
 * Program : dd.c                          Project : smslink
 * Author  : Andrew Worsley <epaanwo@asac.ericsson.se>.
 * Date    : 25/11/03
 * Version : 0.13b
 * Notice  : (c) Andrew Worsley.
 * Comment : Dedicated daemon code.
 *
 * Modification History :
 * - 0.01b (18/02/01) : Initial release.
 * - 0.02b (19/02/01) : Moved all #define to sms_serv.h.
 *   Cosmetics.
 * - 0.03b (05/03/01) : Replaced control of all debug info by
 *   global flags.
 * - 0.04b (06/03/01) : Modified the way a new PIN is selected if
 *   we have to provide the PUK: now sets PIN to the one we have in
 *   conf in order to avoid the need to store the new one (thanks
 *   to Shay Kalderon <shay_kalderon@creoscitex.com> for the
 *   hint). Eventually implemented Andrew's hack to make sure
 *   all messages are always removed from the SIM card (created
 *   a device-level flag 'r4' to control it).
 * - 0.05b (02/04/01) : Modified calls to get_gsm_answer() to
 *   include the struct gsms_def field.
 * - 0.06b (04/05/01) : Improved handling of +CREG-returned
 *   registration status in dd_clear_msgs() & dd_send_sms().
 * - 0.07b (08/05/01) : Added a pause in dd_init() after PIN
 *   processing when using "fast" mode to try and solve the
 *   "+CMS ERROR: 515" issue. Modified dd_init() to also use
 *   the new get_gsm_resp() calls. Cosmetics.
 * - 0.08b (07/10/01) : Replaced calls to blopen_mdm_line()
 *   by calls to lopen_mdm_line(). Advantage being that this
 *   version uses the max. supported speed by the device by
 *   default (the one defined in /etc/modems).
 * - 0.09b (17/05/01) : Modified dd_spawn(), dd_init() and
 *   dd_input() to update the stats counter fields in struct
 *   gsms_def.
 * - 0.10b (05/01/01) : Improved error handling around the
 *   call to set_cpms(). Replaced the AT+CMEE dialog with a
 *   call to set_cmee(). Idem for AT+CPIN, AT+CMGF, AT+CSDH
 *   AT+CREG & AT+CSCA.
 * - 0.11b (23/01/02) : Implemented support for default GSM
 *   alphabet. Replaced static PIN post-processing sleep time
 *   by device-level param. (PINpause).
 * - 0.12b (07/11/02) : Re-engineering of "dd" mode to allow
 *   for parallel use of multiple devices when all of them are
 *   in dd mode.
 * - 0.13b (25/11/03) : Added  test on NO_CREG_CHECK flag to
 *   support Motorola A008 device. Contributed by Nicki de Wet
 *   <nickidw@mighty.co.za>.
 *========================================================*/
#include <unistd.h>
#include <fcntl.h>                        /* for fcntl () */
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>               /* for strcpy & friends */
#include <termios.h>         /* for baudrates definitions */
#include <dial/modems.h>           /* requires 'libmodem' */
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>

#include "sms_serv.h"
#include "gsmdev.h"
#include "pdu.h"

/*========================================================*/
/* For debugging purposes only - comment out for normal compile */
/* #define INCL_DEBUG_CODE */

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/
extern int debug;                    /* debug level/flags */
extern int context;              /* global server context */
extern struct symbols symbols;       /* msg-specific data */
extern int csfd;                      /* client socket FD */
extern int use_fast;        /* enable fast modem response */
extern int default_alphabet;
extern int shmem_sem;


/*========================================================*/
/**********               FUNCTIONS                ********/
/*========================================================*/
void dd_spawn (struct gsms_def *dev)
/*
 * fork and start executing the dedicated daemon code on this device.
 * basically will listen for input from the modem line or commands from the
 * command port and immeadiately call the appropriate routine to process
 * the data.
 */
{
  int fd;
  fd_set rfds;
  int retval;
  struct sockaddr_un s_addr;     /* server socket address */
  struct sockaddr_un c_addr;     /* client socket address */
  int ssfd, csfd;                   /* socket descriptors */
  int c_addr_len;
  int nfd = 0;
  int sockflags;

  /*-------------------------First, let's become a daemon */
  int pid = fork ();
  if (pid == -1)
    syserr ("dd_spawn(): can't fork dedicated daemon");
  else if (pid != 0) {
    return;                             /* parent returns */
  }

  /* set context */
  context = CONTEXT_DDMN;
  if (debug & DEBUG_PRINTF) {
    printf ("PID <%d>, context set to <%d>\n", getpid (), context);
  }
  
  /* open connection with the syslog daemon - announce */
  closelog ();
  openlog ("sms_dd", (LOG_CONS | LOG_PID), FACILITY);
  syslog ((FACILITY | LOG_INFO), "dedicated sms daemon starting on device %s",
         dev->device);

  /* Update struct gsms_def */
  if (sem_wait (shmem_sem) == -1)
    syserr ("dd_spawn(): failed to wait on shared mem. semaphore");
  
  dev->owner = getpid ();
  dev->dd_busy = TRUE;

  if (sem_signal (shmem_sem) == -1)
    syserr ("dd_spawn(): can't signal shared mem. semaphore");

  /* set up the modem line */
  if ((fd = dd_init (dev)) < 0) {
    syslog ((FACILITY | LOG_ERR), "Unable to initialise modem (%d)", fd);
    exit (1);
  }
  if (debug & DEBUG_PRINTF) {
    printf ("Initialised modem - fd=%d sms\n", fd);
  }

  /* Flag device as idle again */
  if (sem_wait (shmem_sem) == -1)
    syserr ("dd_spawn(): failed to wait on shared mem. semaphore");
  
  dev->dd_busy = FALSE;

  if (sem_signal (shmem_sem) == -1)
    syserr ("dd_spawn(): can't signal shared mem. semaphore");

  /* create the Unix domain socket */
  /* see man 4 unix - on linux for doco */
  s_addr.sun_family = AF_LOCAL;
  strncpy (s_addr.sun_path, "/var/spool/smslink/", UNIX_PATH_MAX);
  strncat (s_addr.sun_path, dev->device, UNIX_PATH_MAX);

  /* let's create the socket */
  unlink (s_addr.sun_path); /* remove any old - existing one */
  if ((ssfd = socket (AF_LOCAL, SOCK_STREAM, 0)) == -1)
    syserr ("dd_spawn(): can't create socket(AF_LOCAL, SOCK_STREAM, 0)");

  /* now bind the socket to the server address */
  if (bind (ssfd, (struct sockaddr *) &s_addr, sizeof (s_addr)) == -1) {
    syslog ((FACILITY | LOG_ERR), "can't bind socket - %s", s_addr.sun_path);
    syserr ("dd_spawn(): can't bind socket");
  }

  /* create the client queue on the socket */
  if (listen (ssfd, MAXCLIENTCONN) == -1) {
    syslog ((FACILITY | LOG_ERR), "can't listen to that many connections (%d)", MAXCLIENTCONN);
    syserr ("dd_spawn(): can't listen to that many connections");
  }

  syslog ((FACILITY | LOG_INFO), "unix domain socket %s now ready.",
         s_addr.sun_path);

  /* set the flags to make the socket "non-blocking" */
  if ((sockflags = fcntl (ssfd, F_GETFL, 0)) == -1)
    syserr ("dd_spawn(): can't get unix domain socket descriptor attributes");
  if (fcntl (ssfd, F_SETFL, (O_NONBLOCK | sockflags)) == -1)
    syserr ("dd_spawn(): can't set socket descriptor attributes on unix domain socket");

  if (fd > ssfd)
    nfd = fd + 1;
  else
    nfd = ssfd + 1;

  for (;;) {
    FD_ZERO(&rfds);
    FD_SET(fd, &rfds);
    FD_SET(ssfd, &rfds);

    retval = select (nfd, &rfds, NULL, NULL, (struct timeval *)0);

    if (retval < 0) {
      syslog ((FACILITY | LOG_ERR), "select() failed: %m");
      exit (1);
    }

    if (retval == 0) {
      syslog ((FACILITY | LOG_ERR), "select() timedout? trying again...");
      continue;
    }
    /* input from modem */
    if (FD_ISSET (fd, &rfds)) { 
      dd_input (dev, fd);         /* return value ignored */
    }

    /* input from Unix domain socket */
    if (FD_ISSET(ssfd, &rfds)) { 
      c_addr_len = sizeof (c_addr);
      if ((csfd = accept (ssfd, (struct sockaddr *) &c_addr, &c_addr_len))
	  == -1) {
	/* no connection - check the reason of the "error" */
	syserr ("dd_spawn(): accept failed on unix domain socket!");
      }
      else {
	FILE *fp = fdopen (csfd, "r+");
	char *p, *s;
	char buf[BIGGERBUFF];

	syslog ((FACILITY | LOG_INFO), "accepted Unix domain socket connection");

	/* allocate space for the symbols */
	symbols.smsc = (char *) malloc ((MAXPHNUMLEN + 1) * sizeof (char));
	symbols.destgsm = (char *) malloc ((MAXPHNUMLEN + 1) * sizeof (char));
	symbols.message = (char *) malloc ((MAXMSGLEN + 1) * sizeof (char));
	symbols.pdu = NULL; /* allocated only as needed */
	symbols.user = (char *) malloc ((MAXUIDLEN + 1) * sizeof (char));

	/* set those symbols to their default values */
	strcpy (symbols.smsc, DEFAULTSMSC);
	symbols.destgsm[0] = '\0';
	symbols.message[0] = '\0';
	symbols.user[0] = '\0';
	symbols.mode = SMS_MODE_DEFAULT;

	/* parse input for parameters for the request */
	/* Note: ad-hoc protocol on the unix-domain socket. */
	/* Sender part is in dd_send_wrapper(). */
	while ((s = fgets (buf, BIGGERBUFF, fp)) != 0) {
          if (debug & DEBUG_PRINTF) {
            fprintf (stderr, "Read '%s'\n", buf);
          }
	  if (p = strchr (buf, '=')) {
	    /* we have a variable assignment */
	    *p++ = '\0';
	    if (s = strchr (p, '\n'))
	      *s = '\0';
	    if (strcasecmp (buf, "smsc") == 0)
	      strcpy (symbols.smsc, p);
	    else if (strcasecmp (buf, "dest") == 0)
	      strcpy (symbols.destgsm, p);
	    else if (strcasecmp (buf, "user") == 0)
	      strcpy (symbols.user, p);
	    else if (strcasecmp (buf, "message") == 0)
	      strcpy (symbols.message, p);
	    else if (strcasecmp (buf, "mode") == 0)
	      symbols.mode = atoi (p);
	    else {
	      fprintf (fp, "Unknown parameter ignored: %s\n", buf);
	      fflush (fp);
	    }
	    continue;
	  }                 /* if (p = strchr (buf, '=')) */
          if (strncasecmp (buf, "send", 4) == 0) {
	    break;
	  }
	  /* send complaint i - ignoring unknown command */
	  fprintf (fp, "Unknown command ignored: %s", buf);
	  fflush (fp);
	}                                        /* while */
	if (s) {         /* only if connection not closed */
	  /* Flag device busy */
	  if (sem_wait (shmem_sem) == -1)
	    syserr ("dd_spawn(): failed to wait on shared mem. semaphore");

	  dev->dd_busy = TRUE;

          if (sem_signal (shmem_sem) == -1)
            syserr ("dd_spawn(): can't signal shared mem. semaphore");

	  fflush (fp); /* clear any pending output for dd_send_sms */
	  if (dd_send_sms (dev, fd, &symbols, csfd) < 0) {
	    fprintf (fp, "Transmission failed!\n");
	    /* Update counters */
	    if (sem_wait (shmem_sem) == -1)
	      syserr ("dd_spawn(): failed to wait on shared mem. semaphore");
	    dev->out_fail++;
            if (sem_signal (shmem_sem) == -1)
              syserr ("dd_spawn(): can't signal shared mem. semaphore");
	  }
	  else {
	    fprintf (fp, "Transmission Successful\n");
	    /* Update counters */
	    if (sem_wait (shmem_sem) == -1)
	      syserr ("dd_spawn(): failed to wait on shared mem. semaphore");
	    dev->out_ok++;
            if (sem_signal (shmem_sem) == -1)
              syserr ("dd_spawn(): can't signal shared mem. semaphore");
	  }
	  fclose (fp);
	  if (dd_notify (dev, fd) < 0) {
	    cleanup_mdm ("Unable to switch on notifications", fd);
	    closelog ();
	    exit (1);
	  }
	  /* Flag device idle again */
	  if (sem_wait (shmem_sem) == -1)
	    syserr ("dd_spawn(): failed to wait on shared mem. semaphore");

	  dev->dd_busy = FALSE;

          if (sem_signal (shmem_sem) == -1)
            syserr ("dd_spawn(): can't signal shared mem. semaphore");

	}                                       /* if (s) */

	/* free the symbols struct members */
	if (symbols.smsc) {
	  free (symbols.smsc);
	}
	if (symbols.destgsm) {
	  free (symbols.destgsm);
	}
	if (symbols.message) {
	  free (symbols.message);
	}
	if (symbols.pdu) {
	  free (symbols.pdu);
	}
	if (symbols.user) {
	  free (symbols.user);
	}
      }                   /* if ((csfd = accept (ssfd,... */
    }                       /* if (FD_ISSET(ssfd, &rfds)) */

    /* process other file descriptors if any */
  }                                           /* for (;;) */

  /* fixup - free up the device */
  syslog ((FACILITY | LOG_INFO), "closing down");
  cleanup_mdm ((char *)0, fd);
  closelog ();
  exit (0);
}                                          /* dd_spawn () */
/*========================================================*/
int dd_init (struct gsms_def *gsm)
/*
 * Initialise the modem line for the dedicated daemon - want new message
 * notifications enabled.
 * Returns the file descriptor of the modem line or negative number if it
 * fails.
 */
{
  int fd = -1;
  int pin_just_in;
  int nmsgin;

  /*--------------------------------------Initialisations */
  pin_just_in = FALSE; /* set to true if PIN needed be provided */

  /* open modem line */
  fd = lopen_mdm_line (gsm->device);
  if (fd < 0) {
    syslog ((FACILITY | LOG_ERR), "call to lopen_mdm_line() failed.");
    mdmperror ("sms_dd: lopen_mdm_line() failed");
    return (-1);
  }

  use_fast = (gsm->flags & FAST_RESPONSE);

  /*------------set GSM to "verbose" error reporting mode */
  if (set_cmee (gsm, fd, 0) == -1) {
    return (-1);
  }
  
  /*---------------------------then, check for SIM status */
  if (checkset_cpin (gsm, fd, 0, &pin_just_in) == -1) {
    return (-1);
  }
  
  /*--------------------------------------sleep if needed */
  /* There seems to be a timing issue in "fast" mode when
   * the AT+CNMI command is issued too fast after the last
   * AT+CPIN? and we get a "+CMS ERROR: 515" (= module
   * not ready or busy processing command). Try sleeping a
   * few secs to solve that. */
  if (use_fast && pin_just_in) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "DD: (pause %d secs. for PIN code post-processing)\n",
              gsm->PINpause);
    }
    sleep (gsm->PINpause);
  }
  
  /* clear messages and enable new message notifications */
  if ((nmsgin = dd_clear_msgs (gsm, fd)) < 0) {
    /* increment the error counter */
    if (sem_wait (shmem_sem) == -1)
      syserr ("dd_init(): failed to wait on shared mem. semaphore");
    gsm->in_fail++;
    if (sem_signal (shmem_sem) == -1)
      syserr ("dd_init(): can't signal shared mem. semaphore");
    /* cleanup */
    mdm_unlock (mdmopendevice);
    hangup (fd);
    syslog ((FACILITY | LOG_ERR), "Unable to switch on notifications.");
    mdmperror ("Unable to switch on notifications");
    return (-1);
  }
  else {
    /* update the "in" counter */
    if (sem_wait (shmem_sem) == -1)
      syserr ("dd_init(): failed to wait on shared mem. semaphore");
    gsm->in_ok += nmsgin;
    if (sem_signal (shmem_sem) == -1)
      syserr ("dd_init(): can't signal shared mem. semaphore");
  }

  return (fd);
}                                           /* dd_init () */
/*========================================================*/
int dd_notify (struct gsms_def *gsm, int fd)
/*
 * switch on notifications from the modem
 */
{
  char* scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));

  if (!scratch)
    syserr ("dd_notify(): can't allocate scratch space");
  memset (scratch, 0, (BIGBUFF + 1));
  
  use_fast = (gsm->flags & FAST_RESPONSE);

  /*---------------set "notify for incoming SMS messages" */
  if (set_cnmi_on (gsm, fd, 0) == -1) {
    return (-1);
  }

  /* fixup - should check for any stored messages and return +ve number so we
   * can pick up messages that arrived while the notifications where switched
   * off
   */

  free (scratch);
  return (0);
}                                         /* dd_notify () */
/*========================================================*/
int dd_input (struct gsms_def *dev, int fd)
/*
 * read in some spontanous input data from the modem - if it's a new message
 * notification process it otherwise just consume it and return.
 */
{
  char *scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  char *p;
  int nmsgin;
  size_t len;
  cpl_list cpl;

  cpl_list_init (&cpl);
  cpl_list_insert (&cpl, dev->capmatrix.line_sep);
  cpl_list_insert (&cpl, "\n");
  
  if (!scratch)
    syserr ("dd_input(): can't allocate scratch space");
  memset (scratch, 0, (BIGBUFF + 1));

  use_fast = (dev->flags & FAST_RESPONSE);

  /* grab first line */
  if (!get_gsm_resp (fd, scratch, BIGBUFF, 2, cpl)) {
      cleanup_mdm ("unable to read input data from modem", fd);
      free_cpl_list (&cpl);
      return (-1);
  }
  /* if (!get_gsm_sleep (fd, scratch, BIGBUFF, 2)) {
   * if (!get_gsm_resp (fd, scratch, BIGBUFF, 2, "\r\n")) {
    not reliable because new message notification has 2 \r\n pairs!
   */
  p = strstr (scratch, dev->capmatrix.line_sep);
  len = strlen (scratch);
  if (!strstr (p + 2, dev->capmatrix.line_sep) &&
      !get_gsm_resp (fd, scratch + len, BIGBUFF - len, 2, cpl)) {
    cleanup_mdm ("unable to read 2nd new line data from new notication", fd);
    free_cpl_list (&cpl);
    return (-1);
  }
  /* expect new message notifications with the form: +CMTI: "ME",1
   *  where 1 is the message index and "ME" is the memory area it is in
   */
  if (strstr (scratch, "+CMTI:")) {
    syslog ((FACILITY | LOG_INFO), "new message : %s", scratch);
    free_cpl_list (&cpl);

    /* Flag device busy */
    if (sem_wait (shmem_sem) == -1)
      syserr ("dd_input(): failed to wait on shared mem. semaphore");

    dev->dd_busy = TRUE;

    if (sem_signal (shmem_sem) == -1)
      syserr ("dd_input(): can't signal shared mem. semaphore");

    if ((nmsgin = dd_clear_msgs (dev, fd)) < 0) {
      /* increment the error counter */
      if (sem_wait (shmem_sem) == -1)
        syserr ("dd_input(): failed to wait on shared mem. semaphore");
      dev->in_fail++;
      if (sem_signal (shmem_sem) == -1)
        syserr ("dd_input(): can't signal shared mem. semaphore");
    }
    else {
      /* update the "in" counter */
      if (sem_wait (shmem_sem) == -1)
        syserr ("dd_input(): failed to wait on shared mem. semaphore");
      dev->in_ok += nmsgin;
      if (sem_signal (shmem_sem) == -1)
        syserr ("dd_input(): can't signal shared mem. semaphore");
    }
    /* Flag device idle again */
    if (sem_wait (shmem_sem) == -1)
      syserr ("dd_input(): failed to wait on shared mem. semaphore");

    dev->dd_busy = FALSE;

    if (sem_signal (shmem_sem) == -1)
      syserr ("dd_input(): can't signal shared mem. semaphore");

    return (nmsgin);
  }
  syslog ((FACILITY | LOG_ERR), "unexpected input from modem ignored '%s'",
         scratch);

  free_cpl_list (&cpl);
  return (-1);
}                                          /* dd_input () */
/*========================================================*/
int dd_clear_msgs (struct gsms_def *gsm, int fd)
/*
 * clear out all pending SMS messages 
 */
{
  int nmsgin = 0;
  char *p1;
  char *p2;
  char *cmsgid;
  int nread;
  struct mbox_item *message;
  mbox_list mailbox;
  int i;
  char* scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));


  if (!scratch)
    syserr ("dd_clear_msgs(): can't allocate scratch space");
  memset (scratch, 0, (BIGBUFF + 1));
  mbox_list_init (&mailbox);

  use_fast = (gsm->flags & FAST_RESPONSE);

  /*------------set "no notify for incoming SMS messages" */
  if (set_cnmi_off (gsm, fd, 0) == -1) {
    return (-1);
  }
  
  /*-----------set message format to device-level default */
  if (checkset_cmgf (gsm, fd, 0, gsm->defmode) == -1) {
    return (-1);
  }
  
  /*-------------------------set "full SMS header detail" */
  /* Beware: only meaningfull (and supported) in text mode AND */
  /* even so only on specific models ! (specifically not the */
  /* PCFF900). */
  if ((gsm->defmode == SMS_MODE_TEXT) &&
      (gsm->capmatrix.capflags & HAS_CSDH_1)) {
    if (set_csdh (gsm, fd, 0) == -1) {
      return (-1);
    }
  }                 /* if (gsm->defmode == SMS_MODE_TEXT) */
  
  /*----------------Check stored SCA against default SMSC */
  if (checkset_csca (gsm, fd, 0, gsm->defsca) == -1) {
    return (-1);
  }
  
  /*---------------------Am I registered on the network ? */
  if (!(gsm->capmatrix.capflags & NO_CREG_CHECK))
  {
    if (check_creg (gsm, fd, 0) == -1) {
      return (-1);
    }
  }
  
  /*------------------------------------Set message store */
  /* make sure the message store is accessible */
  if (gsm->capmatrix.capflags & SET_MSG_STORE) {
    if (set_cpms (gsm, fd, 0) == -1) {
      return (-1);
    }
  }

  /*-----------------------Now loop and read incoming SMS */
  if (gsm->defmode == SMS_MODE_TEXT) {
    if (gsm->flags & SMS_READ_ALL) {
      /* hack to clear messages in SIM memory */
      sprintf (scratch, "AT+CMGL=\"ALL\"\r");
    }
    else {
      sprintf (scratch, "AT+CMGL=\"REC UNREAD\"\r");
    }
  }
  else {
    /* SMS_MODE_PDU */
    if (gsm->flags & SMS_READ_ALL) {
      /* hack to clear messages in SIM memory */
      sprintf (scratch, "AT+CMGL=4\r");
    }
    else {
      sprintf (scratch, "AT+CMGL=0\r");
    }
  }                 /* if (gsm->defmode == SMS_MODE_TEXT) */
  tell_gsm (fd, scratch);
  memset (scratch, 0, (BIGBUFF + 1));
  if (get_gsm_answer (fd, scratch, BIGBUFF, 1, gsm)) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "%s\n", scratch);
    }
    /*....................look for "+CMGL:" in GSM answer */
    p1 = scratch;
    while (p1 && ((p1 = strstr (p1, "+CMGL:")) != NULL)) {
      /* got new message - isolate it from the batch */
      p2 = (p1 + 6);
      if ((p2 = strstr (p2, "+CMGL:")) != NULL) {
        /* there's another one behind */
	p1[((p2 - p1) - 1)] = '\0';
      }
      if (gsm->defmode == SMS_MODE_TEXT) {
        message = mbparse (p1, gsm->capmatrix);
      }
      else {
        /* SMS_MODE_PDU */
        message = mbparsePDU (p1, gsm->capmatrix);
      }             /* if (gsm->defmode == SMS_MODE_TEXT) */
      if (message == NULL) {
	free (scratch);
	syslog ((FACILITY | LOG_ERR), "mbparse() failed - internal error.");
	return (-1);
      }
      mbox_list_insert (&mailbox, message);
      nmsgin++;
      p1 = p2;
    }                              /* while ((msgstart... */
    if (nmsgin) {
      /*............dump all messages to the mailbox file */
      if (dumptofile (&mailbox, gsm->device, gsm) != nmsgin) {
	syslog ((FACILITY | LOG_ERR), "failed to save messages to inbox file.");
	mdmperror ("sms_serv: failed to save messages to inbox file");
      }
      /*.................remove the messages from the SIM */
      message = mailbox.head;
      while (message != NULL) {
	sprintf (scratch, "AT+CMGD=%d\r", message->msgid);
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_answer (fd, scratch, BIGBUFF, 1, gsm)) {
          if (debug & DEBUG_PRINTF) {
	    fprintf (stderr, "%s\n", scratch);
          }
	  /* check for "OK" */
	  if (strstr (scratch, "OK") == NULL) {
	    free (scratch);
	    syslog ((FACILITY | LOG_ERR), "error after sending AT+CMGD command.");
	    mdmperror ("sms_serv: error after sending AT+CMGD command");
	    return (-1);
	  }
	}
	else {
	  free (scratch);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
        message = message->next;
      }                        /* while (message != NULL) */
      /*...............clean the mailbox list from memory */
      free_mbox_list (&mailbox);
    }                                      /* if (nmsgin) */
  }
  else {
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
    mdmperror ("sms_serv: GSM not responding");
    return (-1);
  }

  if (dd_notify (gsm, fd) < 0) {
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "Unable to switch on notifications.");
    mdmperror ("sms_serv: Unable to switch on notifications");
    return (-1);
  }

  free (scratch);
  return (nmsgin);
  
}                                     /* dd_clear_msgs () */
/*========================================================*/
int dd_send_sms (struct gsms_def *gsm, int fd, struct symbols *pSMS, int csfd)
/*
 * send an SMS message based on the given parameters. Return negative number on
 * failure otherwise return's the message id. 
 * 	gsm - device we are talking to
 * 	fd - file descriptor of modem line
 *	pMSM - pointer to definition of SMS request
 *	csfd - file descriptor to report progress to
 */
{
  char *scratch;
  char *ptr;
  char *cmsgid;
  int nread;
  int msgid;
  char reqsca[MAXPHNUMLEN + 1];
  int reqmode;
  int pdubytes;
  char reqmodestr[10];
  cpl_list cpl;
  char subprompt[4];
  
  scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  if (!scratch)
    syserr ("dd_send_sms(): can't allocate scratch space");
  memset (scratch, 0, (BIGBUFF + 1));
  
  use_fast = (gsm->flags & FAST_RESPONSE);

  /*------------set "no notify for incoming SMS messages" */
  if (set_cnmi_off (gsm, fd, csfd) == -1) {
    return (-1);
  }
  
  /*--------------Check stored SCA against requested SMSC */
  /* Which SMSC do we require ? */
  if (strcmp (pSMS->smsc, DEFAULTSMSC) == 0) {
    /* user kept default value - let's use device-level default */
    strcpy (reqsca, gsm->defsca);
  }
  else {
    /* user requested specific SMSC - let's use it */
    strcpy (reqsca, pSMS->smsc);
  }
  if (checkset_csca (gsm, fd, csfd, reqsca) == -1) {
    return (-1);
  }
  
  /*-------------Check stored mode against requested mode */
  /* Which mode do we require ? */
  if (pSMS->mode == SMS_MODE_DEFAULT) {
    /* user kept default value - let's use device-level default */
    reqmode = gsm->defmode;
  }
  else {
    /* user requested specific mode - let's use it */
    reqmode = pSMS->mode;
  }
  if (checkset_cmgf (gsm, fd, csfd, reqmode) == -1) {
    return (-1);
  }
  
  /*---------------------Am I registered on the network ? */
  if (check_creg (gsm, fd, csfd) == -1) {
    return (-1);
  }
  
  /*------------------------Now actually send SMS message */
  cpl_list_init (&cpl);
  strcpy (subprompt, gsm->capmatrix.line_sep);
  strcat (subprompt, ">");
  cpl_list_insert (&cpl, subprompt);
  
  switch (reqmode) {
    case SMS_MODE_TEXT:
      if (gsm->capmatrix.capflags & SUBPROMPT_IN_CMGS) {
        /* first send AT command and dest GSM */
        sprintf (scratch, "AT+CMGS=\"%s\"\r", pSMS->destgsm);
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_resp (fd, scratch, BIGBUFF, 4, cpl)) {
	  tellsock (csfd, scratch);
	  /* check for ">" (subprompt char.) */
	  if (strstr (scratch, ">") == NULL) {
	    free (scratch);
            free_cpl_list (&cpl);
	    syslog ((FACILITY | LOG_ERR), "error getting AT+CMGS subprompt.");
	    mdmperror ("sms_serv: error getting AT+CMGS subprompt");
	    return (-1);
	  }
	}
	else {
	  free (scratch);
          free_cpl_list (&cpl);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
	/* then send the message proper... */
        sprintf (scratch, "%s%s", pSMS->message, CTRL_Z);
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_answer (fd, scratch, BIGBUFF, 4, gsm)) {
	  tellsock (csfd, scratch);
	  /* check for "OK" */
	  if (strstr (scratch, "OK") == NULL) {
	    free (scratch);
            free_cpl_list (&cpl);
	    syslog ((FACILITY | LOG_ERR), "error when sending SMS message.");
	    mdmperror ("sms_serv: error when sending SMS message");
	    return (-1);
	  }
	}
	else {
	  free (scratch);
          free_cpl_list (&cpl);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
      }
      else {
        sprintf (scratch, "AT+CMGS=\"%s\"\r%s%s", pSMS->destgsm,
                pSMS->message, CTRL_Z);
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_answer (fd, scratch, BIGBUFF, 4, gsm)) {
	  tellsock (csfd, scratch);
	  /* check for "OK" */
	  if (strstr (scratch, "OK") == NULL) {
	    free (scratch);
            free_cpl_list (&cpl);
	    syslog ((FACILITY | LOG_ERR), "error when sending SMS message.");
	    mdmperror ("sms_serv: error when sending SMS message");
	    return (-1);
	  }
	}
	else {
	  free (scratch);
          free_cpl_list (&cpl);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
      }                                 /* if (SUBPROMPT) */
      break;
      
    case SMS_MODE_PDU:
      /* Convert message to PDU if required */
      sprintf (scratch, "converting message to PDU format... ");
      tellsock (csfd, scratch);
      if ((pSMS->pdu = encode_pdu (pSMS, default_alphabet)) == NULL) {
	sprintf (scratch, "Fail\r\n");
	tellsock (csfd, scratch);
	free (scratch);
        free_cpl_list (&cpl);
	return (-1);
	/* exit */
      }
      sprintf (scratch, "Ok\r\n");
      tellsock (csfd, scratch);
    
      pdubytes = (strlen (pSMS->pdu) / 2);
      if (gsm->capmatrix.capflags & SUBPROMPT_IN_CMGS) {
        /* first send AT command and PDU length */
        sprintf (scratch, "AT+CMGS=%d\r", pdubytes);
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_resp (fd, scratch, BIGBUFF, 4, cpl)) {
	  tellsock (csfd, scratch);
	  /* check for ">" (subprompt char.) */
	  if (strstr (scratch, ">") == NULL) {
	    free (scratch);
            free_cpl_list (&cpl);
	    syslog ((FACILITY | LOG_ERR), "error getting AT+CMGS subprompt.");
	    mdmperror ("sms_serv: error getting AT+CMGS subprompt");
	    return (-1);
	  }
	}
	else {
	  free (scratch);
          free_cpl_list (&cpl);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
	/* then send the PDU string proper... */
	if (gsm->capmatrix.capflags & HAS_SCA_IN_PDU) {
          /* inserts "00" before PDU : GenSM/SM20 specific, means no SCA */
          sprintf (scratch, "00%s%s", pSMS->pdu, CTRL_Z);
	}
	else {
          /* GSM 03.40 compliant PDU format */
          sprintf (scratch, "%s%s", pSMS->pdu, CTRL_Z);
	}                              /* if (SCA_IN_PDU) */
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_answer (fd, scratch, BIGBUFF, 10, gsm)) {
	  tellsock (csfd, scratch);
	  /* check for "OK" */
	  if (strstr (scratch, "OK") == NULL) {
	    free (scratch);
            free_cpl_list (&cpl);
	    syslog ((FACILITY | LOG_ERR), "error when sending SMS message.");
	    mdmperror ("sms_serv: error when sending SMS message");
	    return (-1);
	  }
	}
	else {
	  free (scratch);
          free_cpl_list (&cpl);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
      }
      else {
        /* no subprompt */
	if (gsm->capmatrix.capflags & HAS_SCA_IN_PDU) {
          /* inserts "00" before PDU : GenSM/SM20 specific, means no SCA */
          sprintf (scratch, "AT+CMGS=%d\r00%s%s", pdubytes,
                  pSMS->pdu, CTRL_Z);
	}
	else {
          /* GSM 03.40 compliant PDU format */
          sprintf (scratch, "AT+CMGS=%d\r%s%s", pdubytes,
                  pSMS->pdu, CTRL_Z);
	}                              /* if (SCA_IN_PDU) */
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_answer (fd, scratch, BIGBUFF, 4, gsm)) {
	  tellsock (csfd, scratch);
	  /* check for "OK" */
	  if (strstr (scratch, "OK") == NULL) {
	    free (scratch);
            free_cpl_list (&cpl);
	    syslog ((FACILITY | LOG_ERR), "error when sending SMS message.");
	    mdmperror ("sms_serv: error when sending SMS message");
	    return (-1);
	  }
	}
	else {
	  free (scratch);
          free_cpl_list (&cpl);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
      }                          /* if (SUBPROMPT_IN_PDU) */
      break;
      
    default:
      free (scratch);
      free_cpl_list (&cpl);
      syslog ((FACILITY | LOG_ERR), "internal error: unsupported mode <%d>.",
             reqmode);
      mdmperror ("sms_serv: internal error: unsupported mode");
      return (-1);
  }                                     /* switch reqmode */
  free_cpl_list (&cpl);
  
  /* we have the answer - now extract the message ID from it */
  if ((ptr = strstr (scratch, "+CMGS:")) == NULL) {
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "can't find '+CMGS:' before OK in modem response.");
    mdmperror ("sms_serv: can't find '+CMGS:' before OK in modem response");
    return (-1);
  }
  ptr += 6;                        /* jumps to next space */
  if ((cmsgid = strtok (ptr, " \t\n\r")) == NULL) {
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "can't find message ID (no CrLfHt) after '+CMGS:'.");
    mdmperror ("sms_serv: can't find message ID (no CrLfHt) after '+CMGS:'");
    return (-1);
  }
  else {
    msgid = atoi (cmsgid);
  }                            /* if ((cmsgid = strtok... */
    
  /*----------------------------------Close communication */
  free (scratch);
  return (msgid);
}                                       /* dd_send_sms () */
/*========================================================*/
void dd_send_wrapper (struct gsms_def *dev)
/*
 * Send an SMS via the dedicated daemon through it's Local Domain socket
 * opens socket and sends parameters one per line.
 * newlines changed to spaces in data. -> simple hack.
 * copy back the results to the requisite socket if available.
 * return success/failure of request
 */
{
  int	fd;
  struct sockaddr_un s_addr;     /* server socket address */
  int ssfd;                          /* socket descriptor */
  char *p, *s;
  char buf[BIGGERBUFF];
  FILE *fp;

  /* attempt to connect to the Unix domain socket */
  /* see man 4 unix - on linux for doco */
  s_addr.sun_family = AF_LOCAL;
  strncpy (s_addr.sun_path, "/var/spool/smslink/", UNIX_PATH_MAX);
  strncat (s_addr.sun_path, dev->device, UNIX_PATH_MAX);

  if ((ssfd = socket (AF_LOCAL, SOCK_STREAM, 0)) == -1)
    syserr ("dd_send_wrapper(): can't create socket(AF_LOCAL, SOCK_STREAM, 0)");

  /* now bind the socket to the server address */
  if (connect (ssfd, (struct sockaddr *) &s_addr, sizeof (s_addr)) == -1) {
    syslog ((FACILITY | LOG_ERR), "can't connect to socket - %s",
	   s_addr.sun_path);
    return;
  }

  syslog ((FACILITY | LOG_INFO), "connection to '%s'", s_addr.sun_path);

  if ((fp = fdopen (ssfd, "r+")) == 0) {
    syslog ((FACILITY | LOG_ERR), "can't open FILE on socket - %d", ssfd);
    return;
  }

  /* some default stuff to stop dd_send_wrapper() from core dumping */
  if (symbols.smsc == 0)
      symbols.smsc = strdup (dev->defsca);
  if (symbols.destgsm == 0)
      symbols.destgsm = strdup ("61414200951");
  if (symbols.user == 0)
      symbols.user = strdup ("Andrew");
  if (symbols.message == 0)
      symbols.message = strdup ("This is a test SMS message");

  /* replace newlines with spaces - messages send over socket use newlines
   * as seperators - paranoia
   * could switch to using null if both ends are modified to understand
   * this -> requires implementing something for fgets() call then.
   */
  trChar (symbols.smsc, '\n', ' ');
  trChar (symbols.destgsm, '\n', ' ');
  trChar (symbols.user, '\n', ' ');
  trChar (symbols.message, '\n', ' ');

  fprintf (fp, "dest=%s\n", symbols.destgsm);
  fprintf (fp, "user=%s\n", symbols.user);
  fprintf (fp, "message=%s\n", symbols.message);
  fprintf (fp, "smsc=%s\n", symbols.smsc);
  fprintf (fp, "mode=%d\n", symbols.mode);
  fprintf (fp, "send\n"); /* command */
  fflush (fp);

  /* now just dump out what the daemon sends to us */
  while ((s = fgets (buf, BIGGERBUFF, fp)) != 0) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "Read '%s'\n", buf);
    }
    tellsock (csfd, buf);
  }                             /* while ((s = fgets (... */
  fclose (fp);

  syslog ((FACILITY | LOG_INFO), "connection to dedicated daemon closed");
  return;
}                                   /* dd_send_wrapper () */
/*==========================================================
 * EOF : dd.c
 *===================*/
