/*==========================================================
 * Program : gsmdevices.c                  Project : smslink
 * Author  : Philippe Andersson.
 * Date    : 12/01/05
 * Version : 0.17b
 * Notice  : Shamelessly copied from Riccardo Facchetti's modems.c
 *           in libmodem-1.0.0 (c) 1997 Riccardo Facchetti under GNU GPL.
 *           (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 * Comment : Handling routines for /etc/gsmdevices database.
 *
 * Modification History :
 * - 0.01a (13/08/98) : Initial release.
 * ++++ Switched to Beta ++++
 * - 0.02b (29/03/00) : Modified /etc/gsmdevices record structure
 *   to hold the default sending mode. Adapted demangle_gsmdev_entry()
 *   accordingly.
 * - 0.03b (20/05/00) : Added translatemodname() function to
 *   handle to pseudo-driver file, and adapted demangle_gsmdev_entry()
 *   accordingly. Cosmetics.
 * - 0.04b (21/06/00) : Adapted translatemodname() and
 *   demangle_gsmdev_entry() to handle the new capmatrix structure.
 * - 0.05b (18/02/01) : Modified demangle_gsmdev_entry() to take
 *   the <flags> field into account (contributed by Andrew Worsley 
 *   <epaanwo@asac.ericsson.se>).
 * - 0.06b (20/02/01) : Modified demangle_gsmdev_entry() to support
 *   the "fast" flag. Cosmetics. Contributed by Andrew Worsley 
 *   <epaanwo@asac.ericsson.se>.
 * - 0.07b (03/03/01) : Replaced control of all debug info by 
 *   global flags.
 * - 0.08b (06/03/01) : Modified demangle_gsmdev_entry() to support
 *   the "r4" flag.
 * - 0.09b (31/03/01) : Modified translatemodname() to support
 *   the new modem_okay field + fixed a bug in processing the
 *   date format field.
 * - 0.10b (03/04/01) : Fixed a bug in the way the special
 *   characters were handled in the modem_okay field.
 * - 0.11b (04/04/01) : Modified translatemodname() to support
 *   the new line_sep field.
 * - 0.12b (20/03/01) : Modified demangle_gsmdev_entry() to support
 *   the "db" flag. 
 * - 0.13b (16/05/01) : Adapted demangle_gsmdev_entry() and
 *   translatemodname() to handle the new "model" field in
 *   struct gsms_def.
 * - 0.14b (02/03/02) : Modified translatemodname() to return
 *   -1 in case the requested model is not found in the driver
 *   file. Should have been that way all along.
 * - 0.15b (07/11/02) : Added support for "init" mode (contributed
 *   by Ing. Andrea Vettori <a.vettori@inetronics.com>).
 *   Improved dependencies handling in device-level flags.
 *   Moved gsmdevcpy() from serv_stuff.c to this file. Added
 *   handling of "PINpause" field. Added dd_busy field handling.
 * - 0.16b (12/09/04) : Corrected a string sizing bug in
 *   translatemodname().
 * - 0.17b (12/01/05) : Added support for 'skip' flag.
 *========================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include <dial/modems.h>
#include <dial/mdmerrno.h>
#include <termios.h>

#include "sms_serv.h"
#include "gsmdev.h"

#define BUFSIZE		128

/*========================================================*/
/* For debugging purposes only - comment out for normal compile */
/* #define INCL_DEBUG_CODE */

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/
extern int debug;                    /* debug level/flags */
extern char *gsmdevfile;
static int gsmdeventry = 0;
static FILE *gsmdevfp;

/*========================================================*/
/**********               FUNCTIONS                ********/
/*========================================================*/
/*========================================================*/
/*################# struct gsms_def handling #############*/
/*========================================================*/
int gsmdevcpy (struct gsms_def *dest, struct gsms_def *src)
{
  if ((src != NULL) && (dest != NULL)) {
    dest->free = src->free;
    dest->owner = src->owner;
    dest->dd_busy = src->dd_busy;
    strcpy (dest->device, src->device);
    strcpy (dest->PIN, src->PIN);
    strcpy (dest->PUK, src->PUK);
    strcpy (dest->addr, src->addr);
    strcpy (dest->defsca, src->defsca);
    dest->PINpause = src->PINpause;
    strcpy (dest->provider, src->provider);
    strcpy (dest->model, src->model);
    dest->capmatrix.capflags = src->capmatrix.capflags;
    strcpy (dest->capmatrix.datefmt, src->capmatrix.datefmt);
    strcpy (dest->capmatrix.modem_okay, src->capmatrix.modem_okay);
    strcpy (dest->capmatrix.line_sep, src->capmatrix.line_sep);
    dest->defmode = src->defmode;
    dest->flags = src->flags;
    /* stats counter fields */
    dest->in_ok = src->in_ok;
    dest->in_fail = src->in_fail;
    dest->out_ok = src->out_ok;
    dest->out_fail = src->out_fail;
    dest->unlocked = src->unlocked;
    return (0);
  }
  else {
    return (-1);
  }
}                                         /* gsmdevcpy () */
/*========================================================*/
int translatemodname (struct gsms_def *mdm, char *driverfilename)
{
  FILE *driverfile;
  int found;
  int l;
  char *line;
  char *token;
  
  /*---------------------------------------Initialization */
  line = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (! line) {
    fprintf (stderr, "sms_serv: in translatemodname(): can't malloc()\n");
    return (-1);
  }
  line[0] = '\0';
  found = FALSE;
  /*-------------------------------------Open driver file */
  if ((driverfile = fopen (driverfilename, "r")) == NULL) {
    fprintf (stderr, "sms_serv: in translatemodname(): can't fopen() %s\n",
            driverfilename);
    return (-1);
  }
  /*--------------------------------------Loop and search */
  while ((! found) && (fgets (line, BUFFSIZE, driverfile) != NULL)) {
    if ((line[0] != '#') && (line[0] != '\n') && (line[0] != '\0')){
      /* line is not a comment or a blank */
      if ((token = strtok (line, ":")) == NULL) {
        return (-1);
      }
      if (strcmp (token, mdm->model) == 0) {
        /* we've found it - let's extract the cap. matrix items */
        /*.......................................capflags */
	if ((token = strtok (NULL, ":")) == NULL) {
          return (-1);
        }
	mdm->capmatrix.capflags = atoi (token);
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "GSM Model %s has cap. flags : [%d]\n", mdm->model,
	          mdm->capmatrix.capflags);
        }
        /*....................................date format */
	if ((token = strtok (NULL, ":")) == NULL) {
          return (-1);
        }
	l = strlen (token);
	if (l) {
	  if (l > 3) {
	    strncpy (mdm->capmatrix.datefmt, token, 3);
	  }
	  else {
	    strcpy (mdm->capmatrix.datefmt, token);
	  }
	}
	else {
	  /* empty field - let's default to 'dmy' */
	  strcpy (mdm->capmatrix.datefmt, "dmy");
	}
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "GSM Model %s has date format : [%s]\n", mdm->model,
	          mdm->capmatrix.datefmt);
        }
        /*......................................OK string */
	if ((token = strtok (NULL, ":")) == NULL) {
          return (-1);
        }
	l = strlen (token);
	if (l) {
	  spchar_pp (token);
	  if (l > TINYBUFF) {
	    strncpy (mdm->capmatrix.modem_okay, token, TINYBUFF);
	    mdm->capmatrix.modem_okay[TINYBUFF] = '\0';
	  }
	  else {
	    strcpy (mdm->capmatrix.modem_okay, token);
	  }
	}
	else {
	  /* empty field - let's default to MODEM_OKAY */
	  strcpy (mdm->capmatrix.modem_okay, MODEM_OKAY);
	}
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "GSM Model %s has OK string : [%s]\n", mdm->model,
	          mdm->capmatrix.modem_okay);
	  if (debug & DEBUG_HEXDUMP) {
	    fprintf (stderr, "Hexdump of modem OK string:\n");
	    dump_string (mdm->capmatrix.modem_okay);
	  }
        }
        /*.................................line separator */
	if ((token = strtok (NULL, ":")) == NULL) {
          return (-1);
        }
	l = strlen (token);
	if (l) {
	  spchar_pp (token);
	  if (l > 2) {
	    strncpy (mdm->capmatrix.line_sep, token, 2);
	    mdm->capmatrix.line_sep[2] = '\0';
	  }
	  else {
	    strcpy (mdm->capmatrix.line_sep, token);
	  }
	}
	else {
	  /* empty field - let's default to CRLF */
	  strcpy (mdm->capmatrix.modem_okay, "\r\n");
	}
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "GSM Model %s has line separator string : [%s]\n",
	          mdm->model, mdm->capmatrix.line_sep);
	  if (debug & DEBUG_HEXDUMP) {
	    fprintf (stderr, "Hexdump of modem line_sep string:\n");
	    dump_string (mdm->capmatrix.line_sep);
	  }
        }
	/*................................................*/
        /* last field (model name) is ignored */
	/*................................................*/
	found = TRUE;
      }
    }                              /* if (line[0] != '#') */
  }                                    /* while (fgets... */
  /*------------------------------------------Conclusions */
  fclose (driverfile);
  free (line);
  if (found) {
    return (0);
  }
  else {
    return (-1);
  }
}                                  /* translatemodname () */
/*========================================================*/
static int open_gsmdevs (void)
/*
 * open/close_gsmdevs: open and close the /etc/gsmdevices file
 */
{
    if ((gsmdevfp = fopen (gsmdevfile, "r")) == NULL)
        return FAILURE;
    return SUCCESS;
}                                      /* open_gsmdevs () */
/*========================================================*/
static void close_gsmdevs (void)
{
    fclose (gsmdevfp);
}                                     /* close_gsmdevs () */
/*========================================================*/
struct gsms_def *getgsmdevbynam (char *line)
/*
 * getgsmdevbynam: returns a pointer to the gsmdev entry named
 *            (char *) line, NULL if line is not present in
 *            /etc/gsmdevices.
 */
{
    static struct gsms_def mdm;
    char *buffer;

    buffer = mdmalloc(BUFSIZE);

    if (!buffer) {
	mdmerrno = -EMDMEM;
	return NULL;
    }

    mdmerrno = 0;

    if (open_gsmdevs() == FAILURE) {
	mdmfree (buffer);
        mdmerrno = -ENOMDMFILE;
        return NULL;
    }

    while (fgets (buffer, BUFSIZE, gsmdevfp) != NULL) {
        buffer[strlen (buffer) - 1] = '\0';

        if (buffer[0] == '#' || buffer[0] == '\0' || buffer[0] == '\n')
            continue;

        if (demangle_gsmdev_entry (&mdm, buffer) == FAILURE) {
            mdmerrno = -EMDMBADCONF;
            continue;
        }
        if (!strcmp (mdm.device, line)) {
            close_gsmdevs ();
	    mdmfree (buffer);
            return (&mdm);
        }
    }
    close_gsmdevs ();
    mdmerrno = -ENOMDMLINE;
    return NULL;
}                                    /* getgsmdevbynam () */
/*========================================================*/
int getgsmdevscount (int validation)
/*
 * getgsmdevscount: count the number of valid (?) entries in
 *            /etc/gsmdevices.
 */
{
    static struct gsms_def mdm;
    char *buffer;
    int nvalidentries = 0;

    buffer = mdmalloc(BUFSIZE);

    if (!buffer) {
	mdmerrno = -EMDMEM;
	return (0);
    }

    mdmerrno = 0;

    if (open_gsmdevs() == FAILURE) {
	mdmfree (buffer);
        mdmerrno = -ENOMDMFILE;
        return (0);
    }

    while (fgets (buffer, BUFSIZE, gsmdevfp) != NULL) {
        buffer[strlen (buffer) - 1] = '\0';

        if (buffer[0] == '#' || buffer[0] == '\0' || buffer[0] == '\n')
            continue;

        if (validation && (demangle_gsmdev_entry (&mdm, buffer) == FAILURE)) {
            mdmerrno = -EMDMBADCONF;
            continue;
        }
	nvalidentries++;
    }
    close_gsmdevs ();
    return (nvalidentries);
}                                   /* getgsmdevscount () */
/*========================================================*/
void setgsmdevs (void)
/*
 * setgsmdevs: set the library for modem sequential search from the first to the
 *          end. See getnextgsmdev ().
 */
{
    gsmdeventry = 0;
}                                        /* setgsmdevs () */
/*========================================================*/
struct gsms_def *getnextgsmdev (void)
/*
 * getnextgsmdev: get the next modem entry in /etc/gsmdevices
 *             file returns NULL if we are over the last entry.
 */
{
    static struct gsms_def mdm;
    int i = 0;
    char *buffer;

    buffer = mdmalloc (BUFSIZE);

    if (!buffer) {
	mdmerrno = -EMDMEM;
	return NULL;
    }


    mdmerrno = 0;

    if (open_gsmdevs () == FAILURE) {
	mdmfree (buffer);
        mdmerrno = -ENOMDMFILE;
        return NULL;
    }

    while (fgets (buffer, BUFSIZE, gsmdevfp) != NULL) {
        buffer[strlen (buffer) - 1] = '\0';

        if (buffer[0] == '#' || buffer[0] == '\0' || buffer[0] == '\n')
            continue;

        if (i == gsmdeventry) {
            if (demangle_gsmdev_entry (&mdm, buffer) == FAILURE) {
                mdmerrno = -EMDMBADCONF;
                continue;
            }
            gsmdeventry++;
            close_gsmdevs ();
	    mdmfree (buffer);
            return (&mdm);
        }
        i++;
    }
    close_gsmdevs ();
    mdmfree(buffer);
    return NULL;
}                                     /* getnextgsmdev () */
/*========================================================*/
static int demangle_gsmdev_entry (struct gsms_def *mdm, char *mdmstr)
/*
 * demangle_gsmdev_entry: get a gsmdev pointer that point to an existing
 * gsm_def structure and a string to demangle. Returns FAILURE or SUCCESS.
 * The buffer pointed to by mdmstr is modified in the process.
 */
{
    char *ptr, *pptr, *errptr;
    int more = 0;

    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "raw gsmdev entry: '%s'\n", mdmstr);
    }
    /*------------------------------------Initialisations */
    /*.........................................mdm->flags */
    /* set default value for flags: (poll mode | inbox) */
    mdm->flags = SMS_INBOX;

    /*....................................mdm->free (int) */
    /* is left uninitialized - dynamic data */
    
    /*-----------------------------------------Processing */
    /*...............................mdm->device (char *) */
    if ((ptr = strchr (mdmstr, ':')) == NULL)
        return (FAILURE);
    *ptr = '\0';
    if ((strlen (mdmstr) == 0) || (strlen (mdmstr) > MAXDEVLEN))
        return (FAILURE);
    strcpy (mdm->device, mdmstr);
    ptr++;

    /*..................................mdm->PIN (char *) */
    if (*ptr == ':') {
        /* empty field */
        mdm->PIN[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr (ptr, ':')) == NULL)
            return (FAILURE);
        *pptr = '\0';
        if (strlen (ptr) > PINLEN)
            return (FAILURE);
	strcpy (mdm->PIN, ptr);
        ptr = pptr + 1;
    }

    /*..................................mdm->PUK (char *) */
    if (*ptr == ':') {
        /* empty field */
        mdm->PUK[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr (ptr, ':')) == NULL)
            return (FAILURE);
        *pptr = '\0';
        if (strlen (ptr) > PUKLEN)
            return (FAILURE);
        strcpy (mdm->PUK, ptr);
        ptr = pptr + 1;
    }

    /*.................................mdm->addr (char *) */
    if (*ptr == ':') {
        /* empty field */
        mdm->addr[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr (ptr, ':')) == NULL)
            return (FAILURE);
        *pptr = '\0';
        if (strlen (ptr) > MAXPHNUMLEN)
            return (FAILURE);
        strcpy (mdm->addr, ptr);
        ptr = pptr + 1;
    }

    /*...............................mdm->defsca (char *) */
    if (*ptr == ':') {
        /* empty field */
        mdm->defsca[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr (ptr, ':')) == NULL)
            return (FAILURE);
        *pptr = '\0';
        if (strlen (ptr) > MAXPHNUMLEN)
            return (FAILURE);
        strcpy (mdm->defsca, ptr);
        ptr = pptr + 1;
    }

    /*.................................mdm->defmode (int) */
    if (*ptr == ':') {
        /* empty field */
        mdm->defmode = SMS_MODE_DEFAULT;
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr (ptr, ':')) == NULL)
            return (FAILURE);
        *pptr = '\0';
        if (strlen (ptr) == 0)
            return (FAILURE);
        mdm->defmode = strtol (ptr, &errptr, 0);
	if (*errptr != '\0')
            return (FAILURE);
        ptr = pptr + 1;
    }

    /*................................mdm->PINpause (int) */
    if (*ptr == ':') {
        /* empty field */
        mdm->PINpause = SMS_PINPAUSE_DEFAULT;
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr (ptr, ':')) == NULL)
            return (FAILURE);
        *pptr = '\0';
        if (strlen (ptr) == 0)
            return (FAILURE);
        mdm->PINpause = strtol (ptr, &errptr, 0);
	if (*errptr != '\0')
            return (FAILURE);
        ptr = pptr + 1;
    }

    /*.....................................mdm->capmatrix */
    if (*ptr == ':') {
        /* empty field */
        mdm->capmatrix.capflags = 0;
	mdm->capmatrix.datefmt[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr (ptr, ':')) == NULL)
            return (FAILURE);
        *pptr = '\0';
        if (strlen (ptr) > MAXMODLEN)
            return (FAILURE);
        strcpy (mdm->model, ptr);
	/* Now translate model into capacity matrix */
	if ((translatemodname (mdm, GSMCAPS_FILE)) == -1) {
	  return (FAILURE);
	}
        ptr = pptr + 1;
    }

    /*.............................mdm->provider (char *) */
    /*                   !!! Last field when no flags !!! */
    if (*ptr == '\n' || *ptr == '\0' || *ptr == ':') {
        mdm->provider[0] = '\0';
 	if (*ptr != ':') {
	    /* was the last field - no flags */
 	    return (SUCCESS);
	}
 	ptr++;                    /* go on to flags field */
    }
    else {
        if (pptr = strchr (ptr, ':'))
	    *pptr = '\0';
        if (strlen (ptr) > MAXDEVLEN)
            return (FAILURE);
        strcpy (mdm->provider, ptr);
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "Provider : [%s]\n", mdm->provider);
        }
	if (pptr)
	    ptr = pptr + 1;
	else
	    ptr = 0;
    }

    /*...................................mdm->flags (int) */
    if (ptr != 0 && *ptr != '\n' && *ptr != '\0' && *ptr != ':') {
      if (debug & DEBUG_PRINTF) {
        fprintf (stderr, "GSM Model %s has flags field : [%s]\n",
	        mdm->model, ptr);
      }
      mdm->flags = 0; /* explicitly specifying flags so zero all flags */
      do {
	size_t i = strcspn (ptr, ":,\n");
	more = 0;
	if (i > 0) {
	  pptr = ptr + i;
	  if (*pptr == ',') /* more flags */
	    more = 1;
	  *pptr = '\0';
	}
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "processing flag=%s flags = [%d]\n", ptr, mdm->flags);
        }
	/* could use a table in the future here for future options */
	/* could use strncasecmp(), and check for optional '=' character
	 * which could be used to specify extra info such as program to
	 * run for SMS_PROG or mail box file for SMS_INBOX et cetera 
	 */
	if (strcasecmp (ptr, "dd") == 0) {
	  /* dd mode excludes init mode */
	  if (mdm->flags & SMS_INIT) {
	    mdm->flags &= ~SMS_INIT;
	    fprintf (stderr, "Warning: dd mode conflicts with init mode, overrides it\n");
	  }
	  mdm->flags |= DEDICATED_DAEMON;
	}
	else if (strcasecmp (ptr, "init") == 0) {
	  /* init mode excludes dd mode */
	  if (mdm->flags & DEDICATED_DAEMON) {
	    mdm->flags &= ~DEDICATED_DAEMON;
	    fprintf (stderr, "Warning: init mode conflicts with dd mode, overrides it\n");
	  }
	  mdm->flags |= SMS_INIT;
	}
	else if (strcasecmp (ptr, "poll") == 0) {
	  /* poll mode excludes dd mode */
	  if (mdm->flags & DEDICATED_DAEMON) {
	    mdm->flags &= ~DEDICATED_DAEMON;
	    fprintf (stderr, "Warning: poll mode conflicts with dd mode, overrides it\n");
	  }
	  /* poll mode excludes init mode */
	  if (mdm->flags & SMS_INIT) {
	    mdm->flags &= ~SMS_INIT;
	    fprintf (stderr, "Warning: poll mode conflicts with init mode, overrides it\n");
	  }
	  mdm->flags |= POLL_DEVICE;
	}
	else if (strcasecmp (ptr, "inbox") == 0) {
	  mdm->flags |= SMS_INBOX;
	}
	else if (strcasecmp (ptr, "prog") == 0) {
	  mdm->flags |= SMS_PROG;
	}
	else if (strcasecmp (ptr, "db") == 0) {
	  mdm->flags |= SMS_DB;
	}
	else if (strcasecmp (ptr, "fast") == 0) {
	  mdm->flags |= FAST_RESPONSE;
	}
	else if (strcasecmp (ptr, "r4") == 0) {
	  mdm->flags |= SMS_READ_ALL;
	}
	else if (strcasecmp (ptr, "skip") == 0) {
	  /* skip option excludes inbox mode */
	  if (mdm->flags & SMS_INBOX) {
	    mdm->flags &= ~SMS_INBOX;
	    fprintf (stderr, "Warning: skip option conflicts with inbox mode, overrides it\n");
	  }
	  /* skip option excludes db mode */
	  if (mdm->flags & SMS_DB) {
	    mdm->flags &= ~SMS_DB;
	    fprintf (stderr, "Warning: skip option conflicts with db mode, overrides it\n");
	  }
	  /* skip option excludes prog mode */
	  if (mdm->flags & SMS_PROG) {
	    mdm->flags &= ~SMS_PROG;
	    fprintf (stderr, "Warning: skip option conflicts with prog mode, overrides it\n");
	  }
	  mdm->flags |= SMS_SKIP;
	}
	else {
	  return (FAILURE);
	}

	ptr = pptr + 1;

      } while (more);
    }
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "flags are now: [%d]\n", mdm->flags);
    }

    /*----------------------------------------Conclusions */
    return (SUCCESS);
}                             /* demangle_gsmdev_entry () */
/*==========================================================
 * EOF : gsmdevices.c
 *===================*/
