/*==========================================================
 * Program : gw_stuff.c                    Project : smslink
 * Author  : Philippe Andersson.
 * Date    : 20/04/06
 * Version : 0.27b
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 * Comment : Library of functions for the smslink sms2mail gateway.
 *
 * Modification History :
 * - 0.01b (19/08/99) : Initial release. Moved the mail gateway-
 *   specific functions from stuff.c over here.
 * - 0.02b (28/09/99) : Created mailbox_run().
 * - 0.03b (29/09/99) : Extensive rework. Added tkize_ibox_line(),
 *   reset_mail_struct() and parse_smail().
 * - 0.04b (04/10/99) : Added expand_addr().
 * - 0.05b (17/10/99) : Expanded parse_smail() extensively.
 * - 0.06b (19/10/99) : Debugged tkize_ibox_line(). Increased
 *   debugging output. Created shell for send_mail().
 * - 0.07b (20/10/99) : Added mkfromfield().
 * - 0.08b (21/10/99) : Added is_dotted_quad () and resolve().
 * - 0.09b (08/11/99) : Built dialog in send_mail(). Added
 *   slurp_n_catch().
 * - 0.10b (09/11/99) : Modified parse_smail() to provide a
 *   default subject field if the SMS doesn't contain one. Involved
 *   alter the parameters passed to the function. Cosmetics.
 * - 0.11b (12/11/99) : Improved send_mail() to free() the
 *   allocated space for the nicified date.
 * - 0.12b (30/11/99) : Added temp file creation to mailbox_run().
 * - 0.13b (24/02/00) : Completed the transfer routine to the
 *   new mailbox file in mailbox_run().
 * - 0.14b (26/02/00) : Made sure the lockfile (MBOX_LOCKF) was
 *   deleted on each fatal error, as well as on SIGTERM being
 *   caught (in mailgws_death()). Improved error reporting in
 *   various functions. Cosmetics.
 * - 0.15b (29/02/00) : Added functions to handle recepients
 *   lists (as part of struct email_msg). Heavily modified most
 *   functions (send_mail(), parse_smail(), expand_addr()) and
 *   created sub_tkize_field() to account for changes in the
 *   email_msg struct.
 * - 0.16b (01/03/00) : Replaced the "old" slurp_n_catch() by
 *   a new version inspired by get_gsm_answer() (cf. serv_stuff.c).
 *   This should be more robust and less sensitive to the amount
 *   of data sent back by the server (here: sendmail). Created
 *   the clean_field() function (forgotten when split expand_addr()
 *   from sub_tkize_field()).
 * - 0.17b (28/03/00) : Improved error handling and reporting in
 *   send_mail() and mailbox_run().
 * - 0.18b (05/03/01) : Replaced control of all debug info by
 *   global flags.
 * - 0.19b (06/01/02) : Added a "mailhost" parameter to the
 *   mailbox_run() function, to carry the parameter passed
 *   through the "--smtprelay=" option.
 * - 0.20b (28/01/02) : In send_mail(), replaced all <lf> by
 *   <cr><lf> in the dialog with the SMTP server, as per RFC
 *   822bis, section 2.3 (see <http://cr.yp.to/docs/smtplf.html>
 *   for details). Caused problems when talking to qmail. Thanks
 *   a lot to Nathan Thomas (<nthomas@e-comm.com.au>) for letting
 *   me know.
 * - 0.21b (21/02/03) : In send_mail(), included a line skip
 *   between the last header row and the message body. Suggested
 *   by Denny Prasetya <denny@prasetya.com> to fix a problem
 *   with squirrelmail.
 * - 0.22b (25/07/03) : Moved anything mail-related (send_mail(),
 *   resolve() and all functions for struct email_msg handling)
 *   over to smtp.[ch].
 * - 0.23b (03/08/03) : Corrected a bug that prevented multi-line
 *   inbox entries from being successfully processed. Those are
 *   now supported, whether they be mail or non-mail entries.
 * - 0.24b (19/02/04) : Added debug info in tkize_ibox_line().
 * - 0.25b (07/01/06) : Cosmetics. Improved error checking in
 *   mailbox_run(). Fixed a bug that would lose the first line(s)
 *   of a multiline, non-mail inbox entry if it came after mail
 *   entries.
 * - 0.26b (22/03/06) : Added PID file removal to signal handler.
 * - 0.27b (20/04/06) : Plugged a memory leak in mailbox_run().
 *========================================================*/

#include <unistd.h>
#include <stdio.h>                         /* for fprintf */
#include <stdlib.h>                  /* for errno & stuff */
#include <errno.h>
#include <string.h>
#include <syslog.h>
#include <netdb.h>                 /* for gethostbyname() */
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/time.h>           /* for the struct timeval */
#include <sys/types.h>
#include <sys/wait.h>                /* for WEXITSTATUS() */
#include <sys/ipc.h>
#include <sys/ioctl.h>            /* for the ioctl() call */
#ifdef LINUX_LC6
#  include <limits.h>                     /* for PATH_MAX */
#  include <linux/limits.h>
#else
#  include <limits.h>
#endif

#include "sms_serv.h"
#include "smtp.h"

/*========================================================*/
/* For debugging purposes only - comment out for normal compile */
/* #define INCL_DEBUG_CODE */

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/
extern int debug;                    /* debug level/flags */

/*========================================================*/
/**********               FUNCTIONS                ********/
/*========================================================*/
void mailgws_death ()
{
  extern int inbox_is_locked;
  extern char *pidfile;
  
  /* remove inbox lock file if necessary */
  if (inbox_is_locked) {
    unlink (MBOX_LOCKF);
  }

  /* remove PID file */
  if (unlink (pidfile) == -1) {
    if (errno != ENOENT) {             /* no such file... */
      syserr ("mailgws_death(): can't remove PID file");
    }
  }

  /* log it... */
  syslog ((FACILITY | LOG_INFO), "gateway exiting on SIGTERM.");
  
  /* closes connection to the syslog daemon */
  closelog ();
  
  /* now exits gracefully */
  exit (0);
}                                     /* mailgws_death () */
/*========================================================*/
int is_ibox_header (char *line)
{
  int r;
  int retval = FALSE;
  
  r = strncmp (line, "/dev/", 5);
  
  if (r == 0) {
    retval = TRUE;
  }
  
  return (retval);
}                                    /* is_ibox_header () */
/*========================================================*/
int is_ibox_trailer (char *line)
{
  char *p;
  int r;
  int retval = FALSE;

  p = line + (strlen (line) - 2);
  r = strncmp (p, "\"\n", 2);
  
  if (r == 0) {
    retval = TRUE;
  }
  
  return (retval);
}                                   /* is_ibox_trailer () */
/*========================================================*/
int tkize_ibox_line (char *l, struct inbox_line *tkl)
{
  char *token;
  char *scratch;
  
  /*--------------------------------------Initializations */
  /* copy input string to scratch space - we can't modify it */
  scratch = (char *) malloc ((strlen (l) + 1) * sizeof (char));
  if (! scratch) {
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("tkize_ibox_line(): can't malloc() scratch");
  }
  scratch[0] = '\0';
  strcpy (scratch, l);
  
  /*--------------------------------------Parsing routine */
  /*...............................................Device */
  if ((token = strtok (scratch, ",")) == NULL) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "tkize_ibox_line(): failed to extract <device> field.\n");
      fprintf (stderr, "line = [%s].\n", l);
    }
    free (scratch);
    return (-1);
  }
  strcpy (tkl->device, token);
  
  /*................................................MsgID */
  if ((token = strtok (NULL, ",")) == NULL) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "tkize_ibox_line(): failed to extract <msgID> field.\n");
      fprintf (stderr, "line = [%s].\n", l);
    }
    free (scratch);
    return (-1);
  }
  tkl->msgid = atoi (token);
  
  /*..............................................FromGSM */
  if ((token = strtok (NULL, ",")) == NULL) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "tkize_ibox_line(): failed to extract <fromGSM> field.\n");
      fprintf (stderr, "line = [%s].\n", l);
    }
    free (scratch);
    return (-1);
  }
  strcpy (tkl->fromgsm, token);
  
  /*.................................................Date */
  if ((token = strtok (NULL, ",")) == NULL) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "tkize_ibox_line(): failed to extract <date> field.\n");
      fprintf (stderr, "line = [%s].\n", l);
    }
    free (scratch);
    return (-1);
  }
  strcpy (tkl->date, token);
  
  /*.................................................Time */
  if ((token = strtok (NULL, ",")) == NULL) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "tkize_ibox_line(): failed to extract <time> field.\n");
      fprintf (stderr, "line = [%s].\n", l);
    }
    free (scratch);
    return (-1);
  }
  strcpy (tkl->time, token);
  
  /*.................................................Text */
  /* just grab the rest of the line */
  if ((token = strtok (NULL, "\0")) == NULL) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "tkize_ibox_line(): failed to extract <text> field.\n");
      fprintf (stderr, "line = [%s].\n", l);
    }
    free (scratch);
    return (-1);
  }
  /* remove trailing LF */
  token[strlen (token) - 1] = '\0';
  dequote (token);
  strcpy (tkl->text, token);
  
  /*-------------------------------------------------Exit */
  free (scratch);
  return 0;                                    /* success */
}                                   /* tkize_ibox_line () */
/*========================================================*/
char *expand_addr (char *field, char *defdom)
{
  char *newstring;

  /*---------------------------------------Initialization */
  newstring = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  if (! newstring) {
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("expand_addr(): can't malloc() newstring");
  }
  newstring[0] = '\0';
  
  /*---------------------------------Main processing loop */
  if (strchr (field, '@') == NULL) {
    sprintf (newstring, "%s@%s", field, defdom);
  }
  else {
    strcpy (newstring, field);
  }
  /*------------------------------------------Conclusions */
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "expanded field: [%s]\n", newstring);
  }
  return (newstring);
}                                       /* expand_addr () */
/*========================================================*/
char *clean_field (char *field)
/* For fields that cannot be multi-part (Reply-to:), remove
 * the field header and any additional address. */
{
  char *ptr;
  
  /* The field for sure has a field header - skip it */
  shiftleft (field, 2);
  /* If the first char now is a space, skip that as well */
  if (field[0] == ' ') {
    shiftleft (field, 1);
  }
  /* If the field contains a comma, kill it and all that follows */
  if ((ptr = strchr (field, ',')) != NULL) {
    ptr[0] = '\0';
  }
  
  return (field);
}                                       /* clean_field () */
/*========================================================*/
int sub_tkize_field (rcpt_list *list, char *field, char *defdom)
{
  int nfields = 0;
  char *s;
  char *ptr;
  char *newstring;
  char *item;
  
  /*---------------------------------------Initialization */
  /* take a local copy of "field" to avoid it being modified */
  s = (char *) malloc ((strlen (field) + 1) * sizeof (char));
  if (! s) {
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("sub_tkize_field(): can't malloc() s");
  }
  s[0] = '\0';
  strcpy (s, field);
  ptr = s;
  
  /* s for sure has a field header ("T:" for inst.) - skip it */
  ptr += 2;
  /* if there was a blank between the header and the field, skip it too */
  if (ptr[0] == ' ') {
    ptr++;
  }
  
  /*---------------------------------Main processing loop */
  if ((item = strtok (ptr, ",")) == NULL) {
    /* field header w/o contents */
    free (s);
    return (nfields);
  }
  /* process first item */
  newstring = expand_addr (item, defdom);
  rcpt_list_insert (list, newstring);
  nfields++;
  free (newstring);
  
  /* now tokenize and process the rest */
  while ((item = strtok (NULL, ",")) != NULL) {
    /* process additional item(s) */
    newstring = expand_addr (item, defdom);
    rcpt_list_insert (list, newstring);
    nfields++;
    free (newstring);
  }                                        /* while (...) */
  /*------------------------------------------Conclusions */
  free (s);
  return (nfields);
}                                   /* sub_tkize_field () */
/*========================================================*/
char *mkfromfield (char *srcgsm, char *host, char *defdom)
{
  char *fromfield;
  
  /*--------------------------------------Initializations */
  fromfield = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (! fromfield) {
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("mkfromfield(): can't malloc() fromfield");
  }
  fromfield[0] = '\0';
  
  /*-------------------------------------------Processing */
  sprintf (fromfield, "%s%s@%s.%s", GWUSER, srcgsm, host, defdom);
  
  /*------------------------------------------Conclusions */
  return (fromfield);
}                                       /* mkfromfield () */
/*========================================================*/
int parse_smail (struct inbox_line *ibl, struct email_msg *m, char *host,
                 char *domain)
{
  char *s;
  char *s_saved;
  char *ptr;
  char *s_end;
  int end_found;
  char field_header;
  char end_marker;
  
  /*--------------------------------------Initializations */
  /* struct email fields */
  m->from = NULL;
  rcpt_list_init (&(m->to));
  rcpt_list_init (&(m->cc));
  rcpt_list_init (&(m->bcc));
  m->reply_to = NULL;
  m->subject = NULL;
  body_list_init (&(m->body));
  
  /* copy sms string to scratch space - we can't modify it */
  s = (char *) malloc ((MAXMSGLEN + 1) * sizeof (char));
  if (! s) {
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("parse_smail(): can't malloc() s");
  }
  s[0] = '\0';
  strcpy (s, ibl->text);
  s_saved = s;
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "\ns as received:\n[%s]\n", s);
  }
  
  /*--------------------------------------Main Processing */
  /* First field have to be To: */
  if ((toupper (s[0]) == 'T') && (s[1] == ':')) {
    /* if space between : and address, remove it */
    if (s[2] == ' ') {
      ptr = (s + 2);
      shiftleft (ptr, 1);
    }
    end_found = FALSE;
    ptr = s;
    while (!end_found) {
      if ((ptr = strchr (ptr, ' ')) == NULL) {
        /* To: was the only field */
        syslog ((FACILITY | LOG_ERR), "can't parse, nothing more than To: field.");
	free (s_saved);
	return (-1);
      }
      if ((ptr[-1] == ',') && (ptr[0] != '\0')) {
        /* we have a multipart to: field */
	ptr++;
      }
      else {
        end_found = TRUE;
	s_end = ptr;
      }
    }                               /* while (!end_found) */
    if (ptr[0] != '\0') {
      ptr++;
    }
    s_end[0] = '\0';
    /* sub-tokenize it and save it */
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "to field: [%s]\n", s);
    }
    sub_tkize_field (&(m->to), s, domain);
    
    /* reset s to begin of next field */
    s = ptr;
    
    /* now loop to process the rest */
    while (s[0]) {
      field_header = toupper (s[0]);
      if ((strchr (FIELD_HEADERS, field_header) != NULL) && (s[1] == ':')) {
        /* we have a valid header */
	switch (field_header) {
	  case 'T' : {
	    end_marker = ' ';
	    break;
	  }
	  
	  case 'C' : {
	    end_marker = ' ';
	    break;
	  }
	  
	  case 'R' : {
	    if (m->reply_to) {
	      free (m->reply_to);
	    }
	    end_marker = ' ';
	    break;
	  }
	  
	  case 'B' : {
	    end_marker = ' ';
	    break;
	  }
	  
	  case 'F' : {
	    if (m->from) {
	      free (m->from);
	    }
	    end_marker = ' ';
	    break;
	  }
	  
	  case 'S' : {
	    if (m->subject) {
	      free (m->subject);
	    }
	    end_marker = '#';
	    break;
	  }
	}                        /* switch (field_header) */
	/* if space between : and address, remove it */
	if (s[2] == ' ') {
	  ptr = (s + 2);
	  shiftleft (ptr, 1);
	}
	end_found = FALSE;
	ptr = s;
	while (!end_found) {
	  if ((ptr = strchr (ptr, end_marker)) == NULL) {
	    /* we reached end-of-string */
	    end_found = TRUE;
	    ptr = s + strlen (s);
	    s_end = ptr;
	  }
	  else {
	    if ((ptr[-1] == ',') && (ptr[0] != '\0')) {
	      /* multipart field */
	      ptr++;
	    }
	    else {
	      end_found = TRUE;
	      s_end = ptr;
	    }
	  }                  /* if (end_marker not found) */
	}                           /* while (!end_found) */
	if (ptr[0] != '\0') {
	  ptr++;
	}
	s_end[0] = '\0';
	/* assigns the right field */
	switch (field_header) {
	  case 'T' : {
            if (debug & DEBUG_PRINTF) {
              fprintf (stderr, "field: [%s]\n", s);
            }
            sub_tkize_field (&(m->to), s, domain);
	    break;
	  }
	  
	  case 'C' : {
            if (debug & DEBUG_PRINTF) {
              fprintf (stderr, "field: [%s]\n", s);
            }
            sub_tkize_field (&(m->cc), s, domain);
	    break;
	  }
	  
	  case 'R' : {
            if (debug & DEBUG_PRINTF) {
              fprintf (stderr, "field: [%s]\n", s);
            }
            clean_field (s);
	    m->reply_to = expand_addr (s, domain);
	    break;
	  }
	  
	  case 'B' : {
            if (debug & DEBUG_PRINTF) {
              fprintf (stderr, "field: [%s]\n", s);
            }
            sub_tkize_field (&(m->bcc), s, domain);
	    break;
	  }
	  
	  case 'F' : {
            if (debug & DEBUG_PRINTF) {
              fprintf (stderr, "field: [%s]\n", s);
            }
            clean_field (s);
            m->from = expand_addr (s, domain);
	    break;
	  }
	  
	  case 'S' : {
            if (debug & DEBUG_PRINTF) {
              fprintf (stderr, "field: [%s]\n", s);
            }
	    /* remove field header */
	    s += 2;
	    /* if ':' is followed by ' ', remove it as well */
	    if ((s[0]) && (s[0] == ' ')) {
	      s++;
	    }
	    m->subject = (char *) malloc ((strlen (s) + 1) * sizeof (char));
	    if (! m->subject) {
	      syslog ((FACILITY | LOG_ERR), "can't malloc().");
              unlink (MBOX_LOCKF);
	      syserr ("parse_smail(): can't malloc() m->subject");
	    }
	    m->subject[0] = '\0';
	    strcpy (m->subject, s);
	    break;
	  }
	}                        /* switch (field_header) */
	s = ptr;
      }
      else {
        /* the rest is considered BODY */
	if (s[0]) {
          if (debug & DEBUG_PRINTF) {
            fprintf (stderr, "body: [%s]\n", s);
          }
          body_list_insert (&(m->body), s);
	  ptr = s + strlen (s);
	  s = ptr;
	}                                    /* if (s[0]) */
      }                        /* if (valid header found) */
    }                                     /* while (s[0]) */
  }
  else {
    /* can't even find To: field */
    syslog ((FACILITY | LOG_ERR), "can't parse, can't find To: field.");
    free (s_saved);
    return (-1);
  }                           /* if ('T' followed by ':') */
  
  /*--------------------------Make sure we have a subject */
  if (! m->subject) {
    m->subject = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
    if (! m->subject) {
      syslog ((FACILITY | LOG_ERR), "can't malloc().");
      unlink (MBOX_LOCKF);
      syserr ("parse_smail(): can't malloc() m->subject");
    }
    m->subject[0] = '\0';
    sprintf (m->subject, "An SMS for you from GSM number %s.", ibl->fromgsm);
  }
  /*-------------------------------------------------Exit */
  free (s_saved);
  return 0;                                    /* success */
}                                       /* parse_smail () */
/*========================================================*/
int process_mail (struct email_msg *m, struct inbox_line *ibl, char *mailhost,
                  char *localhost, char *defdom)
{
  struct servent *sent;
  struct in_addr server_ip;
  struct sockaddr_in sockaddr;
  struct rcpt_item *cursor;
  int sockfd;
  int addrlen;
  char *cmdline;
  char *nd;                              /* nicified date */
  int retval;

  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "===========================================\n");
    if (m->from)
      fprintf (stderr, "From: %s\n", m->from);
    if (!empty_rcpt_list (m->to)) {
      fprintf (stderr, "To: recepients\n");
      print_rcpt_list (m->to);
    }
    if (!empty_rcpt_list (m->cc)) {
      fprintf (stderr, "CC: recepients\n");
      print_rcpt_list (m->cc);
    }
    if (!empty_rcpt_list (m->bcc)) {
      fprintf (stderr, "BCC: recepients\n");
      print_rcpt_list (m->bcc);
    }
    if (m->reply_to)
      fprintf (stderr, "Reply-To: %s\n", m->reply_to);
    if (m->subject)
      fprintf (stderr, "Subject: %s\n", m->subject);
    fprintf (stderr, "===========================================\n");
  }
    
  /*--------------------------------------Initializations */
  cmdline = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (!cmdline) {    
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("process_mail(): can't malloc() cmdline");
  }
  cmdline[0] = '\0';

  /* add some system info to the mail body */
  sprintf (cmdline, "\r\n=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-\r\n");
  body_list_insert (&(m->body), cmdline);
  sprintf (cmdline, "This mail has been relayed to you by SMSLink's sms2mailgw\r\n");
  body_list_insert (&(m->body), cmdline);
  sprintf (cmdline, "ver. %s (%s) running on %s.%s.\r\n", SMS_GW_VERSION,
           SMS_GW_DATE, localhost, defdom);
  body_list_insert (&(m->body), cmdline);
  nd = nicedate (ibl->date);
  sprintf (cmdline, "The original message was received on %s at %s\r\n",
           nd, ibl->time);
  body_list_insert (&(m->body), cmdline);
  sprintf (cmdline, "from the GSM number [%s] through device [%s].\r\n", ibl->fromgsm,
           ibl->device);
  body_list_insert (&(m->body), cmdline);
  sprintf (cmdline, "The message ID was [%d].\r\n", ibl->msgid);
  body_list_insert (&(m->body), cmdline);
  
  /* then shoot it out */
  retval = send_mail (m, mailhost, localhost, defdom);
  
  /*------------------------------------------Conclusions */
  /* free what's need to be */
  free (nd);
  free (cmdline);
  
  /* leave - success */
  return (retval);
}                                      /* process_mail () */
/*========================================================*/
int mailbox_run (char *localhost, char *defaultdomain, char* mailhost)
{
  struct email_msg email;
  struct inbox_line tkline;            /* tokenized line */
  char *line;                                /* raw line */
  char *iboxentry;                    /* recomposed line */
  int iboxentrylen;           /* number of lines to skip */
  char *buffline;                       /* transfer line */
  FILE *inbox;
  FILE *newbox;
  int anymailfound = FALSE;
  int filecreatedyet = FALSE;
  int msg_count = 0;
  int lineismail;
  int lcursor;
  int nline = 0;
  int startofentry;
  int got_whole_entry = FALSE;
  char *cmd;
  char *newboxname;
  int i;
  int sysret;
  
  /*---------------------------------Initialize variables */
  line = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  if (! line) {
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("mailbox_run(): can't malloc() line");
  }
  line[0] = '\0';
  iboxentry = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  if (! iboxentry) {
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("mailbox_run(): can't malloc() iboxentry");
  }
  line[0] = '\0';
  buffline = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  if (! buffline) {
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("mailbox_run(): can't malloc() buffline");
  }
  buffline[0] = '\0';
  cmd = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (! cmd) {
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("mailbox_run(): can't malloc() cmd");
  }
  cmd[0] = '\0';
  newboxname = (char *) malloc ((PATH_MAX + 1) * sizeof (char));
  if (! newboxname) {
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("mailbox_run(): can't malloc() newboxname");
  }
  newboxname[0] = '\0';
  sprintf (newboxname, "%s.tmp.%d", MBOX_FILE, getpid());

  /*--------------------------------------------Open file */
  if ((inbox = fopen (MBOX_FILE, "r")) == NULL) {
    syslog ((FACILITY | LOG_ERR), "can't fopen() inbox file.");
    unlink (MBOX_LOCKF);
    syserr ("mailbox_run(): can't fopen() inbox file");
  }

  /*----------------------Read file and process each line */
  while (fgets (line, BIGBUFF, inbox) != NULL) {
    nline++;
    /* what kind of line is it ? (header, continuation, trailer) */
    if (is_ibox_header (line)) {
      strcpy (iboxentry, line);
      iboxentrylen = 1;
      startofentry = nline;
      got_whole_entry = FALSE;
      /* maybe whole entry in one line ? */
      if (is_ibox_trailer (line)) {
        got_whole_entry = TRUE;
      }
    }
    else if (is_ibox_trailer (line)) {
      strcat (iboxentry, line);
      iboxentrylen++;
      got_whole_entry = TRUE;
    }
    else {  /* is "continuation" line */
      iboxentrylen++;
      strcat (iboxentry, line);
    }
    
    /* when a full entry is gathered, process it */
    if (got_whole_entry) {
      /* tokenize line to fill a struct inbox_line */
      if (tkize_ibox_line (iboxentry, &tkline) == -1) {
	syslog ((FACILITY | LOG_ERR), "inbox corruption - can't parse line %d.", nline);
	unlink (MBOX_LOCKF);
	syserr ("mailbox_run(): inbox corruption - can't parse line");
      }

      /* is it a mail message ? */
      if ((toupper (tkline.text[0]) == 'T') && (tkline.text[1] == ':')) {
	if (parse_smail (&tkline, &email, localhost, defaultdomain) == -1) {
          /* reject line after all */
	  lineismail = FALSE;
	  /* log it */
          syslog ((FACILITY | LOG_NOTICE), "inbox line #%d doesn't comply with protocol.",
		 nline);
	}
	else {
	  if (! email.from) {
	    email.from = mkfromfield (tkline.fromgsm, localhost, defaultdomain);
	  }
          if (process_mail (&email, &tkline, mailhost, localhost, defaultdomain) == 0) {
            /* successfully sent */
            lineismail = TRUE;
            anymailfound = TRUE;
	    msg_count++;
	  }
	  else {
	    /* failed to hand mail over to MTA */
            lineismail = FALSE;      /* keeps it in inbox */
            syslog ((FACILITY | LOG_ERR), "inbox line #%d couldn't be handed over to MTA - will retry.",
	           nline);
	  }
	  reset_mail_struct (&email);
	}                         /* if (parse_smail (... */
      }
      else {
	lineismail = FALSE;
      }                            /* if (it is an email) */
      if (anymailfound) {
	if (! filecreatedyet) {
          /* open new inbox file */
	  if ((newbox = fopen (newboxname, "w")) == NULL) {
	    syslog ((FACILITY | LOG_ERR), "can't fopen() temp inbox file.");
            unlink (MBOX_LOCKF);
	    syserr ("mailbox_run(): can't fopen() temp inbox file");
	  }
          filecreatedyet = TRUE;
	  /* transfer up to current line - current not included */
          lcursor = 1;
          /* "rewind" input file */
          if (fseek (inbox, (long) 0, SEEK_SET) == -1) {
	    syslog ((FACILITY | LOG_ERR), "can't fseek() on inbox file.");
            unlink (MBOX_LOCKF);
	    syserr ("mailbox_run(): can't fseek() on inbox file");
          }
          while (lcursor < startofentry) {
            if (fgets (buffline, BIGBUFF, inbox) == NULL) {
	      syslog ((FACILITY | LOG_ERR), "can't read from inbox file.");
              unlink (MBOX_LOCKF);
	      syserr ("mailbox_run(): can't read from inbox file");
            }
            if (fputs (buffline, newbox) == EOF) {
	      syslog ((FACILITY | LOG_ERR), "can't write to new inbox file.");
              unlink (MBOX_LOCKF);
	      syserr ("mailbox_run(): can't write to new inbox file");
            }
            lcursor++;
          }             /* while (lcursor < startofentry) */
          /* re-read line #nline (and any continuation) to skip it */
	  for (i = 0; i < iboxentrylen; i++) {
            if (fgets (buffline, BIGBUFF, inbox) == NULL) {
	      syslog ((FACILITY | LOG_ERR), "can't read from inbox file.");
              unlink (MBOX_LOCKF);
	      syserr ("mailbox_run(): can't read from inbox file");
            }
	  }
	}                        /* if (! filecreatedyet) */
	if (! lineismail) {
          /* transfer current inbox entry to new file */
          if (fputs (iboxentry, newbox) == EOF) {
	    syslog ((FACILITY | LOG_ERR), "can't write to new inbox file.");
            unlink (MBOX_LOCKF);
	    syserr ("mailbox_run(): can't write to new inbox file");
          }
	}                            /* if (! lineismail) */
      }                              /* if (anymailfound) */
    }                             /* if (got_whole_entry) */
  }                                    /* while (fgets... */
  /*------------------------------Close file and conclude */
  if (fclose (inbox) != 0) {
    syslog ((FACILITY | LOG_ERR), "can't close inbox file.");
    unlink (MBOX_LOCKF);
    syserr ("mailbox_run(): can't close inbox file");
  }
  if (anymailfound) {
    if (fclose (newbox) != 0) {
      syslog ((FACILITY | LOG_ERR), "can't close newbox file.");
      unlink (MBOX_LOCKF);
      syserr ("mailbox_run(): can't close newbox file");
    }
    /* now mv newbox inbox */
    sprintf (cmd, "mv %s %s", newboxname, MBOX_FILE);
    if ((sysret = system (cmd)) != 0) {
      switch (sysret) {
        case -1:
	  syslog ((FACILITY | LOG_ERR), "call to system() failed.");
	  unlink (MBOX_LOCKF);
	  syserr ("mailbox_run(): call to system() failed");
	  break;
	
	default:
	  /* mv returned with WEXITSTATUS(sysret) */
	  syslog ((FACILITY | LOG_ERR), "can't move new inbox file - mv returned %d.",
	         WEXITSTATUS (sysret));
	  unlink (MBOX_LOCKF);
	  syserr ("mailbox_run(): can't move new inbox file - mv failed");
	  break;
      }                                /* switch (sysret) */
    }                   /* if ((sysret = system (cmd))... */
  }
  
  /* free what's needs to be */
  free (line);
  free (buffline);
  free (cmd);
  free (newboxname);
  free (iboxentry);
  
  return (msg_count);
}                                       /* mailbox_run () */
/*==========================================================
 * EOF : gw_stuff.c
 *===================*/
