/*==========================================================
 * Program : mbchecker.c                   Project : smslink
 * Author  : Philippe Andersson.
 * Date    : 31/08/06
 * Version : 0.48b
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 * Comment : Mailbox checking functions for the smslink server.
 *
 * Modification History :
 * - 0.01a (05/12/98) : Initial release.
 * - 0.02a (16/05/99) : Added include line for errno.h, to solve
 *   compilation problem on RedHat platform.
 * - 0.03a (06/06/99) : Started building the actual mailbox
 *   check procedure.
 * - 0.04b (28/06/99) : In mbcheck(), moved the AT+CNMI command
 *   after the PIN check routine. PIN-Activated SIM is required
 *   for AT+CNMI. Solved a bug with default SMSC.
 * - 0.05a (03/07/99) : Added list managment functions.
 * - 0.06a (11/07/99) : Added dumptofile() function.
 * ++++ Switch to Beta ++++
 * - 0.07b (20/07/99) : Improved mbcheck() to also remove the
 *   messages from the SIM after having saved them to disk.
 * - 0.08b (17/08/99) : Included code to create and update the
 *   checkpoint file after each successfull mailbox check (needed
 *   for interaction with the SMS to Mail gateway).
 * - 0.09b (02/12/99) : Improved handling for default SMSC in
 *   mbcheck(). Now uses the value kept in /etc/gsmdevices.
 * - 0.10b (25/02/00) : In mbcheck_wrapper(), moved the update
 *   of the checkpoint file outside of the device-checking loop
 *   (first because there's no need to update it more than once,
 *   second because it caused sms2mailgw to fire during the
 *   device checking loop).
 * - 0.11b (04/04/00) : Changed "at+cmgl=" syntax to always use
 *   the "text" parameter (Wavecom WM02 is strict about param.
 *   types in relation with text or PDU mode). Thanks to Guido
 *   Dolci <dolci@sfera.net> for this one. Also added "at+csdh"
 *   command (full headers - thanks to Piero Baudino <piero@aries.it>
 *   for pointing that out) and made sure we use text mode here
 *   (temp solution until full PDU mode support). All of this
 *   in mbcheck().
 * - 0.12b (06/04/00) : Improved mbcheck() by using the chosen
 *   device-level default for message format (aka "mode"). Has
 *   to be text for the meantime anyway.
 * - 0.13b (08/06/00) : In mbcheck(), converted all AT commands
 *   to uppercase. It seems that the command parser in some
 *   Wavecom devices is case-sensitive. Thanks to Andrea Vettori
 *   <av@tservicetlc.net> and Frank Friel <frank_friel@hotmail.com>
 *   for pointing that out.
 * - 0.14b (21/06/00) : In mbparse(), check for extra field (and
 *   discard it) when using a Falcom A2. Thanks to Pierluigi
 *   Mangani <pluigi@sif.it> for letting me know. Adapted
 *   mbcheck() to pass the capmatrix to mbparse(). More
 *   informative error message when parse fail.
 * - 0.15b (26/06/00) : In mbparse(), improved algorithm to
 *   handle case where none of the optional parameters are
 *   present.
 * - 0.16b (11/09/00) : Added dump_string() to help Guido Dolci
 *   <dolci@sfera.net> debug an incoming SMS parsing issue.
 * - 0.17b (11/09/00) : Some modules return text-mode SMS delimited
 *   with '\r', others with '\n' (e.g. WM02). Hence the introduction
 *   of the unixify() function to standardize on '\n'. Thanks to
 *   Guido Dolci <dolci@sfera.net> for his help in tracking this.
 * - 0.18b (18/10/00) : Added alternate CNMI handling for Siemens
 *   S25 / S35 (thanks to Thomas Omerzu <thomas@omerzu.de> for
 *   the fix).
 * - 0.19b (21/11/00) : Modified mbcheck() to correctly handle
 *   SMS reception in PDU mode. Added the mbparsePDU() function.
 * - 0.20b (23/11/00) : In mbcheck(), altered the AT+CPIN command
 *   by putting the PIN value between quotes. Closer to the
 *   standard and required by the PCFF900.
 * - 0.21b (24/11/00) : Modified mbcheck() to account for
 *   ALT_CNMI_2 and HAS_CSDH_1 (PCFF900). Kudos to Bernard Willemot 
 *   <bernard_willemot@cse.creoscitex.com> for lending me the
 *   hardware.
 * - 0.22b (18/02/01) : Modified mbcheck() to account for
 *   ALT_CNMI_3 (GM12). Modified mbparsePDU() to continue processing
 *   when dealing with invalid PDU string (to remove it from the
 *   SIM). Enhanced dumptofile(): now can also feed SMS to an
 *   external prog. Contributed by Andrew Worsley 
 *   <epaanwo@asac.ericsson.se>.
 * - 0.23b (20/02/01) : Added support for "fast modem response"
 *   and "message store" in mbcheck(). Cosmetics. Contributed by 
 *   Andrew Worsley <epaanwo@asac.ericsson.se>.
 * - 0.24b (06/03/01) : Replaced control of all debug info by
 *   global flags. Modified the way a new PIN is selected if we
 *   have to provide the PUK: now sets PIN to the one we have in
 *   conf in order to avoid the need to store the new one (thanks
 *   to Shay Kalderon <shay_kalderon@creoscitex.com> for the
 *   hint). Eventually implemented Andrew's hack to make sure
 *   all messages are always removed from the SIM card (created
 *   a device-level flag 'r4' to control it).
 * - 0.25b (30/03/01) : Moved dump_string() over to stuff.c.
 * - 0.26b (02/04/01) : Modified calls to get_gsm_answer() to
 *   include the struct gsms_def field.
 * - 0.27b (04/04/01) : Replaced calls to get_gsm_answer() by
 *   calls to get_gsm_resp() in mbcheck() when sending PIN code
 *   to try and solve "fast" mode problems.
 * - 0.28b (04/05/01) : Improved handling of +CREG-returned
 *   registration status in mbcheck().
 * - 0.29b (08/05/01) : Added a pause in mbcheck() after PIN
 *   processing when using "fast" mode to try and solve the
 *   "+CMS ERROR: 515" issue.
 * - 0.30b (07/10/01) : Replaced calls to blopen_mdm_line()
 *   by calls to lopen_mdm_line(). Advantage being that this
 *   version uses the max. supported speed by the device by
 *   default (the one defined in /etc/modems).
 * - 0.31b (20/03/01) : Enabled storage of incoming messages in
 *   a back-end MySQL database. Contributed by Ing. Andrea
 *   Vettori <a.vettori@inetronics.com>.
 * - 0.32b (30/04/01) : Added code to update stats counters in
 *   struct gsms_def, both in mbcheck_wrapper() and in
 *   outboxcheck_wrapper().
 * - 0.33b (05/01/02) : Improved debugging info in dumptodb().
 *   Replaced the MySQL-defined strmov() macro by the original
 *   stpcpy() function it calls (WARNING: non-ANSI/POSIX !).
 *   Improved error handling around the call to set_cpms().
 *   Replaced the AT+CMEE dialog with a call to set_cmee().
 *   Idem for AT+CPIN, AT+CMGF, AT+CSDH, AT+CREG & AT+CSCA.
 * - 0.34b (09/12/02) : Started to implement default GSM
 *   alphabet support. Implemented support for "init" mode
 *   (contributed by Ing. Andrea Vettori <a.vettori@inetronics.com>).
 *   Replaced static PIN post-processing sleep time by device-level
 *   param. (PINpause). Corrected parameter order in the
 *   mysql_real_connect() calls. Modularized outbox check.
 *   Replaced calls to mysql_escape_string() by calls to
 *   mysql_real_escape_string() in dumptodb(). Implemented the
 *   file queue processing method in the outbox check. Improved
 *   mbparse() to allow for <LF> chars in the message, as seen
 *   in those sent through websites (contributed by Marcel P.
 *   <marcel_p@canada.com>).
 * - 0.35b (10/06/03) : Fixed initialization bug for query string
 *   in dumptodb() (bug manifested itself under FreeBSD). Fixed
 *   the inbox insert query string to take the "stored" field
 *   into account.
 * - 0.36b (15/06/03) : Added "maxqruns" support to the outbox
 *   processing routine.
 * - 0.37b (28/07/03) : Added bounce_message() in support to
 *   queue item obsolescence admin. warning.
 * - 0.38b (03/09/03) : Corrected bug where the outbox.stored
 *   column would get reset to NOW() on update. Thanks to Andrew
 *   Goldschmidt (<andrew@rsaweb.co.za>) for pointing it out.
 * - 0.39b (25/11/03) : Added  test on NO_CREG_CHECK flag to
 *   support Motorola A008 device. Contributed by Nicki de Wet
 *   <nickidw@mighty.co.za>.
 * - 0.40b (11/12/03) : Replaced defined SMS_PROG_NAME by extern
 *   char *targetprog (--exec command-line parameter).
 * - 0.41b (05/04/04) : Added priority support to queued messages.
 *   Added "processed" field to dumptodb().
 * - 0.42b (12/11/04) : Improved SQL queries according to Marty's
 *   suggestion + solved SIGCHLD bug in 'prog' mode
 *   (<marty@upstart-training.co.uk>).
 * - 0.43b (12/01/05) : Added support for 'skip' flag.
 * - 0.44b (23/10/05) : Solved a silly bug in db_outboxcheck(),
 *   where 'gsm' was used before being initialized.
 * - 0.45b (10/11/05) : Solved the same bug in file_outboxcheck(),
 *   where 'gsm' was used before being initialized.
 * - 0.46b (29/11/05) : Solved a bug in dumptodb(), where cursor
 *   could have been used uninitialised in the case of a MySQL
 *   connection failure (bug spotted by Tarmo Kuuse
 *   <tarmo.kuuse@oskando.ee>).
 * - 0.47b (18/07/06) : Replaced sigignore() with signal(...,SIG_IGN)
 *   to restore compatibility with FreeBSD. Fix contributed by
 *   Peter Forgac (<forgacp@mail.t-mobile.sk>).
 * - 0.48b (31/08/06) : Prevent string overflow when processing
 *   DB queue items that were created outside of SMSLink
 *   (bug reported by Gabriele Zappi). Add missing terminating
 *   NULL in other calls to strncpy().
 *========================================================*/

#include <stdio.h>                         /* for fprintf */
#include <stdlib.h>                  /* for errno & stuff */
#include <ctype.h>                       /* for isdigit() */
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <syslog.h>
#include <signal.h>
#include <sys/stat.h>                      /* for open () */
#include <sys/param.h>                  /* for MAXPATHLEN */
#include <dirent.h>              /* for the struct dirent */
#include <sys/time.h>           /* for the struct timeval */
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ipc.h>
#include <sys/sem.h>                        /* semaphores */
#include <sys/shm.h>                     /* shared memory */
#include <sys/ioctl.h>            /* for the ioctl() call */
#include <termios.h>         /* for baudrates definitions */
#include <dial/modems.h>           /* requires 'libmodem' */
#ifdef WITH_MYSQL
#  include <mysql/mysql.h>	  /* for mysql client api */
#endif

#include "sms_serv.h"
#include "gsmdev.h"
#include "smtp.h"
#include "pdu.h"

/*========================================================*/
/* For debugging purposes only - comment out for normal compile */
/* #define INCL_DEBUG_CODE */

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/
extern int debug;                    /* debug level/flags */
extern int m_timeout;
extern int qmethod;
extern int dbaccess_enabled;
extern int default_alphabet;
int MBC_instance;
int MBC_instance_is_set = FALSE;

/*========================================================*/
/**********               FUNCTIONS                ********/
/*========================================================*/
int bounce_message (char *quid, struct symbols *sym)
/* From: sms_daemon
 * To: postmaster
 * Subject: message ID couldn't be sent
 * "Message <ID> submitted by <user> destined to <dest>
 * couldn't be delivered." */
{
  extern char *localhost;
  extern char *mailhost;
  extern char *defaultdomain;
  extern char *bounce_addr;
  extern int maxqruns;
  struct email_msg email;
  int retval;
  char *p;

  /*------------------------------initialize email struct */
  mail_struct_init (&email);
  
  /*---------------------------------fill-in struct email */
  /*................................................From: */
  p = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (! p) {
    syserr ("bounce_message(): can't malloc() p");
  }
  p[0] = '\0';
  sprintf (p, "%s@%s.%s", DEFLT_SENDER, localhost, defaultdomain);
  email.from = p;

  /*..................................................To: */
  rcpt_list_insert (&(email.to), bounce_addr);

  /*.............................................Subject: */
  p = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (! p) {
    syserr ("bounce_message(): can't malloc() p");
  }
  p[0] = '\0';
  sprintf (p, "SMS queue ID <%s> could not be sent", quid);
  email.subject = p;

  /*.................................................Body */
  p = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  if (! p) {
    syserr ("bounce_message(): can't malloc() p");
  }
  p[0] = '\0';
  sprintf (p, "Queued SMS message ID <%s>, submitted by user <%s>\r\n"
              "with the intended destination of GSM #<%s>\r\n"
	      "couldn't be delivered after %d attempts.\r\n\r\n"
	      "The queued message has been removed.\r\n",
	      quid, sym->user, sym->destgsm, maxqruns);
  body_list_insert (&(email.body), p);


  /* Now send it */
  retval = send_mail (&email, mailhost, localhost, defaultdomain);
  
  reset_mail_struct (&email);
  
  return (retval);
}                                    /* bounce_message () */
/*========================================================*/
void mbox_list_init (mbox_list *list)
{
  list->head = NULL;
  list->tail = NULL;
}                                    /* mbox_list_init () */
/*========================================================*/
int empty_mbox_list (mbox_list list)
{
  return (list.head == NULL);
}                                   /* empty_mbox_list () */
/*========================================================*/
void mbox_list_insert (mbox_list *list, struct mbox_item *element)
{
  /* WARNING : the order of the elements IS relevent - has to
   * be chronological => insert at tail. */
  
  /* chain the element in the list */
  if (empty_mbox_list (*list)) {
    list->head = element;
    list->tail = element;
    element->next = NULL;
    element->prev = NULL;
  }
  else {
    element->next = NULL;
    element->prev = list->tail;
    list->tail->next = element;
    list->tail = element;
  }
}                                  /* mbox_list_insert () */
/*========================================================*/
void free_mbox_list (mbox_list *list)
{
  struct mbox_item *cursor;

  if (!empty_mbox_list (*list)) {
    /* hop to element before last */
    cursor = list->tail->prev;
    /* now go back and clean behind */
    while (cursor != NULL) {
      free (cursor->next);
      cursor->next = NULL;
      list->tail = cursor;
      cursor = cursor->prev;
    }                           /* while (cursor != NULL) */
  }                          /* if (!empty_mbox_list (... */
  /* now clean last element and reset header */
  free (list->head);
  list->head = NULL;
  list->tail = NULL;
}                                    /* free_mbox_list () */
/*========================================================*/
void MBC_unlock_gsm ()
{
  extern int context;
  extern int shmem_sem;
  extern struct gsms_def *gsmdevices;
  extern struct symbols ob_symbols;

  switch (context) {
    case CONTEXT_MBCI: {
      syslog ((FACILITY | LOG_WARNING), "MBC IN being hit by SIGTERM.");
      break;
    }
    
    case CONTEXT_MBCO: {
      syslog ((FACILITY | LOG_WARNING), "MBC OUT being hit by SIGTERM.");
      break;
    }
    
  }                                   /* switch (context) */
  /* set the allocated GSM module back to 'free' status */
  if (MBC_instance_is_set) {
    if (sem_wait (shmem_sem) == -1)
      syserr ("MBC_unlock_gsm(): failed to wait on shared mem. semaphore");

    /* ---- Begin Crit. Sect. #5 ---- */
    if (!gsmdevices[MBC_instance].free) {
      gsmdevices[MBC_instance].free = TRUE;
      gsmdevices[MBC_instance].owner = 0;
      syslog ((FACILITY | LOG_WARNING), "GSM instance </dev/%s> has been unlocked.",
             gsmdevices[MBC_instance].device);
    }
    /* leave crit. sect. */
    if (sem_signal (shmem_sem) == -1)
      syserr ("MBC_unlock_gsm(): can't signal shared mem. semaphore");
    /* ---- End Crit. Sect. #5 ---- */
  }
  /* free what's need to be freed and exit */
  if (context == CONTEXT_MBCO) {
    free (ob_symbols.smsc);
    free (ob_symbols.destgsm);
    free (ob_symbols.message);
    if (ob_symbols.pdu) {
      free (ob_symbols.pdu);
    }
  }
  /*------------------------------------------------------*/
  exit (0);
}                                    /* MBC_unlock_gsm () */
/*========================================================*/
int all_done (int *array, int size)
{
  int retval = TRUE;
  int i;

  for (i = 0; i < size; i++) {
    retval = (retval && array[i]);
  }
  return (retval);
}                                          /* all_done () */
/*========================================================*/
struct mbox_item *mbparse (char *msg_string, cap_matrix capmatrix)
{
  struct mbox_item *msg;
  char *s;
  char *ptr;
  char *token;
  char *dataptr;
  char *q;
  char date[11];
  char time[10];
  char brol[5];
  int year, month, day;
  int datelen;
  int i;
  char okstr[TINYBUFF + 1];
  
  /*----------------Take a local copy of the input string */
  if ((s = (char *) malloc ((strlen (msg_string) + 1) * sizeof (char))) == NULL) {
    syslog ((FACILITY | LOG_ERR), "mbparse() - can't malloc.");
    return (NULL);
  }
  strcpy (s, msg_string);
  /*-----Take a local copy of the OKstring and unixify it */
  strcpy (okstr, capmatrix.modem_okay);
  unixify (okstr);
  /*-----------------------Allocate memory for the struct */
  if ((msg = (struct mbox_item *) malloc (sizeof (struct mbox_item))) == NULL) {
    syslog ((FACILITY | LOG_ERR), "mbparse() - can't malloc.");
    return (NULL);
  }
  /*----------------------------------------Initialize it */
  msg->msgid = 0;
  msg->fromgsm[0] = '\0';
  msg->date[0] = '\0';
  msg->time[0] = '\0';
  msg->text[0] = '\0';
  msg->next = NULL;
  msg->prev = NULL;
  /*----------------------------Parse message string copy */
  if (debug & DEBUG_HEXDUMP) {
    fprintf (stderr, "String <s> as received:\n");
    dump_string (s);
  }
  unixify (s);
  if (debug & DEBUG_HEXDUMP) {
    fprintf (stderr, "String <s> unixified:\n");
    dump_string (s);
  }
  /* ===========> First split-off the data <============= */
  if ((dataptr = strchr (s, '\n')) == NULL) {
    /* Internal structure error */
    syslog ((FACILITY | LOG_ERR), "mbparse() - can't parse SMS (data).");
    return (NULL);
  }
  dataptr[0] = '\0';           /* terminate header string */
  dataptr++;
  /*.........................................Message text */
  if (strlen (dataptr)) {
    /* Locate the true end of the output (in some cases, an  */
    /* "OK" follows the SMS being displayed). Warning: you   */
    /* don't want to loose any <LF> embedded in the message  */
    /* (as in those sent from websites -- thanks Marcel !)   */
    /* We search from the end of the message string to allow */
    /* for an "\nOK\n" embedded in the text.                 */
    q = dataptr + (strlen (dataptr) - strlen (okstr)) - 3;
    if ((ptr = strstr (q, okstr)) != NULL) {
      ptr[0] = '\0';
    }
    /* Message text - trim unwanted characters */
    trim (dataptr);
    if (strlen (dataptr) > MAXMSGLEN) {
      strncpy (msg->text, dataptr, MAXMSGLEN);
      msg->text[MAXMSGLEN] = '\0';
    }
    else {
      strcpy (msg->text, dataptr);
    }
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "data item, message text = [%s]\n", msg->text);
    }
  }                              /* if (strlen (dataptr)) */
  /* else: empty msg -- ok */
  
  /* ==========> Now deal with the headers <============= */
  /* "static" header first... */
  token = s;
  /*...........................................Message ID */
  if ((ptr = strchr (s, ',')) != NULL) {
    ptr[0] = '\0';
    /* Message ID - extract the ID token alone and atoi it */
    while (!isdigit (token[0]) && token[0]) {
      token++;
    }
    msg->msgid = atoi (token);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "token #1, msg ID = [%d]\n", msg->msgid);
    }
  }
  else {
    /* Internal structure error */
    syslog ((FACILITY | LOG_ERR), "mbparse() - can't parse SMS (ID).");
    return (NULL);
  }
  token = ptr + 1;
  /*.......................................Message Status */
  if ((ptr = strchr (token, ',')) != NULL) {
    ptr[0] = '\0';
    /* Message status - drop it */
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "token #2, msg status = [%s] (ignored)\n", token);
    }
  }
  else {
    /* Internal structure error */
    syslog ((FACILITY | LOG_ERR), "mbparse() - can't parse SMS (status).");
    return (NULL);
  }
  token = ptr + 1;
  /*....................................Sender GSM number */
  if ((ptr = strchr (token, ',')) != NULL) {
    ptr[0] = '\0';
  }
  else {
    /* might be the last field -- adjust ptr accordingly */
    ptr = (token + (strlen (token) - 1));
  }
  /* Sender GSM number - dequote it */
  dequote (token);
  if (strlen (token) <= MAXPHNUMLEN) {
    strcpy (msg->fromgsm, token);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "token #3, sender GSM = [%s]\n", msg->fromgsm);
    }
  }
  else {
    /* Internal structure error */
    syslog ((FACILITY | LOG_ERR), "mbparse() - can't parse SMS (oa/da).");
    return (NULL);
  }
  token = ptr + 1;
  /*.................................Variable header part */
  /* loop to parse them -- all are optional */
  while (token[0]) {
    if (token[0] == '"') {
      /*...........................................Quoted */
      /* search for matching quote */
      if ((ptr = strchr ((token + 1), '"')) != NULL) {
        /* ok - now skip to next */
	ptr++;
	/* if ',' or '\0' we're all set */
	if (ptr[0] == ',') {
	  /* there's more to come */
	  ptr[0] = '\0';
	}
	else {
	  if (ptr[0]) {
	    /* Internal structure error - unexpected char */
	    syslog ((FACILITY | LOG_ERR), "mbparse() - can't parse SMS (unexpected char.).");
	    return (NULL);
	  }
	  else {
	    /* end of string - adjust ptr */
	    ptr--;
	  }
	}
	/* now process the field */
	dequote (token);
	if (strlen (token)) {
          /* <scts> ? */
	  if (strlen (token) && (isdigit (token[0]))) {
	    /* Timestamp (<scts>) -- parse it */
	    if (strchr (capmatrix.datefmt, 'Y') != NULL) {
	      /* 4-digit year */
	      datelen = 10;
	    }
	    else {
	      /* 2-digit year */
	      datelen = 8;
	    }
	    /* split date from time */
	    strncpy (date, token, datelen);
	    date[datelen] = '\0';
	    token += (datelen + 1);
	    strncpy (time, token, 9);
	    time[9] = '\0';
	    /* Convert date format from "datefmt" to YYYYMMDD */
	    for (i = 0; i < 3; i++) {
	      switch (capmatrix.datefmt[i]) {
	        case 'd':
                  strncpy (brol, date, 2);         /* day */
		  brol[2] = '\0';
                  day = atoi (brol);
                  shiftleft (date, 3);
		  break;
		  
	        case 'm':
                  strncpy (brol, date, 2);       /* month */
		  brol[2] = '\0';
                  month = atoi (brol);
                  shiftleft (date, 3);
		  break;
		  
	        case 'y':
                  strncpy (brol, date, 2);        /* year */
		  brol[2] = '\0';
                  year = atoi (brol);
                  shiftleft (date, 3);
                  /* take care of the missing century */
                  if (year < 95) {
                    year += 2000;
		  }
                  else {
                    year += 1900;
		  }
		  break;
		  
	        case 'Y':
                  strncpy (brol, date, 4);        /* year */
		  brol[4] = '\0';
                  year = atoi (brol);
                  shiftleft (date, 5);
		  break;
		  
	        default:
	          syslog ((FACILITY | LOG_ERR), "mbparse() - invalid date format (%c).",
		         capmatrix.datefmt[i]);
	          return (NULL);
		  break;
		  
	      }                    /* switch (datefmt[i]) */
	    }                          /* for (i= 0 -> 2) */
            sprintf (msg->date, "%d%02d%02d", year, month, day);
	    /* Force time to HHhMMmSSs format */
	    time[2] = 'h';
	    time[5] = 'm';
	    time[8] = 's';
            /* Copy the time over */
            strcpy (msg->time, time);
            if (debug & DEBUG_PRINTF) {
              fprintf (stderr, "message date = [%s]\n", msg->date);
              fprintf (stderr, "message time = [%s]\n", msg->time);
            }
	  }
	}
	else {
          /* empty field */
          if (debug & DEBUG_PRINTF) {
            fprintf (stderr, "quoted empty field (ignored)\n");
          }
	}                          /* if (strlen (token)) */
      }
      else {
	/* Internal structure error - matching quote not found */
	syslog ((FACILITY | LOG_ERR), "mbparse() - can't parse SMS (unbalanced quote).");
	return (NULL);
      }
    }
    else {
      /*.........................................Unquoted */
      /* numerical field or "other" */
      if ((ptr = strchr (token, ',')) != NULL) {
        ptr[0] = '\0';
      }
      else {
        /* might be the last field - adjust ptr accordingly */
	ptr = (token + (strlen (token) - 1));
      }
      if (isdigit (token[0])) {
        /* <tooa/toda> or <length> */
	/* ignore for now */
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "numerical field = [%s] (ignored)\n", token);
        }
      }
      else {
        if (strlen (token)) {
          /* most likely structural error - log it but don't die */
          syslog ((FACILITY | LOG_NOTICE), "mbparse() - unknown field [%s].",
	         token);
	}
	else {
	  /*empty field -- ok */
          if (debug & DEBUG_PRINTF) {
            fprintf (stderr, "unquoted empty field (ignored)\n");
          }
	}                          /* if (strlen (token)) */
      }                        /* if (isdigit (token[0])) */
    }                             /* if (token[0] == '"') */
    token = ptr + 1;
  }                                   /* while (token[0]) */
  /*-------------------------------------------------Exit */
  free (s);
  return (msg);
}                                           /* mbparse () */
/*========================================================*/
struct mbox_item *mbparsePDU (char *msg_string, cap_matrix capmatrix)
{
  struct mbox_item *msg;
  char *s;
  char *ptr;
  char *token;
  char *dataptr;
  int pdulength;
  char *var1;
  char *var2;

  /*----------------Take a local copy of the input string */
  if ((s = (char *) malloc ((strlen (msg_string) + 1) * sizeof (char))) == NULL) {
    syslog ((FACILITY | LOG_ERR), "mbparsePDU() - can't malloc.");
    return (NULL);
  }
  strcpy (s, msg_string);
  /*-----------------------Allocate memory for the struct */
  if ((msg = (struct mbox_item *) malloc (sizeof (struct mbox_item))) == NULL) {
    syslog ((FACILITY | LOG_ERR), "mbparsePDU() - can't malloc.");
    return (NULL);
  }
  /*----------------------------------------Initialize it */
  msg->msgid = 0;
  msg->fromgsm[0] = '\0';
  msg->date[0] = '\0';
  msg->time[0] = '\0';
  msg->text[0] = '\0';
  msg->next = NULL;
  msg->prev = NULL;
  /*----------------------------Parse message string copy */
  if (debug & DEBUG_HEXDUMP) {
    fprintf (stderr, "PDU-String <s> as received:\n");
    dump_string (s);
  }
  unixify (s);
  if (debug & DEBUG_HEXDUMP) {
    fprintf (stderr, "PDU-String <s> unixified:\n");
    dump_string (s);
  }
  /* ===========> First split-off the data <============= */
  if ((dataptr = strchr (s, '\n')) == NULL) {
    /* Internal structure error */
    syslog ((FACILITY | LOG_ERR), "mbparsePDU() - can't parse SMS (data).");
    return (NULL);
  }
  dataptr[0] = '\0';           /* terminate header string */
  dataptr++;
  /*...........................................PDU String */
  if (strlen (dataptr)) {
    /* chop off any trailing junk */
    if ((ptr = strchr (dataptr, '\n')) != NULL) {
      ptr[0] = '\0';
    }
    /* Go decode the PDU string... */
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "data, raw PDU = [%s]\n", dataptr);
    }
    if (decode_pdu (dataptr, msg, capmatrix, default_alphabet) == -1) {
      syslog ((FACILITY | LOG_ERR), "mbparsePDU() - can't parse SMS (PDU).");
      /* don't return now -- we want to remove the bad PDU from the SIM */
      /* return (NULL); */
    }                      /* if (decode_pdu (...) == -1) */
  }
  else {
    /* missing PDU string -- should not happen */
    syslog ((FACILITY | LOG_ERR), "mbparsePDU() - missing PDU string.");
    /* don't return now -- we want to remove the bad PDU from the SIM */
    /* return (NULL); */
  }                              /* if (strlen (dataptr)) */
  
  /* ==========> Now deal with the headers <============= */
  /* "static" header first... */
  token = s;
  /*...........................................Message ID */
  if ((ptr = strchr (s, ',')) != NULL) {
    ptr[0] = '\0';
    /* Message ID - extract the ID token alone and atoi it */
    while (!isdigit (token[0]) && token[0]) {
      token++;
    }
    msg->msgid = atoi (token);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "token #1, msg ID = [%d]\n", msg->msgid);
    }
  }
  else {
    /* Internal structure error */
    syslog ((FACILITY | LOG_ERR), "mbparsePDU() - can't parse SMS (ID).");
    return (NULL);
  }
  token = ptr + 1;
  /*.......................................Message Status */
  if ((ptr = strchr (token, ',')) != NULL) {
    ptr[0] = '\0';
    /* Message status - drop it */
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "token #2, msg status = [%s] (ignored)\n", token);
    }
  }
  else {
    /* Internal structure error */
    syslog ((FACILITY | LOG_ERR), "mbparsePDU() - can't parse SMS (status).");
    return (NULL);
  }
  token = ptr + 1;
  /*.................................Variable header part */
  /* we can have either one or two fields now. If one, it */
  /* will be <length> (decimal, not null). If two, then   */
  /* first <alpha> (may be empty), then <length>.         */
  /*...................................1st variable field */
  var1 = var2 = NULL;
  if ((ptr = strchr (token, ',')) != NULL) {
    /* not the last field yet... */
    ptr[0] = '\0';
    var1 = token;
    /*.................................2nd variable field */
    token = ptr + 1;
    var2 = token;
  }
  else {
    /* last field */
    var1 = token;
  }
  /* we can now analyse those field(s) */
  if (var2) {
    /* <alpha> is var1 - drop it for now */
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "token #3, <alpha> = [%s] (ignored)\n", var1);
    }
    /* <length> is var2 */
    if (strlen (var2) > 0) {
      pdulength = atoi (var2);
      if (debug & DEBUG_PRINTF) {
        fprintf (stderr, "token #4, PDU length = [%d]\n", pdulength);
      }
    }
    else {
      syslog ((FACILITY | LOG_ERR), "mbparsePDU() - can't parse SMS (length).");
      return (NULL);
    }                          /*  if (length (var2) > 0) */
  }
  else {
    /* <length> is var1 - no <alpha> */
    if (strlen (var1) > 0) {
      pdulength = atoi (var1);
      if (debug & DEBUG_PRINTF) {
        fprintf (stderr, "token #3, PDU length = [%d]\n", pdulength);
      }
    }
    else {
      syslog ((FACILITY | LOG_ERR), "mbparsePDU() - can't parse SMS (length).");
      return (NULL);
    }                          /*  if (length (var1) > 0) */
  }                                          /* if (var2) */
  /*-------------------------------------------------Exit */
  free (s);
  return (msg);
}                                        /* mbparsePDU () */
/*========================================================*/
int runprog (char *prog, char *device, struct mbox_item *cursor)
/*
 * fork and invoke the given program with the given SMS message
 * It expects the program to exit with a zero exit status otherwise
 * it will log an error.
 */
{
  pid_t pid;
  int status;
  int exit_val;
  char buf[MAXINTBUF];

  if ((pid = fork ()) < 0) {
    syslog ((FACILITY | LOG_ERR), "runprog - can't fork : %m");
    return (-1);
  }
  if (pid == 0) {
    sprintf (buf, "%d", cursor->msgid);
    execlp (prog, prog, device, buf, cursor->fromgsm,
	   cursor->date, cursor->time, cursor->text, NULL);
    syslog ((FACILITY | LOG_ERR), "excelp of %s failed : %m", prog);
    exit (1);
  }
  if (waitpid (pid, &status, 0) < 0) {
    syslog ((FACILITY | LOG_ERR), "waitpid on child failed : %m");
    return (-2);
  }
  if (WIFEXITED(status) == 0) {
    if (WIFSIGNALED(status)) 
      syslog ((FACILITY | LOG_ERR),
	     "program %s pid = %d terminated by signal %d", prog, pid,
	     WTERMSIG(status));
    else
      syslog ((FACILITY | LOG_ERR),
	     "program %s exited abnormally : %d", prog, pid);
    return (-2);
  }
  if (exit_val = WEXITSTATUS(status)) {
    syslog ((FACILITY | LOG_ERR),
	   "program %s pid=%d exited with non-zero status %d", prog, pid,
	   exit_val);
    return (-3);
  }
  syslog ((FACILITY | LOG_INFO),
         "SMS sent to program id=%d from=%s date=%s time=%s",
         cursor->msgid, cursor->fromgsm, cursor->date, cursor->time);
  return (0);
}                                           /* runprog () */
/*========================================================*/
#ifdef WITH_MYSQL
int dumptodb (mbox_list *list, char *device) {

  extern char *db_dbhost;
  extern char *db_dbname;
  extern char *db_username;
  extern char *db_password;

  int nline = 0;
  struct mbox_item *cursor;
  MYSQL mysql;
  char query[1000], *end;

  mysql_init (&mysql);

  cursor = list->head;

  if (mysql_real_connect (&mysql, db_dbhost, db_username, db_password, db_dbname, 0, NULL, 0)) {
    syslog ((FACILITY | LOG_INFO), "Connected to mysql as %s@%s", db_username, db_dbname);
    while (cursor != NULL) {
      /* reset query string */
      memset (query, 0, 1000);
      /* create INSERT sql statement */
      end = (char *) l_stpcpy (query, "INSERT INTO inbox (device, fromgsm, msgdate, msgtime, msgtext, processed) VALUES (");
      *end++='\'';
      end += mysql_escape_string (end, device, strlen(device));
      *end++='\''; *end++=','; *end++='\'';
      end += mysql_escape_string (end, cursor->fromgsm, strlen(cursor->fromgsm));
      *end++='\'';
      *end++=',';
      *end++='\'';
      end += mysql_escape_string (end, cursor->date, strlen(cursor->date));
      *end++='\'';
      *end++=',';
      *end++='\'';
      end += mysql_escape_string (end, cursor->time, strlen(cursor->time));
      *end++='\'';
      *end++=',';
      *end++='\'';
      end += mysql_escape_string (end, cursor->text, strlen(cursor->text));
      *end++='\'';
      *end++=',';
      *end++='\'';
      *end++='n';
      *end++='\'';
      *end++=')';

      if (debug & DEBUG_DBACCESS) {
        fprintf (stderr, "Now submitting following query:\n[%s].\n", query);
	if (debug & DEBUG_HEXDUMP) {
	  fprintf (stderr, "Hexdump of the query string:\n");
	  dump_string (query);
	}
      }
      
      if (mysql_real_query (&mysql, query, (unsigned int)(end-query))) {
        syslog ((FACILITY | LOG_ERR), "Mysql query error code: %s", mysql_error(&mysql));
        syslog ((FACILITY | LOG_ERR), "Unable to store message from %s, date %s, time %s, text %s",
	       cursor->fromgsm, cursor->date, cursor->time, cursor->text);
      }
      nline++;
      cursor = cursor->next;
    }
  }
  else {
    syslog ((FACILITY | LOG_ERR), "Can't connect to mysql DB %s. Error code: %s",
           db_dbname, mysql_error(&mysql));
    if (cursor != NULL) {
      syslog ((FACILITY | LOG_ERR), "Unable to store message from %s, date %s, time %s, text %s",
             cursor->fromgsm, cursor->date, cursor->time, cursor->text);
    }
  }

  mysql_close(&mysql);
  return (nline);
}                                          /* dumptodb () */
#endif
/*========================================================*/
int dumptofile (mbox_list *list, char *device, struct gsms_def *gsm)
{
  int nline = 0;
  FILE *mbox_file;
  int lockf_desc;
  struct mbox_item *cursor;
  int save_errno;
  extern char *targetprog;

  /*-------------------------Feed SMS to an external prog */
  if (gsm->flags & SMS_PROG) {
    /* always run program for testing */
    cursor = list->head;
    while (cursor != NULL) {
      if (runprog (targetprog, device, cursor) < 0) {
	syslog ((FACILITY | LOG_ERR),
	       "Unable to submit message %d (from %s)",
	       cursor->msgid, cursor->fromgsm);
      }
      cursor = cursor->next;
      nline++;
    }                           /* while (cursor != NULL) */
  }                         /* if (gsm->flags & SMS_PROG) */

  /*------------------------------Store SMS in a MySQL DB */
  if (gsm->flags & SMS_DB) {
#ifdef WITH_MYSQL
    nline = dumptodb (list, device);
#endif
  }
  
  /*-------------------------------------Dump SMS to file */
  if (gsm->flags & SMS_INBOX) {
    nline = 0;
    lockf_desc = open (MBOX_LOCKF, (O_RDWR | O_CREAT | O_EXCL), 0444);
    if (lockf_desc == -1) {
      syslog ((FACILITY | LOG_ERR), "can't lock the inbox file.");
      mdmperror ("sms_serv: can't lock the inbox file");
    }
    else {
      /* file is now locked */
      if ((mbox_file = fopen (MBOX_FILE, "a")) != NULL) {
	cursor = list->head;
	while (cursor != NULL) {
          fprintf (mbox_file, "/dev/%s,%d,%s,%s,%s,\"%s\"\n",
	          device, cursor->msgid, cursor->fromgsm,
		  cursor->date, cursor->time, cursor->text);
	  nline++;
	  cursor = cursor->next;
	}                       /* while (cursor != NULL) */
	fclose (mbox_file);
      }
      else {
	syslog ((FACILITY | LOG_ERR), "can't open the inbox file.");
	mdmperror ("sms_serv: can't open the inbox file");
      }
      /* Now remove lock file */
      close (lockf_desc);
      if (unlink (MBOX_LOCKF) == -1) {
	syslog ((FACILITY | LOG_ERR), "can't remove the lock file.");
	mdmperror ("sms_serv: can't remove the lock file");
      }
    }                            /* if (lockf_desc == -1) */
  }                        /* if (gsm->flags & SMS_INBOX) */

  /*------------------------------------------Conclusions */
  return (nline);
}                                        /* dumptofile () */
/*========================================================*/
int mbcheck (struct gsms_def *gsm)
{
  extern int use_fast;      /* enable fast modem response */
  int use_init;
  int fd, retval = 0;
  int nmsgin = 0;
  char *scratch;
  char *p1;
  char *p2;
  char *cmsgid;
  int nread;
  int pin_just_in;
  int msgid;
  struct mbox_item *message;
  mbox_list mailbox;
  cpl_list cpl;
  
  /*--------------------------------------Initializations */
  scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  if (!scratch) {
    syslog ((FACILITY | LOG_ERR), "can't allocate scratch space -- dying.");
    syserr ("mbcheck(): can't allocate scratch space");
  }
  memset (scratch, 0, (BIGBUFF + 1));

  use_fast = (gsm->flags & FAST_RESPONSE);
  use_init = (gsm->flags & SMS_INIT);

  mbox_list_init (&mailbox);
  
  pin_just_in = FALSE; /* set to true if PIN needed be provided */

  /* open modem line */
  fd = lopen_mdm_line (gsm->device);
  if (fd < 0) {
    syslog ((FACILITY | LOG_ERR), "call to lopen_mdm_line() failed.");
    mdmperror ("sms_serv: lopen_mdm_line() failed");
    free (scratch);
    return (-1);
  }
  
  /*------------set GSM to "verbose" error reporting mode */
  if (!use_init) {
    if (set_cmee (gsm, fd, 0) == -1) {
      return (-1);
    }
  }
  
  /*---------------------------then, check for SIM status */
  if (!use_init) {
    if (checkset_cpin (gsm, fd, 0, &pin_just_in) == -1) {
      return (-1);
    }
  }
  
  /*--------------------------------------sleep if needed */
  /* There seems to be a timing issue in "fast" mode when
   * the AT+CNMI command is issued too fast after the last
   * AT+CPIN? and we get a "+CMS ERROR: 515" (= module
   * not ready or busy processing command). Try sleeping a
   * few secs to solve that. */
  if (use_fast && pin_just_in) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "MBC: (pause %d secs. for PIN code post-processing)\n",
              gsm->PINpause);
    }
    sleep (gsm->PINpause);
  }
  
  /*------------set "no notify for incoming SMS messages" */
  if (!use_init) {
    if (set_cnmi_off (gsm, fd, 0) == -1) {
      return (-1);
    }
  }
  
  /*-----------set message format to device-level default */
  if (checkset_cmgf (gsm, fd, 0, gsm->defmode) == -1) {
    return (-1);
  }
  
  /*-------------------------set "full SMS header detail" */
  /* Beware: only meaningfull (and supported) in text mode AND */
  /* even so only on specific models ! (specifically not the */
  /* PCFF900). */
  if (!use_init) {
    if ((gsm->defmode == SMS_MODE_TEXT) &&
	(gsm->capmatrix.capflags & HAS_CSDH_1)) {
      if (set_csdh (gsm, fd, 0) == -1) {
	return (-1);
      }
    }                 /* if (gsm->defmode == SMS_MODE_TEXT) */
  }
  
  /*----------------Check stored SCA against default SMSC */
  if (checkset_csca (gsm, fd, 0, gsm->defsca) == -1) {
    return (-1);
  }
  
  /*---------------------Am I registered on the network ? */
  if (!(gsm->capmatrix.capflags & NO_CREG_CHECK))
  {
    if (check_creg (gsm, fd, 0) == -1) {
      return (-1);
    }
  }
  
  /*------------------------------------Set message store */
  /* make sure the message store is accessible */
  if (!use_init) {
    if (gsm->capmatrix.capflags & SET_MSG_STORE) {
      if (set_cpms (gsm, fd, 0) == -1) {
	return (-1);
      }
    }
  }

  /*-----------------------Now loop and read incoming SMS */
  if (gsm->defmode == SMS_MODE_TEXT) {
    if (gsm->flags & SMS_READ_ALL) {
      /* hack to clear messages in SIM memory */
      sprintf (scratch, "AT+CMGL=\"ALL\"\r");
    }
    else {
      sprintf (scratch, "AT+CMGL=\"REC UNREAD\"\r");
    }
  }
  else {
    /* SMS_MODE_PDU */
    if (gsm->flags & SMS_READ_ALL) {
      /* hack to clear messages in SIM memory */
      sprintf (scratch, "AT+CMGL=4\r");
    }
    else {
      sprintf (scratch, "AT+CMGL=0\r");
    }
  }                 /* if (gsm->defmode == SMS_MODE_TEXT) */
  tell_gsm (fd, scratch);
  memset (scratch, 0, (BIGBUFF + 1));
  if (get_gsm_answer (fd, scratch, BIGBUFF, 1, gsm)) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "%s\n", scratch);
    }
    /*....................look for "+CMGL:" in GSM answer */
    p1 = scratch;
    while (p1 && ((p1 = strstr (p1, "+CMGL:")) != NULL)) {
      /* got new message - isolate it from the batch */
      p2 = (p1 + 6);
      if ((p2 = strstr (p2, "+CMGL:")) != NULL) {
        /* there's another one behind */
	p1[((p2 - p1) - 1)] = '\0';
      }
      if (gsm->defmode == SMS_MODE_TEXT) {
        message = mbparse (p1, gsm->capmatrix);
      }
      else {
        /* SMS_MODE_PDU */
        message = mbparsePDU (p1, gsm->capmatrix);
      }             /* if (gsm->defmode == SMS_MODE_TEXT) */
      if (message == NULL) {
	mdm_unlock (mdmopendevice);
	hangup (fd);
	free (scratch);
	syslog ((FACILITY | LOG_ERR), "mbparse() failed - internal error.");
	mdmperror ("sms_serv: mbparse() failed - internal error");
	return (-1);
      }
      mbox_list_insert (&mailbox, message);
      nmsgin++;
      p1 = p2;
    }                              /* while ((msgstart... */
    if (nmsgin) {
      /*............dump all messages to the mailbox file */
      if (dumptofile (&mailbox, gsm->device, gsm) != nmsgin) {
	syslog ((FACILITY | LOG_ERR), "failed to save or forward messages.");
	mdmperror ("sms_serv: failed to save or forward messages");
      }
      /*.................remove the messages from the SIM */
      message = mailbox.head;
      while (message != NULL) {
	sprintf (scratch, "AT+CMGD=%d\r", message->msgid);
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_answer (fd, scratch, BIGBUFF, 1, gsm)) {
          if (debug & DEBUG_PRINTF) {
	    fprintf (stderr, "%s\n", scratch);
          }
	  /* check for "OK" */
	  if (strstr (scratch, "OK") == NULL) {
	    mdm_unlock (mdmopendevice);
	    hangup (fd);
	    free (scratch);
	    syslog ((FACILITY | LOG_ERR), "error after sending AT+CMGD command.");
	    mdmperror ("sms_serv: error after sending AT+CMGD command");
	    return (-1);
	  }
	}
	else {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	  free (scratch);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
        message = message->next;
      }                        /* while (message != NULL) */
      /*...............clean the mailbox list from memory */
      free_mbox_list (&mailbox);
    }                                      /* if (nmsgin) */
  }
  else {
    mdm_unlock (mdmopendevice);
    hangup (fd);
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
    mdmperror ("sms_serv: GSM not responding");
    return (-1);
  }
  
  /*----------------------------------Close communication */
  mdm_unlock (mdmopendevice);
  hangup (fd);
  free (scratch);
  return (nmsgin);
}                                           /* mbcheck () */
/*========================================================*/
void mbcheck_wrapper ()
{
  extern int errno;
  extern int global_sem;
  extern int shmem_sem;
  extern struct gsms_def *gsmdevices;
  extern int ngsmdevs;     /* num. configured GSM devices */
  int nwaitsecs = 0;
  int last_errno;
  int retval;
  char device[MAXDEVLEN];
  struct gsms_def gsm;
  int msgin;
  int i;
  int cpfd;                     /* checkpoint file descr. */
  int gsm_done[ngsmdevs];
  int maxiterat;
  int iteration;
  char *missedg;
  struct sigaction sa_c;

  /*--------------------------------------Initializations */
  for (i = 0; i < ngsmdevs; i++) {
    gsm_done[i] = FALSE;
  }
  maxiterat = (2 * ngsmdevs);
  iteration = 0;
  
  /* We need to mask out child signals from processes we fork
   * through "prog" mode, or they will kill us when terminating
   * and trouble will ensue.
   */
  signal (SIGCHLD,SIG_IGN);
  
  /*--------------------------------------------Main loop */
  while ((!all_done (gsm_done, ngsmdevs)) && (iteration < maxiterat)) {
    syslog ((FACILITY | LOG_INFO), "MBC waits for a free GSM instance (pass %d)...",
           (iteration + 1));
  
    while (((retval = sem_decreq (global_sem)) == -1) &&
          (errno == EAGAIN) &&
	  (nwaitsecs < m_timeout)) {
      sleep (W_STEP);
      nwaitsecs += W_STEP;
    }                                      /* while (...) */
    last_errno = errno;
  
    if (retval == -1) {          /* failed to get a modem */
      switch (last_errno) {
        case EAGAIN: {
          syslog ((FACILITY | LOG_WARNING), "MBC timeout expired (all GSMs busy).");
          break;
        }

        default: {
          syserr ("[MBC] mbcheck_wrapper(): can't decrease global semaphore");
	  break;
        }
      }                            /* switch (last_errno) */
    }
    else {               /* ---- Begin Crit. Sect. #1 --- */
      /* at least 1 GSM module is free - find it */
      if (sem_wait (shmem_sem) == -1)
        syserr ("[MBC] mbcheck_wrapper(): failed to wait on shared mem. semaphore");

      /* ---- Begin Crit. Sect. #2 ---- */
      /* first catch SIGTERM to handle child being killed while GSM's locked */
      sa_c.sa_handler = MBC_unlock_gsm;
      if (sigfillset (&sa_c.sa_mask) == -1)
        syserr ("[MBC] mbcheck_wrapper(): can't fill signal set");
      sa_c.sa_flags = 0;
      if (sigaction (SIGTERM, &sa_c, NULL) == -1)
        syserr ("[MBC] mbcheck_wrapper(): can't catch SIGTERM");

      /* search for MBC_instance both free and not yet processed */
      MBC_instance = 0;
      while (MBC_instance < ngsmdevs) {
	if (gsmdevices[MBC_instance].free && !gsm_done[MBC_instance]) {
	  /* let's process this one */
	  break;
	}
        MBC_instance++;
      }                     /* while (MBC_instance < ngsmdevs) */
      if (MBC_instance < ngsmdevs) {
        /* got it - now lock it (device lockfile handled by libmodem) */
        MBC_instance_is_set = TRUE;
        gsmdevices[MBC_instance].free = FALSE;
        gsmdevices[MBC_instance].owner = getpid ();
      
        /* copy to "local" space, to avoid the shmem balagan */
        if (gsmdevcpy (&gsm, &gsmdevices[MBC_instance]) == -1)
          syserr ("[MBC] mbcheck_wrapper(): error copying GSM MBC_instance");
      
        /* leave crit. sect. */
        if (sem_signal (shmem_sem) == -1)
          syserr ("[MBC] mbcheck_wrapper(): can't signal shared mem. semaphore");
        /* ---- End Crit. Sect. #2 ---- */

        if (gsm.flags & SMS_SKIP) {
	  /* log it */
          syslog ((FACILITY | LOG_NOTICE), "MBC skipping device </dev/%s> as requested.", gsm.device);
	}
	else {
	  /* log it */
          syslog ((FACILITY | LOG_NOTICE), "MBC now polling device </dev/%s>.", gsm.device);

          /* ---->>> Actual GSM Dialogue <<<---- */
          if ((msgin = mbcheck (&gsm)) != -1) {
            /* log it */
            syslog ((FACILITY | LOG_NOTICE), "MBC check ok, got %d msg from </dev/%s>.", msgin, gsm.device);
          }
          else { /* some error occured */
            syslog ((FACILITY | LOG_WARNING), "MBC ERROR: failed to poll </dev/%s>.", gsm.device);
          }
	}                    /* if (gsm.flags & SMS_SKIP) */

	/* mark this GSM as "done" */
	gsm_done[MBC_instance] = TRUE;
    
        /* now free the modem again */
        if (sem_wait (shmem_sem) == -1)
          syserr ("[MBC] mbcheck_wrapper(): failed to wait on shared mem. semaphore");

        /* ---- Begin Crit. Sect. #3 ---- */
        gsmdevices[MBC_instance].free = TRUE;
        gsmdevices[MBC_instance].owner = 0;
	if (msgin != -1) {
	  gsmdevices[MBC_instance].in_ok += msgin;
	}
	else {
	  /* that count might not be accurate ! */
	  gsmdevices[MBC_instance].in_fail++;
	}
        MBC_instance_is_set = FALSE;
        /* leave crit. sect. */
        if (sem_signal (shmem_sem) == -1)
          syserr ("[MBC] mbcheck_wrapper(): can't signal shared mem. semaphore");
        /* ---- End Crit. Sect. #3 ---- */

        /* leave main crit. sect. */
        if (sem_signal (global_sem) == -1)
          syserr ("[MBC] mbcheck_wrapper(): can't signal global semaphore");
	/* ---- End Crit. Sect. #1 ---- */
      }
      else {
        /* leave crit. sect. */
        if (sem_signal (shmem_sem) == -1)
          syserr ("[MBC] mbcheck_wrapper(): can't signal shared mem. semaphore");
        /* ---- End Crit. Sect. #2 ---- */

        /* leave main crit. sect. - release unused resource */
        if (sem_signal (global_sem) == -1)
          syserr ("[MBC] mbcheck_wrapper(): can't signal global semaphore");
	/* ---- End Crit. Sect. #1 ---- */

        syslog ((FACILITY | LOG_WARNING), "MBC all free GSM's already processed");
      }                   /* if (MBC_instance < ngsmdevs) */
    }                                /* if (retval == -1) */
    iteration++;
  }                               /* while ((!all_done... */
  /*-------------------------------------------Conclusion */
  /* update timestamp on checkpoint file */
  if ((cpfd = open (CHECKPOINTF, O_RDWR | O_CREAT | O_TRUNC, 0644)) == -1) {
    perror ("sms_serv(MBC): can't update checkpoint file");
    syslog ((FACILITY | LOG_WARNING), "MBC can't update checkpoint file.");
  }
  close (cpfd);
  
  /* Based on gsm_done, report on missed MBC_instances */
  if (all_done (gsm_done, ngsmdevs)) {
    /* all done */
    syslog ((FACILITY | LOG_NOTICE), "MBC all devices processed.");
  }
  else {
    /* missed some */
    missedg = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
    missedg[0] = '\0';
    for (i = 0; i < ngsmdevs; i++) {
      if (!gsm_done[i]) {
        strcat (missedg, gsmdevices[i].device);
	strcat (missedg, ", ");
      }
    }
    /* remove trailing ", " sequence */
    missedg[strlen (missedg) - 2] = '\0';
    syslog ((FACILITY | LOG_WARNING), "MBC poll missed {%s}.", missedg);
    free (missedg);
  }
}                                   /* mbcheck_wrapper () */
/*========================================================*/
void db_outboxcheck ()
{
#ifdef WITH_MYSQL
  extern char *db_dbhost;      /* MySQL access parameters */
  extern char *db_dbname;
  extern char *db_username;
  extern char *db_password;
  
  extern int ngsmdevs;
  extern struct symbols ob_symbols;
  extern int maxqruns;

  extern int obchklimit;
  extern int global_sem;
  extern int shmem_sem;
  extern struct gsms_def *gsmdevices;
  int nwaitsecs = 0;
  int last_errno;
  int retval;
  char device[MAXDEVLEN];
  struct gsms_def gsm;
  int msgid;
  struct sigaction sa_c;
  int reqmode;
  int qrun_counter;
  my_ulonglong id;
  char quidstr[TINYBUFF];

  MYSQL mysql;
  MYSQL_RES *result;
  MYSQL_ROW row;
  char query[1000];

  /*--------------------------------------Initializations */
  syslog ((FACILITY | LOG_NOTICE), "DB outbox check begins.");
  mysql_init (&mysql);
  query[0] = '\0';
  
  if (mysql_real_connect (&mysql, db_dbhost, db_username, db_password, db_dbname, 0, NULL, 0)) {
    syslog ((FACILITY | LOG_INFO), "DB outbox check: Connected to mysql as %s@%s",
           db_username, db_dbname);
  }
  else {
    syslog ((FACILITY | LOG_ERR), "DB outbox check: Can't connect to mysql. Error code: %s",
           mysql_error (&mysql));
    return;
  }
  
  /*-------------------------------------------Processing */
  /* get messages from outbox */
  if (obchklimit) {
    sprintf (query, "SELECT id, smsc, togsm, userid, mode, msgtext, priority "
           "FROM outbox WHERE status=0 "
	   "ORDER BY priority ASC, id ASC LIMIT %d", obchklimit);
  }
  else {
    l_stpcpy (query, "SELECT id, smsc, togsm, userid, mode, msgtext "
           "FROM outbox WHERE status=0 "
	   "ORDER BY priority ASC, id ASC");
  }

  if (mysql_query (&mysql, query)) {
    syslog ((FACILITY | LOG_ERR), "DB outbox check: cannot query outbox. MySQL error code: %s",
           mysql_error (&mysql));
  }
  else {
    result = mysql_store_result (&mysql);
    syslog ((FACILITY | LOG_NOTICE), "DB outbox check: messages to send: %d",
           mysql_num_rows (result));

    /* loop on outbox messages */
    while ((row = mysql_fetch_row (result)) != NULL) {

      /* parse the return of the SELECT statement */
      quidstr[0] = '\0';
      strcpy (quidstr, row[0]);
      /*..........................................destgsm */
      if (row[2] == NULL) {
        strcpy (ob_symbols.destgsm, "\0");
      }
      else {
        if (strlen (row[2]) > MAXPHNUMLEN) {
	  strncpy (ob_symbols.destgsm, row[2], MAXPHNUMLEN);
	  /* Terminate it manually. */
	  ob_symbols.destgsm[MAXPHNUMLEN] = '\0';
          syslog ((FACILITY | LOG_WARNING), "DB outbox check: destgsm for QID %s was too long. Got truncated!",
                 quidstr);
	}
	else {
	  strcpy (ob_symbols.destgsm, row[2]);
	}
      }
      /*..........................................message */
      if (row[5] == NULL) {
        strcpy (ob_symbols.message, "\0");
      }
      else {
        if (strlen (row[5]) > MAXMSGLEN) {
	  strncpy (ob_symbols.message, row[5], MAXMSGLEN);
	  /* Terminate it manually. */
	  ob_symbols.message[MAXMSGLEN] = '\0';
          syslog ((FACILITY | LOG_WARNING), "DB outbox check: message for QID %s was too long. Got truncated!",
                 quidstr);
	}
	else {
	  strcpy (ob_symbols.message, row[5]);
	}
      }
      /*.............................................smsc */
      if (row[1] == NULL) {
        strcpy (ob_symbols.smsc, DEFAULTSMSC);
      }
      else {
        if (strlen (row[1]) > MAXPHNUMLEN) {
	  strncpy (ob_symbols.smsc, row[1], MAXPHNUMLEN);
	  /* Terminate it manually. */
	  ob_symbols.smsc[MAXPHNUMLEN] = '\0';
          syslog ((FACILITY | LOG_WARNING), "DB outbox check: smsc for QID %s was too long. Got truncated!",
                 quidstr);
	}
	else {
	  strcpy (ob_symbols.smsc, row[1]);
	}
      }
      /*.............................................user */
      if (row[3] == NULL) {
        strcpy (ob_symbols.user, "\0");
      }
      else {
        if (strlen (row[3]) > MAXUIDLEN) {
	  strncpy (ob_symbols.user, row[3], MAXUIDLEN);
	  /* Terminate it manually. */
	  ob_symbols.user[MAXUIDLEN] = '\0';
          syslog ((FACILITY | LOG_WARNING), "DB outbox check: user for QID %s was too long. Got truncated!",
                 quidstr);
	}
	else {
	  strcpy (ob_symbols.user, row[3]);
	}
      }
      /*.............................................mode */
      ob_symbols.mode = ((row[4] == NULL) ? SMS_MODE_DEFAULT : atoi (row[4]));
      if ((ob_symbols.mode < SMS_MODE_DEFAULT) ||
         (ob_symbols.mode > SMS_MODE_TEXT)) {
	syslog ((FACILITY | LOG_ERR), "DB outbox check: unsupported mode value "
	       "<%d> in queue ID <%s>. Mode set to device-level default.",
	       ob_symbols.mode, quidstr);
        ob_symbols.mode = SMS_MODE_DEFAULT;
      }
      /*.........................................priority */
      ob_symbols.priority = atoi (row[6]);

      /* Obtain a modem instance */
      while (((retval = sem_decreq (global_sem)) == -1) &&
            (errno == EAGAIN) &&
	    (nwaitsecs < m_timeout)) {
	sleep (W_STEP);
	nwaitsecs += W_STEP;
      }                                    /* while (...) */
      last_errno = errno;
      if (retval == -1) {        /* failed to get a modem */
	switch (last_errno) {
	  case EAGAIN: {
            syslog ((FACILITY | LOG_WARNING), "DB outbox check: timeout expired (all GSMs busy).");
            break;
	  }
	  default: {
            syserr ("db_outboxcheck(): can't decrease global semaphore");
	    break;
	  }
	}                          /* switch (last_errno) */
      }
      else {             /* ---- Begin Crit. Sect. #1 --- */
	/* at least 1 GSM module is free - find it */
	if (sem_wait (shmem_sem) == -1) {
	  syserr ("db_outboxcheck(): failed to wait on shared mem. semaphore");
	}

	/* ---- Begin Crit. Sect. #2 ---- */
	/* Signal Handling */
	sa_c.sa_handler = MBC_unlock_gsm;
	if (sigfillset (&sa_c.sa_mask) == -1) {
	  syserr ("db_outboxcheck(): can't fill signal set");
	}
	sa_c.sa_flags = 0;
	/* catch SIGTERM to handle child's death while GSM's locked */
	if (sigaction (SIGTERM, &sa_c, NULL) == -1) {
	  syserr ("db_outboxcheck(): can't catch SIGTERM");
	}
	/* catch SIGPIPE to handle client disconnect while GSM's locked */
	if (sigaction (SIGPIPE, &sa_c, NULL) == -1) {
	  syserr ("db_outboxcheck(): can't catch SIGPIPE");
	}

	/* search free instance */
	MBC_instance = 0;
	while (MBC_instance < ngsmdevs) {
	  if (gsmdevices[MBC_instance].free) {
            /* let's process this one */
	    break;
	  }
	  MBC_instance++;
	}                      /* while (MBC_instance...) */
	if (MBC_instance >= ngsmdevs) {
	  fatal ("sms_serv: can't get a free GSM - internal error");
	}

	/* got it - now lock it (device lockfile handled by libmodem) */
	MBC_instance_is_set = TRUE;
	gsmdevices[MBC_instance].free = FALSE;
	gsmdevices[MBC_instance].owner = getpid ();

	/* copy to "local" space, to avoid the shmem balagan */
	if (gsmdevcpy (&gsm, &gsmdevices[MBC_instance]) == -1) {
	  syserr ("db_outboxcheck(): error copying GSM instance");
	}

	/* leave crit. sect. */
	if (sem_signal (shmem_sem) == -1) {
	  syserr ("db_outboxcheck(): can't signal shared mem. semaphore");
	}
	/* ---- End Crit. Sect. #2 ---- */

	/* log it */
	syslog ((FACILITY | LOG_NOTICE), "DB outbox check: using device </dev/%s>.",
	       gsm.device);
        syslog ((FACILITY | LOG_NOTICE), "DB outbox check: trying to send message with queue ID <%s>",
	       row[0]);

	/* Which mode do we require ? */
	if (ob_symbols.mode == SMS_MODE_DEFAULT) {
	  /* user kept default value - let's use device-level default */
	  reqmode = gsm.defmode;
	}
	else {
	  /* user requested specific mode - let's use it */
	  reqmode = ob_symbols.mode;
	}

	/* Convert message to PDU if required */
	if (reqmode == SMS_MODE_PDU) {
          /* free ob_symbols.pdu if needed -- encode_pdu() will allocate */
	  if (ob_symbols.pdu) {
	    free (ob_symbols.pdu);
	  }
          if ((ob_symbols.pdu = encode_pdu (&ob_symbols, default_alphabet)) == NULL) {
  	    /* exit */
            syslog ((FACILITY | LOG_ERR), "DB outbox check: failed to convert message to PDU format.");
          }
	}

        /* ---->>> Actual SEND <<<---- */
        if ((msgid = backend_send_sms (&gsm, &ob_symbols, 0)) != -1) {
          syslog ((FACILITY | LOG_NOTICE), "DB outbox check: message sent OK, message ID is <%d>.", msgid);
          /* set status and sent for the message */
          sprintf (query, "UPDATE outbox SET status=1, sent=NOW(), stored=stored WHERE id=%s", row[0]);
          if (mysql_query (&mysql, query)) {
            syslog ((FACILITY | LOG_ERR), "DB outbox check: cannot mark message %s as sent. MySQL error code: %s",
	           row[0], mysql_error (&mysql));
            syslog ((FACILITY | LOG_WARNING), "DB outbox check: message ID %s may be re-sent at the next queue run");
          }
        }
        else {                      /* some error occured */
          syslog ((FACILITY | LOG_WARNING), "DB outbox check: ERROR: failed to send message.");
	  /* increment qrun counter */
	  if ((qrun_counter = outbox_counter_op (row[0], SMS_FC_INC)) == -1) {
	    fprintf (stderr, "DB outbox check: failed to INC counter on queue item <%s>.\n", row[0]);
	    syslog ((FACILITY | LOG_ERR), "DB outbox check: failed to INC counter on queue item <%s>.", row[0]);
	  }
	  if ((maxqruns > 0) && (qrun_counter >= maxqruns)) {
	    /* mail postmaster about the failure */
	    if (bounce_message (quidstr, &ob_symbols) == -1) {
              syslog ((FACILITY | LOG_ERR), "DB outbox check: failed to send bounced-back mail.");
	    }
	    /* mark the queue item "failed" */
            sprintf (query, "UPDATE outbox SET status=2, sent=NOW(), stored=stored WHERE id=%s", row[0]);
	    if (mysql_query (&mysql, query)) {
              syslog ((FACILITY | LOG_ERR), "DB outbox check: cannot mark message %s as failed. MySQL error code: %s",
	             row[0], mysql_error (&mysql));
	      fprintf (stderr, "DB outbox check: failed to mark message %s as failed.\n", row[0]);
	    }
	    else {
	      syslog ((FACILITY | LOG_INFO), "DB outbox check: queue item <%s> undelivered after %d tries, marked 'failed'.",
		     row[0], maxqruns);
	    }         /* if (mysql_query (&mysql, query)) */
	  }
        }

	/* now free the modem again */
	if (sem_wait (shmem_sem) == -1) {
	  syserr ("db_outboxcheck(): failed to wait on shared mem. semaphore");
	}

	/* ---- Begin Crit. Sect. #3 ---- */
	gsmdevices[MBC_instance].free = TRUE;
	gsmdevices[MBC_instance].owner = 0;
	if (msgid != -1) {
	  gsmdevices[MBC_instance].out_ok++;
	}
	else {
	  gsmdevices[MBC_instance].out_fail++;
	}
	MBC_instance_is_set = FALSE;
	/* leave crit. sect. */
	if (sem_signal (shmem_sem) == -1) {
	  syserr ("db_outboxcheck(): can't signal shared mem. semaphore");
	}
	/* ---- End Crit. Sect. #3 ---- */

	/* leave main crit. sect. */
	if (sem_signal (global_sem) == -1) {
	  syserr ("db_outboxcheck(): can't signal global semaphore");
	}
      }                    /* ---- End Crit. Sect. #1 --- */
    }                          /* loop on outbox messages */
    mysql_free_result (result);
  }                               /* if (mysql_query (... */

  /*------------------------------------------Conclusions */
  mysql_close (&mysql);
  syslog ((FACILITY | LOG_NOTICE), "DB outbox check: finished.");
#endif

}                                    /* db_outboxcheck () */
/*========================================================*/
void file_outboxcheck ()
{
  extern int ngsmdevs;
  extern struct symbols ob_symbols;
  extern int maxqruns;

  extern int obchklimit;
  extern int global_sem;
  extern int shmem_sem;
  extern struct gsms_def *gsmdevices;
  int nwaitsecs = 0;
  int last_errno;
  int retval;
  int nfiles, maxfiles;
  int file_counter;
  char device[MAXDEVLEN];
  struct gsms_def gsm;
  int queueid;
  char quidstr[TINYBUFF];
  int msgid;
  char priority;
  struct sigaction sa_c;
  int reqmode;
  char *fqqfname;
  char *qitem;
  char *user;
  char *dest;
  char *smsc;
  char *mode;
  char *text;
  int parse_error;
  FILE *q_file;
  struct dirent **dentry;
  int nselfiles;
  int i;

  /*--------------------------------------Initializations */
  syslog ((FACILITY | LOG_NOTICE), "FILE outbox check begins.");
  
  parse_error = 0;
  nfiles = 0;
  maxfiles = obchklimit;
  if (! obchklimit) {
    /* If no msg limit has been set, maxfiles will be kept at nfiles + 1 */
    maxfiles = 1;
  }
  
  fqqfname = (char *) malloc ((MAXPATHLEN + 1) * sizeof (char));
  if (! fqqfname) {
    syserr ("file_outboxcheck(): can't malloc() fqqfname");
  }
  fqqfname[0] = '\0';
  
  qitem = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (! qitem) {
    syserr ("file_outboxcheck(): can't malloc() qitem");
  }
  qitem[0] = '\0';
  
  /*--------------------------------------Processing Loop */
  /* get messages from outbox - fill up a sorted list of dir. entries */
  /* WARNING: scandir() is not a POSIX function. */
  if ((nselfiles = scandir (OBOX_SPOOL, &dentry, 0, alphasort)) == -1) {
    syslog ((FACILITY | LOG_ERR), "scandir() failed on outbox spool dir <%s>.",
           OBOX_SPOOL);
    syserr ("file_outboxcheck(): scandir() failed on outbox spool dir");
  }

  i = 0;
  while ((i < nselfiles) && (nfiles < maxfiles)) {
    if (is_queue_file (dentry[i]->d_name)) {
      /* build fully qualified queuefile name */
      sprintf (fqqfname, "%s/%s", OBOX_SPOOL, dentry[i]->d_name);

      /*------------------extract details from queue file */
      /*.......................................Message ID */
      queueid = get_msgid (dentry[i]->d_name);
      quidstr[0] = '\0';
      sprintf (quidstr, "%d", queueid);

      /*.........................................Priority */
      priority = dentry[i]->d_name[1];
      
      /* lock it first */
      if (queue_get_lock (queueid)) {
	/*.................................Contents field */
	if ((q_file = fopen (fqqfname, "r")) == NULL) {
	  syslog ((FACILITY | LOG_ERR), "call to fopen() failed when reading queue file <%s>.",
        	 fqqfname);
	  syserr ("file_outboxcheck(): queue file read failed: can't fopen()");
	}
	memset (qitem, 0, (BUFFSIZE + 1));
	if (fgets (qitem, (BUFFSIZE + 1), q_file) == NULL) {
	  if (debug & DEBUG_PRINTF) {
	    fprintf (stderr, "Empty queue item: <%s>\n", fqqfname);
	  }
	}
	fclose (q_file);

	/* parse line read from queue file */
	if (strchr (qitem, ',')) {
	  if ((user = strtok (qitem, ",")) == NULL) {
	    parse_error++;
	    if (debug & DEBUG_PRINTF) {
	      fprintf (stderr, "Can't extract user field from:\n[%s]\n", qitem);
	    }
	  }
	  if ((dest = strtok (NULL, ",")) == NULL) {
	    parse_error++;
	    if (debug & DEBUG_PRINTF) {
	      fprintf (stderr, "Can't extract dest field from:\n[%s]\n", qitem);
	    }
	  }
	  if ((smsc = strtok (NULL, ",")) == NULL) {
	    parse_error++;
	    if (debug & DEBUG_PRINTF) {
	      fprintf (stderr, "Can't extract smsc field from:\n[%s]\n", qitem);
	    }
	  }
	  if ((mode = strtok (NULL, ",")) == NULL) {
	    parse_error++;
	    if (debug & DEBUG_PRINTF) {
	      fprintf (stderr, "Can't extract mode field from:\n[%s]\n", qitem);
	    }
	  }
	  if ((text = strtok (NULL, "\n")) == NULL) {
	    parse_error++;
	    if (debug & DEBUG_PRINTF) {
	      fprintf (stderr, "Can't extract text field from:\n[%s]\n", qitem);
	    }
	  }
	}
	else {
	  parse_error++;
	  if (debug & DEBUG_PRINTF) {
	    fprintf (stderr, "Can't parse queue item:\n[%s]\n", qitem);
	  }
	}                     /* if (strchr (qitem, ',')) */
	if (! parse_error) {
          dequote (text);
          strcpy (ob_symbols.destgsm, dest);
          strcpy (ob_symbols.message, text);
          strcpy (ob_symbols.smsc, smsc);
          strcpy (ob_symbols.user, user);
          ob_symbols.mode = atoi (mode);
          ob_symbols.priority = ((int) priority - 48);

	  /*--------------------------Obtain a GSM device */
	  while (((retval = sem_decreq (global_sem)) == -1) &&
        	(errno == EAGAIN) &&
		(nwaitsecs < m_timeout)) {
	    sleep (W_STEP);
	    nwaitsecs += W_STEP;
	  }                                /* while (...) */
	  last_errno = errno;
	  if (retval == -1) {    /* failed to get a modem */
	    switch (last_errno) {
	      case EAGAIN: {
        	syslog ((FACILITY | LOG_WARNING), "FILE outbox check: timeout expired (all GSMs busy).");
        	break;
	      }
	      default: {
        	syserr ("file_outboxcheck(): can't decrease global semaphore");
		break;
	      }
	    }                      /* switch (last_errno) */
	  }
	  else {         /* ---- Begin Crit. Sect. #1 --- */
	    /* at least 1 GSM module is free - find it */
	    if (sem_wait (shmem_sem) == -1) {
	      syserr ("file_outboxcheck(): failed to wait on shared mem. semaphore");
	    }

	    /* ---- Begin Crit. Sect. #2 ---- */
	    /* Signal Handling */
	    sa_c.sa_handler = MBC_unlock_gsm;
	    if (sigfillset (&sa_c.sa_mask) == -1) {
	      syserr ("file_outboxcheck(): can't fill signal set");
	    }
	    sa_c.sa_flags = 0;
	    /* catch SIGTERM to handle child's death while GSM's locked */
	    if (sigaction (SIGTERM, &sa_c, NULL) == -1) {
	      syserr ("file_outboxcheck(): can't catch SIGTERM");
	    }
	    /* catch SIGPIPE to handle client disconnect while GSM's locked */
	    if (sigaction (SIGPIPE, &sa_c, NULL) == -1) {
	      syserr ("file_outboxcheck(): can't catch SIGPIPE");
	    }

	    /* search free instance */
	    MBC_instance = 0;
	    while (MBC_instance < ngsmdevs) {
	      if (gsmdevices[MBC_instance].free) {
        	/* let's process this one */
		break;
	      }
	      MBC_instance++;
	    }                  /* while (MBC_instance...) */
	    if (MBC_instance >= ngsmdevs) {
	      fatal ("sms_serv: can't get a free GSM - internal error");
	    }

	    /* got it - now lock it (device lockfile handled by libmodem) */
	    MBC_instance_is_set = TRUE;
	    gsmdevices[MBC_instance].free = FALSE;
	    gsmdevices[MBC_instance].owner = getpid ();

	    /* copy to "local" space, to avoid the shmem balagan */
	    if (gsmdevcpy (&gsm, &gsmdevices[MBC_instance]) == -1) {
	      syserr ("file_outboxcheck(): error copying GSM instance");
	    }

	    /* leave crit. sect. */
	    if (sem_signal (shmem_sem) == -1) {
	      syserr ("file_outboxcheck(): can't signal shared mem. semaphore");
	    }
	    /* ---- End Crit. Sect. #2 ---- */

	    /* log it */
	    syslog ((FACILITY | LOG_NOTICE), "FILE outbox check: using device </dev/%s>.",
	           gsm.device);
            syslog ((FACILITY | LOG_NOTICE), "FILE outbox check: trying to send message with queue ID <%d>",
	           queueid);

	    /* Which mode do we require ? */
	    if (ob_symbols.mode == SMS_MODE_DEFAULT) {
	      /* user kept default value - let's use device-level default */
	      reqmode = gsm.defmode;
	    }
	    else {
	      /* user requested specific mode - let's use it */
	      reqmode = ob_symbols.mode;
	    }

            /* Convert message to PDU if required */
            if (reqmode == SMS_MODE_PDU) {
              /* free ob_symbols.pdu if needed -- encode_pdu() will allocate */
	      if (ob_symbols.pdu) {
		free (ob_symbols.pdu);
	      }
              if ((ob_symbols.pdu = encode_pdu (&ob_symbols, default_alphabet)) == NULL) {
  		/* exit */
        	syslog ((FACILITY | LOG_ERR), "FILE outbox check: failed to convert to PDU codes");
              }
            }

            /* ---->>> Actual SEND <<<---- */
            if ((msgid = backend_send_sms (&gsm, &ob_symbols, 0)) != -1) {
              syslog ((FACILITY | LOG_NOTICE), "FILE outbox check: message sent OK, message ID is <%d>.", msgid);
              /* remove that entry from the queue */
	      if (unlink (fqqfname) != -1) {
		syslog ((FACILITY | LOG_INFO), "FILE outbox check: removed queue file <%s>.", fqqfname);
	      }
	      else {
		fprintf (stderr, "FILE outbox check: failed to remove queue file <%s>.\n", fqqfname);
		syslog ((FACILITY | LOG_ERR), "FILE outbox check: failed to remove queue file <%s>.", fqqfname);
	      }                  /* if (unlink (fqqfname) */
            }
            else {
	      /* some error occured */
              syslog ((FACILITY | LOG_WARNING), "FILE outbox check: ERROR: failed to send queue ID <%d>.", queueid);
	      /* increment file counter */
	      if ((file_counter = file_counter_op (fqqfname, SMS_FC_INC)) == -1) {
		fprintf (stderr, "FILE outbox check: failed to INC counter on queue file <%s>.\n", fqqfname);
		syslog ((FACILITY | LOG_ERR), "FILE outbox check: failed to INC counter on queue file <%s>.", fqqfname);
	      }
	      if ((maxqruns > 0) && (file_counter >= maxqruns)) {
	        /* mail postmaster about the failure */
		if (bounce_message (quidstr, &ob_symbols) == -1) {
        	  syslog ((FACILITY | LOG_ERR), "FILE outbox check: failed to send bounced-back mail.");
		}
		/* remove the queue item */
		if (unlink (fqqfname) != -1) {
		  syslog ((FACILITY | LOG_INFO), "FILE outbox check: queue file <%s> undelivered after %d tries, removed.",
		         fqqfname, maxqruns);
		}
		else {
		  fprintf (stderr, "FILE outbox check: failed to remove queue file <%s>.\n", fqqfname);
		  syslog ((FACILITY | LOG_ERR), "FILE outbox check: failed to remove queue file <%s>.", fqqfname);
		}                  /* if (unlink (fqqfname) */
	      }
            }          /* if ((msgid = backend_send_sms ( */
	    nfiles++;
	    if (! obchklimit) {
	      /* no limit => keeps it at nfiles+1 => test always true */
	      maxfiles++;
	    }

	    /* now free the modem again */
	    if (sem_wait (shmem_sem) == -1) {
	      syserr ("file_outboxcheck(): failed to wait on shared mem. semaphore");
	    }

	    /* ---- Begin Crit. Sect. #3 ---- */
	    gsmdevices[MBC_instance].free = TRUE;
	    gsmdevices[MBC_instance].owner = 0;
	    if (msgid != -1) {
	      gsmdevices[MBC_instance].out_ok++;
	    }
	    else {
	      gsmdevices[MBC_instance].out_fail++;
	    }
	    MBC_instance_is_set = FALSE;
	    /* leave crit. sect. */
	    if (sem_signal (shmem_sem) == -1) {
	      syserr ("file_outboxcheck(): can't signal shared mem. semaphore");
	    }
	    /* ---- End Crit. Sect. #3 ---- */

	    /* leave main crit. sect. */
	    if (sem_signal (global_sem) == -1) {
	      syserr ("file_outboxcheck(): can't signal global semaphore");
	    }
          }                /* ---- End Crit. Sect. #1 --- */
	}
	else {
	  /* Parse error */
          syslog ((FACILITY | LOG_WARNING), "FILE outbox check: can't parse queue ID <%d>, ignored.", queueid);
	}                           /* if (! parse_error) */
        /* remove the lock file */
        if (! queue_remove_lock (queueid)) {
	  syslog ((FACILITY | LOG_ERR), "FILE outbox check: failed to unlock queue ID <%d>.", queueid);
        }
      }
      else {
	/* failed to lock */
        syslog ((FACILITY | LOG_WARNING), "FILE outbox check: failed to lock queue ID <%d>, ignored.", queueid);
      }                  /* if (queue_get_lock (queueid)) */
    }                              /* if (is_queue_file ( */
    free (dentry[i]);
    i++;
  }                             /* while ((i < nselfiles) */
  syslog ((FACILITY | LOG_NOTICE), "FILE outbox check: <%d> queue file(s) processed.", nfiles);

  /*------------------------------------------Conclusions */
  /* free what needs to be */
  free (fqqfname);
  free (qitem);
  free (dentry);

  syslog ((FACILITY | LOG_NOTICE), "FILE outbox check: finished.");

}                                  /* file_outboxcheck () */
/*========================================================*/
void outboxcheck_wrapper ()
{

  switch (qmethod) {
    case SMS_Q_DB:
      db_outboxcheck ();
      break;
      
    case SMS_Q_FILE:
      file_outboxcheck ();
      if (dbaccess_enabled) {
        /* also check the DB -- could be filled by something else */
        db_outboxcheck ();
      }
      break;
      
    case SMS_Q_NONE:
      if (dbaccess_enabled) {
        /* check the DB anyway -- could be filled by something else */
        db_outboxcheck ();
      }
      break;
      
    default:
      syslog ((FACILITY | LOG_WARNING), "outbox check: unknown queueing method <%d>.",
             qmethod);
      break;
      
  }                                   /* switch (qmethod) */
  
}                               /* outboxcheck_wrapper () */
/*========================================================*/

/*==========================================================
 * EOF : mbchecker.c
 *===================*/
