/*==========================================================
 * Program : pdu.c                         Project : smslink
 * Authors : Philippe Andersson.
 *           Franois Baligant <francois@euronet.be>.
 * Date    : 28/05/03
 * Version : 0.18b
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 * Comment : Library of PDU-related functions for the smslink servers.
 *
 * Modification History :
 * - 0.01a (07/04/00) : Initial release.
 * - 0.02a (13/04/00) : Added the ascii7topacked8() function.
 * - 0.03a (26/04/00) : Fixed a misunderstanding in encode_pdu().
 *   TP_UDL should be measured in chars, not in bytes. Thanks
 *   to <eric.guidi@wavecom.fr> for pointing that out.
 * - 0.04a (08/05/00) : Fixed a second misunderstanding (pad
 *   the TP_DA before byteswap, not after. Thanks again
 *   to <eric.guidi@wavecom.fr> for pointing that out.
 * ++++ Switch to Beta ++++
 * - 0.05b (20/11/00) : Included the packed8toascii7() function.
 * - 0.06b (21/11/00) : Added the decode_pdu() function.
 * - 0.07b (22/11/00) : Added the parse_csts_date() and
 *   parse_csts_time() functions.
 * - 0.08b (23/11/00) : Included cap_matrix in the parameters
 *   passed to decode_pdu(), in order to handle those devices
 *   that have the SCA at the start of the PDU string.
 * - 0.09b (24/11/00) : Fixed a bug in the decode_pdu() function
 *   for cases where HAVE_SCA_IN_PDU is true. <scalen> is not
 *   coded the same as <oalen>. Simplified <tpdu> structure in
 *   encode_pdu() function (caused problems for PCFF900). 
 * - 0.10b (18/02/01) : Modified packed8toascii7() so that it
 *   would return in error in case a non-ascii7 character is
 *   found (contributed by Andrew Worsley <epaanwo@asac.ericsson.se>).
 * - 0.11b (05/03/01) : Replaced control of all debug info by
 *   global flags.
 * - 0.12b (11/08/01) : In decode_pdu(), added a debug feature
 *   that would accept any numplan, print a warning and go ahead,
 *   e.g. for numplan #85 [reserved] (requested by Thomas Gutzler
 *   <giggel@hadiko.de>). Changed numplan processing to a
 *   switch structure to make it easier supporting more in the
 *   future.
 * - 0.13b (18/10/01) : Added formal support for numplans #83,
 *   #85, #89, #A1 and #D0. Kudos to Thomas Gutzler
 *   (<giggel@hadiko.de>) for his tremendous help in tracking
 *   that down. Modified the numplan switch cases to hex values.
 *   Made the <OA> byteswap conditional to <numplan> value.
 *   Fixed a bug where the msg->fromgsm string was not always
 *   properly terminated.
 * - 0.14b (12/01/02) : Added support for numplan #C1 (contributed
 *   by Richard Parsons <richard.parsons@namitech.com>).
 * - 0.15b (14/01/02) : Added support for numplan #C8 (ref. found
 *   on <http://www.dreamfabric.com/sms/>, a PDU page by Lars
 *   Pettersson <lars.pettersson@eds.com>).
 * - 0.16b (24/01/02) : Started implementing support for default
 *   GSM alphabet. Implemented transcode(), listsupportedcharsets(),
 *   getcharsetID() & getcharsetname().
 * - 0.17b (09/03/03) : Improved error reporting and end of string
 *   handling in packed8toascii7(). Should solve (at least part)
 *   of the trailing '@' problem.
 * - 0.18b (28/05/03) : Added udl param to packed8toascii7().
 *   Moved extra '@' detection from decode_pdu() to packed8toascii7().
 *========================================================*/

#include <unistd.h>
#include <stdio.h>                         /* for fprintf */
#include <stdlib.h>                  /* for errno & stuff */
#include <errno.h>
#include <string.h>
#include <syslog.h>
#include <math.h>                            /* for pow() */

#include "sms_serv.h"
#include "gsmdev.h"
#include "pdu.h"

/*========================================================*/
/* For debugging purposes only - comment out for normal compile */
/* #define INCL_DEBUG_CODE */

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/
extern int debug;                    /* debug level/flags */
extern int default_alphabet;

charsetdef charsetdeftable[] = {
  {"GSM",         "GSM Default Alphabet",  0},
  {"ISO-8859-1",  "ISO Latin1",            1},
  {"ISO-8859-15", "ISO Latin1 + Euro",     2},
  {"",            "",                     -1}
};

int chartable[][3] = {
/* GSM def. - 8859-1 - 8859-15*/
  {    0,	 64,	 64},	/* @ */
  {    1,	163,	163},	/*  */
  {    2,	 36,	 36},	/* $ */
  {    3,	165,	165},	/* yen */
  {    4,	232,	232},	/*  */
  {    5,	233,	233},	/*  */
  {    6,	250,	250},	/* u + acute */
  {    7,	236,	236},	/* i + grave */
  {    8,	242,	242},	/* o + grave */
  {    9,	199,	199},	/* C + cedilla */
  {   10,	 10,	 10},	/* <lf> */
  {   11,	216,	216},	/* O stroke */
  {   12,	248,	248},	/* o stroke */
  {   13,	 13,	 13},	/* <cr> */
  {   14,	197,	197},	/* A bolle */
  {   15,	229,	229},	/* a bolle */
  {   16,	  0,	  0},	/* DELTA */
  {   17,	 95,	 95},	/* _ */
  {   18,	  0,	  0},	/* PHI */
  {   19,	  0,	  0},	/* GAMMA */
  {   20,	  0,	  0},	/* LAMBDA */
  {   21,	  0,	  0},	/* OMEGA */
  {   22,	  0,	  0},	/* PI */
  {   23,	  0,	  0},	/* PSI */
  {   24,	  0,	  0},	/* SIGMA */
  {   25,	  0,	  0},	/* THETA */
  {   26,	  0,	  0},	/* XI */
  {   27,	  0,	  0},	/* Escape -> Ext Char */
  {   28,	198,	198},	/* AE */
  {   29,	230,	230},	/* ae */
  {   30,	223,	223},	/* sharp s */
  {   31,	202,	202},	/* E + acute */
  {   32,	 32,	 32},	/* space */
  {   33,	 33,	 33},	/* ! */
  {   34,	 34,	 34},	/* " */
  {   35,	 35,	 35},	/* # */
  {   36,	164,	  0},	/* currency */
  {   37,	 37,	 37},	/* % */
  {   38,	 38,	 38},	/* & */
  {   39,	 39,	 39},	/* ' */
  {   40,	 40,	 40},	/* ( */
  {   41,	 41,	 41},	/* ) */
  {   42,	 42,	 42},	/* * */
  {   43,	 43,	 43},	/* + */
  {   44,	 44,	 44},	/* , */
  {   45,	 45,	 45},	/* - */
  {   46,	 46,	 46},	/* . */
  {   47,	 47,	 47},	/* / */
  {   48,	 48,	 48},	/* 0 */
  {   49,	 49,	 49},	/* 1 */
  {   50,	 50,	 50},	/* 2 */
  {   51,	 51,	 51},	/* 3 */
  {   52,	 52,	 52},	/* 4 */
  {   53,	 53,	 53},	/* 5 */
  {   54,	 54,	 54},	/* 6 */
  {   55,	 55,	 55},	/* 7 */
  {   56,	 56,	 56},	/* 8 */
  {   57,	 57,	 57},	/* 9 */
  {   58,	 58,	 58},	/* : */
  {   59,	 59,	 59},	/* ; */
  {   60,	 60,	 60},	/* < */
  {   61,	 61,	 61},	/* = */
  {   62,	 62,	 62},	/* > */
  {   63,	 63,	 63},	/* ? */
  {   64,	161,	161},	/* inverted ! */
  {   65,	 65,	 65},	/* A */
  {   66,	 66,	 66},	/* B */
  {   67,	 67,	 67},	/* C */
  {   68,	 68,	 68},	/* D */
  {   69,	 69,	 69},	/* E */
  {   70,	 70,	 70},	/* F */
  {   71,	 71,	 71},	/* G */
  {   72,	 72,	 72},	/* H */
  {   73,	 73,	 73},	/* I */
  {   74,	 74,	 74},	/* J */
  {   75,	 75,	 75},	/* K */
  {   76,	 76,	 76},	/* L */
  {   77,	 77,	 77},	/* M */
  {   78,	 78,	 78},	/* N */
  {   79,	 79,	 79},	/* O */
  {   80,	 80,	 80},	/* P */
  {   81,	 81,	 81},	/* Q */
  {   82,	 82,	 82},	/* R */
  {   83,	 83,	 83},	/* S */
  {   84,	 84,	 84},	/* T */
  {   85,	 85,	 85},	/* U */
  {   86,	 86,	 86},	/* V */
  {   87,	 87,	 87},	/* W */
  {   88,	 88,	 88},	/* X */
  {   89,	 89,	 89},	/* Y */
  {   90,	 90,	 90},	/* Z */
  {   91,	196,	196},	/* A + diaeresis */
  {   92,	214,	214},	/* O + diaeresis */
  {   93,	209,	209},	/* N + tilde */
  {   94,	220,	220},	/* U + diaeresis */
  {   95,	167,	167},	/*  */
  {   96,	191,	191},	/* inverted ? */
  {   97,	 97,	 97},	/* a */
  {   98,	 98,	 98},	/* b */
  {   99,	 99,	 99},	/* c */
  {  100,	100,	100},	/* d */
  {  101,	101,	101},	/* e */
  {  102,	102,	102},	/* f */
  {  103,	103,	103},	/* g */
  {  104,	104,	104},	/* h */
  {  105,	105,	105},	/* i */
  {  106,	106,	106},	/* j */
  {  107,	107,	107},	/* k */
  {  108,	108,	108},	/* l */
  {  109,	109,	109},	/* m */
  {  110,	110,	110},	/* n */
  {  111,	111,	111},	/* o */
  {  112,	112,	112},	/* p */
  {  113,	113,	113},	/* q */
  {  114,	114,	114},	/* r */
  {  115,	115,	115},	/* s */
  {  116,	116,	116},	/* t */
  {  117,	117,	117},	/* u */
  {  118,	118,	118},	/* v */
  {  119,	119,	119},	/* w */
  {  120,	120,	120},	/* x */
  {  121,	121,	121},	/* y */
  {  122,	122,	122},	/* z */
  {  123,	228,	228},	/* a + diaeresis */
  {  124,	246,	246},	/* o + diaeresis */
  {  125,	241,	241},	/* n + tilde */
  {  126,	252,	252},	/* u + diaeresis */
  {  127,	224,	224},	/* a + grave */
  {27010,	 12,	 12},	/* <ff> */
  {27020,	 94,	 94},	/* ^ */
  {27040,	123,	123},	/* { */
  {27041,	125,	125},	/* } */
  {27047,	 92,	 92},	/* \ */
  {27060,	 91,	 91},	/* [ */
  {27061,	126,	126},	/* ~ */
  {27062,	 93,	 93},	/* ] */
  {27064,	124,	124},	/* | */
  {27101,	  0,	164},	/* Euro */
  {   -1, 	 -1,	 -1}
};

/*========================================================*/
/**********               FUNCTIONS                ********/
/*========================================================*/
inline int dapad (char *string)
{
  int len;
  
  len = strlen (string);
  if ((strlen (string) % 2) == 1) {
    /* str_insert (string, (len - 1), 'F'); */
    strcat (string, "F");
  }
}                                             /* dapad () */
/*========================================================*/
inline int byteswap (char *string)
{
  int len, i;
  char c;
  
  len = strlen (string);
  for (i = 1; i < len; i += 2) {
    c = string[i];
    string[i] = string[i - 1];
    string[i - 1] = c;
  }
}                                          /* byteswap () */
/*========================================================*/
int parse_csts_date (char *instr, char *outstr)
{
  char yy[3];
  char *errptr;
  int y, s;
  
  /* extract the "year" part */
  strncpy (yy, instr, 2);
  yy[2] = '\0';
  
  /* convert it to decimal */
  y = strtol (yy, &errptr, 10);
  if (*errptr != '\0') {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "Invalid input date string format : [%s]\n", instr);
    }
    return (-1);
  }
  
  /* compute the required century */
  if (y > 95) {
    s = 19;
  }
  else {
    s = 20;
  }
  
  /* build output string */
  sprintf (outstr, "%d%s", s, instr);
  
  if (strlen (outstr) != 8) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "Invalid output date string format : [%s]\n", outstr);
    }
    return (-1);
  }
  else {
    return (strlen (outstr));
  }
}                                   /* parse_csts_date () */
/*========================================================*/
int parse_csts_time (char *instr, char *outstr)
{
  /* convert from hhmmssuu to xxhyymzzs */
  char hh[3];
  char mm[3];
  char ss[3];
  
  /* split instr into its components */
  strncpy (hh, instr, 2);
  hh[2] = '\0';
  shiftleft (instr, 2);
  
  strncpy (mm, instr, 2);
  mm[2] = '\0';
  shiftleft (instr, 2);
  
  strncpy (ss, instr, 2);
  ss[2] = '\0';
  
  /* build output string */
  sprintf (outstr, "%sh%sm%ss", hh, mm, ss);
  
  return (strlen (outstr));
}                                   /* parse_csts_time () */
/*========================================================*/
int transcode (int c, int from, int to)
{
  int i = 0;
  
  while ((chartable[i][from] != c) && (chartable[i][from] != -1)) {
    i++;
  }
  
  if (chartable[i][from] == c) {
    return (chartable[i][to]);
  }
  else {
    return (0);
  }
}                                         /* transcode () */
/*========================================================*/
int ascii7topacked8 (char *in, char *out, int da)
{
  int i, inbitlen, outbitlen, outfree, chunklen;
  int stored_c;
  unsigned char cur_c, out_c, mask, trans;
  char scratch[3];

  /*--------------------------------------Initializations */
  i = 0;
  outbitlen = 0;
  out_c = '\0';
  stored_c = 0;

  /*---------------------------------Main processing loop */
  while (in[i]) {
    if (stored_c) {
      /* single char translates to ESC + code */
      /* stored_c stores the code */
      cur_c = stored_c;
      stored_c = 0;
    }
    else {
      cur_c = in[i];
      /* transcode cur_c before packing it */
      stored_c = transcode (cur_c, da, ALPH_GSM);
      if (stored_c > ESC_VALUE) {
        cur_c = 0x1b;                         /* ESC char */
	stored_c -= ESC_VALUE;
      }
      else {
        cur_c = stored_c;
        stored_c = 0;
      }
    }
    inbitlen = 7;

    while (inbitlen) {
      outfree = (8 - outbitlen);
      chunklen = min (outfree, inbitlen);
      if (!chunklen) {
        /* out_c is full - save it */
        sprintf (scratch, "%0.2X", out_c);
        strcat (out, scratch);
        /* re-initialize out_c */
        out_c = '\0';
        outbitlen = 0;
        chunklen = inbitlen;
      }                                 /* if (!chunklen) */
      /* save chunklen bits to trans */
      mask = (pow (2, chunklen) - 1);
      trans = cur_c & mask;
      /* offset what's left of cur_c for further use */
      cur_c = cur_c >> chunklen;
      /* offset trans for merge with out_c */
      trans = trans << outbitlen;
      /* merge trans in out_c */
      out_c |= trans;
      /* update length counters */
      inbitlen -= chunklen;
      outbitlen += chunklen;
    }                                 /* while (inbitlen) */

    if (!stored_c) {
      i++;
    }
  }                                      /* while (in[i]) */
  
  /*------------------------------------------Conclusions */
  /* save the remaining last bits */
  sprintf (scratch, "%0.2X", out_c);
  strcat (out, scratch);

  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Message in packed8 format : [%s]\n", out);
  }
  return (strlen (out) / 2);
}                                   /* ascii7topacked8 () */
/*========================================================*/
int packed8toascii7 (char *in, char *out, int udl, int da)
{
  int inbitlen, outbitlen, outfree, chunklen;
  int textlen;
  unsigned char cur_c, out_c, mask, trans;
  int stored_c;
  char scratch[3];
  char *ptr;
  char *endptr;

  /*--------------------------------------Initializations */
  ptr = in;
  outbitlen = 0;
  out_c = '\0';
  stored_c = 0;

  while (ptr < (in + strlen (in))) {
    /* chop the in-string 2 (hex) char at a time */
    strncpy (scratch, ptr, 2);
    scratch[2] = '\0';
    /* convert hex digit pair to binary value */
    cur_c = strtol (scratch, &endptr, 16);
    if (*endptr != '\0') {
      if (debug & DEBUG_PRINTF) {
        fprintf (stderr, "packed8toascii7() non-hex value found! [%s]\n", scratch);
	if (debug & DEBUG_HEXDUMP) {
	  fprintf (stderr, "Hexdump of the scratch buffer:\n");
	  dump_string (scratch);
	}
      }
      return (-1);                /* non-hex value found! */
    }
    inbitlen = 8;

    while (inbitlen) {
      outfree = (7 - outbitlen);
      chunklen = min (outfree, inbitlen);
      if (!chunklen) {
        /* out_c is full - save it */
	if (out_c & 0x80) {
          if (debug & DEBUG_PRINTF) {
            fprintf (stderr, "packed8toascii7() non-ascii7 character found! 0x%x\n",
	            out_c);
          }
	  return (-1);     /* non-ascii7 character found! */
	}
	/* transcode out_c first */
	if (out_c == 0x1b) {
	  /* got ESC - keep a note of it */
	  stored_c = ESC_VALUE;
	  out_c = '\0';
	}
	else {
	  if (stored_c) {
	    stored_c += out_c;
	    out_c = transcode (stored_c, ALPH_GSM, da);
	    stored_c = 0;
	  }
	  else {
	    out_c = transcode (out_c, ALPH_GSM, da);
	  }
	}
	/* append transcoded char. to result string */
        sprintf (scratch, "%c", out_c);
        strcat (out, scratch);
        /* re-initialize out_c */
        out_c = '\0';
        outbitlen = 0;
        chunklen = min (inbitlen, 7);
      }                                 /* if (!chunklen) */
      /* save chunklen bits to trans */
      mask = (pow (2, chunklen) - 1);
      trans = cur_c & mask;
      /* offset what's left of cur_c for further use */
      cur_c = cur_c >> chunklen;
      /* offset trans for merge with out_c */
      trans = trans << outbitlen;
      /* merge trans in out_c */
      out_c |= trans;
      /* update length counters */
      inbitlen -= chunklen;
      outbitlen += chunklen;
    }                                 /* while (inbitlen) */

    ptr += 2;
  }                                  /* while (ptr < ...) */
  /* save the last bits only when they contain a full character */
  /* ...and then only if it agrees with the <tpudl> value ! */
  if (outbitlen == 7) {
    if (out_c & 0x80) {
      if (debug & DEBUG_PRINTF) {
	fprintf (stderr, "packed8toascii7() trailing non-ascii7 character found! 0x%x\n",
		out_c);
      }
      return (-1);         /* non-ascii7 character found! */
    }
    /* transcode out_c first */
    if (out_c == 0x1b) {
      /* got ESC - keep a note of it */
      /* should never happen on the last char., obviously ! */
      stored_c = ESC_VALUE;
      out_c = '\0';
    }
    else {
      if (stored_c) {
	stored_c += out_c;
	out_c = transcode (stored_c, ALPH_GSM, da);
	stored_c = 0;
      }
      else {
	out_c = transcode (out_c, ALPH_GSM, da);
      }
    }
    /* append transcoded char. to result string */
    sprintf (scratch, "%c", out_c);
    strcat (out, scratch);
  }                                /* if (outbitlen == 7) */

  /* chop out string to <udl> chars (iif <udl> is known) */
  textlen = strlen (out);
  if (udl != TP_UDL_UNKNOWN) {
    if ((textlen == (udl + 1)) && (transcode (out[textlen - 1], da, ALPH_GSM) == 0)) {
      /* no way packed8toascii7() can make the difference between the final '\0' */
      /* and a trailing '@' that is part of the message. So, an unexpected (based */
      /* on <tpudl>) trailing '@' is removed here. Not the cleanest way, I know ! */
      if (debug & DEBUG_PRINTF) {
        fprintf (stderr, "Warning: removed trailing '@' in converted string, was [%s].\n",
                 out);
      }
      out[textlen - 1] = '\0';
      textlen = strlen (out);
    }
  }                         /* if (udl != TP_UDL_UNKNOWN) */

  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Raw message after packed8toascii7() : [%s]\n", out);
  }
  return (textlen);
}                                   /* packed8toascii7 () */
/*========================================================*/

/*========================================================*/
/*#################### main PDU functions ################*/
/*========================================================*/
char *encode_pdu (struct symbols *symbols, int da)
{
  /* This function is largely based on proof-of-concept code *
   * provided by Franois Baligant <francois@euronet.be>.    */
   
  char *pdu;
  char scratch[MAXPHNUMLEN + 1];
  unsigned int tpdu;
  int skip;
  char numplan[3];
  char dalen[3];
  char *buff;
  int tp_udl;
  
  /*---------------------------------------Initialization */
  pdu = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (!pdu) {
    syslog ((FACILITY | LOG_ERR), "can't malloc() for PDU string.");
    return (NULL);
  }
  pdu[0] = '\0';
  
  /*--------------------------------------Field #1 : TPDU */
  tpdu = (TP_MTI_SMS_SUBMIT | TP_RD_REJECT);
  sprintf (scratch, "%0.2X", tpdu);
  strcat (pdu, scratch);
  
  /*----------------------------------------Field #2 : MR */
  /* Message Reference left NULL - should be set by the phone */
  strcat (pdu, "00");

  /*----------------------------------------Field #3 : DA */
  /* Destination Address : Address Length, Numbering Plan, Number */
  if (symbols->destgsm[0] == '+') {
    /* international number */
    skip = 1;
    strcpy (numplan, "91");
  }
  else if ((symbols->destgsm[0] == '0') && (symbols->destgsm[1] == '0')) {
    /* international number */
    skip = 2;
    strcpy (numplan, "91");
  }
  else {
    /* local number */
    skip = 0;
    strcpy (numplan, "81");
  }
  
  /* remove the international call prefix - if any */
  strcpy (scratch, symbols->destgsm);
  if (skip > 0) {
    shiftleft (scratch, skip);
  }
  
  /* Now store address length as hex value */
  sprintf (dalen, "%0.2X", strlen (scratch));
  strcat (pdu, dalen);
  
  /* Add the numbering plan */
  strcat (pdu, numplan);
  
  /* Finally append the padded, byte-swapped, destination */
  dapad (scratch);
  byteswap (scratch);
  strcat (pdu, scratch);

  /*---------------------------------------Field #4 : PID */
  sprintf (scratch, "%0.2X", PID_SMS);
  strcat (pdu, scratch);
  
  /*---------------------------------------Field #5 : DCS */
  sprintf (scratch, "%0.2X", DCS_PCCP437);
  strcat (pdu, scratch);
  
  /*----------------------------------------Field #6 : VP */
  if (tpdu & TP_VPF_RELATIVE) {
    strcat (pdu, "AA");                         /* 4 days */
  }
  
  /*----------------------------Field #7 : TP_UDL + TP_UD */
  /* TP_UDL is the length of TP_UD measured in chars (NOT in  */
  /* bytes). Thanks to  <eric.guidi@wavecom.fr>).             */
  buff = (char *) malloc (((2 * MAXMSGLEN) + 1) * sizeof (char));
  buff[0] = '\0';
  
  tp_udl = strlen (symbols->message);
  ascii7topacked8 (symbols->message, buff, da);

  sprintf (scratch, "%0.2X", tp_udl);
  strcat (pdu, scratch);
  strcat (pdu, buff);
  /*-------------------------------------------Conclusion */
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "PDU-encoded string : [%s]\n", pdu);
  }
  free (buff);
  return (pdu);
}                                        /* encode_pdu () */
/*========================================================*/
int decode_pdu (char *pdu, struct mbox_item *msg, cap_matrix capmatrix, int da)
{
  char *scratch;
  char *ptr;
  char *errptr;
  int scalen, scalen_p, scalen_t;
  char numplan[3];
  int n_numplan;
  char sca[MAXPHNUMLEN + 1];
  int tpdu;
  int oalen, oalen_p;
  int pid;
  int dcs;
  char *datestr;
  char *timestr;
  int tpudl;
  int textlen;

  /*---------------------------------------Initialization */
  scratch = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (!scratch) {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't malloc() (scratch).");
    return (-1);
  }
  scratch[0] = '\0';
  
  datestr = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
  if (!datestr) {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't malloc() (datestr).");
    return (-1);
  }
  datestr[0] = '\0';
  
  timestr = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
  if (!timestr) {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't malloc() (timestr).");
    return (-1);
  }
  timestr[0] = '\0';
  ptr = pdu;
  n_numplan = 0;
  
  /*--------------------------------Field #0 : SCA (opt.) */
  /* only appears on specific GSM models */
  if (capmatrix.capflags & HAS_SCA_IN_PDU) {
    /* first extract <scalen> - 1 byte */
    /* Warning: <scalen> is measured in PDU "bytes", <numplan> and */
    /* padding char. included */
    if (strlen (ptr) >= 2) {
      strncpy (scratch, ptr, 2);
      scratch[2] = '\0';
    }
    else {
      syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse PDU (scalen).");
      return (-1);
    }
    /* convert it to decimal */
    scalen = strtol (scratch, &errptr, 16);
    if (*errptr != '\0') {
      syslog ((FACILITY | LOG_ERR), "decode_pdu() - invalid scalen value.");
      return (-1);
    }
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "SCALEN value : [%d]\n", scalen);
    }
    /* jump to next */
    ptr += 2;

    /* then extract the numbering plan - 1 byte */
    if (strlen (ptr) >= 2) {
      strncpy (numplan, ptr, 2);
      numplan[2] = '\0';
    }
    else {
      syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse PDU (sca numplan).");
      return (-1);
    }
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "SCA NUMPLAN value : [%s]\n", numplan);
    }
    /* jump to next */
    ptr += 2;

    /* now get the SCA number - (scalen - 1) * 2 char */
    scalen_p = ((scalen - 1) * 2);
    if (strlen (ptr) >= scalen_p) {
      strncpy (scratch, ptr, scalen_p);
      scratch[scalen_p] = '\0';
    }
    else {
      syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse PDU (sca).");
      return (-1);
    }
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "Raw SCA value : [%s]\n", scratch);
    }
    /* byteswap the SCA field */
    byteswap (scratch);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "Byteswaped SCA value : [%s]\n", scratch);
    }
    /* correct scalen_p according to presence of padding char. */
    if (scratch[scalen_p - 1] == 'F') {
      /* pad char. present */
      scalen_t = scalen_p - 1;
    }
    else {
      /* no pad char. */
      scalen_t = scalen_p;
    }
    /* build the SCA ph. number */
    if (strcmp (numplan, "81") == 0) {
      /* local number - nothing special to do */
      strncpy (sca, scratch, scalen_t);
    }
    else if (strcmp (numplan, "91") == 0) {
      /* international number */
      strcpy (sca, "+");
      strncat (sca, scratch, scalen_t);
    }
    else {
      syslog ((FACILITY | LOG_ERR), "decode_pdu() - unsupported SCA numplan (%s).",
             numplan);
      return (-1);
    }
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "Final SCA value : [%s]\n", sca);
    }
    /* jump to next */
    ptr += scalen_p;
  }           /* if (capmatrix.capflags & HAS_SCA_IN_PDU) */
  
  /*--------------------------------------Field #1 : TPDU */
  /* extract the first byte */
  if (strlen (ptr) >= 2) {
    strncpy (scratch, ptr, 2);
    scratch[2] = '\0';
  }
  else {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse PDU (tpdu).");
    return (-1);
  }
  /* convert it to decimal */
  tpdu = strtol (scratch, &errptr, 16);
  if (*errptr != '\0') {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - invalid tpdu value.");
    return (-1);
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "TPDU value : [%d]\n", tpdu);
  }
  /* analyse it - later */
  /* jump to next */
  ptr += 2;
  
  /*----------------------------------------Field #2 : OA */
  /*.......................first extract <oalen> - 1 byte */
  /* Warning: <oalen> is measured in characters, <numplan> and padding */
  /* character excluded */
  if (strlen (ptr) >= 2) {
    strncpy (scratch, ptr, 2);
    scratch[2] = '\0';
  }
  else {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse PDU (oalen).");
    return (-1);
  }
  /* convert it to decimal */
  oalen = strtol (scratch, &errptr, 16);
  if (*errptr != '\0') {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - invalid oalen value.");
    return (-1);
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "OALEN value : [%d]\n", oalen);
  }
  /* jump to next */
  ptr += 2;
  
  /*.............then extract the numbering plan - 1 byte */
  if (strlen (ptr) >= 2) {
    strncpy (numplan, ptr, 2);
    numplan[2] = '\0';
  }
  else {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse PDU (numplan).");
    return (-1);
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "NUMPLAN value : [%s]\n", numplan);
  }
  /* convert it for further use (switch) */
  n_numplan = strtol (numplan, (char **)NULL, 16);
  /* jump to next */
  ptr += 2;
  
  /*......now get the origin. number - oalen (+ 1 ?) char */
  if ((oalen % 2) == 1) {
    /* odd length => pad char. present */
    oalen_p = oalen + 1;
  }
  else {
    /* no pad char. */
    oalen_p = oalen;
  }
  if (strlen (ptr) >= oalen_p) {
    strncpy (scratch, ptr, oalen_p);
    scratch[oalen_p] = '\0';
  }
  else {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse PDU (oa).");
    return (-1);
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Raw OA value : [%s]\n", scratch);
  }

  /* byteswap the OA field when needed */
  if (n_numplan != 0xD0) {
    byteswap (scratch);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "Byteswaped OA value : [%s]\n", scratch);
    }
  }
  else {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "No byteswap needed for numplan [%s]\n", numplan);
    }
  }

  /* build and save the source ph. number according to <numplan> */
  switch (n_numplan) {
    case (0x81): {
      /* local number - nothing special to do */
      strncpy (msg->fromgsm, scratch, oalen);
      msg->fromgsm[oalen] = '\0';
      break;
    }

    case (0x91): {
      /* international number */
      strcpy (msg->fromgsm, "+");
      strncat (msg->fromgsm, scratch, oalen);
      msg->fromgsm[oalen + 1] = '\0';
      break;
    }

    case (0x83): {
      /* Data numbering plan (X.121) */
      if ((oalen > 2) && (scratch[0] == '0') && (scratch[0] == '0')) {
        /* International number w/ leading "00" - replace w/ '+' */
	strcpy (msg->fromgsm, "+");
	strncat (msg->fromgsm, (scratch + 2), (oalen - 2));
	msg->fromgsm[oalen - 1] = '\0';
      }
      else if ((oalen > 1) && (scratch[0] == '0')) {
        /* Single leading '0' -> local number - keep as is */
        strncpy (msg->fromgsm, scratch, oalen);
        msg->fromgsm[oalen] = '\0';
      }
      else {
        /* International number (?) */
	strcpy (msg->fromgsm, "+");
	strncat (msg->fromgsm, scratch, oalen);
	msg->fromgsm[oalen + 1] = '\0';
      }
      break;
    }

    case (0xA1): {
      /* National number - nothing special to do */
      strncpy (msg->fromgsm, scratch, oalen);
      msg->fromgsm[oalen] = '\0';
      break;
    }

    case (0xC1): {
      /* Subscriber number / ISDN-tel */
      /* <oa> field might contain voicemail number */
      strncpy (msg->fromgsm, scratch, oalen);
      msg->fromgsm[oalen] = '\0';
      break;
    }

    case (0xC8): {
      /* Subscriber number / National numbering plan */
      strncpy (msg->fromgsm, scratch, oalen);
      msg->fromgsm[oalen] = '\0';
      break;
    }

    case (0xD0): {
      /* Alphanum. value in GSM TS 03.38 7bit alphabet */
      /* likely to contain the originating website name */
      packed8toascii7 (scratch, msg->fromgsm, TP_UDL_UNKNOWN, da);
      break;
    }

    case (0x85):
    case (0x89): {
      /* reserved value or "private" numplan */
      /* <OA> field likely just garbage - nuke it */
      strcpy (msg->fromgsm, "000000000000");
      syslog ((FACILITY | LOG_NOTICE), "decode_pdu() - reserved numplan (%s), <OA> was [%s] - ignored.",
             numplan, scratch);
      break;
    }

    default: {
      if (debug & DEBUG_PDU) {
	/* unusual value, assumed local - nothing special to do */
	strncpy (msg->fromgsm, scratch, oalen);
        msg->fromgsm[oalen] = '\0';
	/* log a warning */
	syslog ((FACILITY | LOG_NOTICE), "decode_pdu() - unknown numplan (%s), assumed local.",
               numplan);
	/* print a debug message */
	fprintf (stderr, "Unknown numplan (%s), assumed local. PLEASE INVESTIGATE !\n",
	        numplan);
	fprintf (stderr, "Originating phone number: <%s>\n", msg->fromgsm);
	fprintf (stderr, "Proceeding anyway.\n");
	break;
      }
      else {
	syslog ((FACILITY | LOG_ERR), "decode_pdu() - unsupported numplan (%s).",
               numplan);
	return (-1);
      }
    }
  }                                 /* switch (n_numplan) */
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Final OA value : [%s]\n", msg->fromgsm);
  }
  /* jump to next */
  ptr += oalen_p;

  /*---------------------------------------Field #3 : PID */
  /* extract the <pid> field - 1 byte */
  if (strlen (ptr) >= 2) {
    strncpy (scratch, ptr, 2);
    scratch[2] = '\0';
  }
  else {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse PDU (pid).");
    return (-1);
  }
  /* convert it to decimal */
  pid = strtol (scratch, &errptr, 16);
  if (*errptr != '\0') {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - invalid pid value (%s).",
            scratch);
    return (-1);
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "PID value : [%d]\n", pid);
  }
  /* analyse it - later */
  /* jump to next */
  ptr += 2;

  /*---------------------------------------Field #4 : DCS */
  /* extract the <dcs> field - 1 byte */
  if (strlen (ptr) >= 2) {
    strncpy (scratch, ptr, 2);
    scratch[2] = '\0';
  }
  else {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse PDU (dcs).");
    return (-1);
  }
  /* convert it to decimal */
  dcs = strtol (scratch, &errptr, 16);
  if (*errptr != '\0') {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - invalid dcs value (%s).",
            scratch);
    return (-1);
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "DCS value : [%d]\n", dcs);
  }
  /* analyse it - later */
  /* jump to next */
  ptr += 2;

  /*--------------------------------------Field #5 : CSTS */
  /* extract the <csts> field - 7 byte */
  if (strlen (ptr) >= 14) {
    strncpy (scratch, ptr, 14);
    scratch[14] = '\0';
  }
  else {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse PDU (csts).");
    return (-1);
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Raw CSTS value : [%s]\n", scratch);
  }
  /* byteswap the CSTS field */
  byteswap (scratch);
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Byteswaped CSTS value : [%s]\n", scratch);
  }
  /* now split the byteswaped <csts> string into raw date and time */
  strncpy (datestr, scratch, 6);
  datestr[6] = '\0';
  shiftleft (scratch, 6);
  strcpy (timestr, scratch);
  /* parse each of them */
  if (parse_csts_date (datestr, msg->date) == -1) {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse <csts> date.");
    return (-1);
  }
  if (parse_csts_time (timestr, msg->time) == -1) {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse <csts> time.");
    return (-1);
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "message date = [%s]\n", msg->date);
    fprintf (stderr, "message time = [%s]\n", msg->time);
  }

  /* jump to next */
  ptr += 14;

  /*-------------------------------------Field #6 : TP-UD */
  /* first extract <tp-udl> - 1 byte */
  if (strlen (ptr) >= 2) {
    strncpy (scratch, ptr, 2);
    scratch[2] = '\0';
  }
  else {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - can't parse PDU (tp-udl).");
    return (-1);
  }
  /* convert it to decimal */
  tpudl = strtol (scratch, &errptr, 16);
  if (*errptr != '\0') {
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - invalid tp-udl value.");
    return (-1);
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "TP-UDL value : [%d]\n", tpudl);
  }
  /* jump to next */
  ptr += 2;
  
  /* the rest is TP-UD - decode it */
  strcpy (scratch, ptr);
  textlen = packed8toascii7 (scratch, msg->text, tpudl, da);
  if (textlen != tpudl) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "Decoded TP-UD : [%s] expected <tpdul>: %d, got %d, length: %d (corruption ?)\n",
              msg->text, tpudl, textlen, strlen (msg->text));
    }
    syslog ((FACILITY | LOG_ERR), "decode_pdu() - corrupt data, <tpudl> don't match, expected %d, got %d.",
           tpudl, textlen);
    return (-1);
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Decoded TP-UD string : [%s]\n", msg->text);
  }

  /*-------------------------------------------Conclusion */
  free (scratch);
  free (datestr);
  free (timestr);
  return (strlen (msg->text));
}                                        /* decode_pdu () */
/*========================================================*/
/*############ charset table handling functions ##########*/
/*========================================================*/
void listsupportedcharsets ()
{
  int i = 0;
  
  while (strlen (charsetdeftable[i].name)) {
    printf ("  %s %s (%s)\n", ((i == default_alphabet)? "*" : "-"),
           charsetdeftable[i].name, charsetdeftable[i].descr);
    i++;
  }
}                             /* listsupportedcharsets () */
/*========================================================*/
int getcharsetID (char *name)
/* Returns charset ID, -1 if not found */
{
  int i = 0;
  
  strupper (name);
  
  while ((strcmp (name, charsetdeftable[i].name) != 0) &&
        (strlen (charsetdeftable[i].name))) {
    i++;
  }                                           /* while () */
  
  if (debug & DEBUG_PRINTF) {
    if (charsetdeftable[i].id != -1) {
      fprintf (stderr, "Selected charset: %s (%s), ID = %d\n",
              charsetdeftable[i].name,
              charsetdeftable[i].descr,
              charsetdeftable[i].id);
    }
    else {
      fprintf (stderr, "Requested charset %s not supported\n", name);
    }
  }
  
  return (charsetdeftable[i].id);
}                                      /* getcharsetID () */
/*========================================================*/
char *getcharsetname (int id)
/* Returns charset name (pointer to allocated string), NULL if not found */
{
  int i = 0;
  char *name = NULL;
  
  while ((charsetdeftable[i].id != id) &&
        (charsetdeftable[i].id != -1)) {
    i++;
  }                                           /* while () */
  
  if (charsetdeftable[i].id == id) {
    name = (char *) malloc ((strlen (charsetdeftable[i].name) + 1) * sizeof (char));
    if (!name) {
      syserr ("getcharsetname(): can't malloc() name");
    }
    name[0] = '\0';
    strcpy (name, charsetdeftable[i].name);
  }
  
  return (name);
}                                    /* getcharsetname () */
/*========================================================*/

/*==========================================================
 * EOF : pdu.c
 *===================*/
