/*==========================================================
 * Program : serv_stuff.c                  Project : smslink
 * Author  : Philippe Andersson.
 * Date    : 22/03/06
 * Version : 0.23b
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 * Comment : Library of functions for the smslink server.
 *
 * Modification History :
 * - 0.01b (19/08/99) : Initial release. Moved the SMS server-
 *   specific functions from stuff.c over here.
 * - 0.02b (21/10/99) : Moved tellsock() over to stuff.c (also
 *   used by sms2mailgw for communication with sendmail).
 * - 0.03b (24/01/00) : The definition for union semun disappeared
 *   from the include files coming with the latest version of
 *   glibc2. Fixed the code for daemons_death() and set_semvalue()
 *   to account for this.
 * - 0.04b (31/03/00) : Adapted gsmdevcpy() to accomodate for
 *   structure change (defmode field).
 * - 0.05b (17/05/00) : Adapted gsmdevcpy() to accomodate for
 *   structure change (capmatrix field).
 * - 0.06b (22/06/00) : Adapted gsmdevcpy() to accomodate for
 *   structure change (capmatrix structure).
 * - 0.07b (26/09/00) : Added a signal handler for SIGCHLD to prevent
 *   zombie formation (contributed by Stanley Hopcroft 
 *   <Stanley.Hopcroft@IPAustralia.Gov.AU>). Modified it to log more
 *   info about the exit status of the child.
 * - 0.08b (03/10/00) : Removed all logging from sig_chld(). The
 *   server died after a few days in production, and it might have
 *   been due to the use of unsafe functions within a signal
 *   handler.
 * - 0.09b (18/02/01) : Included debug level handling. Modified
 *   gsmdevcpy() to accomodate new <flags> field. Improved
 *   debug info in tell_gsm() and get_gsm_answer(). Added fn
 *   cleanup_mdm(). Contributed by Andrew Worsley 
 *   <epaanwo@asac.ericsson.se>.
 * - 0.10b (19/02/01) : Added support for fast modem response,
 *   and added set_cpms() to set the "message store". Contributed 
 *   by Andrew Worsley <epaanwo@asac.ericsson.se>.
 * - 0.11b (05/03/01) : Replaced control of all debug info by
 *   global flags.
 * - 0.12b (30/03/01) : Added hexdump facility for gsm answers,
 *   controlled by global debug level, to ease debugging "fast
 *   modem response" mode.
 * - 0.13b (02/04/01) : Modified gsmdevcpy() to support new
 *   modem_okay field. Passed the struct gsms_def as param. to
 *   get_gsm_answer().
 * - 0.14b (04/04/01) : Added functions for cpl_list handling.
 *   Altered get_gsm_resp(), get_gsm_answer() and get_gsm_fast()
 *   to be passed a cpl_list parameter instead of a simple char*.
 * - 0.15b (03/05/01) : Added getregstat() function.
 * - 0.16b (05/05/01) : Fixed a small bug in the debugging code
 *   in getregstat() function.
 * - 0.17b (30/04/01) : Adapted gsmdevcpy() to handle new counter
 *   fields in struct gsms_def.
 * - 0.18b (16/05/01) : Adapted gsmdevcpy() to handle new model
 *   field in struct gsms_def.
 * - 0.19b (08/09/02) : In set_cpms(), added missing cleanup code
 *   in case of error + cosmetics. Added the set_cmee(),
 *   checkset_cpin() and checkset_csca() functions.
 *   Moved gsmdevcpy() over to gsmdevices.c. Replaced static
 *   PIN post-processing sleep time by device-level param.
 *   (PINpause). Merged 0.54b-9 changes in set_cnmi() (T65
 *   support contributed by Ritesh Shah <riteshs@rediff.co.in>).
 *   Implemented support for the file queueing method.
 * - 0.20b (25/11/03) : Added  test on NO_CREG_CHECK flag to
 *   support Motorola A008 device. Contributed by Nicki de Wet
 *   <nickidw@mighty.co.za>.
 * - 0.21b (29/01/04) : Added "obc_list" handling functions.
 * - 0.22b (07/04/04) : Added priority support to queued messages.
 *   Solved a nasty bug in obc_list_insert().
 * - 0.23b (22/03/06) : Added PID file removal to signal handler.
 *========================================================*/

#include <unistd.h>
#include <stdio.h>                         /* for fprintf */
#include <stdlib.h>                  /* for errno & stuff */
#include <errno.h>
#include <string.h>
#include <syslog.h>
#include <utime.h>                         /* for utime() */
#include <time.h>                /* for time_t and time() */
#include <sys/time.h>           /* for the struct timeval */
#include <sys/types.h>                     /* for open () */
#include <sys/stat.h>                      /* for open () */
#include <sys/param.h>                  /* for MAXPATHLEN */
#include <fcntl.h>                         /* for open () */
#include <dirent.h>              /* for the struct dirent */
#include <sys/wait.h>                    /* for waitpid() */
#include <sys/ipc.h>
#include <sys/sem.h>                        /* semaphores */
#include <sys/shm.h>                     /* shared memory */
#include <sys/ioctl.h>            /* for the ioctl() call */
#include <dial/modems.h>           /* requires 'libmodem' */

#include "sms_serv.h"
#include "gsmdev.h"

/*========================================================*/
/**********             LOCAL DEFINES              ********/
/*========================================================*/
/* For debugging purposes only - comment out for normal compile */
/* #define INCL_DEBUG_CODE */

/* For the random generator routines */
#define LOCAL_RAND_MAX 9999999.0
#define MAXINT_32 4294967295.0
#define MAXPID_F 32767.0

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/
extern int debug;                    /* debug level/flags */
extern int use_fast;        /* enable fast modem response */
extern int context;              /* global server context */

/*========================================================*/
/**********               FUNCTIONS                ********/
/*========================================================*/
/*========================================================*/
/*################### obc_list handling ##################*/
/*========================================================*/
void obc_list_init (obc_list *list)
{
  list->head = NULL;
  list->tail = NULL;
  list->count = 0;
}                                     /* obc_list_init () */
/*========================================================*/
int empty_obc_list (obc_list list)
{
  return (list.head == NULL);
}                                    /* empty_obc_list () */
/*========================================================*/
void obc_list_insert (obc_list *list, int pid)
{
  struct obc_proc *element;

  /* alloc memory for new element */
  element = (struct obc_proc *) malloc (sizeof (struct obc_proc));
  if (!element)
    syserr ("obc_list_insert(): can't malloc() for new OBC entry");
  
  /* initialize fields for new element */
  element->pid = pid;
  
  /* chain it in the list */
  if (empty_obc_list (*list)) {
    list->head = element;
    list->tail = element;
    list->count++;
    element->next = NULL;
    element->prev = NULL;
  }
  else {
    element->next = NULL;
    element->prev = list->tail;
    list->count++;
    list->tail->next = element;
    list->tail = element;
  }
}                                   /* obc_list_insert () */
/*========================================================*/
void obc_snip (obc_list *list, struct obc_proc *item)
{
  if (item->prev) {
    item->prev->next = item->next;
  }
  else {
    /* was first in list */
    list->head = item->next;
  }
  
  if (item->next) {
    item->next->prev = item->prev;
  }
  else {
    /* was last in list */
    list->tail = item->prev;
  }
  
  free (item);
}                                          /* obc_snip () */
/*========================================================*/
void obc_remove (obc_list *list, int pid)
{
  struct obc_proc *cursor;

  if (!empty_obc_list (*list)) {
    cursor = list->head;
    if (cursor->pid == pid) {
      obc_snip (list, cursor);
      list->count--;
      return;
    }
    while (cursor->next) {
      cursor = cursor->next;
      if (cursor->pid == pid) {
	obc_snip (list, cursor);
	list->count--;
	return;
      }
    }
  }
}                                        /* obc_remove () */
/*========================================================*/
void dump_obc_list (obc_list list, char *dumpstr)
{
  struct obc_proc *cursor;
  int item = 0;
  char *scratch;

  /*---------------------------------------Initialisation */
  scratch = (char *) malloc ((20 + 1) * sizeof (char));
  if (! scratch) {
    syserr ("dump_obc_list(): can't malloc() scratch");
  }
  scratch[0] = '\0';
  /*-------------------------------------------Processing */
  if (!empty_obc_list (list)) {
    cursor = list.head;
    sprintf (scratch, "Child #%d: <%d> ", item, cursor->pid);
    strcat (dumpstr, scratch);
    while (cursor->next != NULL) {
      cursor = cursor->next;
      item++;
      sprintf (scratch, "Child #%d: <%d> ", item, cursor->pid);
      strcat (dumpstr, scratch);
    }
  }
  else {
    sprintf (dumpstr, "Empty 'OBC' list.");
  }
  /*-------------------------------------------Conclusion */
  free (scratch);
}                                     /* dump_obc_list () */
/*========================================================*/
void free_obc_list (obc_list *list)
{
  struct obc_proc *cursor;

  if (!empty_obc_list (*list)) {
    /* hop to element before last */
    cursor = list->tail->prev;
    /* now go back and clean behind */
    while (cursor != NULL) {
      free (cursor->next);
      cursor->next = NULL;
      list->tail = cursor;
      cursor = cursor->prev;
    }                           /* while (cursor != NULL) */
  }                           /* if (!empty_obc_list (... */
  /* now clean last element and reset header */
  free (list->head);
  list->head = NULL;
  list->tail = NULL;
  list->count = 0;
}                                     /* free_obc_list () */
/*========================================================*/
/*################### cpl_list handling ##################*/
/*========================================================*/
void cpl_list_init (cpl_list *list)
{
  list->head = NULL;
  list->tail = NULL;
}                                     /* cpl_list_init () */
/*========================================================*/
int empty_cpl_list (cpl_list list)
{
  return (list.head == NULL);
}                                    /* empty_cpl_list () */
/*========================================================*/
void cpl_list_insert (cpl_list *list, char *c_phrase)
{
  struct cpl_item *element;

  /* alloc memory for new element */
  element = (struct cpl_item *) malloc (sizeof (struct cpl_item));
  if (!element)
    syserr ("cpl_list_insert(): can't malloc() for new CPL entry");
  
  /* initialize fields for new element */
  element->catch_phrase = (char *) malloc ((strlen (c_phrase) + 1) * sizeof (char));;
  if (! element->catch_phrase)
    syserr ("cpl_list_insert(): can't malloc() for new CPL entry");
  strcpy (element->catch_phrase, c_phrase);
  
  /* chain it in the list */
  if (empty_cpl_list (*list)) {
    list->head = element;
    list->tail = element;
    element->next = NULL;
    element->prev = NULL;
  }
  else {
    element->next = NULL;
    element->prev = list->tail;
    list->tail->next = element;
    list->tail = element;
  }
}                                   /* cpl_list_insert () */
/*========================================================*/
int cpl_match_any (char *s, cpl_list list)
/* Returns TRUE whenever any of the strings stored in list
   is matches a substring of s.
   Returns FALSE otherwise. */
{
  struct cpl_item *cursor;
  int found = FALSE;
  
  if (!empty_cpl_list (list)) {
    cursor = list.head;
    found = (strstr (s, cursor->catch_phrase) != 0);
    while ((cursor->next != NULL) && !found) {
      cursor = cursor->next;
      found = (strstr (s, cursor->catch_phrase) != 0);
    }
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "cpl_match_any(): %s\n",
            (found ? "match found" : "no match found"));
  }
  return (found);
}                                     /* cpl_match_any () */
/*========================================================*/
void dump_cpl_list (cpl_list list)
{
  struct cpl_item *cursor;
  int item = 0;

  if (!empty_cpl_list (list)) {
    cursor = list.head;
    fprintf (stderr, "Item <%d> ", item);
    dump_string (cursor->catch_phrase);
    while (cursor->next != NULL) {
      cursor = cursor->next;
      item++;
      fprintf (stderr, "Item <%d> ", item);
      dump_string (cursor->catch_phrase);
    }
  }
  else {
    fprintf (stderr, "sms_serv: empty 'CPL' list.\n");
  }
}                                     /* dump_cpl_list () */
/*========================================================*/
void free_cpl_list (cpl_list *list)
{
  struct cpl_item *cursor;

  if (!empty_cpl_list (*list)) {
    /* hop to element before last */
    cursor = list->tail->prev;
    /* now go back and clean behind */
    while (cursor != NULL) {
      free (cursor->next->catch_phrase);
      free (cursor->next);
      cursor->next = NULL;
      list->tail = cursor;
      cursor = cursor->prev;
    }                           /* while (cursor != NULL) */
  }                           /* if (!empty_cpl_list (... */
  /* now clean last element and reset header */
  free (list->head->catch_phrase);
  free (list->head);
  list->head = NULL;
  list->tail = NULL;
}                                     /* free_cpl_list () */
/*========================================================*/
/*############ processes and signals handling ############*/
/*========================================================*/
void daemons_death ()
{
#ifdef _SEM_SEMUN_UNDEFINED           /* see <bits/sem.h> */
  union semun {
    int val;				/* value for SETVAL */
    struct semid_ds *buf;		/* buffer for IPC_STAT & IPC_SET */
    unsigned short int *array;		/* array for GETALL & SETALL */
    struct seminfo *__buf;		/* buffer for IPC_INFO */
  } sem_union;
#else  
  union semun sem_union;
#endif                     /* #ifdef _SEM_SEMUN_UNDEFINED */
  extern int shmem_sem, global_sem;
  extern int shmem_id;
  extern void *shared_memory;
  extern acl_list gsm_acl;
  extern obc_list childprocs;
  extern int errno;
  extern char *pidfile;
  
  /* free what's need to be freed */
  free_acl_list (&gsm_acl);
  free_obc_list (&childprocs);
  
  /* release shared memory */
  if (shmdt (shared_memory) == -1)
    syserr ("daemons_death(): can't detach from shared memory segment");
  if (shmctl (shmem_id, IPC_RMID, 0) == -1)
    syserr ("daemons_death(): can't remove shared memory segment");
  
  /* delete semaphores */
  if (semctl (shmem_sem, 0, IPC_RMID, sem_union) == -1)
    syserr ("daemons_death(): can't delete shmem semaphore");
  
  if (semctl (global_sem, 0, IPC_RMID, sem_union) == -1)
    syserr ("daemons_death(): can't delete global semaphore");
  
  /* remove checkpoint file */
  if (unlink (CHECKPOINTF) == -1) {
    if (errno != ENOENT) {             /* no such file... */
      syserr ("daemons_death(): can't remove checkpoint file");
    }
  }

  /* remove PID file */
  if (unlink (pidfile) == -1) {
    if (errno != ENOENT) {             /* no such file... */
      syserr ("daemons_death(): can't remove PID file");
    }
  }

  /* log it... */
  syslog ((FACILITY | LOG_INFO), "server exiting on SIGTERM.");

  /* closes connection to the syslog daemon */
  closelog ();
  
  /* now exits gracefully */
  exit (0);
}                                     /* daemons_death () */
/*========================================================*/
void sig_chld ()
{
  extern obc_list childprocs;
  int stat;
  pid_t pid;
  /*char cause[20];*/
  
  /* this is supposed to hunt down zombies */
  if ((pid = waitpid (-1, &stat, 0)) <= 0)
    syserr ("sig_chld(): waitpid error in sig_chld()");
  
  /* remove pid from list (if present) */
  obc_remove (&childprocs, pid);
  
  /* child's autopsy */
  /* those functions might not be safe to use in a signal handler...
  if (WIFEXITED (stat)) {
    sprintf (cause, "exit code %d", WEXITSTATUS (stat));
  }
  else if (WIFSIGNALED (stat)) {
    sprintf (cause, "signal %d", WTERMSIG (stat));
  }
  else {
    sprintf (cause, "cause unknown");
  }
  */
  
  /* log the event */
  /* those functions might not be safe in a signal handler...
  syslog ((FACILITY | LOG_INFO), "child [%d] just terminated (%s).", pid,
          cause);
  */
  if (debug & DEBUG_ALL) {
    syslog ((FACILITY | LOG_INFO), "child [%d] just terminated.", pid);
  }
  
  /* go on... */
  return;
}                                          /* sig_chld () */
/*========================================================*/
/*################ Dialogue with GSM module ##############*/
/*========================================================*/
int tell_gsm (int fd, char *command)
{
  int len;
  
  len = strlen (command);
  if (write (fd, command, len) != len) {
    syslog ((FACILITY | LOG_ERR), "error writing to the GSM module.");
    mdmperror ("error writing to the GSM module");
    mdm_unlock (mdmopendevice);
    hangup (fd);
    free (command);
    exit (-1);
  }
  if (debug & DEBUG_TELL) {
    syslog ((FACILITY | LOG_DEBUG), "tell_gsm(%d, \"%s\")", fd, command);
  }
}                                          /* tell_gsm () */
/*========================================================*/
int get_gsm_resp (int fd, char *answer, int size, int resptime,
                 cpl_list cpl)
/*
 * get a response from the GSM modem, will select sleep or fast method
 * dependant on the global variable setting
 */
{
  if (use_fast)
    return (get_gsm_fast (fd, answer, size, resptime, cpl));

  return (get_gsm_sleep (fd, answer, size, resptime));
}                                      /* get_gsm_resp () */
/*========================================================*/
int get_gsm_answer (int fd, char *answer, int size, int resptime,
                    struct gsms_def *gsm)
{
  cpl_list cpl;       /* to store magic terminator string */
  int retval;
  
  cpl_list_init (&cpl);
  cpl_list_insert (&cpl, gsm->capmatrix.modem_okay);
  
  if (use_fast) {
    retval = get_gsm_fast (fd, answer, size, resptime, cpl);
  }
  else {
    retval = get_gsm_sleep (fd, answer, size, resptime);
  }
  
  free_cpl_list (&cpl);
  return (retval);
}                                    /* get_gsm_answer () */
/*========================================================*/
int get_gsm_fast (int fd, char *answer, int size, int resptime,
                 cpl_list cpl)
{
  int nread, retval, previous;
  fd_set inputs;
  struct timeval timeout;
  char *buffer;
  char *answerPtr = answer; /* next available character in the answer area */
      /* last available spot in answer area -> save it for '\0' */
  char *answerEnd = answer + size - 1;
  size_t spaceLeft = size; /* space left in buffer */
  
  /*--------------------------------------Initializations */
  nread = previous = 0;
   
  FD_ZERO (&inputs);
  FD_SET (fd, &inputs);
  
  timeout.tv_sec = 10;
  timeout.tv_usec = 0;
  
  /* non-buffering solution. 
   * reads into the given buffer until terminating string seen or timeout. If
   * too many characters read it falls back to original behavour sleeping and
   * calling FIONREAD and discarding extra characters by reading them into
   * a temporary buffer
   */
  if (debug & DEBUG_HEXDUMP) {
    fprintf (stderr, "Hexdump of current catch phrase list:\n");
    dump_cpl_list (cpl);
  }
  for (;;) {
    if (answerPtr >= answerEnd) /* we have no space left in the buffer */
	break; /* fall through - just read in rest of data and drop it */

    spaceLeft = answerEnd - answerPtr - 1;
    FD_SET (fd, &inputs);
    retval = select (FD_SETSIZE, &inputs, NULL, NULL, &timeout);
    switch (retval) {
      case 0:
        syslog ((FACILITY | LOG_WARNING), "timeout while waiting for GSM answer.");
        return (0);                 /* return zero on error */

      case -1:
        syserr ("get_gsm_fast(): call to select() failed");
        return (0);               /* return zero on error */
      
      default:
	break;                           /* got some data */
    }                                  /* switch (retval) */

    ioctl (fd, FIONREAD, &nread);
    if (spaceLeft < nread) /* more data than space -> just fill buffer */
	nread = spaceLeft;
    if ((nread = read (fd, answerPtr, nread)) <= 0) {
      syslog ((FACILITY | LOG_ERR), "read failed: %m");
      return (0);                 /* return zero on error */
    }
    answerPtr[nread] = '\0'; /* terminate the string for strstr() */
    if (debug & DEBUG_GET_GSM_ANSWER) {
      fprintf (stderr, "get_gsm_fast() timeout %d.%d read in %d bytes '%s'\n",
	      timeout.tv_sec, timeout.tv_usec, nread, answerPtr);
      if (debug & DEBUG_HEXDUMP) {
	fprintf (stderr, "Hexdump of receive buffer:\n");
	dump_string (answerPtr);
      }
    }
    answerPtr += nread;
    if (cpl_match_any (answer, cpl)) { /* found end of string */ 
      if (debug & DEBUG_GET_GSM_ANSWER) {
	syslog ((FACILITY | LOG_DEBUG),
	       "get_gsm_answer(%d, \"%s\", %d, %d)",
	       fd, answer, size, resptime);
      }
      return (answerPtr - answer);
    }
  }                                           /* for (;;) */
  /* junk rest of data */
  ioctl (fd, FIONREAD, &nread);
  while (nread != previous) {
      sleep (resptime);
      previous = nread;
      ioctl (fd, FIONREAD, &nread);
  }                          /* while (nread != previous) */
  /* we know what's the data size - alloc space for it */
  buffer = (char *) malloc ((nread + 1) * sizeof (char));
  if (!buffer)
      syserr ("get_gsm_fast(): can't allocate buffer space");
  /* now we can finally read this data */
  nread = read (fd, buffer, nread);
  switch (nread) {
    case 0:
      /* EOF */
      syslog ((FACILITY | LOG_WARNING), "got no data from GSM module.");
      break;

    case -1:
      syserr ("get_gsm_fast(): error while reading answer from GSM");
      break;

    default:
      buffer[nread] = '\0';
      syslog((FACILITY | LOG_WARNING),
	  "too much data, truncating following %d bytes '%s'", nread, buffer);
      if (debug & DEBUG_GET_GSM_ANSWER) {
	fprintf (stderr, "get_gsm_fast(), truncating following %d bytes '%s'\n",
		nread, buffer);
	if (debug & DEBUG_HEXDUMP) {
	  fprintf (stderr, "Hexdump of receive buffer:\n");
	  dump_string (buffer);
	}
      }
      break;
  }                                     /* switch (nread) */

  free (buffer);
  return (answerPtr - answer);
}                                      /* get_gsm_fast () */
/*========================================================*/
int get_gsm_sleep (int fd, char *answer, int size, int resptime)
{
  int nread, retval, previous;
  fd_set inputs;
  struct timeval timeout;
  char *buffer;
  
  /*--------------------------------------Initializations */
  nread = previous = 0;
  
  FD_ZERO (&inputs);
  FD_SET (fd, &inputs);
  
  timeout.tv_sec = 10;
  timeout.tv_usec = 0;
  
  /* wait for data to arrive on the line */
  retval = select (FD_SETSIZE, &inputs, NULL, NULL, &timeout);
  switch (retval) {
    case 0:
      syslog ((FACILITY | LOG_WARNING), "timeout while waiting for GSM answer.");
      break;
    
    case -1:
      syserr ("get_gsm_sleep(): call to select() failed");
      break;
      
    default:
      if (FD_ISSET (fd, &inputs)) {
        /* first wait for all data to be ready */
	ioctl (fd, FIONREAD, &nread);
	while (nread != previous) {
	  sleep (resptime);
	  previous = nread;
	  ioctl (fd, FIONREAD, &nread);
	}                    /* while (nread != previous) */
	/* we know what's the data size - alloc space for it */
	buffer = (char *) malloc ((nread + 1) * sizeof (char));
	if (!buffer)
	  syserr ("get_gsm_sleep(): can't allocate buffer space");
	/* now we can finally read this data */
	nread = read (fd, buffer, nread);
	switch (nread) {
	  case 0:
	    /* EOF */
	    syslog ((FACILITY | LOG_WARNING), "got no data from GSM module.");
	    break;
	    
	  case -1:
	    syserr ("get_gsm_sleep(): error while reading answer from GSM");
	    break;
	    
	  default:
	    buffer[nread] = '\0';
            if (debug & DEBUG_GET_GSM_ANSWER) {
	      fprintf (stderr, "get_gsm_sleep(), pid<%d> Got : [%s] (%d char)\n",
	              getpid (), buffer, nread);
	      if (debug & DEBUG_HEXDUMP) {
		fprintf (stderr, "Hexdump of receive buffer:\n");
		dump_string (buffer);
	      }
            }
	    /* here we could pre-process it (remove Ctrl-Z) */
	    /* copy it over to out string param. */
	    if (nread > size) {
	      syslog ((FACILITY | LOG_WARNING), "too much data, truncation will occur");
	      strncpy (answer, buffer, size);
	      answer[size] = '\0';
	    }
	    else {
	      strcpy (answer, buffer);
	      if (debug & DEBUG_GET_GSM_ANSWER) {
		syslog ((FACILITY | LOG_DEBUG),
		       "get_gsm_answer(%d, \"%s\", %d, %d)",
		       fd, answer, size, resptime);
	      }
	    }
	    break;
	}                               /* switch (nread) */
      }                                /* if (FD_ISSET... */
      free (buffer);
      break;
  }                                    /* switch (retval) */
  return (nread);
}                                     /* get_gsm_sleep () */
/*========================================================*/
int getregstat (char *answer)
/*
 * Parses the +CREG: answer and returns the <stat> value
 * (= registration status).
 * Returns -1 on non-parseable input.
 */
{
  char *p1;
  char *p2;
  char *p3;
  int stat;
  int mode;
  
  /* find beginning of "+CREG:" token */
  if ((p1 = strstr (answer, "+CREG:")) == 0) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "getregstat(): can't find +CREG: header in [%s]\n",
             answer);
      if (debug & DEBUG_HEXDUMP) {
	fprintf (stderr, "Hexdump of +CREG answer:\n");
	dump_string (answer);
      }
    }
    return (-1);
  }
  
  /* Hop to first char after the ':' (should be a WS) */
  p1 += 6;
  
  /* Locate the (first ?) comma */
  if ((p2 = strchr (p1, ',')) == 0) {
    /* there should be at least 2 fields */
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "getregstat(): can't locate start of 2nd field in +CREG: string [%s]\n",
             answer);
      if (debug & DEBUG_HEXDUMP) {
	fprintf (stderr, "Hexdump of +CREG answer:\n");
	dump_string (answer);
      }
    }
    return (-1);
  }
  
  /* Terminate the <mode> field */
  if (strlen (p2) <= 1) {
    /* nothing after the comma */
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "getregstat(): 2nd field looks empty in +CREG: string [%s]\n",
             answer);
      if (debug & DEBUG_HEXDUMP) {
	fprintf (stderr, "Hexdump of +CREG answer:\n");
	dump_string (answer);
      }
    }
    return (-1);
  }
  p2[0] = '\0';
  
  /* p1 is now <mode> field (ignored) */
  mode = atoi (p1);
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "getregstat(): found <mode> field: [%d] (ignored)\n",
           mode);
  }
  
  /* Hop to the first char. of the <stat> field */
  p2++;
  
  /* skip any whitespace */
  while ((p2[0] == ' ') || (p2[0] == '\t')) {
    p2++;
  }
  
  /* locate the end of the <stat> field */
  /* any additional field or non-digit char. is ignored */
  p3 = p2;
  while (isdigit (p3[0])) {
    p3++;
  }

  /* Terminate the <stat> field */
  p3[0] = '\0';
  
  /* p2 is now <stat> field -- convert it */
  stat = atoi (p2);
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "getregstat(): found <stat> field: [%d] (returned)\n",
           stat);
  }
  
  return (stat);
}                                        /* getregstat () */
/*========================================================*/
int set_cmee (struct gsms_def *gsm, int fd, int csfd)
/*
 * Set GSM to "verbose" error reporting mode
 */
{
  char* scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));

  if (!scratch) {
    syslog ((FACILITY | LOG_ERR), "can't allocate scratch space -- dying.");
    syserr ("set_cmee(): can't allocate scratch space");
  }
  memset (scratch, 0, (BIGBUFF + 1));
  
  sprintf (scratch, "AT+CMEE=1\r");
  tell_gsm (fd, scratch);
  memset (scratch, 0, (BIGBUFF + 1));
  if (get_gsm_answer (fd, scratch, BIGBUFF, 1, gsm)) {
    tellsock (csfd, scratch);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "GSM answer [%s]\n", scratch);
    }
    /* check for "OK" */
    if (strstr (scratch, "OK") == NULL) {
      mdm_unlock (mdmopendevice);
      hangup (fd);
      free (scratch);
      syslog ((FACILITY | LOG_ERR), "error after sending AT+CMEE command.");
      mdmperror ("sms_serv: error after sending AT+CMEE command");
      return (-1);
    }
  }
  else {
    mdm_unlock (mdmopendevice);
    hangup (fd);
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
    mdmperror ("sms_serv: GSM not responding");
    return (-1);
  }

  free (scratch);
  return (0);
}                                          /* set_cmee () */
/*========================================================*/
int set_cnmi_off (struct gsms_def *gsm, int fd, int csfd)
/*
 * Set "no notify for incoming SMS messages"
 */
{
  char* scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));

  if (!scratch) {
    syslog ((FACILITY | LOG_ERR), "can't allocate scratch space -- dying.");
    syserr ("set_cnmi_off(): can't allocate scratch space");
  }
  memset (scratch, 0, (BIGBUFF + 1));
  
  if (gsm->capmatrix.capflags & ALT_CNMI)
    sprintf (scratch, "AT+CNMI=0,0,0,0,1\r");
  else if (gsm->capmatrix.capflags & ALT_CNMI_2)
    sprintf (scratch, "AT+CNMI=1,0,0,0,0\r");
  else if (gsm->capmatrix.capflags & ALT_CNMI_3)
    sprintf (scratch, "AT+CNMI=3,0,0,0\r");
  else if (gsm->capmatrix.capflags & ALT_CNMI_4)
    sprintf (scratch, "AT+CNMI=3,0,0,0,0\r");
  else
    sprintf (scratch, "AT+CNMI=0,0,0,0,0\r");

  tell_gsm (fd, scratch);
  memset (scratch, 0, (BIGBUFF + 1));
  if (get_gsm_answer (fd, scratch, BIGBUFF, 1, gsm)) {
    tellsock (csfd, scratch);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "GSM answer [%s]\n", scratch);
    }
    /* check for "OK" */
    if (strstr (scratch, "OK") == 0) {
      if (context != CONTEXT_DDMN) {
        mdm_unlock (mdmopendevice);
        hangup (fd);
      }
      free (scratch);
      syslog ((FACILITY | LOG_ERR), "error after sending AT+CNMI command.");
      mdmperror ("sms_serv: error after sending AT+CNMI command");
      return (-1);
    }
  }
  else {
    if (context != CONTEXT_DDMN) {
      mdm_unlock (mdmopendevice);
      hangup (fd);
    }
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
    mdmperror ("sms_serv: GSM not responding");
    return (-1);
  }

  free (scratch);
  return (0);
}                                      /* set_cnmi_off () */
/*========================================================*/
int set_cnmi_on (struct gsms_def *gsm, int fd, int csfd)
/*
 * Set "notify ON for incoming SMS messages"
 */
{
  char* scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));

  if (!scratch) {
    syslog ((FACILITY | LOG_ERR), "can't allocate scratch space -- dying.");
    syserr ("set_cnmi_on(): can't allocate scratch space");
  }
  memset (scratch, 0, (BIGBUFF + 1));
  
  if (gsm->capmatrix.capflags & ALT_CNMI)
    sprintf (scratch, "AT+CNMI=0,1,0,0,1\r");
  else if (gsm->capmatrix.capflags & ALT_CNMI_2)
    sprintf (scratch, "AT+CNMI=1,1,0,0,0\r");
  else if (gsm->capmatrix.capflags & ALT_CNMI_3)
    sprintf (scratch, "AT+CNMI=3,1,0,0\r");
  else if (gsm->capmatrix.capflags & ALT_CNMI_4)
    sprintf (scratch, "AT+CNMI=3,1,0,0,0\r");
  else
    sprintf (scratch, "AT+CNMI=0,1,0,0,0\r");

  tell_gsm (fd, scratch);
  memset (scratch, 0, (BIGBUFF + 1));
  if (get_gsm_answer (fd, scratch, BIGBUFF, 1, gsm)) {
    tellsock (csfd, scratch);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "GSM answer [%s]\n", scratch);
    }
    /* check for "OK" */
    if (strstr (scratch, "OK") == 0) {
      if (context != CONTEXT_DDMN) {
        mdm_unlock (mdmopendevice);
        hangup (fd);
      }
      free (scratch);
      syslog ((FACILITY | LOG_ERR), "error after sending AT+CNMI command.");
      mdmperror ("sms_serv: error after sending AT+CNMI command");
      return (-1);
    }
  }
  else {
    if (context != CONTEXT_DDMN) {
      mdm_unlock (mdmopendevice);
      hangup (fd);
    }
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
    mdmperror ("sms_serv: GSM not responding");
    return (-1);
  }

  free (scratch);
  return (0);
}                                       /* set_cnmi_on () */
/*========================================================*/
int set_cpms (struct gsms_def *gsm, int fd, int csfd)
/*
 * force preferred message storage area to be ME - which is only area
 * accessible via AT+CMGL on the GM-12.
 */
{
  char* scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));

  if (!scratch) {
    syslog ((FACILITY | LOG_ERR), "can't allocate scratch space -- dying.");
    syserr ("set_cpms(): can't allocate scratch space");
  }
  memset (scratch, 0, (BIGBUFF + 1));
  
  /* set "message store for reading" to where incomming messages are stored ME */
  sprintf (scratch, "AT+CPMS=\"ME\",\"ME\"\r");
  tell_gsm (fd, scratch);
  memset (scratch, 0, (BIGBUFF + 1));
  if (get_gsm_answer (fd, scratch, BIGBUFF, 1, gsm)) {
    tellsock (csfd, scratch);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "GSM answer: [%s]\n", scratch);
    }
    /* check for "OK" */
    if (strstr (scratch, "OK") == NULL) {
      if (context != CONTEXT_DDMN) {
        mdm_unlock (mdmopendevice);
        hangup (fd);
      }
      free (scratch);
      syslog ((FACILITY | LOG_ERR), "error after sending AT+CPMS command.");
      mdmperror ("sms_serv: error after sending AT+CPMS command");
      return (-1);
    }
  }
  else {
    if (context != CONTEXT_DDMN) {
      mdm_unlock (mdmopendevice);
      hangup (fd);
    }
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
    mdmperror ("sms_serv: GSM not responding");
    return (-1);
  }

  free (scratch);
  return (0);
}                                          /* set_cpms () */
/*========================================================*/
int set_csdh (struct gsms_def *gsm, int fd, int csfd)
/*
 * sets "full SMS header detail" - Beware: only meaningfull
 * (and supported) in text mode AND even so only on specific
 * models ! (specifically not the PCFF900).
 * Returns -1 on failure.
 */
{
  char* scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));

  if (!scratch) {
    syslog ((FACILITY | LOG_ERR), "can't allocate scratch space -- dying.");
    syserr ("set_csdh(): can't allocate scratch space");
  }
  memset (scratch, 0, (BIGBUFF + 1));
  
  sprintf (scratch, "AT+CSDH=1\r");
  tell_gsm (fd, scratch);
  memset (scratch, 0, (BIGBUFF + 1));
  if (get_gsm_answer (fd, scratch, BIGBUFF, 1, gsm)) {
    tellsock (csfd, scratch);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "%s\n", scratch);
    }
    /* check for "OK" */
    if (strstr (scratch, "OK") == NULL) {
      if (context != CONTEXT_DDMN) {
        mdm_unlock (mdmopendevice);
        hangup (fd);
      }
      free (scratch);
      syslog ((FACILITY | LOG_ERR), "error after sending AT+CSDH command.");
      mdmperror ("sms_serv: error after sending AT+CSDH command");
      return (-1);
    }
  }
  else {
    if (context != CONTEXT_DDMN) {
      mdm_unlock (mdmopendevice);
      hangup (fd);
    }
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
    mdmperror ("sms_serv: GSM not responding");
    return (-1);
  }

  free (scratch);
  return (0);
}                                          /* set_csdh () */
/*========================================================*/
int checkset_cpin (struct gsms_def *gsm, int fd, int csfd, int *pji)
/* 
 * Checks for SIM status (set [= READY], PIN requested, PUK
 * requested) and sets the PIN or PUK as requested if not
 * READY. Retries upon error, forces PIN to stored value if
 * requested to provide PUK.
 * Returns -1 on failure.
 */
{
  int newpin;
  int retries;
  cpl_list cpl;
  char* scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));

  if (!scratch) {
    syslog ((FACILITY | LOG_ERR), "can't allocate scratch space -- dying.");
    syserr ("checkset_cpin(): can't allocate scratch space");
  }
  memset (scratch, 0, (BIGBUFF + 1));
  
  /* fill the cpl_list with likely replies */
  cpl_list_init (&cpl);
  cpl_list_insert (&cpl, "READY");
  cpl_list_insert (&cpl, "SIM PIN");
  cpl_list_insert (&cpl, "SIM PUK");
  
  sprintf (scratch, "AT+CPIN?\r");
  tell_gsm (fd, scratch);
  memset (scratch, 0, (BIGBUFF + 1));
  if (get_gsm_resp (fd, scratch, BIGBUFF, 1, cpl)) {
    tellsock (csfd, scratch);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "%s\n", scratch);
    }
    /* check for "READY" -- if not, make it so */
    retries = MAXATTEMPTS;
    while ((strstr (scratch, "READY") == 0) && retries--) {
      /* not ready yet -> asking for PIN or PUK ? */
      /* flag pin_just_in to TRUE anyway */
      *pji = TRUE;
      if (strstr (scratch, "SIM PIN")) {
        /* send PIN */
	sprintf (scratch, "AT+CPIN=\"%s\"\r", gsm->PIN);
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_answer (fd, scratch, BIGBUFF, 15, gsm)) {
          tellsock (csfd, scratch);
          if (debug & DEBUG_PRINTF) {
            fprintf (stderr, "%s\n", scratch);
          }
	  /* check for "OK" */
	  if (strstr (scratch, "OK") == 0) {
	    mdm_unlock (mdmopendevice);
	    hangup (fd);
	    free (scratch);
            free_cpl_list (&cpl);
            syslog ((FACILITY | LOG_ERR), "can't send PIN to GSM or wrong PIN.");
	    mdmperror ("sms_serv: can't send PIN to GSM or wrong PIN");
	    return (-1);
	  }
	}
	else {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	  free (scratch);
          free_cpl_list (&cpl);
          syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
      }
      else if (strstr (scratch, "SIM PUK")) {
        /* send PUK - set new PIN (use the one we already have in conf) */
	newpin = atoi (gsm->PIN);
	syslog ((FACILITY | LOG_WARNING), "I'll try to set new PIN for </dev/%s>.", gsm->device);
	sprintf (scratch, "AT+CPIN=\"%s\",\"%04d\"\r", gsm->PUK, newpin);
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_answer (fd, scratch, BIGBUFF, 20, gsm)) {
          tellsock (csfd, scratch);
          if (debug & DEBUG_PRINTF) {
            fprintf (stderr, "%s\n", scratch);
          }
	  /* check for "OK" */
	  if (strstr (scratch, "OK") == 0) {
	    mdm_unlock (mdmopendevice);
	    hangup (fd);
	    free (scratch);
            free_cpl_list (&cpl);
            syslog ((FACILITY | LOG_ERR), "can't send PUK to GSM or wrong PUK.");
	    mdmperror ("sms_serv: can't send PUK to GSM or wrong PUK");
	    return (-1);
	  }
	}
	else {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	  free (scratch);
          free_cpl_list (&cpl);
          syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
	/* store new PIN in conf. file (GSMDEVFILE) -- no need ! */
	/* log it */
	syslog ((FACILITY | LOG_WARNING), "new PIN for </dev/%s> set to [%04d].", gsm->device, newpin);
      }
      else {                                     /* ERROR */
	mdm_unlock (mdmopendevice);
	hangup (fd);
	free (scratch);
        free_cpl_list (&cpl);
        syslog ((FACILITY | LOG_ERR), "unable to get SIM status info.");
	mdmperror ("sms_serv: unable to get SIM status info");
	return (-1);
      }                 /* if "PIN" elseif "PUK" else ... */
      /* query the status again */
      sprintf (scratch, "AT+CPIN?\r");
      tell_gsm (fd, scratch);
      memset (scratch, 0, (BIGBUFF + 1));
      if (!get_gsm_resp (fd, scratch, BIGBUFF, 1, cpl)) {
	mdm_unlock (mdmopendevice);
	hangup (fd);
	free (scratch);
        free_cpl_list (&cpl);
        syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	mdmperror ("sms_serv: GSM not responding");
	return (-1);
      }
      tellsock (csfd, scratch);
      if (debug & DEBUG_PRINTF) {
        fprintf (stderr, "%s\n", scratch);
      }
    }                                 /* while (!"READY") */
  }
  else {
    mdm_unlock (mdmopendevice);
    hangup (fd);
    free (scratch);
    free_cpl_list (&cpl);
    syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
    mdmperror ("sms_serv: GSM not responding");
    return (-1);
  }
  
  free_cpl_list (&cpl);
  free (scratch);
  return (0);
}                                     /* checkset_cpin () */
/*========================================================*/
int checkset_csca (struct gsms_def *gsm, int fd, int csfd,
                   char *reqsca)
/*
 * Checks whether the default SCA (Service Center Address) stored
 * in the SIM corresponds to the one requested by the server
 * and sets it if it's not the case.
 * Returns -1 on failure.
 */
{
  char* scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));

  if (!scratch) {
    syslog ((FACILITY | LOG_ERR), "can't allocate scratch space -- dying.");
    syserr ("checkset_csca(): can't allocate scratch space");
  }
  memset (scratch, 0, (BIGBUFF + 1));
  
  sprintf (scratch, "AT+CSCA?\r");
  tell_gsm (fd, scratch);
  memset (scratch, 0, (BIGBUFF + 1));
  if (get_gsm_answer (fd, scratch, BIGBUFF, 1, gsm)) {
    tellsock (csfd, scratch);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "%s\n", scratch);
    }
    /* compare with requested SMSC */
    if (strstr (scratch, reqsca) == 0) {
      /* let's set it then */
      sprintf (scratch, "AT+CSCA=\"%s\"\r", reqsca);
      tell_gsm (fd, scratch);
      memset (scratch, 0, (BIGBUFF + 1));
      if (get_gsm_answer (fd, scratch, BIGBUFF, 2, gsm)) {
        tellsock (csfd, scratch);
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "%s\n", scratch);
        }
	/* check for "OK" */
	if (strstr (scratch, "OK") == 0) {
          if (context != CONTEXT_DDMN) {
	    mdm_unlock (mdmopendevice);
	    hangup (fd);
	  }
	  free (scratch);
          syslog ((FACILITY | LOG_ERR), "error after sending AT+CSCA= command.");
	  mdmperror ("sms_serv: error after sending AT+CSCA= command");
	  return (-1);
	}
      }
      else {
        if (context != CONTEXT_DDMN) {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	}
	free (scratch);
        syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	mdmperror ("sms_serv: GSM not responding");
	return (-1);
      }
    }
  }
  else {
    if (context != CONTEXT_DDMN) {
      mdm_unlock (mdmopendevice);
      hangup (fd);
    }
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
    mdmperror ("sms_serv: GSM not responding");
    return (-1);
  }

  free (scratch);
  return (0);
}                                     /* checkset_csca () */
/*========================================================*/
int checkset_cmgf (struct gsms_def *gsm, int fd, int csfd,
                   int reqmode)
/*
 * Checks whether the default mode (text mode or PDU) stored
 * in the SIM corresponds to the one requested by the server
 * and sets it if it's not the case.
 * Returns -1 on failure.
 */
{
  char reqmodestr[10];
  char* scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));

  if (!scratch) {
    syslog ((FACILITY | LOG_ERR), "can't allocate scratch space -- dying.");
    syserr ("checkset_cmgf(): can't allocate scratch space");
  }
  memset (scratch, 0, (BIGBUFF + 1));
  
  /* Build catch string */
  sprintf (reqmodestr, "+CMGF: %d", reqmode);

  sprintf (scratch, "AT+CMGF?\r");
  tell_gsm (fd, scratch);
  memset (scratch, 0, (BIGBUFF + 1));
  if (get_gsm_answer (fd, scratch, BIGBUFF, 1, gsm)) {
    tellsock (csfd, scratch);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "%s\n", scratch);
    }
    /* compare with requested mode */
    if (strstr (scratch, reqmodestr) == 0) {
      /* let's set it then */
      sprintf (scratch, "AT+CMGF=%d\r", reqmode);
      tell_gsm (fd, scratch);
      memset (scratch, 0, (BIGBUFF + 1));
      if (get_gsm_answer (fd, scratch, BIGBUFF, 2, gsm)) {
        tellsock (csfd, scratch);
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "%s\n", scratch);
        }
	/* check for "OK" */
	if (strstr (scratch, "OK") == 0) {
          if (context != CONTEXT_DDMN) {
	    mdm_unlock (mdmopendevice);
	    hangup (fd);
	  }
	  free (scratch);
          syslog ((FACILITY | LOG_ERR), "error after sending AT+CMGF= command.");
	  mdmperror ("sms_serv: error after sending AT+CMGF= command");
	  return (-1);
	}
      }
      else {
        if (context != CONTEXT_DDMN) {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	}
	free (scratch);
        syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	mdmperror ("sms_serv: GSM not responding");
	return (-1);
      }
    }
  }
  else {
    if (context != CONTEXT_DDMN) {
      mdm_unlock (mdmopendevice);
      hangup (fd);
    }
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
    mdmperror ("sms_serv: GSM not responding");
    return (-1);
  }

  free (scratch);
  return (0);
}                                     /* checkset_cmgf () */
/*========================================================*/
int check_creg (struct gsms_def *gsm, int fd, int csfd)
/*
 * Gets and checks registration status. When a "registered"
 * status is found, 0 is returned.
 * Returns -1 on failure ("not registered" or "unknown").
 */
{
  int regstat;
  char sname[10];
  char* scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));

  if (!scratch) {
    syslog ((FACILITY | LOG_ERR), "can't allocate scratch space -- dying.");
    syserr ("check_creg(): can't allocate scratch space");
  }
  memset (scratch, 0, (BIGBUFF + 1));
  
  /* server name */
  switch (context) {
    case CONTEXT_SEND: {
      strcpy (sname, "SRV CHLD");
      break;
    }
    
    case CONTEXT_MBCI:
    case CONTEXT_MBCO: {
      strcpy (sname, "MBC");
      break;
    }
    
    case CONTEXT_DDMN: {
      strcpy (sname, "DD MBC");
      break;
    }
    
    default: {
      strcpy (sname, "[??]");
      break;
    }
  }
  
  sprintf (scratch, "AT+CREG?\r");
  tell_gsm (fd, scratch);
  memset (scratch, 0, (BIGBUFF + 1));
  if (get_gsm_answer (fd, scratch, BIGBUFF, 1, gsm)) {
    tellsock (csfd, scratch);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "%s\n", scratch);
    }
    /* check for 1 or 5 status (= registered) */
    if (strstr (scratch, "+CREG:") == 0) {
      if (context != CONTEXT_DDMN) {
        mdm_unlock (mdmopendevice);
        hangup (fd);
      }
      free (scratch);
      syslog ((FACILITY | LOG_ERR), "didn't get +CREG: info - can't send.");
      mdmperror ("sms_serv: didn't get +CREG: info - can't send");
      return (-1);
    }
    regstat = getregstat (scratch);
    switch (regstat) {
      case -1: {                           /* parse error */
        if (context != CONTEXT_DDMN) {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	}
        sprintf (scratch, "(can't parse registration status)\r\n");
	tellsock (csfd, scratch);
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "%s: (can't parse registration status)\n", sname);
        }
	free (scratch);
	syslog ((FACILITY | LOG_ERR), "can't parse reg. status - can't send.");
	mdmperror ("sms_serv: can't parse reg. status - can't send");
	return (-1);
      }
      
      case 0: {                         /* not registered */
        if (context != CONTEXT_DDMN) {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	}
        sprintf (scratch, "(not registered on network, not searching)\r\n");
	tellsock (csfd, scratch);
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "%s: (not registered on network, not searching)\n",
	          sname);
        }
	free (scratch);
	syslog ((FACILITY | LOG_ERR), "not registered, not searching - can't send.");
	mdmperror ("sms_serv: not registered, not searching - can't send");
	return (-1);
      }
      
      case 1: {                           /* home network */
        sprintf (scratch, "(Registered, Home network)\r\n");
	tellsock (csfd, scratch);
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "%s: (Registered, Home network)\n", sname);
        }
	break;
      }
      
      case 2: {                              /* searching */
        if (context != CONTEXT_DDMN) {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	}
        sprintf (scratch, "(not registered yet, searching -- try again)\r\n");
	tellsock (csfd, scratch);
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "%s: (not registered yet, searching -- try again)\n",
	          sname);
        }
	free (scratch);
	syslog ((FACILITY | LOG_ERR), "not registered yet, searching - can't send.");
	mdmperror ("sms_serv: not registered yet, searching - can't send");
	return (-1);
      }
      
      case 3: {                                 /* denied */
        if (context != CONTEXT_DDMN) {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	}
        sprintf (scratch, "(registration on network denied)\r\n");
	tellsock (csfd, scratch);
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "%s: (registration on network denied)\n", sname);
        }
	free (scratch);
	syslog ((FACILITY | LOG_ERR), "registration denied - can't send.");
	mdmperror ("sms_serv: registration denied - can't send");
	return (-1);
      }
      
      case 4: {                                 /* failed */
        if (context != CONTEXT_DDMN) {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	}
        sprintf (scratch, "(registration failed, cause unknown)\r\n");
	tellsock (csfd, scratch);
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "%s: (registration failed, cause unknown)\n", sname);
        }
	free (scratch);
	syslog ((FACILITY | LOG_ERR), "registration failed, cause unknown - can't send.");
	mdmperror ("sms_serv: registration failed, cause unknown - can't send");
	return (-1);
      }
      
      case 5: {                                /* roaming */
        sprintf (scratch, "(Registered, Roaming)\r\n");
	tellsock (csfd, scratch);
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "%s: (Registered, Roaming)\n", sname);
        }
	break;
      }
      
      default: {                               /* unknown */
        if (context != CONTEXT_DDMN) {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	}
        sprintf (scratch, "(registration failed, unexpected status %d)\r\n",
	        regstat);
	tellsock (csfd, scratch);
        if (debug & DEBUG_PRINTF) {
          fprintf (stderr, "%s: (registration failed, unexpected status %d)\n",
	          sname, regstat);
        }
	free (scratch);
	syslog ((FACILITY | LOG_ERR), "registration failed, unexpected status %d - can't send.",
	       regstat);
	mdmperror ("sms_serv: registration failed, unexpected status - can't send");
	return (-1);
      }
    }                                 /* switch (regstat) */
  }
  else {
    if (context != CONTEXT_DDMN) {
      mdm_unlock (mdmopendevice);
      hangup (fd);
    }
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
    mdmperror ("sms_serv: GSM not responding");
    return (-1);
  }

  free (scratch);
  return (0);
}                                        /* check_creg () */
/*========================================================*/
int initialize_gsm (struct gsms_def *gsm)
{
  int fd;
  int pin_just_in;

  /*--------------------------------------Initializations */
  use_fast = (gsm->flags & FAST_RESPONSE);

  pin_just_in = FALSE; /* set to true if PIN needed be provided */

  /* open modem line */
  fd = lopen_mdm_line (gsm->device);
  if (fd < 0) {
    syslog ((FACILITY | LOG_ERR), "call to lopen_mdm_line() failed.");
    mdmperror ("sms_serv: lopen_mdm_line() failed");
    return (-1);
  }

  /*------------set GSM to "verbose" error reporting mode */
  if (set_cmee (gsm, fd, 0) == -1) {
    return (-1);
  }

  /*---------------------------then, check for SIM status */
  if (checkset_cpin (gsm, fd, 0, &pin_just_in) == -1) {
    return (-1);
  }

  /*--------------------------------------sleep if needed */
  /* There seems to be a timing issue in "fast" mode when
   * the AT+CNMI command is issued too fast after the last
   * AT+CPIN? and we get a "+CMS ERROR: 515" (= module
   * not ready or busy processing command). Try sleeping a
   * few secs to solve that. */
  if (use_fast && pin_just_in) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "INIT: (pause %d secs. for PIN code post-processing)\n",
              gsm->PINpause);
    }
    sleep (gsm->PINpause);
  }
  
  /*------------set "no notify for incoming SMS messages" */
  if (set_cnmi_off (gsm, fd, 0) == -1) {
    return (-1);
  }

  /*-----------set message format to device-level default */
  if (checkset_cmgf (gsm, fd, 0, gsm->defmode) == -1) {
    return (-1);
  }

  /*-------------------------set "full SMS header detail" */
  /* Beware: only meaningfull (and supported) in text mode AND */
  /* even so only on specific models ! (specifically not the */
  /* PCFF900 */
  if ((gsm->defmode == SMS_MODE_TEXT) &&
      (gsm->capmatrix.capflags & HAS_CSDH_1)) {
    if (set_csdh (gsm, fd, 0) == -1) {
      return (-1);
    }
  }                 /* if (gsm->defmode == SMS_MODE_TEXT) */

  /*----------------Check stored SCA against default SMSC */
  if (checkset_csca (gsm, fd, 0, gsm->defsca) == -1) {
    return (-1);
  }

  /*---------------------Am I registered on the network ? */
  if (!(gsm->capmatrix.capflags & NO_CREG_CHECK))
  {
    if (check_creg (gsm, fd, 0) == -1) {
      return (-1);
    }
  }

  /*------------------------------------Set message store */
  /* make sure the message store is accessible */
  if (gsm->capmatrix.capflags & SET_MSG_STORE) {
    if (set_cpms (gsm, fd, 0) == -1) {
      return (-1);
    }
  }

  /*----------------------------------Close communication */
  mdm_unlock (mdmopendevice);
  hangup (fd);
  return (0);
}                                    /* initialize_gsm () */
/*========================================================*/
/*################## Queue files handling ################*/
/*========================================================*/
int file_counter_op (char *fn, int op)
/*
 * Gets, resets or increments a counter located in the "seconds"
 * part of a file last modification timestamp.
 * Returns the current counter value when successfull, -1 otherwise.
 */
{
  struct stat stat_info;
  struct tm *tm;
  time_t new_time;
  struct utimbuf timbuf;

  if (stat (fn, &stat_info) == -1) {
    return (-1);
  }

  tm = localtime (&stat_info.st_mtime);
  if (! tm) {
    return (-1);
  }
  
  switch (op) {
    case SMS_FC_RST: {
      tm->tm_sec = 0;
      break;
    }
    
    case SMS_FC_INC: {
      tm->tm_sec++;
      if (tm->tm_sec > 59) {
	tm->tm_sec = 0;
      }
      break;
    }
    
    case SMS_FC_GET: {
      return (tm->tm_sec);
      break;
    }
    
    default: {
      fprintf (stderr, "sms_serv: file_counter_op(): unsupported op <%d>\n", op);
      return (-1);
      break;
    }
  }                                        /* switch (op) */
  
  new_time = mktime (tm);
  
  timbuf.actime = stat_info.st_atime;
  timbuf.modtime = new_time;
  
  if (utime (fn, &timbuf) == -1) {
    return (-1);
  }

  return (tm->tm_sec);
}                                   /* file_counter_op () */
/*========================================================*/
int make_queue_file (int pri)
/*
 * Attempts to create a spool queue file with a randomly
 * generated file name. Returns the random file ID in case
 * of success, -1 in the event of failure.
 */
{
  extern int errno;
  unsigned int seed;
  time_t t;
  int msgid;
  int rand_value, crv;
  char *queuefile;
  int queuefd;
  int retry;

  /*--------------------------------------Initializations */
  msgid = -1;
  retry = 0;
  
  queuefile = (char *) malloc ((MAXPATHLEN + 1) * sizeof (char));
  if (! queuefile) {
    syserr ("make_queue_file(): can't malloc() queuefile");
  }
  queuefile[0] = '\0';
  
  /* add some entropy to the random generator */
  time (&t);
  seed = 1 + (unsigned int) ((MAXINT_32 * t * getpid ()) / ((MAXINT_32 * MAXPID_F) + 1.0));
  srand (seed);

  /*-------------------------------------------Processing */
  /* now draw a number */
  rand_value = rand ();
  /* fit it to the required scope (1 -> LOCAL_RAND_MAX) */
  crv =  1 + (int) (LOCAL_RAND_MAX * rand_value / (RAND_MAX + 1.0));
  /* make sure the first digit is the message priority */
  crv = (pri * (LOCAL_RAND_MAX + 1)) + crv;
  /* build candidate queue file name */
  sprintf (queuefile, "%s/q%08d", OBOX_SPOOL, crv);
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "trying to create queue file <%s>\n", queuefile);
  }
  while (((queuefd = open (queuefile, (O_RDWR | O_CREAT | O_EXCL), 0600)) == -1) &&
         (retry < MAXRETRIES)) {
    switch (errno) {
      case EEXIST: {
	/* draw a new number */
	rand_value = rand ();
	/* fit it to the required scope (1 -> LOCAL_RAND_MAX) */
	crv =  1 + (int) (LOCAL_RAND_MAX * rand_value / (RAND_MAX + 1.0));
        /* make sure the first digit is the message priority */
        crv = (pri * (LOCAL_RAND_MAX + 1)) + crv;
	/* build candidate queue file name */
	sprintf (queuefile, "%s/q%08d", OBOX_SPOOL, crv);
	if (debug & DEBUG_PRINTF) {
	  fprintf (stderr, "trying to create queue file <%s>\n", queuefile);
	}
        break;
      }
      
      default: {
        syserr ("make_queue_file(): can't create queue file");
      }
    }                                   /* switch (errno) */
    retry++;
  }                        /* while ((queuefd = open (... */
  /* did we get it ? */
  if (queuefd != -1) {
    close (queuefd);
    msgid = crv;
    syslog ((FACILITY | LOG_INFO), "created queue file <%s>.", queuefile);
  }

  /*-------------------------------------------Conclusion */
  free (queuefile);
  return (msgid);
}                                   /* make_queue_file () */
/*========================================================*/
int queue_get_lock (int msgid)
/*
 * Attempts to create a spool lock file for the requested
 * queue file ID. Returns TRUE in case of success, FALSE
 * in the event of failure.
 */
{
  extern int errno;
  char *ctrlfile;
  int ctrlfd;
  int retval = TRUE;

  /*--------------------------------------Initializations */
  ctrlfile = (char *) malloc ((MAXPATHLEN + 1) * sizeof (char));
  if (! ctrlfile) {
    syserr ("queue_get_lock(): can't malloc() ctrlfile");
  }
  ctrlfile[0] = '\0';
  
  /*-------------------------------------------Processing */
  /* build candidate control file name */
  sprintf (ctrlfile, "%s/c%08d", OBOX_SPOOL, msgid);
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "trying to create control file <%s>\n", ctrlfile);
  }
  if ((ctrlfd = open (ctrlfile, (O_RDWR | O_CREAT | O_EXCL), 0600)) == -1) {
    switch (errno) {
      case EEXIST: {
	if (debug & DEBUG_PRINTF) {
	  fprintf (stderr, "can't create control file <%s> - already there\n", ctrlfile);
	}
	retval = FALSE;
        break;
      }
      
      default: {
        syserr ("queue_get_lock(): can't create queue control file (lock)");
      }
    }                                   /* switch (errno) */
  }                            /* if ((ctrlfd = open (... */
  /* did we get it ? */
  if (ctrlfd != -1) {
    close (ctrlfd);
    syslog ((FACILITY | LOG_INFO), "created control file <%s>.", ctrlfile);
  }

  /*-------------------------------------------Conclusion */
  free (ctrlfile);
  return (retval);
}                                    /* queue_get_lock () */
/*========================================================*/
int queue_remove_lock (int msgid)
/*
 * Attempts to remove a spool lock file for the requested
 * queue file ID. Returns TRUE in case of success, FALSE
 * in the event of failure.
 */
{
  char *ctrlfile;
  int retval = FALSE;

  /*--------------------------------------Initializations */
  ctrlfile = (char *) malloc ((MAXPATHLEN + 1) * sizeof (char));
  if (! ctrlfile) {
    syserr ("queue_remove_lock(): can't malloc() ctrlfile");
  }
  ctrlfile[0] = '\0';
  
  /*-------------------------------------------Processing */
  /* build candidate control file name */
  sprintf (ctrlfile, "%s/c%08d", OBOX_SPOOL, msgid);
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "trying to remove control file <%s>\n", ctrlfile);
  }
  if (file_exist (ctrlfile)) {
    if (unlink (ctrlfile) != -1) {
      syslog ((FACILITY | LOG_INFO), "removed control file <%s>.", ctrlfile);
      retval = TRUE;
    }
  }

  /*-------------------------------------------Conclusion */
  free (ctrlfile);
  return (retval);
}                                 /* queue_remove_lock () */
/*========================================================*/
int sms_queue_file ()
{
  int msgid;
  extern struct symbols symbols;
  FILE *q_file;
  char *q_filename;
  
  /*--------------------------------------Initializations */
  q_filename = (char *) malloc ((MAXPATHLEN + 1) * sizeof (char));
  if (! q_filename) {
    syserr ("sms_queue_file(): can't malloc() q_filename");
  }
  q_filename[0] = '\0';
  
  /*-------------------------------------------Processing */
  if ((msgid = make_queue_file (symbols.priority)) != -1) {
    /* build the queue file name based on the msgid */
    sprintf (q_filename, "%s/q%08d", OBOX_SPOOL, msgid);
    /* try to get a lock on it */
    if (! queue_get_lock (msgid)) {
      syslog ((FACILITY | LOG_ERR), "can't lock queue file <%s> for writing.",
             q_filename);
      syserr ("sms_queue_file(): can't lock queue file for writing");
    }
    /* Now open the queue file for writing */
    if ((q_file = fopen (q_filename, "w+")) == NULL) {
      syslog ((FACILITY | LOG_ERR), "call to fopen() failed when writing queue file <%s>.",
             q_filename);
      syserr ("sms_queue_file(): queue file open failed: can't fopen()");
    }
    fprintf (q_file, "%s,%s,%s,%d,\"%s\"\n", symbols.user, symbols.destgsm,
            symbols.smsc, symbols.mode, symbols.message);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "queue file <%s> written\n", q_filename);
    }
    fclose (q_file);
    /* initialize its "queue run" counter */
    if (file_counter_op (q_filename, SMS_FC_RST) == -1) {
      syslog ((FACILITY | LOG_ERR), "can't initialize QR-counter for queue file <%s>.",
             q_filename);
      syserr ("sms_queue_file(): can't initialize QR-counter for queue file");
    }
    
    /* try to remove the lock file */
    if (! queue_remove_lock (msgid)) {
      syslog ((FACILITY | LOG_ERR), "can't unlock queue file <%s>.",
             q_filename);
      syserr ("sms_queue_file(): can't unlock queue file");
    }
  }
  
  /*-------------------------------------------Conclusion */
  free (q_filename);
  return (msgid);
}                                    /* sms_queue_file () */
/*========================================================*/
int is_queue_file (char *filename)
{
  char *p;
  int i;
  
  p = filename;
  if (p[0] != 'q') {
    return (FALSE);
  }
  p++;
  if (strlen (p) != 8) {
    return (FALSE);
  }
  for (i = 0; i < 8; i++) {
    if (! isdigit (p[i])) {
      return (FALSE);
    }
  }
  return (TRUE);
}                                     /* is_queue_file () */
/*========================================================*/
int get_msgid (char *filename)
{
  char *p;
  
  p = filename;
  if (p[0] != 'q') {
    return (-1);
  }
  p++;
  
  return (atoi (p));
}                                         /* get_msgid () */
/*========================================================*/
int file_list_queue (int csfd)
{
  char *buffer;
  char *qitem;
  char *qtime;
  char *fqqfname;
  char *fqlfname;
  char parse_error[] = "***??***";
  char empty_string[] = "";
  char *user;
  char *dest;
  char *text;
  int qlen = 0;
  struct dirent **dentry;
  int nselfiles;
  int i;
  struct stat stat_info;
  struct tm *tm;
  int msgid;
  char priority;
  int file_counter;
  char ls;
  FILE *q_file;
  
  /*--------------------------------------Initializations */
  ls = ' ';
  
  buffer = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (! buffer) {
    syserr ("file_list_queue(): can't malloc() buffer");
  }
  buffer[0] = '\0';
  
  qitem = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (! qitem) {
    syserr ("file_list_queue(): can't malloc() qitem");
  }
  qitem[0] = '\0';
  
  qtime = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
  if (! qtime) {
    syserr ("file_list_queue(): can't malloc() qtime");
  }
  qtime[0] = '\0';
  
  fqqfname = (char *) malloc ((MAXPATHLEN + 1) * sizeof (char));
  if (! fqqfname) {
    syserr ("file_list_queue(): can't malloc() fqqfname");
  }
  fqqfname[0] = '\0';
  
  fqlfname = (char *) malloc ((MAXPATHLEN + 1) * sizeof (char));
  if (! fqlfname) {
    syserr ("file_list_queue(): can't malloc() fqlfname");
  }
  fqlfname[0] = '\0';
  
  /*--------------------------------------Processing Loop */
  sprintf (buffer, "ID       | User         | Destination    | Stored       | Len. | QR | P \r\n"
                   "---------+--------------+----------------+--------------+------+----+---\r\n");
  tellsock (csfd, buffer);
  
  /* fill up a sorted list of directory entries */
  /* WARNING: scandir() is not a POSIX function. */
  if ((nselfiles = scandir (OBOX_SPOOL, &dentry, 0, alphasort)) == -1) {
    syslog ((FACILITY | LOG_ERR), "scandir() failed on outbox spool dir <%s>.",
           OBOX_SPOOL);
    syserr ("file_list_queue(): scandir() failed on outbox spool dir");
  }
  
  for (i = 0; i < nselfiles; i++) {
    if (is_queue_file (dentry[i]->d_name)) {
      /* build fully qualified queuefile name */
      sprintf (fqqfname, "%s/%s", OBOX_SPOOL, dentry[i]->d_name);
      
      /*------------------extract details from queue file */
      /*.......................................Message ID */
      msgid = get_msgid (dentry[i]->d_name);
      
      /*.........................................Priority */
      priority = dentry[i]->d_name[1];
      
      /*.........................................Locked ? */
      /* build fully qualified control file name */
      sprintf (fqlfname, "%s/c%08d", OBOX_SPOOL, msgid);
      if (file_exist (fqlfname)) {
        ls = '*';
      }
      
      /*.....................................Queued since */
      if (stat (fqqfname, &stat_info) == -1) {
        syslog ((FACILITY | LOG_ERR), "can't stat() queue file <%s>.",
               fqqfname);
	syserr ("file_list_queue(): can't stat() queue file");
      }
      tm = localtime (&stat_info.st_mtime);
      strftime (qtime, MINIBUFF, "%Y%m%d%H%M", tm);
      
      /*...................................Contents field */
      if ((q_file = fopen (fqqfname, "r")) == NULL) {
	syslog ((FACILITY | LOG_ERR), "call to fopen() failed when reading queue file <%s>.",
               fqqfname);
	syserr ("file_list_queue(): queue file read failed: can't fopen()");
      }
      memset (qitem, 0, (BUFFSIZE + 1));
      if (fgets (qitem, (BUFFSIZE + 1), q_file) == NULL) {
	if (debug & DEBUG_PRINTF) {
	  fprintf (stderr, "Empty queue item: <%s>\n", fqqfname);
	}
      }
      fclose (q_file);
      /* parse line */
      if (strchr (qitem, ',')) {
	if ((user = strtok (qitem, ",")) == NULL) {
          user = parse_error;
	  if (debug & DEBUG_PRINTF) {
	    fprintf (stderr, "Can't extract user field from:\n[%s]\n", qitem);
	    fprintf (stderr, "User now set to: [%s]\n", user);
	  }
	}
	if ((dest = strtok (NULL, ",")) == NULL) {
          dest = parse_error;
	  if (debug & DEBUG_PRINTF) {
	    fprintf (stderr, "Can't extract dest field from:\n[%s]\n", qitem);
	    fprintf (stderr, "Dest now set to: [%s]\n", dest);
	  }
	}
	strtok (NULL, ",");               /* smsc -- ignore */
	strtok (NULL, ",");               /* mode -- ignore */
	if (text = strtok (NULL, "\n")) {
          dequote (text);
	}
	else {
          text = empty_string;
	}
      }
      else {
        user = parse_error;
        dest = parse_error;
        text = empty_string;
	if (debug & DEBUG_PRINTF) {
	  fprintf (stderr, "Can't parse queue item:\n[%s]\n", qitem);
	}
      }
      
      /*.....................................File counter */
      if ((file_counter = file_counter_op (fqqfname, SMS_FC_GET)) == -1) {
        syslog ((FACILITY | LOG_ERR), "can't stat() queue file <%s>.",
               fqqfname);
	syserr ("file_list_queue(): can't stat() queue file");
      }
      
      /*------------------------------------------display */
      sprintf (buffer, "%8d%c| %-12s | %-14s | %-12s | %4d | %2d | %c\n",
              msgid, ls, user, dest, qtime, strlen (text), file_counter, priority);
      tellsock (csfd, buffer);
      qlen++;
    }                            /* if (is_queue_file ()) */
    free (dentry[i]);
  }                                             /* for () */
  if (qlen) {
    sprintf (buffer, "---------+--------------+----------------+--------------+------+----+---\r\n");
    tellsock (csfd, buffer);
  }

  /*------------------------------------------Conclusions */
  /* free what needs to be */
  free (buffer);
  free (qitem);
  free (qtime);
  free (fqqfname);
  free (fqlfname);
  free (dentry);
  return (qlen);
}                                   /* file_list_queue () */
/*========================================================*/
int file_queue_delete (int msgid)
/*
 * Removes the requested message ID from the queue. Returns
 * 0 on success, -1 on failure.
 */
{
  int retval = 0;
  char *fqqfname;
  
  fqqfname = (char *) malloc ((MAXPATHLEN + 1) * sizeof (char));
  if (! fqqfname) {
    syserr ("file_queue_delete(): can't malloc() fqqfname");
  }
  fqqfname[0] = '\0';
  
  if (queue_get_lock (msgid)) {
    /* build fully qualified queue file name */
    sprintf (fqqfname, "%s/q%08d", OBOX_SPOOL, msgid);
    /* queue file is locked -- remove it */
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "trying to remove queue file <%s>\n", fqqfname);
    }
    if (file_exist (fqqfname)) {
      if (unlink (fqqfname) != -1) {
	syslog ((FACILITY | LOG_INFO), "removed queue file <%s> on user's request.", fqqfname);
      }
      else {
        retval = -1;
      }
    }
    /* remove the lock file */
    if (! queue_remove_lock (msgid)) {
      retval = -1;
    }
  }
  else {
    /* can't lock queue file */
    retval = -1;
  }

  free (fqqfname);
  return (retval);
}                                 /* file_queue_delete () */
/*========================================================*/
/*####################### Semaphores #####################*/
/*========================================================*/
int set_semvalue (int sem_id, int value)
/* returns 0 on success, -1 on failure */
{
#ifdef _SEM_SEMUN_UNDEFINED		/* see <bits/sem.h> */
  union semun {
    int val;				/* value for SETVAL */
    struct semid_ds *buf;		/* buffer for IPC_STAT & IPC_SET */
    unsigned short int *array;		/* array for GETALL & SETALL */
    struct seminfo *__buf;		/* buffer for IPC_INFO */
  } sem_union;
#else  
  union semun sem_union;
#endif                     /* #ifdef _SEM_SEMUN_UNDEFINED */
  
  sem_union.val = value;
  return (semctl (sem_id, 0, SETVAL, sem_union));
}                                      /* set_semvalue () */
/*========================================================*/
int sem_wait (int sem_id)
/* waits for the semaphore to be decreased */
{
  struct sembuf sem_b;
  
  sem_b.sem_num = 0;                   /* sem array index */
  sem_b.sem_op = -1;             /* value to inc. sem. by */
  sem_b.sem_flg = SEM_UNDO;
  
  /* --- can sleep here --- */
  return (semop (sem_id, &sem_b, 1));
}                                      /* int sem_wait () */
/*========================================================*/
int sem_decreq (int sem_id)
/* attempts to decrease the semaphore - return error if fails */
{
  struct sembuf sem_b;
  
  sem_b.sem_num = 0;                   /* sem array index */
  sem_b.sem_op = -1;             /* value to inc. sem. by */
  sem_b.sem_flg = (IPC_NOWAIT | SEM_UNDO);
  
  return (semop (sem_id, &sem_b, 1));
}                                    /* int sem_decreq () */
/*========================================================*/
int sem_signal (int sem_id)
/* increases the semaphore - always succeeds */
{
  struct sembuf sem_b;
  
  sem_b.sem_num = 0;                   /* sem array index */
  sem_b.sem_op = 1;              /* value to inc. sem. by */
  sem_b.sem_flg = SEM_UNDO;
  
  return (semop (sem_id, &sem_b, 1));
}                                    /* int sem_signal () */
/*========================================================*/
/*######################### Misc #########################*/
/*========================================================*/
void cleanup_mdm (const char *msg, int fd)
/*
 * report error and close the given modem line if the file descriptor is >= 0
 * and the global modem device. Can be used with a null message to close down
 * the connection properly when there is no error.
 */
{
    if (msg) {
      /* needed straight away before error code is overwritten ?? */
      mdmperror ((char *) msg);
      syslog ((FACILITY | LOG_ERR), msg);
    }
    mdm_unlock (mdmopendevice);
    if (fd >= 0) {
      hangup (fd);
    }
}                                       /* cleanup_mdm () */
/*==========================================================
 * EOF : serv_stuff.c
 *===================*/

