%{
/*==========================================================
 * Program : server.l                      Project : smslink
 * Author  : Philippe Andersson.
 * Date    : 11/04/04
 * Version : 0.18b
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 * Comment : Lexer module for the parser (server.y).
 *
 * Modification History :
 * - 0.01a (19/08/98) : Initial release.
 * - 0.02a (21/08/98) : Made it recognize text type for some
 *   returned token value (T_PHNUM & T_TEXT).
 * - 0.03a (22/08/98) : Added recognition for '+' in PHNUM,
 *   and for "he" as short for "help".
 * - 0.04a (28/08/98) : Removed inclusion of server.h (was
 *   useless anyway), since was merged into sms_serv.h.
 * - 0.05a (30/08/98) : Added handling for both [LF] and [CR][LF]
 *   sequences ("\r\n"), cause [return] translates to \r\n when
 *   sent through telnet. Moved yyerror () from server.y to here.
 * - 0.06a (24/09/98) : Changed [LF] to [CR][LF] in messages
 *   sent through the socket.
 * - 0.07a (20/10/98) : Added support for 'devicelist' command.
 * ++++ Switched to Beta ++++
 * - 0.08b (21/10/98) : Noticed quoted 'phnum' were considered
 *   'text'. So added optional quote-endquote sequence in 'phnum'
 *   syntax and moved 'phnum' pattern before 'text', otherwize
 *   'text' would play catch-all and hide it.
 * - 0.09b (23/10/98) : Added ' ' as a legal char. in a 'phnum'.
 * - 0.10b (16/05/99) : Added "uptime" command.
 * - 0.11b (02/12/99) : Added "aclist" command.
 * - 0.12b (28/03/00) : Added the SCHAR pattern ("single char"),
 *   that returns the T_SCHAR token when matched. Used for "set
 *   mode = x" support.
 * - 0.13b (14/05/01) : Added a PARAM element to the "dl"
 *   command (to support the "long" form).
 * - 0.14b (08/08/02) : Added "queue" command.
 * - 0.15b (03/09/02) : Added "post" command.
 * - 0.16b (05/09/02) : Added "qstat" and "qdel" commands.
 * - 0.17b (08/04/04) : Added "purge" command + MPARAM support.
 * - 0.18b (11/04/04) : Added a "BEGIN INITIAL" statement to
 *   resynchronize on EOL and fix some parser flakyness. Finally
 *   found a way to reset the input buffer on parse errors (call
 *   to yy_flush_buffer() in yyerror()).
 *========================================================*/

#include "server.tab.h"
#include "sms_serv.h"

#undef YY_INPUT
#define YY_INPUT(b, r, ms)  (r = my_yyinput(b, ms))

/*-------------------------------------External variables */
/* program specific */
extern int csfd;                      /* client socket FD */
extern char *buffer;    /* scratch space for C/S dialogue */

int yyparse ();
%}

%s PARAM
%s MPARAM
%s ASSIG
%s NUMID

WS		[ \t]+
PHNUM		(\")?(\+)?[0-9]+[\/\.\- 0-9]*(\")?
TEXT		\"[^\"\n]*\"
VIDENT		[a-zA-Z][a-zA-Z0-9]+
SCHAR		[a-zA-Z]

%%

{WS}		;

^he |
^help		{
		BEGIN PARAM;
		return T_HELP;
		}

^clear		{
		BEGIN PARAM;
		return T_CLEAR;
		}

^send		{ return T_SEND; }

^queue		{ return T_QUEUE; }

^post		{ return T_POST; }

^dl |
^devicelist	{
		BEGIN PARAM;
		return T_DEVLIST;
		}

^ut |
^uptime		{ return T_UPTIME; }

^acl |
^aclist		{ return T_ACLIST; }

^quit |
^exit |
^bye		{ return T_QUIT; }

^stats		{ return T_STATS; }

^sh |
^show		{
		BEGIN PARAM;
		return T_SHOW;
		}

^set		{
		BEGIN ASSIG;
		return T_SET;
		}

^qstat		{ return T_QSTAT; }

^qdel		{
		BEGIN NUMID;
		return T_QDEL;
		}

^purge		{
		BEGIN MPARAM;
		return T_PURGE;
		}

<PARAM>[a-zA-Z]+ { return T_PARAM; }

<MPARAM>[a-zA-Z]+([ \t]+[a-zA-Z]+)* { return T_PARAM; }

<ASSIG>{VIDENT}	{ return T_NAME; }

<NUMID>[0-9]+	{ return T_NUMID; }

{PHNUM}		{
		yylval.string = (char *) strdup (yytext);
		return T_PHNUM;
		}

{TEXT}		{
		yylval.string = (char *) strdup (yytext);
		return T_TEXT;
		}

{SCHAR}		{
		yylval.string = (char *) strdup (yytext);
		return T_SCHAR;
		}

\r\n |
\n		{
                BEGIN INITIAL;
                return (0);
		}

.		{ return (yytext[0]); }


%%
/*========================================================*/
void yyerror (msg)
const char *msg;
{
  sprintf (buffer, "ERROR: %s near <%s>\r\n", msg, yytext);
  tellsock (csfd, buffer);
  /* The rest of the input is not parseable either: flush it */
  yy_flush_buffer (YY_CURRENT_BUFFER);
}
/*==========================================================
 * EOF : server.l
 *===================*/
