%{
/*==========================================================
 * Program : server.y                      Project : smslink
 * Author  : Philippe Andersson.
 * Date    : 09/04/04
 * Version : 0.67b
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 * Comment : Parser module for SMS server
 *
 * Modification History :
 * - 0.01a (19/08/98) : Initial release.
 * - 0.02a (21/08/98) : Implemented "variable" assignment and
 *   individual clearing.
 * - 0.03a (21/08/98) : Added support for prompt.
 * - 0.04a (21/08/98) : Added dequoting for strings.
 * - 0.05a (22/08/98) : Added detailed help handling and individ.
 *   show for items. Also implemented a separate exit function.
 * - 0.06a (24/08/98) : Completed help system. Also stuffed the
 *   conclude() function a bit.
 * - 0.10a (27/08/98) : Merged the parser module into the main
 *   sms_serv program. Moved yyerror () to server.l.
 * - 0.11a (01/09/98) : Completed output migration to socket
 *   (printf's to sprintf's).
 * - 0.12a (01/09/98) : Start implementing restricted access
 *   to non-sharable resources (modems) through semaphores.
 * - 0.13a (03/09/98) : Completed implementation of crit.
 *   sections through semaphores and shared memory.
 * - 0.14a (06/09/98) : Inserted #ifdef's to have it compile
 *   under RedHat (GNU Libc 6). Also implemented minimal validation
 *   on message parameters (presence checking).
 * - 0.15a (24/09/98) : Changed [LF] to [CR][LF] in messages
 *   sent through the socket.
 * - 0.16a (27/09/98) : Added positive acknowledgement for
 *   variable setting ("Ok") for error checking on client side.
 *   Improved some help messages in give_help().
 * - 0.17a (01/10/98) : Added logging detail (variable setting).
 * - 0.18a (19/10/98) : Changed 'struct modem_def' to 'struct
 *   gsms_def' and 'modemsdef' to 'gsmdevices'. Cosmetics.
 * - 0.20a (20/10/98) : Merge the 'spit.c' source into this.
 *   Implement the actual SMS sending code.
 * - 0.21a (20/10/98) : Implemented 'devicelist' command.
 * - 0.22a (21/10/98) : Added a signal trap for SIGPIPE, in
 *   order to avoid getting locked GSM modules when a client
 *   dies or quits unexpectedly (in the critical section).
 * ++++ Switched to Beta ++++
 * - 0.23b (21/10/98) : Added deslashdot() function to clean
 *   phone numbers before sending them to the GSM.
 * - 0.24b (22/10/98) : Improved SIGPIPE handling by adding
 *   'instance_is_set' variable telling whether the value of
 *   'instance' should be considered relevant.
 * - 0.25b (23/10/98) : Added handling for 'user' parameter,
 *   and improved checking on phone number validity. 'user'
 *   parameter also be made required. Corrected timeout values
 *   in PIN / PUK transmission, and a bug in CSCA-related sprintf
 *   format.
 * - 0.26b (05/11/98) : Corrected an inconsistency in the error
 *   message related to catching SIGPIPE signal.
 * - 0.27b (13/12/98) : Moved to sigaction() signal handling.
 *   Added handler for SIGTERM in crit. section. Included
 *   handling of new member "owner" in struct gsms_def.
 * - 0.28b (14/12/98) : Corrected a boundary-checking bug in the
 *   search for a free GSM instance loop.
 * - 0.29b (16/05/99) : Added "uptime" command.
 * - 0.30b (18/05/99) : Refined the uptime display. Solved a
 *   bug related to the buffer length in "help" command.
 * - 0.31b (06/06/99) : Added "days" counter to uptime display.
 *   Adapted on-line help accordingly. Also added a command to 
 *   turn off notification for incoming SMS messages in send_sms().
 * - 0.32b (28/06/99) : In send_sms(), moved the AT+CNMI command
 *   after the PIN check routine. PIN-Activated SIM is required
 *   for AT+CNMI.
 * - 0.33b (29/06/99) : Moved 3 functions over to stuff.c and
 *   made them public (misc. string handling functions).
 * - 0.34b (20/09/99) : Improved the "show" command to also
 *   display the current length of the message string.
 * - 0.35b (02/12/99) : Improved default SMSC handling in
 *   send_sms(). Now takes its default value from the configuration
 *   file only (/etc/gsmdevices). Adapted on-line help accordingly.
 *   Added "aclist" command and related help. Cosmetics.
 * - 0.36b (28/03/00) : Implemented preliminary support for
 *   SMS sending mode. Adapted "set", "show" and "clear"
 *   syntax. Updated on-line help accordingly.
 * - 0.37b (31/03/00) : In send_sms(), added the mode selection
 *   code to the device dialog. Added defmode display in
 *   print_device_list().
 * - 0.38b (04/04/00) : Adapted "devicelist" on-line help to
 *   reflect the changes.
 * - 0.39b (07/04/00) : Adapted conclude() and unlock_gsm()
 *   to include freeing for pdu string. Added code in 
 *   send_sms_wrapper() to call encode_pdu() when needed.
 * - 0.40b (13/04/00) : Moved the min() macro to pdu.h. Modified
 *   send_sms() to use the requested message format.
 * - 0.41b (18/05/00) : Solved a bug in the definition of RAND_MAX
 *   that would have prevented the election of a new PIN to work,
 *   and that used to generate warning messages at compile time.
 * - 0.42b (21/05/00) : Started to implement support for different
 *   brand of modules through capacity matrices (first in
 *   send_sms()).
 * - 0.43b (08/06/00) : In send_sms(), converted all AT commands
 *   to uppercase. It seems that the command parser in some
 *   Wavecom devices is case-sensitive. Thanks to Andrea Vettori
 *   <av@tservicetlc.net> and Frank Friel <frank_friel@hotmail.com>
 *   for pointing that out.
 * - 0.44b (21/06/00) : In send_sms(), adapted use of capmatrix
 *   to its new structure.
 * - 0.45b (22/06/00) : In send_sms(), added support for devices
 *   featuring a "subprompt" in AT+CMGS (e.g. SM20). Thanks to
 *   Silvano Menegoni <silver@fly.srk.fer.hr> for the tip.
 * - 0.46b (27/06/00) : Added subprompt handling in PDU-mode
 *   send procedure as well. Thanks a lot to Silvano Menegoni 
 *   <silver@fly.srk.fer.hr> for the fine tests he made on his
 *   equipment.
 * - 0.47b (26/09/00) : Included <sys/types.h> due to <netinet/in.h>
 *   demanding it on FreeBSD. Included other FreeBSD compile tests.
 * - 0.48b (27/09/00) : Improved error checking in send_sms() when
 *   parsing the module's reply after +CMGS. Thanks to Stanley Hopcroft 
 *   <Stanley.Hopcroft@IPAustralia.Gov.AU> for this.
 * - 0.49b (18/10/00) : Added alternate CNMI handling for Siemens
 *   S25 / S35 (thanks to Thomas Omerzu <thomas@omerzu.de> for
 *   the fix).
 * - 0.50b (23/11/00) : In sendsms(), altered the AT+CPIN command
 *   by putting the PIN value between quotes. Closer to the
 *   standard and required by the PCFF900.
 * - 0.51b (24/11/00) : Modified sendsms() to account for
 *   ALT_CNMI_2 (PCFF900). Kudos to Bernard Willemot 
 *   <bernard_willemot@cse.creoscitex.com> for lending me the
 *   hardware.
 * - 0.52b (18/02/01) : Modified sendsms() to account for
 *   ALT_CNMI_3 (GM12). Modified sms_send_wrapper() to allow
 *   use of "dedicated device" for sending in case all others
 *   are in use. Contributed by Andrew Worsley 
 *   <epaanwo@asac.ericsson.se>.
 * - 0.53b (19/02/01) : Added support for "fast modem response"
 *   in send_sms(). Contributed by Andrew Worsley 
 *   <epaanwo@asac.ericsson.se>.
 * - 0.54b (06/03/01) : Included definition of extern int
 *   debug (in prep. for debug info ctrl by global flags).
 *   Replaced 'NULL' by '0' in strstr() comparisions and added
 *   2 (char *) casts to eliminate a slew of gcc warnings.
 *   Added a "debug" variable that can be used to interactively
 *   modify the debug level through 'set', 'clear' and 'show' commands.
 *   Adapted the on-line help accordingly. Modified the way a new
 *   PIN is selected if we have to provide the PUK: now sets PIN
 *   to the one we have in conf in order to avoid the need to store
 *   the new one (thanks to Shay Kalderon <shay_kalderon@creoscitex.com>
 *   for the hint).
 * - 0.55b (02/04/01) : Modified calls to get_gsm_answer() to
 *   include the struct gsms_def field.
 * - 0.56b (04/04/01) : Replaced calls to get_gsm_answer() by
 *   calls to get_gsm_resp() in send_sms() when sending PIN code
 *   to try and solve "fast" mode problems.
 * - 0.57b (03/05/01) : Improved handling of +CREG-returned
 *   registration status in send_sms().
 * - 0.58b (05/05/01) : Solved a bug with "fast modem response"
 *   mode in send_sms() when modem has SUBPROMPT_IN_CMGS.
 * - 0.59b (07/05/01) : Added a pause in send_sms() after PIN
 *   processing when using "fast" mode to try and solve the
 *   "+CMS ERROR: 515" issue. Added cpl_list de-allocation code
 *   in send_sms() (back-ported from 0.55b).
 * - 0.60b (07/10/01) : Replaced calls to blopen_mdm_line()
 *   by calls to lopen_mdm_line(). Advantage being that this
 *   version uses the max. supported speed by the device by
 *   default (the one defined in /etc/modems).
 * - 0.61b (27/03/01) : Made send_sms() a wrapper around
 *   backend_send_sms(), in order for it to be called by
 *   outbox_check_wrapper. Contributed by Ing. Andrea
 *   Vettori <a.vettori@inetronics.com>.
 * - 0.62b (30/04/01) : Implemented the stats command using
 *   the new counter fields in the struct gsms_def. Adapted the
 *   online help accordingly.
 * - 0.63b (15/05/01) : Added support for the "long" version
 *   of the "devicelist" command. Adapted on-line help
 *   accordingly. Killed an old compiler warning by giving a
 *   large constant as a hex value instead of decimal (kudos
 *   to Renaud <renaud.florquin@iba.be>).
 * - 0.64b (05/01/02) : Replaced the AT+CMEE dialog with a
 *   call to set_cmee(). Idem for AT+CPIN, AT+CMGF, AT+CREG
 *   & AT+CSCA.
 * - 0.65b (09/11/02) : Implemented support for default GSM
 *   alphabet. Implemented support for "init" mode (contributed
 *   by Ing. Andrea Vettori <a.vettori@inetronics.com>). Replaced
 *   static PIN post-processing sleep time by device-level
 *   param. (PINpause). Added support for the "queue" command.
 *   Added support for the "post" command. Added "qstat" &
 *   "qdel" commands. Improved device selection algorithm to
 *   allow for parallel use of "dd" devices.
 * - 0.66b (25/11/03) : Added test on NO_CREG_CHECK flag to
 *   support Motorola A008 device. Contributed by Nicki de Wet
 *   <nickidw@mighty.co.za>.
 * - 0.67b (09/04/04) : Added priority support to queued messages.
 *   Added support for "purge" command + adapted on-line help.
 *========================================================*/

#include <stdio.h>
#include <stdlib.h>                  /* for errno & div() */
#if (defined (LINUX_LC6) || defined (FREEBSD) || defined (FREEBSD5))
#  include <errno.h>
#endif                                /* #ifdef LINUX_LC6 */
#include <strings.h>
#include <signal.h>
#include <time.h>                       /* for difftime() */
#include <math.h>                            /* for log() */
#include <sys/types.h>                     /* for FreeBSD */
#include <netinet/in.h>                    /* for ntohl() */
#include <termios.h>         /* for baudrates definitions */
#include <dial/modems.h>           /* requires 'libmodem' */

#include "sms_serv.h"
#include "gsmdev.h"
#include "pdu.h"

/*-------------------------------------External variables */
/* for lex & yacc/bison */
extern int errno;
extern int yyleng;
extern char *yytext;
extern char myinput[];
extern char *myinputptr;          /* cur. pos. in myinput */
extern char *myinputlim;                   /* end of data */

/* program specific */
extern int debug;                    /* debug level/flags */
extern char *gsmdevfile;
extern int m_timeout;
extern int qmethod;
extern int dbaccess_enabled;
extern int csfd;                      /* client socket FD */
extern char *buffer;    /* scratch space for C/S dialogue */
extern struct symbols symbols;
extern int ngsmdevs;       /* num. configured GSM devices */
extern int default_alphabet;

/*---------------------------------------Global variables */
char *paramname = NULL;
char *idname = NULL;
int numid;
int priority;
int affrows;
char *p;
char scope;
int nscope;
/* char *buffer = NULL; */
int i;
int instance;
int instance_is_set = FALSE;
int vtype;
char mode;
char cmode[10];
int newdebug;

/*------------------------------------For the help system */
/* Be carefull to match the defines here with the array below */
#define H_HELP		0
#define H_CLEAR		1
#define H_SET		2
#define H_SEND		3
#define H_QUEUE		4
#define H_POST		5
#define H_QUIT		6
#define H_EXIT		7
#define H_BYE		8
#define H_SHOW		9
#define H_STATS		10
#define H_DEVLIST	11
#define H_UPTIME        12
#define H_ACLIST        13
#define H_QSTAT		14
#define H_QDEL		15
#define H_PURGE		16
#define H_MAX		17            /* keeps (last + 1) */

char *help_topics[(H_MAX + 1)] = {"help",
                                  "clear",
		                  "set",
		                  "send",
		                  "queue",
		                  "post",
			          "quit",
			          "exit",
			          "bye",
			          "show",
			          "stats",
			          "devicelist",
			          "uptime",
				  "aclist",
				  "qstat",
				  "qdel",
				  "purge",
			          NULL};

/*----------------------------Local Function declarations */
int shiftleft (char *, int);
int dequote (char *);
int deslashdot (char *);
int give_help (char *);
int help_lookup (char *);
int load_help ();
void free_help ();
void print_device_list ();
void print_long_device_list ();
void print_device_stats ();
void print_access_list ();
void print_queue_stat ();
void delete_queue_item (int);
void compute_uptime ();
int validate_msg_param (struct symbols *);
int backend_send_sms (struct gsms_def *, struct symbols *, int);
int send_sms (struct gsms_def *, struct symbols *);
void sms_send_wrapper ();
void sms_post_wrapper ();
void sms_queue_wrapper ();
void unlock_gsm ();
void conclude ();

/*========================================================*/
%}

%union {
  char *string;          /* string buffer for token value */
}

/* commands */
%token T_HELP
%token T_CLEAR
%token T_SEND
%token T_POST
%token T_QUEUE
%token T_DEVLIST
%token T_ACLIST
%token T_UPTIME
%token T_SET
%token T_QUIT
%token T_STATS
%token T_SHOW
%token T_QSTAT
%token T_QDEL
%token T_PURGE


/* others */
%token T_PARAM
%token T_NUMID
%token <string> T_TEXT
%token <string> T_PHNUM
%token <string> T_SCHAR
%token T_NAME

%type <string> value

%%

statement:	single_command
	|	param_command
	|	assignment
	|	/* allow empty input */
	;

single_command:	T_HELP		{
				sprintf (buffer, "\r\nSMS Server Help\r\n"
				        "===============\r\n"
				        "he | help [command] : displays help\r\n"
				        "set item = value    : sets message parameters\r\n"
				        "clear [item]        : clears item (all items)\r\n");
				tellsock (csfd, buffer);
				sprintf (buffer, "send                : sends SMS right away\r\n"
				        "post                : attempts to send, queues if it fails\r\n"
				        "queue               : queues SMS for later delivery\r\n");
				tellsock (csfd, buffer);
				sprintf (buffer, "dl | devicelist [l] : lists configured GSM devices\r\n"
					"acl | aclist        : dumps the registered ACL's to screen\r\n"
					"ut | uptime         : displays server uptime\r\n"
				        "quit | exit | bye   : closes connection and exits\r\n"
				        "stats               : displays server stats\r\n"
				        "sh | show [item]    : displays item (all items)\r\n");
				tellsock (csfd, buffer);
				sprintf (buffer, "qstat               : displays queue status & contents\r\n"
				        "qdel <id>           : removes <id> from queue\r\n"
				        "purge (in|out) [sc] : removes processed / sent items from DB specif. table\r\n\r\n");
				tellsock (csfd, buffer);
				}
	|	T_CLEAR		{
				strcpy (symbols.smsc, DEFAULTSMSC);
				symbols.destgsm[0] = '\0';
				symbols.user[0] = '\0';
				symbols.message[0] = '\0';
				symbols.mode = SMS_MODE_DEFAULT;
				symbols.priority = SMS_DEF_Q_PRI;
				sprintf (buffer, "All items cleared...\r\n");
				tellsock (csfd, buffer);
				}
	|	T_SEND		{
				if (validate_msg_param (&symbols))
				  sms_send_wrapper ();
				else {
				  sprintf (buffer, "ERROR: Not all required items are filled in.\r\n");
				  tellsock (csfd, buffer);
				}
				}
	|	T_POST		{
				if (validate_msg_param (&symbols))
				  sms_post_wrapper ();
				else {
				  sprintf (buffer, "ERROR: Not all required items are filled in.\r\n");
				  tellsock (csfd, buffer);
				}
				}
	|	T_QUEUE		{
				if (validate_msg_param (&symbols))
				  sms_queue_wrapper ();
				else {
				  sprintf (buffer, "ERROR: Not all required items are filled in.\r\n");
				  tellsock (csfd, buffer);
				}
				}
	|	T_DEVLIST	{
				print_device_list ();
				}
	|	T_ACLIST	{
				print_access_list ();
				}
	|	T_UPTIME	{
				compute_uptime ();
				}
	|	T_STATS		{
				print_device_stats ();
				}
	|	T_SHOW		{
				sprintf (buffer, "\r\nCurrent settings:\r\n"); 
				tellsock (csfd, buffer);
				sprintf (buffer, "smsc = [%s]\r\n", symbols.smsc);
				tellsock (csfd, buffer);
				sprintf (buffer, "user = [%s]\r\n", symbols.user);
				tellsock (csfd, buffer);
				sprintf (buffer, "dest = [%s]\r\n", symbols.destgsm);
				tellsock (csfd, buffer);
				sprintf (buffer, "msg  = [%s] - Length: %d\r\n", 
				         symbols.message, strlen (symbols.message));
				tellsock (csfd, buffer);
				switch (symbols.mode) {
				  case SMS_MODE_PDU: {
				    strcpy (cmode, "pdu");
				    break;
				  }
				  case SMS_MODE_TEXT: {
				    strcpy (cmode, "text");
				    break;
				  }
				  case SMS_MODE_DEFAULT: {
				    strcpy (cmode, "default");
				    break;
				  }
				  default: {
				    strcpy (cmode, "<unknown>");
				    break;
				  }
				} /* switch (symbols.mode) */
				sprintf (buffer, "mode = <%s> (%d)\r\n",
				        cmode, symbols.mode);
				tellsock (csfd, buffer);
				sprintf (buffer, "prio = %d\r\n\r\n",
				        symbols.priority);
				tellsock (csfd, buffer);
				}
	|	T_QUIT		{
	                        conclude ();
				}
	|	T_QSTAT		{
				print_queue_stat ();
				}
	;

param_command:  T_HELP parameter
				{
				give_help (paramname);
				free (paramname);
				paramname = NULL;
				}
	|	T_SHOW parameter
				{
				/* convert shorthands first */
				if (strcmp (paramname, "msg") == 0) {
				  free (paramname);
				  paramname = (char *) malloc (8 * sizeof (char));
				  strcpy (paramname, "message");
				}
				  
				if (strcmp (paramname, "prio") == 0) {
				  free (paramname);
				  paramname = (char *) malloc (9 * sizeof (char));
				  strcpy (paramname, "priority");
				}
				  
				if (strcmp (paramname, "smsc") == 0) {
				  sprintf (buffer, "smsc = [%s]\r\n", symbols.smsc);
				  tellsock (csfd, buffer);
				}
				else if (strcmp (paramname, "user") == 0) {
				  sprintf (buffer, "user = [%s]\r\n", symbols.user);
				  tellsock (csfd, buffer);
				}
				else if (strcmp (paramname, "dest") == 0) {
				  sprintf (buffer, "dest = [%s]\r\n", symbols.destgsm);
				  tellsock (csfd, buffer);
				}
				else if (strcmp (paramname, "message") == 0) {
				  sprintf (buffer, "msg  = [%s] - Length: %d\r\n", 
				           symbols.message, strlen (symbols.message));
				  tellsock (csfd, buffer);
				}
				else if (strcmp (paramname, "mode") == 0) {
				  switch (symbols.mode) {
				    case SMS_MODE_PDU: {
				      strcpy (cmode, "pdu");
				      break;
				    }
				    case SMS_MODE_TEXT: {
				      strcpy (cmode, "text");
				      break;
				    }
				    case SMS_MODE_DEFAULT: {
				      strcpy (cmode, "default");
				      break;
				    }
				    default: {
				      strcpy (cmode, "<unknown>");
				      break;
				    }
				  } /* switch (symbols.mode) */
				  sprintf (buffer, "mode = <%s> (%d)\r\n",
				          cmode,
					  symbols.mode);
				  tellsock (csfd, buffer);
				}
				else if (strcmp (paramname, "priority") == 0) {
				  sprintf (buffer, "prio = %d\r\n", 
				           symbols.priority);
				  tellsock (csfd, buffer);
				}
				else if (strcmp (paramname, "debug") == 0) {
				  sprintf (buffer, "Current debug level  = [%d]\r\n", 
				           debug);
				  tellsock (csfd, buffer);
				}
				else {
				  sprintf (buffer, "ERROR: unknown item <%s>\r\n", paramname);
				  tellsock (csfd, buffer);
				}

				free (paramname);
				paramname = NULL;
				}
	|	T_DEVLIST parameter
				{
				if ((strcmp (paramname, "long") == 0) ||
				   (strcmp (paramname, "l") == 0)) {
				  print_long_device_list ();
				}
				else {
				  sprintf (buffer, "ERROR: unknown parameter <%s>\r\n", paramname);
				  tellsock (csfd, buffer);
				}
				free (paramname);
				paramname = NULL;
				}
	|	T_CLEAR parameter
				{
				/* convert shorthands first */
				if (strcmp (paramname, "msg") == 0) {
				  free (paramname);
				  paramname = (char *) malloc (8 * sizeof (char));
				  strcpy (paramname, "message");
				}
				  
				if (strcmp (paramname, "prio") == 0) {
				  free (paramname);
				  paramname = (char *) malloc (9 * sizeof (char));
				  strcpy (paramname, "priority");
				}
				  
				if (strcmp (paramname, "smsc") == 0) {
				  strcpy (symbols.smsc, DEFAULTSMSC);
				}
				else if (strcmp (paramname, "user") == 0) {
				  symbols.user[0] = '\0';
				}
				else if (strcmp (paramname, "dest") == 0) {
				  symbols.destgsm[0] = '\0';
				}
				else if (strcmp (paramname, "message") == 0) {
				  symbols.message[0] = '\0';
				}
				else if (strcmp (paramname, "mode") == 0) {
				  symbols.mode = SMS_MODE_DEFAULT;
				}
				else if (strcmp (paramname, "priority") == 0) {
				  symbols.priority = SMS_DEF_Q_PRI;
				}
				else if (strcmp (paramname, "debug") == 0) {
				  debug = 0;
				}
				else {
				  sprintf (buffer, "ERROR: unknown item <%s>\r\n", paramname);
				  tellsock (csfd, buffer);
				}
				free (paramname);
				paramname = NULL;
				}
	|	T_QDEL parameter
	                        {
				delete_queue_item (numid);
				}
	|	T_PURGE parameter
	                        {
#ifdef WITH_MYSQL
				/* split parameters if required */
				scope = '\0';
				nscope = DB_OUT_PURGE_ALL;
				trim (paramname);
				if ((p = strchr (paramname, ' ')) != NULL) {
				  scope = p[1];
				  /* cut paramname at the space */
				  p[0] = '\0';
				  switch (scope) {
				    case 'a': {
				      nscope = DB_OUT_PURGE_ALL;
				      break;
				    }
				    case 's': {
				      nscope = DB_OUT_PURGE_SENT;
				      break;
				    }
				    case 'f': {
				      nscope = DB_OUT_PURGE_FAIL;
				      break;
				    }
				    default: {
				      nscope = -1;
				      break;
				    }
				  }     /* switch (scope) */
				}      /* if (space sep.) */
				
				/* then convert shorthands */
				if (strcmp (paramname, "in") == 0) {
				  free (paramname);
				  paramname = (char *) malloc (6 * sizeof (char));
				  strcpy (paramname, "inbox");
				}
				  
				if (strcmp (paramname, "out") == 0) {
				  free (paramname);
				  paramname = (char *) malloc (7 * sizeof (char));
				  strcpy (paramname, "outbox");
				}
				  
				/* Now start processing */  
				if (strcmp (paramname, "inbox") == 0) {
				  if ((affrows = db_purge_inbox ()) == -1) {
				    sprintf (buffer, "ERROR: purge query failed\r\n");
				    tellsock (csfd, buffer);
				  }
				  else {
				    sprintf (buffer, "Inbox purged: %d entries removed.\r\n", affrows);
				    tellsock (csfd, buffer);
				  }
				}
				else if (strcmp (paramname, "outbox") == 0) {
				  if (nscope != -1) {
				    if ((affrows = db_purge_outbox (nscope)) == -1) {
				      sprintf (buffer, "ERROR: purge query failed\r\n");
				      tellsock (csfd, buffer);
				    }
				    else {
				      sprintf (buffer, "Outbox purged: %d entries removed.\r\n", affrows);
				      tellsock (csfd, buffer);
				    }
				  }
				  else {
				    sprintf (buffer, "ERROR: unsupported sub-parameter <%c>\r\n", scope);
				    tellsock (csfd, buffer);
				  }
				}
				else {
				  sprintf (buffer, "ERROR: unsupported parameter <%s>\r\n", paramname);
				  tellsock (csfd, buffer);
				}
				free (paramname);
				paramname = NULL;
#else
				sprintf (buffer, "Command unavailable. Required DB backend not present.\r\n");
				tellsock (csfd, buffer);
#endif                               /* #ifdef WITH_MYSQL */
				}
	;

assignment:	T_SET variable '=' value
				{
				/* convert shorthands first */
				if (strcmp (idname, "msg") == 0) {
				  free (idname);
				  idname = (char *) malloc (8 * sizeof (char));
				  strcpy (idname, "message");
				}
				  
				if (strcmp (idname, "prio") == 0) {
				  free (idname);
				  idname = (char *) malloc (9 * sizeof (char));
				  strcpy (idname, "priority");
				}
				  
				if (strcmp (idname, "smsc") == 0) {
				  /* Assign smsc */
				  if (strlen ($4) > MAXPHNUMLEN) {
				    strncpy (symbols.smsc, $4, MAXPHNUMLEN);
				    symbols.smsc[MAXPHNUMLEN] = '\0';
				    sprintf (buffer, "WARNING: max. smsc length is %d, value now truncated\r\n",
				           MAXPHNUMLEN);
			            tellsock (csfd, buffer);
				  }
				  else
				    strcpy (symbols.smsc, $4);
				  sprintf (buffer, "Ok\r\n");
				  tellsock (csfd, buffer);
                                  syslog ((FACILITY | LOG_NOTICE),
				         "got new smsc <%s>.", symbols.smsc);
				}
				
				else if (strcmp (idname, "user") == 0) {
				  /* Assign user */
				  if (strlen ($4) > MAXUIDLEN) {
				    strncpy (symbols.user, $4, MAXUIDLEN);
				    symbols.user[MAXUIDLEN] = '\0';
				    sprintf (buffer, "WARNING: max. sender ID length is %d, value now truncated\r\n",
				           MAXUIDLEN);
			            tellsock (csfd, buffer);
				  }
				  else
				    strcpy (symbols.user, $4);
				  sprintf (buffer, "Ok\r\n");
				  tellsock (csfd, buffer);
                                  syslog ((FACILITY | LOG_NOTICE),
				         "sender ID is [%s].", symbols.user);
				}
				
				else if (strcmp (idname, "dest") == 0) {
				  /* Check whether we have a T_PHNUM */
				  if (vtype != T_PHNUM) {
				    sprintf (buffer, "Expected a phone number\r\n");
				    tellsock (csfd, buffer);
				  }
				  else {
				    /* Assign dest */
				    if (strlen ($4) > MAXPHNUMLEN) {
				      strncpy (symbols.destgsm, $4, MAXPHNUMLEN);
				      symbols.destgsm[MAXPHNUMLEN] = '\0';
				      sprintf (buffer, "WARNING: max. dest length is %d, value now truncated\r\n",
				             MAXPHNUMLEN);
			              tellsock (csfd, buffer);
				    }
				    else {
				      strcpy (symbols.destgsm, $4);
				    }
				    sprintf (buffer, "Ok\r\n");
				    tellsock (csfd, buffer);
                                    syslog ((FACILITY | LOG_NOTICE),
				           "got destgsm <%s>.", symbols.destgsm);
			          }
				}
				
				else if (strcmp (idname, "message") == 0) {
				  /* Assign message */
				  if (strlen ($4) > MAXMSGLEN) {
				    strncpy (symbols.message, $4, MAXMSGLEN);
				    symbols.message[MAXMSGLEN] = '\0';
				    sprintf (buffer, "WARNING: max. message length is %d, value now truncated\r\n",
				           MAXMSGLEN);
			            tellsock (csfd, buffer);
				  }
				  else
				    strcpy (symbols.message, $4);
				  sprintf (buffer, "Ok\r\n");
				  tellsock (csfd, buffer);
                                  syslog ((FACILITY | LOG_NOTICE),
				         "message to transmit is set.");
                                  if (debug & DEBUG_PRINTF) {
                                    fprintf (stderr, "Message received as [%s].\n", symbols.message);
				  }
				}
				
				else if (strcmp (idname, "mode") == 0) {
				  /* Check whether we have a T_SCHAR */
				  if (vtype != T_SCHAR) {
				    sprintf (buffer, "Expected a single char\r\n");
				    tellsock (csfd, buffer);
				  }
				  else {
				    /* Assign mode */
				    mode = toupper ($4[0]);
				    switch (mode) {
				      case 'P': /* fall through */
				      case 'B': {
				        symbols.mode = SMS_MODE_PDU;
				        sprintf (buffer, "Ok\r\n");
				        tellsock (csfd, buffer);
                                        syslog ((FACILITY | LOG_NOTICE),
				               "got mode <%c>.", mode);
				        break;
				      }
				      case 'T': /* fall through */
				      case 'A': {
				        symbols.mode = SMS_MODE_TEXT;
				        sprintf (buffer, "Ok\r\n");
				        tellsock (csfd, buffer);
                                        syslog ((FACILITY | LOG_NOTICE),
				               "got mode <%c>.", mode);
				        break;
				      }
				      default: {
				        sprintf (buffer, "ERROR: unsupported mode <%c>\r\n", mode);
				        tellsock (csfd, buffer);
				        break;
				      }
				    }    /* switch (mode) */
			          }
				}
				
				else if (strcmp (idname, "priority") == 0) {
				  /* Check whether we have a T_PHNUM */
				  if (vtype != T_PHNUM) {
				    sprintf (buffer, "Expected a single digit\r\n");
				    tellsock (csfd, buffer);
				  }
				  else {
				    priority = atoi ($4);
				    /* Assign priority */
				    if ((priority < SMS_Q_PRI_FLOOR) || (priority > SMS_Q_PRI_CEIL)) {
				      sprintf (buffer, "Priority should be between %d and %d (%d = highest).\r\n",
				             SMS_Q_PRI_FLOOR, SMS_Q_PRI_CEIL, SMS_Q_PRI_FLOOR);
			              tellsock (csfd, buffer);
				    }
				    else {
				      symbols.priority = priority;
				      sprintf (buffer, "Ok\r\n");
				      tellsock (csfd, buffer);
                                      syslog ((FACILITY | LOG_NOTICE),
				             "got priority <%d>.", symbols.priority);
				    }
			          }
				}
				
				else if (strcmp (idname, "debug") == 0) {
				  /* Set debug level */
				  newdebug = 0;
				  newdebug = atoi ($4);
				  if ((newdebug >= 0) && (newdebug <= DEBUG_ALL)) {
				    sprintf (buffer, "Debug level was %d, is now %d.\r\n",
				           debug, newdebug);
			            tellsock (csfd, buffer);
				    debug = newdebug;
				    sprintf (buffer, "Ok\r\n");
				    tellsock (csfd, buffer);
                                    syslog ((FACILITY | LOG_NOTICE),
				           "debug level set to <%d>.", debug);
				  }
				  else {
				    sprintf (buffer, "Invalid debug level (%s).\r\n",
				           $4);
			            tellsock (csfd, buffer);
				    sprintf (buffer, "Error\r\n");
				    tellsock (csfd, buffer);
				  }
				}
				
				else {
				  sprintf (buffer, "ERROR: unknown item <%s>\r\n", idname);
				  tellsock (csfd, buffer);
				}
				
				free (idname);
				idname = NULL;
				free ($4);
				}
	;

parameter:	T_PARAM		{
                                if (paramname)
				  free (paramname);
				paramname = (char *) malloc ((yyleng + 1) * sizeof (char));
				paramname[0] = '\0';
				strcpy (paramname, yytext);
				}
	|	T_NUMID         {
                                numid = atoi (yytext);
	                        }
	;

variable:	T_NAME		{ if (idname)
				    free (idname);
				 idname = (char *) malloc ((yyleng + 1) * sizeof (char));
				 idname[0] = '\0';
				 strcpy (idname, yytext);
				 }
	;

value:		T_TEXT		{
				/* let's dequote here */
				dequote ($1);
				vtype = T_TEXT;
				}
	|	T_PHNUM		{
				/* let's dequote and remove [/.] here */
				dequote ($1);
				deslashdot ($1);
				vtype = T_PHNUM;
				}
	|	T_SCHAR		{
				vtype = T_SCHAR;
				}
	;

%%
/*########################################################*/
/* Main entry point for the parser module                 */
int parser ()
{
  /*======================================Initialisations */

  /*==============================Main Call to the Parser */
  yyparse ();
}                                            /* parser () */
/*########################################################*/

/*========================================================*/
/*                     Local Functions                    */
/*========================================================*/
int my_yyinput (buf, max_size)
char *buf;
int max_size;
{
  int n = min (max_size, (myinputlim - myinputptr));
  
  if (n > 0) {
    memcpy (buf, myinputptr, n);
    myinputptr += n;
  }
  
  return (n);
}                                        /* my_yyinput () */
/*========================================================*/
int give_help (p)
char *p;
{
  int topic_id;

  topic_id = help_lookup (p);

  switch (topic_id) {
    case H_HELP: {
      sprintf (buffer, "\r\nSyntax: he | help [command]\r\n"
              "Used with no option, gives a list of all available commands.\r\n"
              "When a command is provided, gives detailed help on this command.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    case H_CLEAR: {
      sprintf (buffer, "\r\nSyntax: clear [item]\r\n"
              "Where [item] is one of:\r\n"
              "- smsc: SMS Message Center (reverts it to \"%s\")\r\n"
              "- user: sender identification (user name) for logging purposes\r\n"
              "- dest: destination GSM number\r\n", DEFAULTSMSC);
      tellsock (csfd, buffer);
      sprintf (buffer, "- mode: SMS sending mode (reverts it to default mode)\r\n"
              "- message | msg: message text to be sent to destination\r\n"
              "- priority | prio: queued message priority (reverts to default)\r\n"
	      "- debug: server-side debug level\r\n\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "Used with no option, clears all items (except debug).\r\n"
              "When an item is specified, clears only this one.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    case H_SET: {
      sprintf (buffer, "\r\nSyntax: set item = value\r\n"
              "Where [item] is one of:\r\n"
              "- smsc: SMS Message Center (see your provider or use default)\r\n"
              "- user: sender identification (user name) for logging purposes\r\n"
              "- dest: destination GSM number\r\n"
	      "- mode: SMS sending mode\r\n"
              "- message | msg: message text to be sent to destination\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "- priority | prio: message priority (relevant only when queued locally)\r\n"
              "- debug: server-side debug level\r\n\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "Note 1: text must be quoted, GSM numbers can be quoted, GSM numbers can\r\n"
              "contain any of [/.-+].\r\n\r\n"
	      "Note 2: for SMSC, a value of \"%s\" means the default SMSC\r\n"
	      "as defined per device in %s. Set it to a valid phone\r\n"
	      "number to use a different one. Use the \"clear\" command to reset it\r\n"
	      "to \"%s\".\r\n\r\n", DEFAULTSMSC, gsmdevfile, DEFAULTSMSC);
      tellsock (csfd, buffer);
      sprintf (buffer, "Note 3: mode is a single letter (a, b, p or t), and is not\r\n"
              "case-sensitive. a and t stand for text-mode (= ascii), b and p\r\n"
	      "stand for pdu-mode (=binary).\r\n\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "Note 4: debug is any integer value between 0 and DEBUG_ALL\r\n"
              "(%d). Any change in its value will affect the current session only\r\n"
	      "(the child process, not its parent or any other child).\r\n\r\n",
	      DEBUG_ALL);
      tellsock (csfd, buffer);
      break;
    }
    case H_SEND: {
      sprintf (buffer, "\r\nSyntax: send\r\n"
              "When all parameters are filled in as required, use this command to\r\n"
              "send the SMS message. You'll wait during the time it takes to send.\r\n"
              "An acknowlegment will be returned when done.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    case H_POST: {
      sprintf (buffer, "\r\nSyntax: post\r\n"
              "When all parameters are filled in as required, use this command to\r\n"
              "attempt an immediate send of the SMS message. You'll wait during the\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "time it takes to send. If the sending fails (and queueing is enabled),\r\n"
	      "the message is queued for later delivery.\r\n"
              "An acknowlegment will be returned when done.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    case H_QUEUE: {
      sprintf (buffer, "\r\nSyntax: queue\r\n"
              "When all parameters are filled in as required, use this command to\r\n"
              "queue the SMS message for later delivery.\r\n"
              "An acknowlegment will be returned when done.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    case H_QUIT: {
      /* just jump to next */
    }
    case H_EXIT: {
      /* just jump to next */
    }
    case H_BYE: {
      sprintf (buffer, "\r\nSyntax: quit | exit | bye\r\n"
              "Closes the connection and leaves the program.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    case H_SHOW: {
      sprintf (buffer, "\r\nSyntax: sh | show [item]\r\n"
              "Where [item] is one of:\r\n"
              "- smsc: SMS Message Center (see your provider or use default)\r\n"
              "- user: sender identification (user name) for logging purposes\r\n"
              "- dest: destination GSM number\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "- mode: SMS sending mode\r\n"
              "- message | msg: message text to be sent to destination\r\n"
              "- priority | prio: queued message priority\r\n"
	      "- debug: server-side debug level\r\n\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "Used without parameter, the command displays the current setting\r\n"
              "for all items (except debug). When an item is specified, only\r\n"
              "this one is displayed.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    case H_STATS: {
      sprintf (buffer, "\r\nSyntax: stats\r\n"
               "Shows per-device server-side statistics and counters.\r\n"
	       "The fields provided in the output are:\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "- device:     Unix device name\r\n"
	      "- Sent OK:    number of SMS successfully sent by that device\r\n"
	      "- Sent FAIL:  number of sending failures on that device\r\n"
	      "- Recvd OK:   number of SMS successfully received by that device\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "- Recvd FAIL: number of reception failures on that device\r\n"
              "- Unlock:     number of UNLOCK events on that device\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "The server uptime is then displayed, to allow one to\r\n"
              "know the amount of time covered by the statistics that were just\r\n"
              "listed.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    case H_DEVLIST: {
      sprintf (buffer, "\r\nSyntax: devicelist | dl [long | l]\r\n"
              "Lists registered GSM devices.\r\n"
	      "Those are usually configured through <%s>.\r\n"
	      "The \"long\" parameter switches between a short or long list.\r\n"
	      "The fields provided in the short output are:\r\n", GSMDEVFILE);
      tellsock (csfd, buffer);
      sprintf (buffer, "- L:        device lock status (. = free, * = locked)\r\n"
              "- owner:    pid owning a lock on it (0 = not locked)\r\n"
	      "- device:   Unix device name\r\n"
	      "- CallerID: the GSM's network addr. (its phone number)\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "- Def SCA:  the default Service Center Addr. (or SMSC)\r\n"
	      "- M:        the default SMS sending mode for the device\r\n"
	      "- Provider: Network Provider Name\r\n\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "The long output lists all fields the application keeps\r\n"
              "related to the GSM devices, and gives a detailed description\r\n"
	      "for each.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    case H_ACLIST: {
      sprintf (buffer, "\r\nSyntax: aclist | acl\r\n"
              "Dumps the registered ACL entries to screen.\r\n"
	      "ACL's are Access Control Lists that define which machine\r\n"
	      "is allowed to use the server and which isn't. ACL's are\r\n"
	      "usually configured through <%s>.\r\n"
	      "man 5 gsmaccess for more details.\r\n\r\n", ACCESSFILE);
      tellsock (csfd, buffer);
      break;
    }
    case H_UPTIME: {
      sprintf (buffer, "\r\nSyntax: uptime | ut\r\n"
              "Displays the SMS server process uptime in days, hours, \r\n"
	      "minutes and seconds.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    case H_QSTAT: {
      sprintf (buffer, "\r\nSyntax: qstat\r\n"
              "Displays the status of the queueing mechanism (enabled or not),\r\n"
              "the queueing method that will be used (when applicable), and lists\r\n"
              "all pending requests in the queue.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    case H_QDEL: {
      sprintf (buffer, "\r\nSyntax: qdel <id>\r\n"
              "This will remove the specified request <id> from the queue. The <id>\r\n"
              "parameter is mandatory and doesn't support any regexp at this time.\r\n\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "Warning: when the current queueing method is FILE, but DB access is\r\n"
              "enabled, 'qdel' will only be able to remove entries from the file queue.\r\n"
              "To manage the DB-based outbox in this case, use the MySQL client.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    case H_PURGE: {
      sprintf (buffer, "\r\nSyntax: purge (inbox|outbox) [scope]\r\n"
              "This will remove all sent / processed items from the specified DB table.\r\n"
              "'in' is short for inbox, 'out' is short for outbox. When requesting to\r\n"
	      "purge the outbox, an additional 'scope' parameter can be given.\r\n\r\n");
      tellsock (csfd, buffer);
      sprintf (buffer, "'scope' determines whether all, only 'sent' or only 'failed' outbox\r\n"
              "items will be removed ('all' means 'sent' + 'failed', of course).\r\n"
              "You can shorten the scope by its first letter ('a', 's' or 'f').\r\n"
	      "When 'scope' is omitted, 'all' is implied.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    default: {
      sprintf (buffer, "ERROR: don't know about <%s>. Can't help.\r\n", p);
      tellsock (csfd, buffer);
      break;
    }
  }                                  /* switch (topic_id) */
  return (topic_id);
}                                         /* give_help () */
/*========================================================*/
int help_lookup (p)
char *p;
{
  int i;

  for (i = 0; (help_topics[i] != NULL) && (strcmp (p, help_topics[i]) != 0); i++);
  return (i);
}                                       /* help_lookup () */
/*========================================================*/
void print_device_list ()
{
  extern int shmem_sem;
  extern struct gsms_def *gsmdevices;
  int i;

  sprintf (buffer, "\r\n"
                   "L | Owner | Device        | CallerID       | Def. SCA       | M | Provider   \r\n"
                   "--+-------+---------------+----------------+----------------+---+------------\r\n");
  tellsock (csfd, buffer);
  
  /* get exclusive access to shared mem. for safety's sake */
  if (sem_wait (shmem_sem) == -1)
    syserr ("print_device_list(): failed to wait on shared mem. semaphore");

  /* ---- Begin Crit. Sect. #4 ---- */
  for (i = 0; i < ngsmdevs; i++) {
    sprintf (buffer, "%s | %5d | /dev/%-8s | %-14s | %-14s | %d | %s\r\n",
             (gsmdevices[i].free ? "." : "*"),
	     gsmdevices[i].owner,
	     gsmdevices[i].device,
             gsmdevices[i].addr,
             gsmdevices[i].defsca,
             gsmdevices[i].defmode,
	     gsmdevices[i].provider);
    tellsock (csfd, buffer);
  }                                          /* for (i... */

  /* leave crit. sect. */
  if (sem_signal (shmem_sem) == -1)
    syserr ("print_device_list(): can't signal shared mem. semaphore");
  /* ---- End Crit. Sect. #4 ---- */
  
  sprintf (buffer, "\r\n");
  tellsock (csfd, buffer);
}                                 /* print_device_list () */
/*========================================================*/
void print_long_device_list ()
{
  extern int shmem_sem;
  extern struct gsms_def *gsmdevices;
  char mode[10];
  int i, j;

  mode[0] = '\0';
  
  /* get exclusive access to shared mem. for safety's sake */
  if (sem_wait (shmem_sem) == -1)
    syserr ("print_long_device_list(): failed to wait on shared mem. semaphore");

  /* ---- Begin Crit. Sect. #4 ---- */
  for (i = 0; i < ngsmdevs; i++) {
    sprintf (buffer, "/dev/%s\r\n", gsmdevices[i].device);
    tellsock (csfd, buffer);
    sprintf (buffer, "\t- Address:    %s\r\n", gsmdevices[i].addr);
    tellsock (csfd, buffer);
    sprintf (buffer, "\t- Def. SMSC:  %s\r\n", gsmdevices[i].defsca);
    tellsock (csfd, buffer);
    sprintf (buffer, "\t- Provider:   %s\r\n", gsmdevices[i].provider);
    tellsock (csfd, buffer);
    sprintf (buffer, "\t- PIN & PUK:  [<hidden>] - [<hidden>]\r\n");
    /*sprintf (buffer, "\t- PIN & PUK:  [%s] - [%s]\r\n", gsmdevices[i].PIN,
            gsmdevices[i].PUK);*/
    tellsock (csfd, buffer);
    sprintf (buffer, "\t- Driver:     %s\r\n", gsmdevices[i].model);
    tellsock (csfd, buffer);
    switch (gsmdevices[i].defmode) {
      case SMS_MODE_PDU: {
        strcpy (mode, "PDU");
	break;
      }
      
      case SMS_MODE_TEXT: {
        strcpy (mode, "Text");
        break;
      }
      
      default: {
        strcpy (mode, "Unknown");
        break;
      }
    }
    sprintf (buffer, "\t- Def. mode:  %s\r\n", mode);
    tellsock (csfd, buffer);
    sprintf (buffer, "\t- PINpause:   %d secs.\r\n", gsmdevices[i].PINpause);
    tellsock (csfd, buffer);
    if (gsmdevices[i].free) {
      sprintf (buffer, "\t- Status:     free\r\n");
    }
    else {
      sprintf (buffer, "\t- Status:     locked by pid <%d>\r\n", gsmdevices[i].owner);
    }
    tellsock (csfd, buffer);

    /*.......................................Option flags */
    sprintf (buffer, "\t- Enabled optional features:\r\n");
    tellsock (csfd, buffer);
    /* first handle the "poll" special case */
    if (!(gsmdevices[i].flags & DEDICATED_DAEMON)) {
      sprintf (buffer, "\t\t* poll\r\n");
      tellsock (csfd, buffer);
    }
    /* then test for all other flags */
    j = 0;
    while (option_flags_list[j].name != NULL) {
      if (gsmdevices[i].flags & option_flags_list[j].flag) {
        sprintf (buffer, "\t\t* %s\r\n", option_flags_list[j].name);
        tellsock (csfd, buffer);
      }
      j++;
    }                                         /* while () */

    /*.....................................Stats counters */
    sprintf (buffer, "\t- Counters:\r\n");
    tellsock (csfd, buffer);
    sprintf (buffer, "\t\t* Sent OK         = %6d\r\n", gsmdevices[i].out_ok);
    tellsock (csfd, buffer);
    sprintf (buffer, "\t\t* Sent failed     = %6d\r\n", gsmdevices[i].out_fail);
    tellsock (csfd, buffer);
    sprintf (buffer, "\t\t* Received OK     = %6d\r\n", gsmdevices[i].in_ok);
    tellsock (csfd, buffer);
    sprintf (buffer, "\t\t* Received failed = %6d\r\n", gsmdevices[i].in_fail);
    tellsock (csfd, buffer);
    sprintf (buffer, "\t\t* UNLOCK events   = %6d\r\n", gsmdevices[i].unlocked);
    tellsock (csfd, buffer);

    /*..........................................Skip line */
    sprintf (buffer, "\r\n");
    tellsock (csfd, buffer);
  }                                          /* for (i... */

  /* leave crit. sect. */
  if (sem_signal (shmem_sem) == -1)
    syserr ("print_long_device_list(): can't signal shared mem. semaphore");
  /* ---- End Crit. Sect. #4 ---- */
  
}                            /* print_long_device_list () */
/*========================================================*/
void print_device_stats ()
{
  extern int shmem_sem;
  extern struct gsms_def *gsmdevices;
  extern time_t starttime;
  double upsecs;
  int days;
  int hours;
  int rem;
  div_t dt;

  sprintf (buffer, "\r\n"
                   "Device        | Sent OK    | Sent FAIL  | Recvd OK   | Recvd FAIL | Unlock \r\n"
                   "--------------+------------+------------+------------+------------+--------\r\n");
  tellsock (csfd, buffer);
  
  /* get exclusive access to shared mem. for safety's sake */
  if (sem_wait (shmem_sem) == -1)
    syserr ("print_device_stats(): failed to wait on shared mem. semaphore");

  /* ---- Begin Crit. Sect. #4 ---- */
  for (i = 0; i < ngsmdevs; i++) {
    sprintf (buffer, "/dev/%-8s | %10d | %10d | %10d | %10d | %6d \r\n",
	     gsmdevices[i].device,
             gsmdevices[i].out_ok,
             gsmdevices[i].out_fail,
             gsmdevices[i].in_ok,
	     gsmdevices[i].in_fail,
	     gsmdevices[i].unlocked);
    tellsock (csfd, buffer);
  }                                          /* for (i... */

  /* leave crit. sect. */
  if (sem_signal (shmem_sem) == -1)
    syserr ("print_device_stats(): can't signal shared mem. semaphore");
  /* ---- End Crit. Sect. #4 ---- */
  
  /* compute the uptime in seconds */
  upsecs = difftime (time (NULL), starttime);
  
  /* first extract the number of days */
  dt = div ((int) upsecs, 86400);
  days = dt.quot;
  rem = dt.rem;

  /* now extract the number of hours */
  dt = div (rem, 3600);
  hours = dt.quot;
  rem = dt.rem;
  
  /* then extract the number of minutes and seconds */
  dt = div (rem, 60);
  
  sprintf (buffer, "\r\nAccumulating stats since: %d days %d hours %d min %d secs.\r\n\r\n", 
          days, hours, dt.quot, dt.rem);
  tellsock (csfd, buffer);
}                                /* print_device_stats () */
/*========================================================*/
void print_queue_stat ()
{
  int items;
  extern int maxqruns;
  
  switch (qmethod) {
    case SMS_Q_NONE: {
      sprintf (buffer, "Queueing disabled.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    
    case SMS_Q_FILE: {
      if (maxqruns > 0) {
	sprintf (buffer, "Queueing enabled, method is: FILE (max QR = %d).\r\n", maxqruns);
	tellsock (csfd, buffer);
      }
      else {
	sprintf (buffer, "Queueing enabled, method is: FILE (QR unlimited).\r\n");
	tellsock (csfd, buffer);
      }
      if (dbaccess_enabled) {
        sprintf (buffer, "(the outbox check will poll the DB as well).\r\n");
        tellsock (csfd, buffer);
      }
      sprintf (buffer, "\r\n");
      tellsock (csfd, buffer);
      items = file_list_queue (csfd);
      sprintf (buffer, "%d items listed.\r\n\r\n", items);
      tellsock (csfd, buffer);
      if (dbaccess_enabled) {
        sprintf (buffer, "Database-stored queue items:\r\n");
        tellsock (csfd, buffer);
	items = db_list_queue (csfd);
	sprintf (buffer, "%d items listed (can't be removed with 'qdel').\r\n\r\n", items);
	tellsock (csfd, buffer);
      }
      break;
    }
    
    case SMS_Q_DB: {
      if (maxqruns > 0) {
	sprintf (buffer, "Queueing enabled, method is: DB (max QR = %d).\r\n\r\n", maxqruns);
	tellsock (csfd, buffer);
      }
      else {
	sprintf (buffer, "Queueing enabled, method is: DB (QR unlimited).\r\n\r\n");
	tellsock (csfd, buffer);
      }
      items = db_list_queue (csfd);
      sprintf (buffer, "%d items listed.\r\n\r\n", items);
      tellsock (csfd, buffer);
      break;
    }
    
    default: {
      sprintf (buffer, "Unknown queueing method <%d> -- internal error\r\n\r\n",
	      qmethod);
      tellsock (csfd, buffer);
      break;
    }    
  }                                   /* switch (qmethod) */
}                                  /* print_queue_stat () */
/*========================================================*/
void delete_queue_item (int reqid)
{
  switch (qmethod) {
    case SMS_Q_NONE: {
      sprintf (buffer, "Queueing disabled -- nothing to delete.\r\n\r\n");
      tellsock (csfd, buffer);
      break;
    }
    
    case SMS_Q_FILE: {
      if (file_queue_delete (reqid) == -1) {
        sprintf (buffer, "Can't delete request <%d>.\r\n", reqid);
        tellsock (csfd, buffer);
      }
      else {
        sprintf (buffer, "Request <%d> deleted.\r\n", reqid);
        tellsock (csfd, buffer);
      }
      if (dbaccess_enabled) {
        sprintf (buffer, "Warning: DB-based queue items (if any) can't be removed with 'qdel'.\r\n");
        tellsock (csfd, buffer);
      }
      else {
        sprintf (buffer, "\r\n");
        tellsock (csfd, buffer);
      }
      break;
    }
    
    case SMS_Q_DB: {
      if (db_queue_delete (reqid) == -1) {
        sprintf (buffer, "Can't delete request <%d>.\r\n\r\n", reqid);
        tellsock (csfd, buffer);
      }
      else {
        sprintf (buffer, "Request <%d> deleted.\r\n\r\n", reqid);
        tellsock (csfd, buffer);
      }
      break;
    }
    
    default: {
      sprintf (buffer, "Unknown queueing method <%d> -- internal error\r\n\r\n",
	      qmethod);
      tellsock (csfd, buffer);
      break;
    }    
  }                                   /* switch (qmethod) */
}                                 /* delete_queue_item () */
/*========================================================*/
void print_access_list ()
{
  extern acl_list gsm_acl;
  struct acl_item *cursor;
  unsigned long nomask;
  int netmask;
  double power;

  sprintf (buffer, "\r\nCurrent Access Control List:\r\n"
          "============================\r\n");
  tellsock (csfd, buffer);

  if (!empty_acl_list (gsm_acl)) {
    cursor = gsm_acl.head;
    /* convert mask value */
    nomask = ntohl (cursor->nomask);
    if (nomask < (0xFFFFFFFF)) {
      nomask++;
      power = (log (nomask) / log (2));
      netmask = (32 - power);
    }
    else {
      netmask = 0;
    }
    sprintf (buffer, "%-5s : [%s] / %d\r\n", 
            (cursor->action == ACL_ALLOW) ? "ALLOW" : "DENY", 
	    inet_ntoa (cursor->network),
            netmask);
    tellsock (csfd, buffer);
    while (cursor->next != NULL) {
      cursor = cursor->next;
      /* convert mask value */
      nomask = ntohl (cursor->nomask);
      if (nomask < (0xFFFFFFFF)) {
	nomask++;
	power = (log (nomask) / log (2));
	netmask = (32 - power);
      }
      else {
	netmask = 0;
      }
      sprintf (buffer, "%-5s : [%s] / %d\r\n", 
              (cursor->action == ACL_ALLOW) ? "ALLOW" : "DENY", 
	      inet_ntoa (cursor->network),
              netmask);
    tellsock (csfd, buffer);
    }
  }
  else {
    sprintf (buffer, "Nothing to print: empty 'ACL' list.\r\n");
    tellsock (csfd, buffer);
  }

  sprintf (buffer, "\r\n");
  tellsock (csfd, buffer);
}                                 /* print_access_list () */
/*========================================================*/
void compute_uptime ()
{
  extern time_t starttime;
  double upsecs;
  int days;
  int hours;
  int rem;
  div_t dt;

  /* compute the uptime in seconds */
  upsecs = difftime (time (NULL), starttime);
  
  /* first extract the number of days */
  dt = div ((int) upsecs, 86400);
  days = dt.quot;
  rem = dt.rem;

  /* now extract the number of hours */
  dt = div (rem, 3600);
  hours = dt.quot;
  rem = dt.rem;
  
  /* then extract the number of minutes and seconds */
  dt = div (rem, 60);
  
  sprintf (buffer, "\r\nServer uptime: %d days %d hours %d min %d secs.\r\n\r\n", 
          days, hours, dt.quot, dt.rem);
  tellsock (csfd, buffer);
}                                    /* compute_uptime () */
/*========================================================*/
int validate_msg_param (struct symbols *msg)
{
  int retval;

  retval = ((msg->smsc[0] != '\0') && (msg->destgsm[0] != '\0') && 
           (msg->message[0] != '\0') && (msg->user[0] != '\0'));
  return (retval);
}                                /* validate_msg_param () */
/*========================================================*/
int send_sms (struct gsms_def *gsm, struct symbols *symbols)
{
  return backend_send_sms (gsm, symbols, csfd);
}                                          /* send_sms () */
/*========================================================*/
int backend_send_sms (struct gsms_def *gsm, struct symbols *symbols, int csfd)
{
  extern int use_fast;      /* enable fast modem response */
  int use_init;
  int fd, retval = 0;
  char *scratch;
  char *ptr;
  char *cmsgid;
  int nread;
  int pin_just_in;
  int msgid;
  int reqmode;
  int pdubytes;
  char reqsca[MAXPHNUMLEN + 1];
  char reqmodestr[10];
  char subprompt[4];
  cpl_list cpl;
  
  /*--------------------------------------Initialisations */
  scratch = (char *) malloc ((BIGBUFF + 1) * sizeof (char));
  if (!scratch) {
    syslog ((FACILITY | LOG_ERR), "can't allocate scratch space -- dying.");
    syserr ("backend_send_sms(): can't allocate scratch space");
  }
  memset (scratch, 0, (BIGBUFF + 1));
  
  use_fast = (gsm->flags & FAST_RESPONSE);
  use_init = (gsm->flags & SMS_INIT);

  pin_just_in = FALSE; /* set to true if PIN needed be provided */

  /* open modem line */
  fd = lopen_mdm_line (gsm->device);
  if (fd < 0) {
    syslog ((FACILITY | LOG_ERR), "call to lopen_mdm_line() failed.");
    mdmperror ("sms_serv: lopen_mdm_line() failed");
    free (scratch);
    return (-1);
  }
  
  /*------------set GSM to "verbose" error reporting mode */
  if (!use_init) {
    if (set_cmee (gsm, fd, csfd) == -1) {
      return (-1);
    }
  }
  
  /*---------------------------then, check for SIM status */
  if (!use_init) {
    if (checkset_cpin (gsm, fd, csfd, &pin_just_in) == -1) {
      return (-1);
    }
  }
  
  /*--------------------------------------sleep if needed */
  /* There seems to be a timing issue in "fast" mode when
   * the AT+CNMI command is issued too fast after the last
   * AT+CPIN? and we get a "+CMS ERROR: 515" (= module
   * not ready or busy processing command). Try sleeping a
   * few secs to solve that. */
  if (use_fast && pin_just_in) {
    sprintf (scratch, "(pause %d secs. for PIN code post-processing)\r\n",
            gsm->PINpause);
    tellsock (csfd, scratch);
    sleep (gsm->PINpause);
  }
  
  /*------------set "no notify for incoming SMS messages" */
  if (!use_init) {
    if (set_cnmi_off (gsm, fd, csfd) == -1) {
      return (-1);
    }
  }
  
  /*--------------Check stored SCA against requested SMSC */
  /* Which SMSC do we require ? */
  if (strcmp (symbols->smsc, DEFAULTSMSC) == 0) {
    /* user kept default value - let's use device-level default */
    strcpy (reqsca, gsm->defsca);
  }
  else {
    /* user requested specific SMSC - let's use it */
    strcpy (reqsca, symbols->smsc);
  }
  if (checkset_csca (gsm, fd, csfd, reqsca) == -1) {
    return (-1);
  }
  
  /*-------------Check stored mode against requested mode */
  /* Which mode do we require ? */
  if (symbols->mode == SMS_MODE_DEFAULT) {
    /* user kept default value - let's use device-level default */
    reqmode = gsm->defmode;
  }
  else {
    /* user requested specific mode - let's use it */
    reqmode = symbols->mode;
  }
  if (checkset_cmgf (gsm, fd, csfd, reqmode) == -1) {
    return (-1);
  }
  
  /*---------------------Am I registered on the network ? */
  if (!(gsm->capmatrix.capflags & NO_CREG_CHECK))
  {
    if (check_creg (gsm, fd, csfd) == -1) {
      return (-1);
    }
  }
  
  /*------------------------Now actually send SMS message */
  switch (reqmode) {
    case SMS_MODE_TEXT:
      if (gsm->capmatrix.capflags & SUBPROMPT_IN_CMGS) {
        /* fill list with expected subprompt */
        cpl_list_init (&cpl);
	strcpy (subprompt, gsm->capmatrix.line_sep);
	strcat (subprompt, ">");
        cpl_list_insert (&cpl, subprompt);
        /* first send AT command and dest GSM */
        sprintf (scratch, "AT+CMGS=\"%s\"\r", symbols->destgsm);
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_resp (fd, scratch, BIGBUFF, 4, cpl)) {
	  tellsock (csfd, scratch);
	  /* check for ">" (subprompt char.) */
	  if (strstr (scratch, ">") == 0) {
	    mdm_unlock (mdmopendevice);
	    hangup (fd);
	    free (scratch);
	    free_cpl_list (&cpl);
	    syslog ((FACILITY | LOG_ERR), "error getting AT+CMGS subprompt.");
	    mdmperror ("sms_serv: error getting AT+CMGS subprompt");
	    return (-1);
	  }
	}
	else {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	  free (scratch);
	  free_cpl_list (&cpl);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
	/* then send the message proper... */
        sprintf (scratch, "%s%s", symbols->message, CTRL_Z);
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_answer (fd, scratch, BIGBUFF, 4, gsm)) {
	  tellsock (csfd, scratch);
	  /* check for "OK" */
	  if (strstr (scratch, "OK") == 0) {
	    mdm_unlock (mdmopendevice);
	    hangup (fd);
	    free (scratch);
	    free_cpl_list (&cpl);
	    syslog ((FACILITY | LOG_ERR), "error when sending SMS message.");
	    mdmperror ("sms_serv: error when sending SMS message");
	    return (-1);
	  }
	}
	else {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	  free (scratch);
	  free_cpl_list (&cpl);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
	free_cpl_list (&cpl);
      }
      else {
        /* no subprompt */
        sprintf (scratch, "AT+CMGS=\"%s\"\r%s%s", symbols->destgsm,
                symbols->message, CTRL_Z);
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_answer (fd, scratch, BIGBUFF, 4, gsm)) {
	  tellsock (csfd, scratch);
	  /* check for "OK" */
	  if (strstr (scratch, "OK") == 0) {
	    mdm_unlock (mdmopendevice);
	    hangup (fd);
	    free (scratch);
	    syslog ((FACILITY | LOG_ERR), "error when sending SMS message.");
	    mdmperror ("sms_serv: error when sending SMS message");
	    return (-1);
	  }
	}
	else {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	  free (scratch);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
      }                                 /* if (SUBPROMPT) */
      break;
      
    case SMS_MODE_PDU:
      pdubytes = (strlen (symbols->pdu) / 2);
      if (gsm->capmatrix.capflags & SUBPROMPT_IN_CMGS) {
        /* fill list with expected subprompt */
        cpl_list_init (&cpl);
	strcpy (subprompt, gsm->capmatrix.line_sep);
	strcat (subprompt, ">");
        cpl_list_insert (&cpl, subprompt);
        /* first send AT command and PDU length */
        sprintf (scratch, "AT+CMGS=%d\r", pdubytes);
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_resp (fd, scratch, BIGBUFF, 4, cpl)) {
	  tellsock (csfd, scratch);
	  /* check for ">" (subprompt char.) */
	  if (strstr (scratch, ">") == 0) {
	    mdm_unlock (mdmopendevice);
	    hangup (fd);
	    free (scratch);
	    free_cpl_list (&cpl);
	    syslog ((FACILITY | LOG_ERR), "error getting AT+CMGS subprompt.");
	    mdmperror ("sms_serv: error getting AT+CMGS subprompt");
	    return (-1);
	  }
	}
	else {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	  free (scratch);
	  free_cpl_list (&cpl);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
	/* then send the PDU string proper... */
	if (gsm->capmatrix.capflags & HAS_SCA_IN_PDU) {
          /* inserts "00" before PDU : GenSM/SM20 specific, means no SCA */
          sprintf (scratch, "00%s%s", symbols->pdu, CTRL_Z);
	}
	else {
          /* GSM 03.40 compliant PDU format */
          sprintf (scratch, "%s%s", symbols->pdu, CTRL_Z);
	}                              /* if (SCA_IN_PDU) */
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_answer (fd, scratch, BIGBUFF, 4, gsm)) {
	  tellsock (csfd, scratch);
	  /* check for "OK" */
	  if (strstr (scratch, "OK") == 0) {
	    mdm_unlock (mdmopendevice);
	    hangup (fd);
	    free (scratch);
	    free_cpl_list (&cpl);
	    syslog ((FACILITY | LOG_ERR), "error when sending SMS message.");
	    mdmperror ("sms_serv: error when sending SMS message");
	    return (-1);
	  }
	}
	else {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	  free (scratch);
	  free_cpl_list (&cpl);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
	free_cpl_list (&cpl);
      }
      else {
        /* no subprompt */
	if (gsm->capmatrix.capflags & HAS_SCA_IN_PDU) {
          /* inserts "00" before PDU : GenSM/SM20 specific, means no SCA */
          sprintf (scratch, "AT+CMGS=%d\r00%s%s", pdubytes,
                  symbols->pdu, CTRL_Z);
	}
	else {
          /* GSM 03.40 compliant PDU format */
          sprintf (scratch, "AT+CMGS=%d\r%s%s", pdubytes,
                  symbols->pdu, CTRL_Z);
	}                              /* if (SCA_IN_PDU) */
	tell_gsm (fd, scratch);
	memset (scratch, 0, (BIGBUFF + 1));
	if (get_gsm_answer (fd, scratch, BIGBUFF, 4, gsm)) {
	  tellsock (csfd, scratch);
	  /* check for "OK" */
	  if (strstr (scratch, "OK") == 0) {
	    mdm_unlock (mdmopendevice);
	    hangup (fd);
	    free (scratch);
	    syslog ((FACILITY | LOG_ERR), "error when sending SMS message.");
	    mdmperror ("sms_serv: error when sending SMS message");
	    return (-1);
	  }
	}
	else {
	  mdm_unlock (mdmopendevice);
	  hangup (fd);
	  free (scratch);
	  syslog ((FACILITY | LOG_ERR), "GSM module not responding.");
	  mdmperror ("sms_serv: GSM not responding");
	  return (-1);
	}
      }                          /* if (SUBPROMPT_IN_PDU) */
      break;
      
    default:
      mdm_unlock (mdmopendevice);
      hangup (fd);
      free (scratch);
      syslog ((FACILITY | LOG_ERR), "internal error: unsupported mode <%d>.",
             reqmode);
      mdmperror ("sms_serv: internal error: unsupported mode");
      return (-1);
      break;
  }                                     /* switch reqmode */
  /* we have the answer - now extract the message ID from it */
  if ((ptr = (char *) strstr (scratch, "+CMGS:")) == 0) {
    mdm_unlock (mdmopendevice);
    hangup (fd);
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "can't find '+CMGS:' before OK in modem response.");
    mdmperror ("sms_serv: can't find '+CMGS:' before OK in modem response");
    return (-1);
  }
  ptr += 6;                        /* jumps to next space */
  if ((cmsgid = (char *) strtok (ptr, " \t\n\r")) == NULL) {
    mdm_unlock (mdmopendevice);
    hangup (fd);
    free (scratch);
    syslog ((FACILITY | LOG_ERR), "can't find message ID (no CrLfHt) after '+CMGS:'.");
    mdmperror ("sms_serv: can't find message ID (no CrLfHt) after '+CMGS:'");
    return (-1);
  }
  else {
    msgid = atoi (cmsgid);
  }                            /* if ((cmsgid = strtok... */
    
  /*----------------------------------Close communication */
  mdm_unlock (mdmopendevice);
  hangup (fd);
  free (scratch);
  return (msgid);
}                                  /* backend_send_sms () */
/*========================================================*/
void sms_send_wrapper ()
{
  extern int global_sem;
  extern int shmem_sem;
  extern struct gsms_def *gsmdevices;
  int nwaitsecs = 0;
  int last_errno;
  int retval;
  /* int instance; */
  char device[MAXDEVLEN];
  struct gsms_def gsm;
  int msgid;
  struct sigaction sa_c;
  int reqmode;
  int dd_instance; /* any instances which have a dedicated daemon on them */
  
  sprintf (buffer, "Waiting for a free GSM instance...\r\n");
  tellsock (csfd, buffer);
  sprintf (buffer, ".");
  
  while (((retval = sem_decreq (global_sem)) == -1) &&
        (errno == EAGAIN) &&
	(nwaitsecs < m_timeout)) {
    sleep (W_STEP);
    nwaitsecs += W_STEP;
    tellsock (csfd, buffer);               /* display dot */
  }                                        /* while (...) */
  last_errno = errno;
  sprintf (buffer, "\r\n");
  tellsock (csfd, buffer);
  
  if (retval == -1) {            /* failed to get a modem */
    switch (last_errno) {
      case EAGAIN: {
        syslog ((FACILITY | LOG_WARNING), "timeout expired (all GSMs busy).");
        sprintf (buffer, "timeout expired (all GSMs busy) - try again later\r\n");
        tellsock (csfd, buffer);
        break;
      }
      default: {
        syserr ("sms_send_wrapper(): can't decrease global semaphore");
	break;
      }
    }                              /* switch (last_errno) */
  }
  else {                 /* ---- Begin Crit. Sect. #1 --- */
    /* at least 1 GSM module is free - find it */
    if (sem_wait (shmem_sem) == -1)
      syserr ("sms_send_wrapper(): failed to wait on shared mem. semaphore");

    /* ---- Begin Crit. Sect. #2 ---- */
    /* Signal Handling */
    sa_c.sa_handler = unlock_gsm;
    if (sigfillset (&sa_c.sa_mask) == -1)
      syserr ("sms_send_wrapper(): can't fill signal set");
    sa_c.sa_flags = 0;
    /* catch SIGTERM to handle child's death while GSM's locked */
    if (sigaction (SIGTERM, &sa_c, NULL) == -1)
      syserr ("sms_send_wrapper(): can't catch SIGTERM");
    /* catch SIGPIPE to handle client disconnect while GSM's locked */
    if (sigaction (SIGPIPE, &sa_c, NULL) == -1)
      syserr ("sms_send_wrapper(): can't catch SIGPIPE");

    /* search free instance */
    instance = 0;
    dd_instance = -1;
    while (instance < ngsmdevs) {
      if (gsmdevices[instance].free) {
        /* let's process this one */
	break;
      }
      if ((gsmdevices[instance].flags & DEDICATED_DAEMON) &&
         (! gsmdevices[instance].dd_busy)) {
	/* store this in case there aren't any other devices */
	dd_instance = instance;
      }
      instance++;
    }                               /* while (instance...) */

    if (instance >= ngsmdevs) {
      if (dd_instance >= 0) {
	/* okay we have a dedicated daemon we can use that instead */

	/* if you want to support share load across multiple dedicated devices
	 * use an array and randomly choose one element from the array
	 */

	/* copy to "local" space, to avoid the shmem balagan */
	if (gsmdevcpy (&gsm, &gsmdevices[dd_instance]) == -1)
	  syserr ("sms_send_wrapper(): error copying GSM instance");

	/* leave crit. sect. */
	if (sem_signal (shmem_sem) == -1)
	  syserr ("sms_send_wrapper(): can't signal shared mem. semaphore");
	/* ---- End Crit. Sect. #2 ---- */

	/* leave main crit. sect. */
	if (sem_signal (global_sem) == -1)
	  syserr ("sms_send_wrapper(): can't signal global semaphore");

	/* log it */
	syslog ((FACILITY | LOG_NOTICE), "using dedicated device </dev/%s>.",
	       gsm.device);
	sprintf (buffer,
	        "resorting to dedicated device </dev/%s>, now sending...\r\n",
	        gsm.device);
	tellsock (csfd, buffer);
	dd_send_wrapper(&gsm);
	return;
      }
      else {
        /* really no device available */
	fatal ("sms_serv: can't get a free GSM - internal error");
      }                          /* if (dd_instance >= 0) */
    }                        /* if (instance >= ngsmdevs) */
    
    /* got it - now lock it (device lockfile handled by libmodem) */
    instance_is_set = TRUE;
    gsmdevices[instance].free = FALSE;
    gsmdevices[instance].owner = getpid ();
    
    /* copy to "local" space, to avoid the shmem balagan */
    if (gsmdevcpy (&gsm, &gsmdevices[instance]) == -1)
      syserr ("sms_send_wrapper(): error copying GSM instance");
    
    /* leave crit. sect. */
    if (sem_signal (shmem_sem) == -1)
      syserr ("sms_send_wrapper(): can't signal shared mem. semaphore");
    /* ---- End Crit. Sect. #2 ---- */

    /* log it */
    syslog ((FACILITY | LOG_NOTICE), "using device </dev/%s>.", gsm.device);
    sprintf (buffer, "got </dev/%s>, now sending...\r\n", gsm.device);
    tellsock (csfd, buffer);
    
    /* Which mode do we require ? */
    if (symbols.mode == SMS_MODE_DEFAULT) {
      /* user kept default value - let's use device-level default */
      reqmode = gsm.defmode;
    }
    else {
      /* user requested specific mode - let's use it */
      reqmode = symbols.mode;
    }
    
    /* Convert message to PDU if required */
    if (reqmode == SMS_MODE_PDU) {
      sprintf (buffer, "converting message to PDU format... ");
      tellsock (csfd, buffer);
      if ((symbols.pdu = encode_pdu (&symbols, default_alphabet)) == NULL) {
	sprintf (buffer, "Fail\r\n");
	tellsock (csfd, buffer);
	/* exit */
      }
      sprintf (buffer, "Ok\r\n");
      tellsock (csfd, buffer);
    }
    
    /* ---->>> Actual SEND <<<---- */
    if ((msgid = send_sms (&gsm, &symbols)) != -1) {
      syslog ((FACILITY | LOG_NOTICE), "message sent OK, message ID is <%d>.", msgid);
      sprintf (buffer, "message sent OK, message ID is <%d>\r\n", msgid);
      tellsock (csfd, buffer);
    }
    else { /* some error occured */
      syslog ((FACILITY | LOG_WARNING), "ERROR: failed to send message.");
      sprintf (buffer, "ERROR: failed to send message - try again later\r\n");
      tellsock (csfd, buffer);
    }
    
    /* now free the modem again and update counters */
    if (sem_wait (shmem_sem) == -1)
      syserr ("sms_send_wrapper(): failed to wait on shared mem. semaphore");

    /* ---- Begin Crit. Sect. #3 ---- */
    gsmdevices[instance].free = TRUE;
    gsmdevices[instance].owner = 0;
    if (msgid != -1) {
      gsmdevices[instance].out_ok++;
    }
    else {
      gsmdevices[instance].out_fail++;
    }
    instance_is_set = FALSE;
    /* leave crit. sect. */
    if (sem_signal (shmem_sem) == -1)
      syserr ("sms_send_wrapper(): can't signal shared mem. semaphore");
    /* ---- End Crit. Sect. #3 ---- */

    /* leave main crit. sect. */
    if (sem_signal (global_sem) == -1)
      syserr ("sms_send_wrapper(): can't signal global semaphore");
  }                        /* ---- End Crit. Sect. #1 --- */
}                                  /* sms_send_wrapper () */
/*========================================================*/
void sms_post_wrapper ()
{
  extern int global_sem;
  extern int shmem_sem;
  extern struct gsms_def *gsmdevices;
  int nwaitsecs = 0;
  int last_errno;
  int retval;
  char device[MAXDEVLEN];
  struct gsms_def gsm;
  int msgid;
  struct sigaction sa_c;
  int reqmode;
  int dd_instance; /* any instances which have a dedicated daemon on them */
  
  if (qmethod == SMS_Q_NONE) {
    sprintf (buffer, "Error: queueing facility disabled. Use \"send\" command instead.\r\n");
    tellsock (csfd, buffer);
  }
  else {
    /* First attempt immediate send */
    msgid = -1;
    sprintf (buffer, "Waiting for a free GSM instance...\r\n");
    tellsock (csfd, buffer);
    sprintf (buffer, ".");

    while (((retval = sem_decreq (global_sem)) == -1) &&
          (errno == EAGAIN) &&
	  (nwaitsecs < m_timeout)) {
      sleep (W_STEP);
      nwaitsecs += W_STEP;
      tellsock (csfd, buffer);             /* display dot */
    }                                      /* while (...) */
    last_errno = errno;
    sprintf (buffer, "\r\n");
    tellsock (csfd, buffer);

    if (retval == -1) {          /* failed to get a modem */
      switch (last_errno) {
	case EAGAIN: {
          syslog ((FACILITY | LOG_WARNING), "timeout expired (all GSMs busy).");
          sprintf (buffer, "timeout expired (all GSMs busy) - try again later\r\n");
          tellsock (csfd, buffer);
          break;
	}
	default: {
          syserr ("sms_post_wrapper(): can't decrease global semaphore");
	  break;
	}
      }                            /* switch (last_errno) */
    }
    else {               /* ---- Begin Crit. Sect. #1 --- */
      /* at least 1 GSM module is free - find it */
      if (sem_wait (shmem_sem) == -1)
	syserr ("sms_post_wrapper(): failed to wait on shared mem. semaphore");

      /* ---- Begin Crit. Sect. #2 ---- */
      /* Signal Handling */
      sa_c.sa_handler = unlock_gsm;
      if (sigfillset (&sa_c.sa_mask) == -1)
	syserr ("sms_post_wrapper(): can't fill signal set");
      sa_c.sa_flags = 0;
      /* catch SIGTERM to handle child's death while GSM's locked */
      if (sigaction (SIGTERM, &sa_c, NULL) == -1)
	syserr ("sms_post_wrapper(): can't catch SIGTERM");
      /* catch SIGPIPE to handle client disconnect while GSM's locked */
      if (sigaction (SIGPIPE, &sa_c, NULL) == -1)
	syserr ("sms_post_wrapper(): can't catch SIGPIPE");

      /* search free instance */
      instance = 0;
      dd_instance = -1;
      while (instance < ngsmdevs) {
	if (gsmdevices[instance].free) {
          /* let's process this one */
	  break;
	}
	if ((gsmdevices[instance].flags & DEDICATED_DAEMON) &&
           (! gsmdevices[instance].dd_busy)) {
	  /* store this in case there aren't any other devices */
	  dd_instance = instance;
	}
	instance++;
      }                            /* while (instance...) */

      if (instance >= ngsmdevs) {
	if (dd_instance >= 0) {
	  /* okay we have a dedicated daemon we can use that instead */

	  /* if you want to support share load across multiple dedicated devices
	   * use an array and randomly choose one element from the array
	   */

	  /* copy to "local" space, to avoid the shmem balagan */
	  if (gsmdevcpy (&gsm, &gsmdevices[dd_instance]) == -1)
	    syserr ("sms_post_wrapper(): error copying GSM instance");

	  /* leave crit. sect. */
	  if (sem_signal (shmem_sem) == -1)
	    syserr ("sms_post_wrapper(): can't signal shared mem. semaphore");
	  /* ---- End Crit. Sect. #2 ---- */

	  /* leave main crit. sect. */
	  if (sem_signal (global_sem) == -1)
	    syserr ("sms_post_wrapper(): can't signal global semaphore");

	  /* log it */
	  syslog ((FACILITY | LOG_NOTICE), "using dedicated device </dev/%s>.",
		 gsm.device);
	  sprintf (buffer,
	          "resorting to dedicated device </dev/%s>, now sending...\r\n",
	          gsm.device);
	  tellsock (csfd, buffer);
	  dd_send_wrapper(&gsm);
	  return;
	}
	else {
          /* really no device available */
	  fatal ("sms_serv: can't get a free GSM - internal error");
	}                        /* if (dd_instance >= 0) */
      }                      /* if (instance >= ngsmdevs) */

      /* got it - now lock it (device lockfile handled by libmodem) */
      instance_is_set = TRUE;
      gsmdevices[instance].free = FALSE;
      gsmdevices[instance].owner = getpid ();

      /* copy to "local" space, to avoid the shmem balagan */
      if (gsmdevcpy (&gsm, &gsmdevices[instance]) == -1)
	syserr ("sms_post_wrapper(): error copying GSM instance");

      /* leave crit. sect. */
      if (sem_signal (shmem_sem) == -1)
	syserr ("sms_post_wrapper(): can't signal shared mem. semaphore");
      /* ---- End Crit. Sect. #2 ---- */

      /* log it */
      syslog ((FACILITY | LOG_NOTICE), "using device </dev/%s>.", gsm.device);
      sprintf (buffer, "got </dev/%s>, now sending...\r\n", gsm.device);
      tellsock (csfd, buffer);

      /* Which mode do we require ? */
      if (symbols.mode == SMS_MODE_DEFAULT) {
	/* user kept default value - let's use device-level default */
	reqmode = gsm.defmode;
      }
      else {
	/* user requested specific mode - let's use it */
	reqmode = symbols.mode;
      }

      /* Convert message to PDU if required */
      if (reqmode == SMS_MODE_PDU) {
	sprintf (buffer, "converting message to PDU format... ");
	tellsock (csfd, buffer);
	if ((symbols.pdu = encode_pdu (&symbols, default_alphabet)) == NULL) {
	  sprintf (buffer, "Fail\r\n");
	  tellsock (csfd, buffer);
	  /* exit */
	}
	sprintf (buffer, "Ok\r\n");
	tellsock (csfd, buffer);
      }

      /* ---->>> Actual SEND <<<---- */
      if ((msgid = send_sms (&gsm, &symbols)) != -1) {
	syslog ((FACILITY | LOG_NOTICE), "message sent OK, message ID is <%d>.", msgid);
	sprintf (buffer, "message sent OK, message ID is <%d>\r\n", msgid);
	tellsock (csfd, buffer);
      }
      else { /* some error occured */
	syslog ((FACILITY | LOG_WARNING), "ERROR: failed to send message.");
	sprintf (buffer, "ERROR: failed to send - message will be queued instead\r\n");
	tellsock (csfd, buffer);
      }

      /* now free the modem again and update counters */
      if (sem_wait (shmem_sem) == -1)
	syserr ("sms_post_wrapper(): failed to wait on shared mem. semaphore");

      /* ---- Begin Crit. Sect. #3 ---- */
      gsmdevices[instance].free = TRUE;
      gsmdevices[instance].owner = 0;
      if (msgid != -1) {
	gsmdevices[instance].out_ok++;
      }
      else {
	gsmdevices[instance].out_fail++;
      }
      instance_is_set = FALSE;
      /* leave crit. sect. */
      if (sem_signal (shmem_sem) == -1)
	syserr ("sms_post_wrapper(): can't signal shared mem. semaphore");
      /* ---- End Crit. Sect. #3 ---- */

      /* leave main crit. sect. */
      if (sem_signal (global_sem) == -1)
	syserr ("sms_post_wrapper(): can't signal global semaphore");
    }                      /* ---- End Crit. Sect. #1 --- */
    if (msgid == -1) {
      /* either no free GSM, or sending failed */
      /*----------------------------------Put it in queue */
      switch (qmethod) {
	case SMS_Q_DB:
	  if ((msgid = sms_queue_db ()) == -1) {
            sprintf (buffer, "Error: failed to store outgoing message in database.\r\n");
            tellsock (csfd, buffer);
	  }
	  else {
            sprintf (buffer, "message stored OK, message ID is <%d>.\r\n", msgid);
            tellsock (csfd, buffer);
	  }
	  break;

	case SMS_Q_FILE:
	  if ((msgid = sms_queue_file ()) == -1) {
            sprintf (buffer, "Error: failed to create outgoing message spool file.\r\n");
            tellsock (csfd, buffer);
	  }
	  else {
            sprintf (buffer, "message stored OK, message ID is <%d>.\r\n", msgid);
            tellsock (csfd, buffer);
	  }
	  break;

	case SMS_Q_NONE:
	  sprintf (buffer, "Error: queueing facility disabled. Use \"send\" command instead.\r\n");
	  tellsock (csfd, buffer);
	  break;

	default:
	  sprintf (buffer, "Error: unknown queueing method <%d>.\r\n",
        	  qmethod);
	  tellsock (csfd, buffer);
	  sprintf (buffer, "Queueing facility unavailable. Use \"send\" command instead.\r\n");
	  tellsock (csfd, buffer);
	  syslog ((FACILITY | LOG_WARNING), "unknown queueing method <%d>.",
        	 qmethod);
	  break;

      }                               /* switch (qmethod) */
    }                                 /* if (msgid == -1) */
  }                         /* if (qmethod == SMS_Q_NONE) */
}                                  /* sms_post_wrapper () */
/*========================================================*/
void sms_queue_wrapper ()
{
  int msg_id;

  /*--------------------------------------Put it in queue */
  switch (qmethod) {
    case SMS_Q_DB:
      if ((msg_id = sms_queue_db ()) == -1) {
        sprintf (buffer, "Error: failed to store outgoing message in database.\r\n");
        tellsock (csfd, buffer);
      }
      else {
        sprintf (buffer, "message stored OK, message ID is <%d>.\r\n", msg_id);
        tellsock (csfd, buffer);
      }
      break;
      
    case SMS_Q_FILE:
      if ((msg_id = sms_queue_file ()) == -1) {
        sprintf (buffer, "Error: failed to create outgoing message spool file.\r\n");
        tellsock (csfd, buffer);
      }
      else {
        sprintf (buffer, "message stored OK, message ID is <%d>.\r\n", msg_id);
        tellsock (csfd, buffer);
      }
      break;
      
    case SMS_Q_NONE:
      sprintf (buffer, "Error: queueing facility disabled. Use \"send\" command instead.\r\n");
      tellsock (csfd, buffer);
      break;
      
    default:
      sprintf (buffer, "Error: unknown queueing method <%d>.\r\n",
              qmethod);
      tellsock (csfd, buffer);
      sprintf (buffer, "Queueing facility unavailable. Use \"send\" command instead.\r\n");
      tellsock (csfd, buffer);
      syslog ((FACILITY | LOG_WARNING), "unknown queueing method <%d>.",
             qmethod);
      break;
      
  }                                   /* switch (qmethod) */
  
}                                 /* sms_queue_wrapper () */
/*========================================================*/
void unlock_gsm ()
{
  extern int shmem_sem;
  extern struct gsms_def *gsmdevices;

  syslog ((FACILITY | LOG_WARNING), "Socket connection unexpectedly lost.");
  /* set the allocated GSM module back to 'free' status */
  if (instance_is_set) {
    if (sem_wait (shmem_sem) == -1)
      syserr ("unlock_gsm(): failed to wait on shared mem. semaphore");

    /* ---- Begin Crit. Sect. #5 ---- */
    if (!gsmdevices[instance].free) {
      gsmdevices[instance].free = TRUE;
      gsmdevices[instance].owner = 0;
      gsmdevices[instance].unlocked++;
      syslog ((FACILITY | LOG_WARNING), "GSM instance </dev/%s> has been unlocked.",
             gsmdevices[instance].device);
    }
    /* leave crit. sect. */
    if (sem_signal (shmem_sem) == -1)
      syserr ("unlock_gsm(): can't signal shared mem. semaphore");
    /* ---- End Crit. Sect. #5 ---- */
  }
  /* free what's need to be freed and exit */
  free (symbols.smsc);
  free (symbols.destgsm);
  free (symbols.message);
  if (symbols.pdu) {
    free (symbols.pdu);
  }
  free (buffer);
  /*------------------------------------------------------*/
  exit (0);
}                                        /* unlock_gsm () */
/*========================================================*/
void conclude ()
{
  syslog ((FACILITY | LOG_NOTICE), "Connection closed by client.");
  /* free what's need to be freed and exit */
  free (symbols.smsc);
  free (symbols.destgsm);
  free (symbols.message);
  if (symbols.pdu) {
    free (symbols.pdu);
  }
  free (buffer);
  /*------------------------------------------------------*/
  exit (0);
}                                          /* conclude () */
/*========================================================*/

/*==========================================================
 * EOF : server.y
 *===================*/
