/*==========================================================
 * Program : sms2mailgw.c                  Project : smslink
 * Authors : Philippe Andersson.
 * Date    : 20/04/06
 * Version : 0.20b
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 *           contributions (c) Internet Access AG, 1999.
 * Comment : SMS to Mail Gateway.
 *
 * Modification History :
 * - 0.01a (17/08/99) : Initial release.
 * - 0.02a (18/10/99) : Improved code and increased logging
 *   for default domain.
 * ++++ Switched to Beta ++++
 * - 0.03b (09/11/99) : Mail sending code added. Ready to start
 *   beta-test. No change to this file.
 * - 0.04b (24/02/00) : Finalized the transfer code to the "new"
 *   mailbox file (in gw_stuff.c). Added a syslog report of the
 *   amount of messages sent.
 * - 0.05b (26/02/00) : Added a global variable (inbox_is_locked)
 *   to control lockfile removal on server's signalled exit.
 * - 0.06b (01/03/00) : Solved a few bugs in the SMTP session
 *   handling. No change in this file.
 * - 0.07b (02/03/00) : Modified the synchronisation mechanism. In
 *   case the synch. is lost, the daemon will wait 1/10th of an
 *   interval only (to avoid loosing a full cycle before delivering
 *   the mail). After a mailbox run, the daemon goes back to sleep
 *   for the rest of the cycle _+ 1 min._. This should prevent it
 *   colliding with the next MBC.
 * - 0.08b (28/03/00) : Improved error handling in gw_stuff.c.
 *   No change in this file.
 * - 0.09b (26/09/00) : Added conditional compile for FreeBSD.
 * - 0.10b (05/03/01) : Added command-line parameter processing
 *   with GNU getopt(). Added support for flags-driven debug
 *   level ('-d' parameter).
 * - 0.11b (19/03/01) : Added "mbchkinterval" command-line
 *   parameter to allow this daemon to follow sms_serv.
 * - 0.12b (06/01/02) : Replaced the old domain determination
 *   strategy that involved NIS by a more technically correct
 *   approach. Added the "--defaultdomain=" commandline option
 *   for easy override. Added the "--smtprelay=" option for
 *   added flexibility in mail routing setup. 
 * - 0.13b (28/01/02) : In gw_stuff (send_mail() function),
 *   replaced all <lf> by <cr><lf> in the dialog with the SMTP
 *   server, as per RFC 822bis, section 2.3 (see
 *   <http://cr.yp.to/docs/smtplf.html> for details). Caused
 *   problems when talking to qmail. Thanks a lot to Nathan
 *   Thomas (<nthomas@e-comm.com.au>) for letting me know. No
 *   change in this file.
 * - 0.14b (21/02/03) : Added a call to dequalify() on localhost
 *   to prevent the domain to appear twice in the from: address.
 * - 0.15b (25/07/03) : Moved everything email-related to smtp.[ch]
 *   (to be able to share it w/ sms_serv). No change in this file.
 * - 0.16b (03/08/03) : Corrected a bug that prevented multi-line
 *   inbox entries from being successfully processed. Those are
 *   now supported, whether they be mail or non-mail entries.
 *   No change in this file.
 * - 0.17b (19/02/04) : Improved debugging info.
 * - 0.18b (23/10/05) : Improved on-line help info.
 * - 0.19b (22/03/06) : Solved the "can't move new inbox file -
 *   system() failed" crash by removing the ignore on SIGCHLD.
 *   This prevented proper execution of the system() call with
 *   more recent versions of glibc (and was unnecessary anyway).
 *   Added PID file generation. Cosmetics.
 * - 0.20b (20/04/06) : Plugged a memory leak in mailbox_run().
 *   Minor changes in this file.
 *========================================================*/

#ifdef HPUX
#  include <sys/unistd.h>                /* for getopt () */
#else
#  include <unistd.h>                    /* for getopt () */
#endif                                      /* ifdef HPUX */
#include <sys/stat.h>                      /* for stat () */
#include <sys/types.h>
#include <sys/param.h>                  /* for MAXPATHLEN */
#include <sys/socket.h>
#include <sys/ipc.h>
#include <netinet/in.h>     /* for AF_INET domain sockets */
#include <arpa/inet.h>                /* for inet_addr () */
#include <netdb.h>                /* for gethostbyname () */
#include <fcntl.h>                        /* for fcntl () */
#include <time.h>                      /* for difftime () */
#include <syslog.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>               /* for strcpy & friends */
#include <getopt.h>                         /* GNU Getopt */

/*--------------------------------------Personal includes */
#include "sms_serv.h"
#include "smtp.h"

/*========================================================*/
/**********             LOCAL DEFINES              ********/
/*========================================================*/

/* For debugging purposes only - comment out for normal compile */
/* #define INCL_DEBUG_CODE */

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/

int csfd;                             /* client socket FD */
char *buffer;                    /* read and write buffer */
int inbox_is_locked = FALSE;
/* for command-line parameters */
int debug = 0;                         /* debugging level */
char *progname = NULL;
int context = CONTEXT_MLGW;
int mbchkinterval = MBCHKINTERVAL;
char *pidfile = NULL;

/*========================================================*/
/**********               FUNCTIONS                ********/
/*========================================================*/

/*========================================================*/
/**********           MAIN PROGRAM LOOP            ********/
/*========================================================*/

int main (int argc, char** argv)
{
  extern int errno;
  extern int h_errno;
  struct hostent *ipnode;
  time_t lastchked;
  struct stat filestat;              /* filled by stat () */
  int waiting;
  int sleeptime;
  int nread;
  char *localhost;
  char *p;
  char *domain;
  char *defaultdomain = NULL;
  char *mailhost = NULL;
  int lockf_desc;
  int ntry;
  int pid, ppid, gppid;             /* for fork() testing */
  int i;
  struct sigaction sa_f;
  int nmsg;
  /*..............................GNU getopt() management */
  int c=0, option_index=0;
  extern int opterr;
  static struct option long_options[] = {
    {"mbchkinterval", 1, 0, 'c'},
    {"defaultdomain", 1, 0, 'F'},
    {"smtprelay", 1, 0, 'M'},
    {"pidfile", 1, 0, 'i'},
    {"debug", 2, 0, 'd'},
    {"version", 0, 0, 'v'},
    {"help", 0, 0, 'h'},
    {0, 0, 0, 0}
  };
  
  /*---------------------------------------Store own name */
  progname = (char *) malloc ((strlen (argv[0]) + 1) * sizeof (char));
  if (! progname) {
    syserr("main(): can't malloc() progname");
  }
  strcpy (progname, argv[0]);
  l_basename (progname);
  
  /*----------------------Process command-line parameters */
  opterr = 0;
  while ((c=getopt_long (argc, argv, "c:F:M:vhd::", long_options, &option_index)) && c != -1) {
    switch (c) {
      case 'c':
        mbchkinterval = atoi (optarg);
        break;

      case 'M':
        if (optarg) {
          mailhost = (char *) malloc ((strlen (optarg) + 1) * sizeof (char));
	  if (! mailhost)
	    syserr ("main(): can't malloc() mailhost");
          strcpy (mailhost, optarg);
        }
        break;

      case 'F':
        if (optarg) {
          defaultdomain = (char *) malloc ((strlen (optarg) + 1) * sizeof (char));
	  if (! defaultdomain)
	    syserr ("main(): can't malloc() defaultdomain");
          strcpy (defaultdomain, optarg);
        }
        break;

      case 'i':
        if (optarg) {
          pidfile = (char *) malloc ((strlen (optarg) + 1) * sizeof (char));
	  if (! pidfile)
	    syserr ("main(): can't malloc() pidfile");
          strcpy (pidfile, optarg);
        }
        break;

      case 'd':
        debug = optarg ? atoi (optarg) : DEBUG_ALL;
        break;

      case 'v':
	printf ("sms2mailgw for Linux, ver %s (%s)\n", SMS_GW_VERSION, SMS_GW_DATE);
        exit (0);
        break;

      case 'h':
	printf ("sms2mailgw for Linux, ver %s (%s)\n", SMS_GW_VERSION, SMS_GW_DATE);
        printf ("\n");
        printf ("Usage: sms2mailgw [options]\n\n");
        printf ("Where [options] is one or more of the following:\n");
        printf ("--mbchkinterval=n, -cn	Sets inbox check interval (in secs)\n");
        printf ("--defaultdomain=s, -Fs	Overrides the default DNS domain\n");
        printf ("--smtprelay=s, -Ms	Overrides the default SMTP relay (%s)\n", MAILHOST);
        printf ("--pidfile=s, -is	Uses 's' as pidfile instead of default\n");
        printf ("--debug[=n], -d[n]	Sets debugging level\n");
        printf ("--help, -h		Shows this help text\n");
        printf ("--version, -v	  	Displays the version and exits\n");
        printf ("\n\n");
        exit (0);
        break;

      case '?':
	printf ("%s: unsupported option -%c\n", argv[0], optopt);
        exit (-1);
        break;

      case ':':
	printf ("%s: missing required parameter for option -%c\n", argv[0], optopt);
        exit (-1);
        break;

      default:
        break;
    }                                       /* switch (c) */
  }                         /* while ((c=getopt_long (... */

  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Current options value:\n");
    fprintf (stderr, "- mbchkinterval .... %d\n", mbchkinterval);
    fprintf (stderr, "- defaultdomain .... %s\n", defaultdomain);
    fprintf (stderr, "- mailhost ......... %s\n", mailhost);
    if (pidfile) {
      fprintf (stderr, "- PID file ........ [%s]\n", pidfile);
    }
    else {
      fprintf (stderr, "- PID file ........ using default\n");
    }
    fprintf (stderr, "- debug ............ %d\n", debug);
  }
  
  /*-------------------------First, let's become a daemon */
  gppid = fork ();
  if (gppid == -1)
    syserr ("main(): grand-father can't fork");
  else if (gppid != 0)
         exit (0);                /* grand-father's death */
  
  /* open connection with the syslog daemon - announce */
  openlog ("sms2mailgw", (LOG_CONS | LOG_PID), FACILITY);
  syslog ((FACILITY | LOG_INFO), "gateway process starting (v. %s)...", SMS_GW_VERSION);
  
  /* let's become group leader (thereby loosing my tty) */
#if (defined (FREEBSD) || defined (FREEBSD5))
  if (setsid () == -1)
#else
  if (setpgrp () == -1)
#endif
    syserr ("main(): can't become group leader");
  
  /*---------------------Validate command-line parameters */
  /*.............................................mailhost */
  if (! mailhost) {
    mailhost = (char *) malloc ((strlen (MAILHOST) + 1) * sizeof (char));
    if (! mailhost) {
      syserr ("main(): can't malloc() mailhost");
    }
    mailhost[0] = '\0';
    strcpy (mailhost, MAILHOST);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "Mailhost set to: [%s] (compiled-in default).\n",
              mailhost);
    }
  }                                    /* if (! mailhost) */
  
  if (! pidfile) {
    /* pidfile name not set yet -- build default value */
    pidfile = (char *) malloc ((MAXPATHLEN + 1) * sizeof (char));
    if (! pidfile) {
      syserr ("main(): can't malloc() pidfile");
    }
    pidfile[0] = '\0';
    sprintf (pidfile, "/var/run/%s.pid", progname);
  }                                     /* if (! pidfile) */

  /*........................................mbchkinterval */
  if ((mbchkinterval < 0) || ((mbchkinterval > 0) && (mbchkinterval < 60))) {
    syslog ((FACILITY | LOG_ERR), "config. error: mbchkinterval should be 0 or >= 60 (now %d), exiting.",
           mbchkinterval);
    syserr ("main(): config. error, invalid mbchkinterval value");
  }
  /* Warn if mailbox check is disabled */
  if (mbchkinterval == 0) {
    syslog ((FACILITY | LOG_NOTICE), "mailbox check function disabled - exiting.");
    /* no point in going any further, then */
    fprintf (stderr, "sms2mailgw: mailbox check disabled, nothing to process - exiting.\n");
    exit (0);
  }
  else {
    syslog ((FACILITY | LOG_INFO), "mailbox check function enabled, first check in %d secs.",
           mbchkinterval);
  }

  /*--------------------------------------Initialisations */
  /* Signal Handling - ignore some */
  sa_f.sa_handler = SIG_IGN;
  if (sigemptyset (&sa_f.sa_mask) == -1)
    syserr ("main(): can't empty signal set");
  sa_f.sa_flags = 0;
  if (sigaction (SIGHUP, &sa_f, NULL) == -1)    /* hangup */
    syserr ("main(): can't ignore SIGHUP");
  /* Don't ignore SIGCHLD -- it bothers the system() call */

  /* now do something meaningfull on SIGTERM */
  sa_f.sa_handler = mailgws_death;
  if (sigfillset (&sa_f.sa_mask) == -1)
    syserr ("main(): can't fill signal set");
  sa_f.sa_flags = 0;
  if (sigaction (SIGTERM, &sa_f, NULL) == -1)
    syserr ("main(): can't catch SIGTERM");
    
  /* first fork */
  ppid = fork ();
  if (ppid == -1)
    syserr ("main(): father can't fork");
  else if (ppid != 0)
         exit (0);                      /* father's death */
  /*-------------------------------Start of daemon itself */
  /* set the file creation mask */
  /* umask (0); */
  /* change directory to ...? */
#ifndef INCL_DEBUG_CODE       /* we want to keep stdout for printf's */
  /* close unused file descriptors */
#endif

  /* Create PID file */
  lay_pid_file (pidfile);
  syslog ((FACILITY | LOG_INFO), "PID file [%s] created.", pidfile);
  
  /*================================Real start of program */
  /* get the hostname I'm running on (to fill in the From: part) */
  localhost = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
  if (gethostname (localhost, MINIBUFF) == -1) {
    syserr ("main(): can't get host name");
  }
  /* make sure localhost is not qualified */
  dequalify (localhost);
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Local host name is: [%s]\n", localhost);
  }

  /* set the default domain (to allow for short names in To: fields) */
  if (! defaultdomain) {
    /* not been provided through CLI */
    if (strcmp (DEFAULTDOMAIN, ".") == 0) {
      /* means we need to get it from the system */
      if ((ipnode = gethostbyname (localhost)) == NULL) {
	fprintf (stderr, "sms2mailgw: call to gethostbyname() failed:\n");
	switch (h_errno) {
	  case HOST_NOT_FOUND:
            fprintf (stderr, "            host not found\n");
	    break;

	  case NO_ADDRESS:
            fprintf (stderr, "            name valid but no answer returned\n");
	    break;

	  case NO_RECOVERY:
            fprintf (stderr, "            NS returned permanent failure\n");
	    break;

	  case TRY_AGAIN:
            fprintf (stderr, "            NS temporary failure - try again\n");
	    break;

	  default:
            fprintf (stderr, "            unknown error code <%d>\n", h_errno);
	    break;
	}                             /* switch (h_errno) */
	syserr ("main(): can't get domain name - aborting");
      }              /* if ((ipnode = gethostbyname (...) */
      if (debug & DEBUG_PRINTF) {
        fprintf (stderr, "Official host name: %s\n", ipnode->h_name);
      }
      /* Now extract the domain from the official host name */
      if ((p = strchr (ipnode->h_name, '.')) == NULL) {
	fprintf (stderr, "gethostbyname() reported %s\n", ipnode->h_name);
	syserr ("main(): not a fully qualified host name - aborting");
      }
      domain = p + 1;
      defaultdomain = (char *) malloc ((strlen (domain) + 1) * sizeof (char));
      if (! defaultdomain) {
	syserr ("main(): can't malloc() defaultdomain");
      }
      defaultdomain[0] = '\0';
      strcpy (defaultdomain, domain);
    }
    else {
      defaultdomain = (char *) malloc ((strlen (DEFAULTDOMAIN) + 1) * sizeof (char));
      if (! defaultdomain) {
	syserr ("main(): can't malloc() defaultdomain");
      }
      defaultdomain[0] = '\0';
      strcpy (defaultdomain, DEFAULTDOMAIN);
    }
  }                               /* if (! defaultdomain) */
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Working for domain [%s]\n", defaultdomain);
  }
  syslog ((FACILITY | LOG_INFO), "working for domain [%s]", defaultdomain);
  
  /* initialize "last checked" time */
  lastchked = 0;

  syslog ((FACILITY | LOG_INFO), "server ready.");

  /*---------------------------Start real processing loop */
  while (TRUE) {
    waiting = TRUE;
    while (waiting) {
      /* if checkpoint file not present, wait for it */
      while (stat (CHECKPOINTF, &filestat) == -1) {
        switch (errno) {
          case ENOENT: {                  /* no such file */
	    sleep (5);
	    break;
	  }
	
          default: {                       /* real error */
            syslog ((FACILITY | LOG_WARNING), "can't stat() checkpoint file.");
            syserr ("main(): can't stat() checkpoint file");
	  }
	
        }                               /* switch (errno) */
      }                               /* while (stat (... */
      /* I have a checkpoint file - is it worth processing ? */
      if (filestat.st_mtime > lastchked) {
        /* waited long enough - this is the one */
	waiting = FALSE;
      }
      else {
        /* sleep for the rest of the interval */
        syslog ((FACILITY | LOG_NOTICE), "sync. lost - sleeping for 1/10th interval.");
	sleep (mbchkinterval / 10);
      }
    }                                  /* while (waiting) */
    /* now process the inbox file */
    syslog ((FACILITY | LOG_INFO), "now synchronized - starting mailbox run.");
    
    /* lock inbox file */
    ntry = 0;
    while (((lockf_desc = open (MBOX_LOCKF, (O_RDWR | O_CREAT | O_EXCL), 0444)) == -1) &&
          (ntry < MAXRETRIES)) {
      ntry++;
      sleep (2);
    }
    if (lockf_desc == -1) {
      syslog ((FACILITY | LOG_ERR), "can't lock the inbox file.");
      syserr ("main(): can't lock the inbox file");
    }
    
    /*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
    /* Inbox file now locked */
    inbox_is_locked = TRUE;
    
    /* extract mail and send it */
    nmsg = mailbox_run (localhost, defaultdomain, mailhost);
    syslog ((FACILITY | LOG_NOTICE), "mailbox run finished - %d email(s) sent.",
           nmsg);
    
    /* unlock mailbox file -- clean lockfile */
    close (lockf_desc);
    inbox_is_locked = FALSE;
    if (unlink (MBOX_LOCKF) == -1) {
      syslog ((FACILITY | LOG_ERR), "can't remove the lock file.");
      syserr ("main(): can't remove the lock file");
    }
    /*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-*/
    
    /* reset "last checked" time */
    lastchked = time (NULL);

    /* give the CPU some rest - wait for next mailbox check */
    sleeptime = (mbchkinterval - (time (NULL) - filestat.st_mtime) + 60);
    if (sleeptime < 0)
      sleeptime = mbchkinterval;
    sleep (sleeptime);
  }                                          /* while (1) */
  /*------------------------------------------Conclusions */
  /* let's clean what's allocated on the heap */
  free (localhost);
  free (defaultdomain);
  free (mailhost);
  free (progname);
  free (pidfile);
  /*------------------------------------------End program */
  exit (0);
}                                              /* main () */

/*==========================================================
 * EOF : sms2mailgw.c
 *===================*/
