/*==========================================================
 * Program : sms_serv.c                    Project : smslink
 * Authors : Philippe Andersson.
 *           Philipp Klaus <pklaus@access.ch>.
 * Date    : 02/10/06
 * Version : 0.56b-3
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 *           contributions (c) Internet Access AG, 1999.
 * Comment : SMS Server for Linux. Main source file (daemon core).
 *
 * Modification History :
 * - 0.01a (12/08/98) : Initial release.
 * - 0.02a (13/08/98) : Simmultaneous clients handling (fork).
 * - 0.03a (25/08/98) : Daemonized it. Started to implement
 *   use of syslog facility. Added handling function for SIGTERM.
 * - 0.10a (27/28/98) : Merge sms_serv with the parser module.
 * - 0.11a (01/09/98) : Completed output migration to socket
 *   (printf's to sprintf's).
 * - 0.12a (01/09/98) : Start implementing restricted access
 *   to non-sharable resources (modems) through semaphores.
 * - 0.13a (03/09/98) : Completed implementation of crit.
 *   sections through semaphores and shared memory.
 * - 0.14a (24/09/98) : Improved error handling when forking.
 * - 0.15a (27/09/98) : Display local hostname on server announce.
 * - 0.16a (30/09/98) : Reordered two lines - cosmetics.
 * - 0.17a (19/10/98) : Changed 'struct modem_def' to 'struct
 *   gsms_def' ; changed 'modemsdef' to 'gsmdevices'. Added
 *   GSM devices array init. from config file (GSMDEVFILE) for
 *   improved flexibility. Cosmetics.
 * - 0.20a (20/10/98) : Merge test program into main source
 *   tree. Implements the actual SMS sending.
 * ++++ Switch to Beta ++++
 * - 0.21b (21/10/98) : First beta release.
 * - 0.22b (23/10/98) : Added 'user' member in struct 'symbols'.
 * - 0.30b (22/11/98) : Start implementing logic to regularly
 *   check on incoming messages.
 * - 0.31b (13/12/98) : Moved to sigaction() signal handling.
 *   Adapted gsmdevices initialization for new "owner" member.
 * - 0.32b (14/12/98) : Inserted a 1 sec. sleep time in the
 *   main loop. Prevents sms_serv eating up all the CPU cycles.
 *   The side effect is a slight delay when connecting - ok.
 * - 0.33b (06/02/99) : Included an access-list mechanism
 *   based on the client's IP address. Contributed by Philipp
 *   Klaus <pklaus@access.ch>.
 * - 0.34b (14/02/99) : Implemented a more flexible version
 *   of the ACL handling.
 * - 0.40b (16/05/99) : Added "starttime" variable for uptime
 *   computation.
 * - 0.41b (20/07/99) : Finalized incoming messages processing.
 *   No change to this file.
 * - 0.42b (18/08/99) : Added handling for a checkpoint file
 *   (for interaction with a SMS to Mail gateway module). Also
 *   modified a bit the order of appearance for unistd.h. Removed
 *   the HPUX-specific code.
 * - 0.43b (08/09/99) : Included version info in the server's
 *   initial syslog message.
 * - 0.44b (25/02/00) : Improved handling of the ckeckpoint file
 *   by mbchecker.c. No change in this file.
 * - 0.45b (28/03/00) : Included preliminary support for SMS mode
 *   handling. Initialized mode field in struct symbols.
 * - 0.46b (30/03/00) : Mailbox check now disabled when interval
 *   == 0. Aborts if silly value found (neg. or < 60 secs). Log
 *   it as well.
 * - 0.47b (07/04/00) : Started implementing PDU code. Added PDU
 *   field in struct symbols and initialization code here.
 * - 0.48b (23/05/00) : Implemented a pseudo-driver concept
 *   through capacity flags for the modules, in order to
 *   support PDU mode on all of them (hopefully). No change in
 *   this file.
 * - 0.48b-2 (08/06/00) : Bugfix release (see comments in server.y
 *   and mbchecker.c). No change in this file.
 * - 0.48b-3 (22/06/00) : Bugfix release (see comments in server.y
 *   and mbchecker.c). Cosmetics.
 * - 0.48b-4 (11/09/00) : Bugfix release (see comments in
 *   mbchecker.c). No change in this file.
 * - 0.49b (26/09/00) : Added handler for SIGCHLD (see serv_stuff.c
 *   for details). Included some conditional compile for FreeBSD,
 *   and initialized c_addr_len before the accept() call (helps
 *   on FreeBSD, can't harm elsewhere). Cosmetics.
 * - 0.50b (18/10/00) : The SIGCHLD handler seemed to still let
 *   the server die on Linux after a while, so I isolated it in
 *   a conditional compile for FreeBSD where it seems to solve
 *   problems with zombies.
 * - 0.51b (24/11/00) : Added support for Siemens GSM modules
 *   S25 / S35 (thanks to Thomas Omerzu <thomas@omerzu.de> for
 *   the fix). Added support for PDU-mode SMS retrieval. Added
 *   support for Option FirstFone GSM 900 PCcard. Kudos to Bernard
 *   Willemot <bernard_willemot@cse.creoscitex.com> for lending me
 *   the hardware. No code change in this file.
 * - 0.52b (29/11/00) : Port of the client to Win32 + Cygwin1.1.
 *   No change to the server.
 * - 0.52b-2 (30/12/00) : Improved driver definitions for Wavecom
 *   devices (Thanks to David Martinez <david.martinez@wavecom.fr>
 *   from Wavecom France for his excellent technical advice).
 *   No change to any server source file.
 * - 0.53b (20/02/01) : Added global debug level handling. Added
 *   handling of "dedicated daemons" and support for "fast modem
 *   response". Cosmetics. Contributed by Andrew Worsley
 *   <epaanwo@asac.ericsson.se>.
 * - 0.54b (06/03/01) : Started implementing some mods by
 *   Ing. Andrea Vettori <a.vettori@inetronics.com>, namely
 *   his command-line params. parsing with getopt, to prepare
 *   for future extensions. Replaced control of all debugging
 *   code by global flags (-d parameter).
 * - 0.54b-2 (30/03/01) : Added a DEBUG_HEXDUMP feature to
 *   help debugging "fast modem response" mode on some devices.
 *   No change in this file.
 * - 0.54b-3 (04/04/01) : Debugged the "fast modem response"
 *   mode, included "modem_okay" and "line_sep" in the device
 *   "driver". Added a test on debug level being < 0, and logged
 *   current debug level when != 0.
 * - 0.54b-4 (05/05/01) : Improved handling of +CREG: returned
 *   status. No changes to this file.
 * - 0.54b-5 (05/05/01) : Solved a bug with "fast modem response"
 *   mode when modem has SUBPROMPT_IN_CMGS. No changes in this
 *   file.
 * - 0.54b-6 (08/05/01) : Added a pause (in server.y, mbchecker.c
 *   and dd.c) after PIN processing when using "fast" mode to
 *   try and solve the "+CMS ERROR: 515" issue. Added de-allocation
 *   code in server.y (back-ported from 0.55b). No changes in
 *   this file.
 * - 0.54b-7 (26/09/01) : Made numplan support more flexible and
 *   added debug feature for it in pdu.c (requested by Thomas
 *   Gutzler <giggel@hadiko.de>). No changes to this file.
 * - 0.54b-8 (07/10/01) : Replaced calls to blopen_mdm_line()
 *   by calls to lopen_mdm_line(). Advantage being that this
 *   version uses the max. supported speed by the device by
 *   default (the one defined in /etc/modems). Added formal
 *   support for numplan #85. No changes to this file.
 * - 0.54b-9 (19/02/02) : Added preliminary support for Ericsson
 *   T65 device. Contributed by Ritesh Shah <riteshs@rediff.co.in>.
 *   Sending is supposed to work, reception still problematic.
 *   No changes to this file. Ver. 0.54b-9 was never officially
 *   released.
 * - 0.55b (03/08/03) : Went ahead with mods by Ing. Andrea
 *   Vettori <a.vettori@inetronics.com> (more command-line
 *   parameters, access to back-end database, "initialized"
 *   mode support, etc.). Added "context" concept. Added
 *   initialization and update code for the new stats fields
 *   in struct gsms_def. Added #include for "gsmdev.h".
 *   Implemented the "stats" & "dl long" commands. Added queueing
 *   support. Added preliminary support for Ericsson DI27
 *   device (an IR plug-in modem module for the T10 and T18).
 *   Contributed by Frank Weis <frank.weis@sitel.lu>. Improved
 *   concurrent access to multiple devices in "dd" mode. Restored
 *   FreeBSD portability (with help from Greg May <greg@ult.co.za>).
 * - 0.55b-2 (10/10/03) : Corrected bug where the outbox.stored
 *   column would get reset to NOW() on update. Thanks to Andrew
 *   Goldschmidt (<andrew@rsaweb.co.za>) for pointing it out.
 *   No change in this file. Fixed compilation on FreeBSD 4.8.
 *   Thanks to Nicki de Wet (<klitsgras@ananzi.co.za>) for the patch.
 * - 0.56b (11/04/04) : Corrected TPDU header flags. Added
 *   --exec command-line parameter. Added structure to control
 *   outbox check child procs concurrency. Added queued messages
 *   priority support. Added "purge" command. Added version
 *   info to debug output. Fixed an old bug in the parser code.
 * - 0.56b-2 (22/03/06) : Improved dump_string() debugging function.
 *   Improved startup message. Added PID file generation. Cosmetics.
 * - 0.56b-3 (02/10/06) : Fixes in sms2mailgw. Changed DB column
 *   name to support MySQL v5.0 (contributed by Tony S. Sykes
 *   <Tony.Sykes@BCPSoftware.com>). Restored compatibility with
 *   FreeBSD (fix contributed by Peter Forgac <forgacp@mail.t-mobile.sk>).
 *   Prevent string overflow when processing DB queue items that
 *   were created outside of SMSLink (bug reported by Gabriele Zappi).
 *   Limited OBC child processes to 1 (instead of ngsmdevs) to
 *   prevent multiple send of same message. Bug reported by
 *   Tony S. Sykes.
 *========================================================*/

#include <unistd.h>                      /* for getopt () */
#include <sys/types.h>
#include <sys/param.h>                  /* for MAXPATHLEN */
#include <sys/socket.h>
#include <sys/sem.h>           /* for semaphore functions */
#include <sys/shm.h>       /* for shared memory functions */
#include <sys/ipc.h>
#include <netinet/in.h>     /* for AF_INET domain sockets */
#include <arpa/inet.h>                /* for inet_addr () */
#include <netdb.h>                /* for gethostbyname () */
#include <fcntl.h>                        /* for fcntl () */
#include <time.h>                      /* for difftime () */
#include <syslog.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>               /* for strcpy & friends */
#include <getopt.h>                         /* GNU Getopt */

/*--------------------------------------Personal includes */
#include "sms_serv.h"
#include "gsmdev.h"
#include "smtp.h"
#include "pdu.h"

/*========================================================*/
/**********             LOCAL DEFINES              ********/
/*========================================================*/

/* For debugging purposes only - comment out for normal compile */
/* #define INCL_DEBUG_CODE */

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/

int csfd;                             /* client socket FD */
char *buffer;                    /* read and write buffer */
struct symbols symbols;              /* msg-specific data */
struct symbols ob_symbols;    /* outbox msg-specific data */
int context = 0;
int global_sem;       /* sem. for global send crit. sect. */
int shmem_sem;             /* sem. for shared mem. access */
int shmem_id;                   /* shared mem. identifier */
void *shared_memory = (void *)0;     /* shared mem. addr. */
time_t starttime;               /* for uptime calculation */
struct gsms_def *gsmdevices;
acl_list gsm_acl;
int ngsmdevs = 0;                /* # defined GSM devices */
int use_fast = 0;     /* fast modem response (def. = off) */
obc_list childprocs;        /* max # parallel child procs */
char *accessfile = NULL;
char *dbaccessfile = NULL;
char *gsmdevfile = NULL;
char *localhost;
char *defaultdomain = NULL;
char *mailhost = NULL;
char *bounce_addr = NULL;

/* for command-line parameters */
int debug = 0;                         /* debugging level */
char *progname = NULL;
int mbchkinterval = MBCHKINTERVAL;
int obchkinterval = OBCHKINTERVAL;
int maxqruns = MAXQRUNS;
int obchklimit = OBCHKLIMIT;
char *db_dbhost = NULL;        /* MySQL access parameters */
char *db_dbname = NULL;
char *db_username = NULL;
char *db_password = NULL;
int default_alphabet = DEFAULT_ALPHABET;
int dbaccess_enabled = FALSE;
int m_timeout = M_TIMEOUT;
int qmethod = SMS_Q_FILE;
char *targetprog = NULL;
char *pidfile = NULL;

/* for flex input */
char myinput[];
char *myinputptr;              /* current pos. in myinput */
char *myinputlim;                   /* ptr to end of data */

/*========================================================*/
/**********               FUNCTIONS                ********/
/*========================================================*/

/*========================================================*/
/**********           MAIN PROGRAM LOOP            ********/
/*========================================================*/

int main (int argc, char** argv)
{
  struct hostent *ipnode;
  struct servent ssent;
  struct servent *sent;
  struct sockaddr_in s_addr;     /* server socket address */
  struct sockaddr_in c_addr;     /* client socket address */
  int ssfd;                           /* server socket FD */
  int sockflags;                /* socket attribute flags */
  time_t lastchked;                              /* inbox */
  time_t ob_lastchked;                          /* outbox */
  int c_addr_len;  /* size of struct returned by accept() */
  int smstcpport = 0;          /* non-default server port */
  int nread;
  char *destgsm;
  char *message;
  char *p;
  char *temp;
  char *domain;
  char *csname = NULL;
  char *filesuffix = NULL;
  char *scratch;
  int suffixlen;
  int pid, ppid, gppid;             /* for fork() testing */
  int i;
  int optval;                     /* INA setsockopt value */
  struct gsms_def *gsmitem;
  struct sigaction sa_f;
  /*..............................GNU getopt() management */
  int c=0, option_index=0;
  extern int opterr;
  static struct option long_options[] = {
    {"mbchkinterval", 1, 0, 'c'},
    {"obchkinterval", 1, 0, 'o'},
    {"obchklimit", 1, 0, 'l'},
    {"maxqruns", 1, 0, 'r'},
    {"dbhost", 1, 0, 'D'},
    {"dbname", 1, 0, 'B'},
    {"dbuser", 1, 0, 'u'},
    {"dbpass", 1, 0, 'k'},
    {"port", 1, 0, 'p'},
    {"pidfile", 1, 0, 'i'},
    {"filesuffix", 1, 0, 'f'},
    {"mtimeout", 1, 0, 't'},
    {"qmethod", 1, 0, 'q'},
    {"exec", 1, 0, 'X'},
    {"charset", 1, 0, 'z'},
    {"defaultdomain", 1, 0, 'F'},
    {"smtprelay", 1, 0, 'M'},
    {"bounceaddr", 1, 0, 'b'},
    {"debug", 2, 0, 'd'},
    {"version", 0, 0, 'v'},
    {"help", 0, 0, 'h'},
    {0, 0, 0, 0}
  };
  
  /*---------------------------------------Store own name */
  progname = (char *) malloc ((strlen (argv[0]) + 1) * sizeof (char));
  if (! progname) {
    syserr("main(): can't malloc() progname");
  }
  progname[0] = '\0';
  strcpy (progname, argv[0]);
  l_basename (progname);
  
  /*-----------------------------------Deal w/ targetprog */
  targetprog = (char *) malloc ((MAXPATHLEN + 1) * sizeof (char));
  if (! targetprog) {
    syserr("main(): can't malloc() targetprog");
  }
  targetprog[0] = '\0';
  strcpy (targetprog, SMS_PROG_NAME);
  
  /*----------------------Process command-line parameters */
  opterr = 0;
  while ((c=getopt_long (argc, argv, "c:o:l:r:D:B:u:k:p:i:f:t:q:X:F:M:b:z:vhd::", long_options, &option_index)) && c != -1) {
    switch (c) {
      case 'c':
        mbchkinterval = atoi (optarg);
        break;

      case 'o':
        obchkinterval = atoi (optarg);
        break;

      case 'l':
        obchklimit = atoi (optarg);
        break;

      case 'r':
        maxqruns = atoi (optarg);
        break;

      case 'D':
        if (optarg) {
          db_dbhost = (char *) malloc ((strlen (optarg) + 1) * sizeof (char));
	  if (! db_dbhost)
	    syserr ("main(): can't malloc() db_dbhost");
          strcpy (db_dbhost, optarg);
        }
        break;

      case 'B':
        if (optarg) {
          db_dbname = (char *) malloc ((strlen (optarg) + 1) * sizeof (char));
	  if (! db_dbname)
	    syserr ("main(): can't malloc() db_dbname");
          strcpy (db_dbname, optarg);
        }
        break;

      case 'u':
        if (optarg) {
          db_username = (char *) malloc ((strlen (optarg) + 1) * sizeof (char));
	  if (! db_username)
	    syserr ("main(): can't malloc() db_username");
          strcpy (db_username, optarg);
        }
        break;

      case 'k':
        if (optarg) {
          db_password = (char *) malloc ((strlen (optarg) + 1) * sizeof (char));
	  if (! db_password)
	    syserr ("main(): can't malloc() db_password");
          strcpy (db_password, optarg);
        }
        break;

      case 'f':
        if (optarg) {
          filesuffix = (char *) malloc ((strlen (optarg) + 1) * sizeof (char));
	  if (! filesuffix)
	    syserr ("main(): can't malloc() filesuffix");
          strcpy (filesuffix, optarg);
        }
        break;

      case 'p':
        smstcpport = atoi (optarg);
        break;

      case 'i':
        if (optarg) {
          pidfile = (char *) malloc ((strlen (optarg) + 1) * sizeof (char));
	  if (! pidfile)
	    syserr ("main(): can't malloc() pidfile");
          strcpy (pidfile, optarg);
        }
        break;

      case 't':
        m_timeout = atoi (optarg);
        break;

      case 'q':
        if (strcasecmp (optarg, "none") == 0) {
	  qmethod = SMS_Q_NONE;
	}
	else if (strcasecmp (optarg, "file") == 0) {
	  qmethod = SMS_Q_FILE;
	}
	else if (strcasecmp (optarg, "db") == 0) {
	  qmethod = SMS_Q_DB;
	}
	else {
          fprintf (stderr, "Requested queueing method %s not supported\n", optarg);
	  exit (-1);
	}
        break;

      case 'X':
        if (optarg) {
          if (strlen (optarg) > MAXPATHLEN) {
            fprintf (stderr, "-X parameter too long (max. %d)\n", MAXPATHLEN);
	    exit (-1);
	  }
	  else {
	    strcpy (targetprog, optarg);
	  }
        }
        break;

      case 'M':
        if (optarg) {
          mailhost = (char *) malloc ((strlen (optarg) + 1) * sizeof (char));
	  if (! mailhost)
	    syserr ("main(): can't malloc() mailhost");
          strcpy (mailhost, optarg);
        }
        break;

      case 'F':
        if (optarg) {
          defaultdomain = (char *) malloc ((strlen (optarg) + 1) * sizeof (char));
	  if (! defaultdomain)
	    syserr ("main(): can't malloc() defaultdomain");
          strcpy (defaultdomain, optarg);
        }
        break;

      case 'b':
        if (optarg) {
          bounce_addr = (char *) malloc ((strlen (optarg) + 1) * sizeof (char));
	  if (! bounce_addr)
	    syserr ("main(): can't malloc() bounce_addr");
          strcpy (bounce_addr, optarg);
        }
        break;

      case 'z':
        if (optarg) {
          if ((default_alphabet = getcharsetID (optarg)) == -1) {
            fprintf (stderr, "Requested charset %s not supported\n", optarg);
	    exit (-1);
	  }
        }
        break;

      case 'd':
        debug = optarg ? atoi (optarg) : DEBUG_ALL;
	if (debug < 0) {
	  debug = DEBUG_ALL;
	}
        break;

      case 'v':
	printf ("SMS-Link for Linux, ver %s (%s)\n", SMS_SERV_VERSION, SMS_SERV_DATE);
        exit (0);
        break;

      case 'h':
	printf ("SMS-Link for Linux, ver %s (%s)\n", SMS_SERV_VERSION, SMS_SERV_DATE);
        printf ("\n");
        printf ("Usage: sms_serv [options]\n\n");
        printf ("Where [options] is one or more of the following:\n");
        printf ("--mbchkinterval=n, -cn Sets inbox check interval (in secs) [def. %d]\n",
	       mbchkinterval);
        printf ("--obchkinterval=n, -on Sets outbox check interval (in secs) [def. %d]\n",
	       obchkinterval);
        printf ("--obchklimit=n, -ln    Sets max messages processed per queue run [def. %d]\n",
	       obchklimit);
        printf ("--maxqruns=n, -rn      Sets queue items longevity (in queue runs) [def. %d]\n",
	       maxqruns);
        printf ("--dbhost=s, -Ds        Server name where the DB lives\n");
        printf ("--dbname=s, -Bs        Database instance / name\n");
        printf ("--dbuser=s, -us        Username for connecting to the DB\n");
        printf ("--dbpass=s, -ks        Password for connecting to the DB (unsafe)\n");
        printf ("--port=n, -pn          Listens on tcp port n instead of default\n");
        printf ("--pidfile=s, -is       Uses 's' as pidfile instead of default\n");
        printf ("--filesuffix=s, -fs    Adds suffix '.s' to all config. files.\n");
        printf ("--mtimeout=n, -tn      Sets free modem wait timeout to n secs [def. %d]\n",
	       m_timeout);
        printf ("--qmethod=s, -qs       Selects queueing method (file|db|none)\n");
        printf ("--exec=s, -Xs          Sets the external program to be used by 'prog' flag\n");
        printf ("                         (default: [%s]).\n", targetprog);
        printf ("--defaultdomain=s, -Fs Overrides the default DNS domain\n");
        printf ("--smtprelay=s, -Ms     Specifyies the SMTP relay (default: %s)\n", MAILHOST);
        printf ("--bounceaddr=s, -bs    Who gets mails about undeliverables (default: %s)\n", BOUNCE_ADDR);
        printf ("--charset=s, -zs       Sets character set for PDU processing\n");
        printf ("--debug[=n], -d[n]     Sets debugging level\n");
        printf ("--help, -h             Shows this help text\n");
        printf ("--version, -v          Displays the version and exits\n");
        printf ("\n");
        printf ("Supported character sets:\n");
	listsupportedcharsets ();
        printf ("    (* indicates compiled-in default).\n");
        printf ("\n");
        exit (0);
        break;

      case ':':
	printf ("%s: missing required parameter for option -%c\n", argv[0], optopt);
        exit (-1);
        break;

      case '?':
	printf ("%s: unsupported option -%c\n", argv[0], optopt);
        exit (-1);
        break;

      default:
        break;
    }                                       /* switch (c) */
  }                         /* while ((c=getopt_long (... */

  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Running server info:\n");
    fprintf (stderr, "- version ......... [%s]\n", SMS_SERV_VERSION);
    fprintf (stderr, "- release date .... [%s]\n", SMS_SERV_DATE);
    fprintf (stderr, "Current options value:\n");
    fprintf (stderr, "- mbchkinterval ... %d\n", mbchkinterval);
    fprintf (stderr, "- obchkinterval ... %d\n", obchkinterval);
    fprintf (stderr, "- obchklimit ...... %d\n", obchklimit);
    fprintf (stderr, "- maxqruns ........ %d\n", maxqruns);
    fprintf (stderr, "- dbhost .......... [%s]\n", db_dbhost);
    fprintf (stderr, "- dbname .......... [%s]\n", db_dbname);
    fprintf (stderr, "- dbuser .......... [%s]\n", db_username);
    fprintf (stderr, "- dbpass .......... [%s]\n", db_password);
    fprintf (stderr, "- mtimeout ........ %d\n", m_timeout);
    fprintf (stderr, "- qmethod ......... %d\n", qmethod);
    fprintf (stderr, "- exec ............ [%s]\n", targetprog);
    fprintf (stderr, "- defaultdomain ... [%s]\n", defaultdomain);
    fprintf (stderr, "- mailhost ........ [%s]\n", mailhost);
    fprintf (stderr, "- bounce addr ..... [%s]\n", bounce_addr);
    fprintf (stderr, "- charset ID ...... %d\n", default_alphabet);
    if (smstcpport) {
      fprintf (stderr, "- TCP port ........ %d\n", smstcpport);
    }
    else {
      fprintf (stderr, "- TCP port ........ using default\n");
    }
    if (pidfile) {
      fprintf (stderr, "- PID file ........ [%s]\n", pidfile);
    }
    else {
      fprintf (stderr, "- PID file ........ using default\n");
    }
    if (filesuffix) {
      fprintf (stderr, "- file suffix ..... [.%s]\n", filesuffix);
    }
    else {
      fprintf (stderr, "- file suffix ..... none\n");
    }
    fprintf (stderr, "- debug ........... %d\n", debug);
  }

  /*-------------------------First, let's become a daemon */
  gppid = fork ();
  if (gppid == -1)
    syserr ("main(): grand-father can't fork");
  else if (gppid != 0)
         exit (0);                /* grand-father's death */
  
  /* open connection with the syslog daemon - announce */
  openlog ("sms_serv", (LOG_CONS | LOG_PID), FACILITY);
  syslog ((FACILITY | LOG_INFO), "server starting (ver. %s [%s])...",
         SMS_SERV_VERSION, SMS_SERV_DATE);
  
  /* let's become group leader (thereby loosing my tty) */
#if (defined (FREEBSD) || defined (FREEBSD5))
  if (setsid () == -1)
#else
  if (setpgrp () == -1)
#endif
    syserr ("main(): can't become group leader");
  
  /*---------------------Validate command-line parameters */
  /*.............................................mailhost */
  if (! mailhost) {
    mailhost = (char *) malloc ((strlen (MAILHOST) + 1) * sizeof (char));
    if (! mailhost) {
      syserr ("main(): can't malloc() mailhost");
    }
    mailhost[0] = '\0';
    strcpy (mailhost, MAILHOST);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "Mailhost set to: [%s] (compiled-in default).\n",
              mailhost);
    }
  }                                    /* if (! mailhost) */
  
  /*..........................................bounce_addr */
  if (! bounce_addr) {
    bounce_addr = (char *) malloc ((strlen (BOUNCE_ADDR) + 1) * sizeof (char));
    if (! mailhost) {
      syserr ("main(): can't malloc() bounce_addr");
    }
    bounce_addr[0] = '\0';
    strcpy (bounce_addr, BOUNCE_ADDR);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "Bounce_addr set to: [%s] (compiled-in default).\n",
              bounce_addr);
    }
  }                                 /* if (! bounce_addr) */
  
  /*--------------------------------------Initialisations */
  /* Configuration files */
  suffixlen = 0;
  if (filesuffix) {
    suffixlen = strlen (filesuffix) + 1;
  }

  if (! pidfile) {
    /* pidfile name not set yet -- build default value */
    pidfile = (char *) malloc ((MAXPATHLEN + 1) * sizeof (char));
    if (! pidfile) {
      syserr ("main(): can't malloc() pidfile");
    }
    pidfile[0] = '\0';
    if (suffixlen) {
      sprintf (pidfile, "/var/run/%s.pid.%s", progname, filesuffix);
    }
    else {
      sprintf (pidfile, "/var/run/%s.pid", progname);
    }
  }
  else {
    if (suffixlen) {
      /* realloc pidfile, concatenate suffix */
      temp = (char *) malloc ((strlen (pidfile) + suffixlen + 1) * sizeof (char));
      if (! temp) {
	syserr ("main(): can't malloc() temp");
      }
      temp[0] = '\0';
      strcpy (temp, pidfile);
      strcat (temp, ".");
      strcat (temp, filesuffix);
      free (pidfile);
      pidfile = temp;
    }
  }                                     /* if (! pidfile) */

  accessfile = (char *) malloc ((strlen (ACCESSFILE) + suffixlen + 1) * sizeof (char));
  if (! accessfile) {
    syserr ("main(): can't malloc() accessfile");
  }
  strcpy (accessfile, ACCESSFILE);
  if (filesuffix) {
    strcat (accessfile, ".");
    strcat (accessfile, filesuffix);
  }

  dbaccessfile = (char *) malloc ((strlen (DBACCESSFILE) + suffixlen + 1) * sizeof (char));
  if (! dbaccessfile) {
    syserr ("main(): can't malloc() dbaccessfile");
  }
  strcpy (dbaccessfile, DBACCESSFILE);
  if (filesuffix) {
    strcat (dbaccessfile, ".");
    strcat (dbaccessfile, filesuffix);
  }
  
  gsmdevfile = (char *) malloc ((strlen (GSMDEVFILE) + suffixlen + 1) * sizeof (char));
  if (! gsmdevfile) {
    syserr ("main(): can't malloc() gsmdevfile");
  }
  strcpy (gsmdevfile, GSMDEVFILE);
  if (filesuffix) {
    strcat (gsmdevfile, ".");
    strcat (gsmdevfile, filesuffix);
  }
  
  /* Signal Handling - ignore some */
  sa_f.sa_handler = SIG_IGN;
  if (sigemptyset (&sa_f.sa_mask) == -1)
    syserr ("main(): can't empty signal set");
  sa_f.sa_flags = 0;
  if (sigaction (SIGHUP, &sa_f, NULL) == -1)    /* hangup */
    syserr ("main(): can't ignore SIGHUP");

  /* handle SIGCHLD to prevent zombie formation */
  sa_f.sa_handler = sig_chld;
  if (sigfillset (&sa_f.sa_mask) == -1)
    syserr ("main(): can't fill signal set");
  sa_f.sa_flags = 0;
  if (sigaction (SIGCHLD, &sa_f, NULL) == -1) 
    syserr ("main(): can't catch SIGCHLD");

  /* now do something meaningfull on SIGTERM */
  sa_f.sa_handler = daemons_death;
  if (sigfillset (&sa_f.sa_mask) == -1)
    syserr ("main(): can't fill signal set");
  sa_f.sa_flags = 0;
  if (sigaction (SIGTERM, &sa_f, NULL) == -1)
    syserr ("main(): can't catch SIGTERM");
    
  /* first fork */
  ppid = fork ();
  if (ppid == -1)
    syserr ("main(): father can't fork");
  else if (ppid != 0)
         exit (0);                      /* father's death */

  /*-------------------------------Start of daemon itself */
  /* set the file creation mask */
  /* umask (0); */
  /* change directory to ...? */
#ifndef INCL_DEBUG_CODE       /* we want to keep stdout for printf's */
  /* close unused file descriptors */
#endif

  /* Create PID file */
  lay_pid_file (pidfile);
  syslog ((FACILITY | LOG_INFO), "PID file [%s] created.", pidfile);
  
  /* Load MySQL credentials from file /etc/gsmdbcreds (if present) */
  /* Any value given through CL option will NOT be overriden */
  /* This _HAS_ to happen before struct gsms_def init - for dd_spawn() */
#ifdef WITH_MYSQL
  if (got_any_creds () || file_exist (dbaccessfile)) {
    if (! load_db_creds ()) {
      syslog ((FACILITY | LOG_ERR), "config. error: can't parse DB creds file %s or no creds found.",
             dbaccessfile);
      syserr ("main(): config. error, can't parse DB creds file or no creds found");
    }
    if (! validate_db_creds ()) {
      syslog ((FACILITY | LOG_ERR), "config. error: invalid DB credentials %s@%s/%s.",
             db_username, db_dbhost, db_dbname);
      syserr ("main(): config. error, invalid DB credentials");
    }
    dbaccess_enabled = TRUE;
    syslog ((FACILITY | LOG_INFO), "DB access enabled with credentials: %s@%s/%s",
           db_username, db_dbhost, db_dbname);
    if (debug & DEBUG_DBACCESS) {
      fprintf (stderr, "DB access enabled with credentials: %s@%s/%s.\n",
              db_username, db_dbhost, db_dbname);
    }
  }
  else {
    dbaccess_enabled = FALSE;
    syslog ((FACILITY | LOG_INFO), "DB access disabled.");
    if (debug & DEBUG_DBACCESS) {
      fprintf (stderr, "DB access disabled.\n");
    }
  }
#endif

  /* how many GSM devices do we have ? */
  syslog ((FACILITY | LOG_INFO), "loading GSM module definitions...");
  if ((ngsmdevs = getgsmdevscount (TRUE)) == 0) {
    syslog ((FACILITY | LOG_ERR), "no GSM module definition found, exiting.");
    syserr ("main(): no defined GSM device");
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "getgsmdevscount() reported %d valid configs\n", ngsmdevs);
  }
  if (ngsmdevs > MAXMODEMS) {
    syslog ((FACILITY | LOG_ERR), "too many GSM module definitions found, exiting.");
    syserr ("main(): hardware can't support that many serial devices");
  }
    
  /* create and initialize semaphores */
  if ((global_sem = semget (IPC_PRIVATE, 1, (0660 | IPC_CREAT | IPC_EXCL))) == -1)
    syserr ("main(): can't create global semaphore");
  if (set_semvalue (global_sem, ngsmdevs) == -1)
    syserr ("main(): unable to initialize global semaphore");
    
  if ((shmem_sem = semget (IPC_PRIVATE, 1, (0660 | IPC_CREAT | IPC_EXCL))) == -1)
    syserr ("main(): can't create shmem semaphore");
  if (set_semvalue (shmem_sem, 1) == -1)
    syserr ("main(): unable to initialize shmem semaphore");
  
  /* create shared memory segment */
  if ((shmem_id = shmget (IPC_PRIVATE, (ngsmdevs * sizeof (struct gsms_def)),
     (0660 | IPC_CREAT))) == -1)
    syserr ("main(): can't create shared memory segment");
  if ((shared_memory = shmat (shmem_id, (void *)0, 0)) == (void *)-1)
    syserr ("main(): can't attach shared memory segment");
  gsmdevices = (struct gsms_def *)shared_memory;
  
  /* now initialize shared memory with GSM devices definition */
  setgsmdevs;
  for (i = 0; i < ngsmdevs; i++) {
    if ((gsmitem = getnextgsmdev ()) != NULL) {
      if (gsmdevcpy (&gsmdevices[i], gsmitem) == -1)
        syserr ("main(): error copying GSM instance");
      gsmdevices[i].owner = 0;
      gsmdevices[i].in_ok = 0;
      gsmdevices[i].in_fail = 0;
      gsmdevices[i].out_ok = 0;
      gsmdevices[i].out_fail = 0;
      gsmdevices[i].unlocked = 0;
      gsmdevices[i].dd_busy = FALSE;
      if (gsmdevices[i].flags & DEDICATED_DAEMON) {
	gsmdevices[i].free = FALSE;
	/* forks off a daemon to handle this */
	dd_spawn (gsmdevices + i);
      }
      else if (gsmdevices[i].flags & SMS_INIT) {
	gsmdevices[i].free = FALSE;
	/* Initialize this instance */
        syslog ((FACILITY | LOG_INFO), "calling INIT for device [%s].", gsmdevices[i].device);
	if (initialize_gsm (&gsmdevices[i]) == -1) {
	  /* log failure and abort */
          syslog ((FACILITY | LOG_ERR), "GSM module [%s] initialization failed.",
                 gsmdevices[i].device);
          syserr ("main(): GSM module initialization failed");
	}
	gsmdevices[i].free = TRUE;
      }
      else {
	gsmdevices[i].free = TRUE;
      }
      if (debug & DEBUG_PRINTF) {
        fprintf (stderr, "Loaded GSM instance %d: device %s - flags = %d\n",
	        i, gsmdevices[i].device, gsmdevices[i].flags);
      }
    }
    else {
      syserr ("main(): error while reading GSM config. file");
    }
  }                                      /* for (i = 0... */
  if (i != ngsmdevs) {
    syslog ((FACILITY | LOG_ERR), "internal error: expected %d, found %d definition(s).",
           ngsmdevs, i);
    syserr ("main(): internal error, not all GSM modules accounted for");
  }
    
  syslog ((FACILITY | LOG_INFO), "found %d GSM module definition(s)", ngsmdevs);
  
  /* Log current charset */
  csname = getcharsetname (default_alphabet);
  if (default_alphabet == 0) {
    /* GSM default alphabet is not meant to be used on the server side */
    fprintf (stderr, "WARNING: charset set to GSM default alphabet.\n");
    fprintf (stderr, "         This may not have the intended result !\n");
    syslog ((FACILITY | LOG_WARNING), "charset set to GSM default alphabet. Please check!!");
  }
  else {
    syslog ((FACILITY | LOG_INFO), "using charset %s", csname);
  }
  if (csname) {
    free (csname);
  }
  
  /* Check for m_timeout value validity */
  if (m_timeout < 0) {
    syserr ("main(): config. error, invalid mtimeout value (must be >= 0)");
  }
  /* Log any altered value */
  if (m_timeout != M_TIMEOUT) {
    syslog ((FACILITY | LOG_INFO), "Changed default free modem wait timeout to %d secs",
           m_timeout);
  }
  
  /* Check for mbchkinterval value validity */
  if ((mbchkinterval < 0) || ((mbchkinterval > 0) && (mbchkinterval < 60))) {
    syslog ((FACILITY | LOG_ERR), "config. error: mbchkinterval should be 0 or >= 60 (now %d), exiting.",
           mbchkinterval);
    syserr ("main(): config. error, invalid mbchkinterval value");
  }
  /* Warn if mailbox check is disabled */
  if (mbchkinterval == 0) {
    syslog ((FACILITY | LOG_NOTICE), "incoming mailbox check function disabled.");
  }
  else {
    syslog ((FACILITY | LOG_INFO), "incoming mailbox check function enabled, first check in %d secs.",
           mbchkinterval);
  }
  
  /* Check for obchkinterval value validity */
  if ((obchkinterval < 0) || ((obchkinterval > 0) && (obchkinterval < 60))) {
    syslog ((FACILITY | LOG_ERR), "config. error: obchkinterval should be 0 or >= 60 (now %d), exiting.",
           obchkinterval);
    syserr ("main(): config. error, invalid obchkinterval value");
  }
  /* Warn if outgoing mailbox check is disabled */
  if (obchkinterval == 0) {
    syslog ((FACILITY | LOG_NOTICE), "outgoing mailbox check function disabled.");
    /* disable queueing as well, then (no point, if it won't be processed) */
    qmethod = SMS_Q_NONE;
  }
  else {
    syslog ((FACILITY | LOG_INFO), "outgoing mailbox check function enabled, first check in %d secs.",
           obchkinterval);
  }
  
  /* Check for obchklimit value validity */
  if (obchklimit < 0) {
    syslog ((FACILITY | LOG_ERR), "config. error: obchklimit should be >= 0 (now %d), exiting.",
           obchklimit);
    syserr ("main(): config. error, invalid obchklimit value");
  }
  /* Warn if outbox check limit is enabled */
  if ((obchkinterval > 0) && (obchklimit > 0)) {
    syslog ((FACILITY | LOG_WARNING), "max %d messages will be processed per queue run.",
           obchklimit);
  }

  /* Check for maxqruns value validity */
  if ((maxqruns < 0) || (maxqruns > 59)) {
    syslog ((FACILITY | LOG_ERR), "config. error: maxqruns should be >= 0 & <= 59 (now %d), exiting.",
           maxqruns);
    syserr ("main(): config. error, invalid maxqruns value");
  }
  /* Warn if maxqruns limit is disabled */
  if ((obchkinterval > 0) && (maxqruns == 0)) {
    syslog ((FACILITY | LOG_WARNING), "no limit on queue items life expectancy.",
           obchklimit);
  }

  /* Correct queueing method if required */
  if ((qmethod == SMS_Q_DB) && (! dbaccess_enabled)) {
    fprintf (stderr, "Warning: queueing method switched from db to file (dbaccess disabled)!\n");
    syslog ((FACILITY | LOG_WARNING), "queueing method switched from db to file (dbaccess disabled)!");
    qmethod = SMS_Q_FILE;
  }
  
  /* log selected queueing method */
  switch (qmethod) {
    case SMS_Q_NONE:
      syslog ((FACILITY | LOG_INFO), "queueing facility disabled.");
      break;
      
    case SMS_Q_FILE:
      syslog ((FACILITY | LOG_INFO), "queueing enabled, using files.");
      break;
      
    case SMS_Q_DB:
      syslog ((FACILITY | LOG_INFO), "queueing enabled, using DB.");
      break;
      
    default:
      syslog ((FACILITY | LOG_WARNING), "unknown queueing facility <%d>.", qmethod);
      break;
      
  }                                   /* switch (qmethod) */

  if (debug > 0) {
    syslog ((FACILITY | LOG_INFO), "debugging enabled, level is <%d>.",
           debug);
  }

  /*================================Real start of program */
  /* get the hostname I'm running on */
  localhost = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
  if (gethostname (localhost, MINIBUFF) == -1)
    syserr ("main(): can't get host name");
  /* make sure localhost is not qualified */
  dequalify (localhost);
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Local host name is: [%s]\n", localhost);
  }

  /* get the default domain (for bounced mails sending) */
  if (! defaultdomain) {
    /* not been provided through CLI */
    if (strcmp (DEFAULTDOMAIN, ".") == 0) {
      /* means we need to get it from the system */
      if ((ipnode = gethostbyname (localhost)) == NULL) {
	fprintf (stderr, "sms_serv: call to gethostbyname() failed:\n");
	switch (h_errno) {
	  case HOST_NOT_FOUND:
            fprintf (stderr, "            host not found\n");
	    break;

	  case NO_ADDRESS:
            fprintf (stderr, "            name valid but no answer returned\n");
	    break;

	  case NO_RECOVERY:
            fprintf (stderr, "            NS returned permanent failure\n");
	    break;

	  case TRY_AGAIN:
            fprintf (stderr, "            NS temporary failure - try again\n");
	    break;

	  default:
            fprintf (stderr, "            unknown error code <%d>\n", h_errno);
	    break;
	}                             /* switch (h_errno) */
	syserr ("main(): can't get domain name - aborting");
      }              /* if ((ipnode = gethostbyname (...) */
      if (debug & DEBUG_PRINTF) {
        fprintf (stderr, "Official host name: %s\n", ipnode->h_name);
      }
      /* Now extract the domain from the official host name */
      if ((p = strchr (ipnode->h_name, '.')) == NULL) {
	fprintf (stderr, "gethostbyname() reported %s\n", ipnode->h_name);
	syserr ("main(): not a fully qualified host name - aborting");
      }
      domain = p + 1;
      defaultdomain = (char *) malloc ((strlen (domain) + 1) * sizeof (char));
      if (! defaultdomain) {
	syserr ("main(): can't malloc() defaultdomain");
      }
      defaultdomain[0] = '\0';
      strcpy (defaultdomain, domain);
    }
    else {
      defaultdomain = (char *) malloc ((strlen (DEFAULTDOMAIN) + 1) * sizeof (char));
      if (! defaultdomain) {
	syserr ("main(): can't malloc() defaultdomain");
      }
      defaultdomain[0] = '\0';
      strcpy (defaultdomain, DEFAULTDOMAIN);
    }
  }                               /* if (! defaultdomain) */
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "Working for domain [%s]\n", defaultdomain);
  }
  syslog ((FACILITY | LOG_INFO), "working for domain [%s]", defaultdomain);
  
  /* get the port number we should listen to */
  if (smstcpport == 0) {
    /* not provided on command-line -- get it from system */
    if ((sent = getservbyname ("sms", "tcp")) == NULL) {
      syserr ("main(): can't get service port info");
    }
    if (debug & DEBUG_PRINTF) {
      printf ("found port <%d> for service sms\n", ntohs (sent->s_port));
    }
  }
  else {
    /* store provided value in struct servent after conversion */
    sent = &ssent;
    sent->s_port = htons (smstcpport);
    syslog ((FACILITY | LOG_INFO), "binding socket to non-standard TCP port <%d>.",
           smstcpport);
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "using provided sms tcp port <%d> to bind to.\n",
              smstcpport);
    }
  }                               /* if (smstcpport == 0) */
    
  /* now fill in the socket address structure */
  s_addr.sin_family = AF_INET;
  s_addr.sin_port = sent->s_port;
  s_addr.sin_addr.s_addr = htonl (INADDR_ANY);
  /* INADDR_ANY allows connection to any valid IP address for this
     host, including the loopback address. */
  
  /* let's create the socket */
  if ((ssfd = socket (AF_INET, SOCK_STREAM, IPPROTO_IP)) == -1)
    syserr ("main(): can't create socket");
  
  /* set socket options (INA)*/
  optval = 1;
  if (setsockopt (ssfd, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof (optval))) {
       syserr ("main(): can't set socket options to reuse");
  }

  /* load access control list */
  acl_list_init (&gsm_acl);
  if (! read_acl (&gsm_acl)) {
    syserr ("main(): can't read access control list");
  }
  /* end INA */

  /* now bind the socket to the server address */
  if (bind (ssfd, (struct sockaddr *) &s_addr, sizeof (s_addr)) == -1)
    syserr ("main(): can't bind socket");
    
  /* create the client queue on the socket */
  if (listen (ssfd, MAXCLIENTCONN) == -1)
    syserr ("main(): can't listen to that many connections");

  /* set the flags to make the socket "non-blocking" */
  if ((sockflags = fcntl (ssfd, F_GETFL, 0)) == -1)
    syserr ("main(): can't get socket descriptor attributes");
  if (fcntl (ssfd, F_SETFL, (O_NONBLOCK | sockflags)) == -1)
    syserr ("main(): can't set socket descriptor attributes");

  /* initialize "last checked" time for IN- and OUT-box */
  lastchked = time (NULL);
  ob_lastchked = time (NULL);
  starttime = lastchked;
  obc_list_init (&childprocs);
  
  syslog ((FACILITY | LOG_INFO), "server now ready to answer requests.");
  /*---------------------------Start real processing loop */
  while (TRUE) {
    /* wait for a client to connect */
    c_addr_len = sizeof (c_addr);
    if ((csfd = accept (ssfd, (struct sockaddr *) &c_addr, &c_addr_len)) == -1) {
      /* no connection - check the reason of the "error" */
      switch (errno) {
        case EWOULDBLOCK:
	  /* no error - time to check the mailbox ? */
	  if ((mbchkinterval > 0) &&
	     (difftime (time (NULL), lastchked) >= mbchkinterval)) {
	    syslog ((FACILITY | LOG_INFO), "initiating INBOX check.");

	    /* now fork a child to handle mailbox check */
	    pid = fork ();
	    switch (pid) {
	      case (-1):                         /* error */
		syserr ("main(): can't fork for MBCI");
	        break;

	      case (0):
	        /* ###########< I'm the child >########## */
	        /*----Initialize child-specific variables */
	        context = CONTEXT_MBCI;
                if (debug & DEBUG_PRINTF) {
                  printf ("PID <%d>, context set to <%d>\n", getpid (), context);
                }
		/*------------------------------Now do it */
		mbcheck_wrapper ();
		/*----------------------------Child exits */
        	if (debug & DEBUG_PRINTF) {
        	  printf ("PID <%d>, context <%d> -- child exits.\n", getpid (), context);
        	}
		exit (0);
	        break;           /* Just to be consistent */

	      default:
	        /* ##########< I'm the father >########## */
		/* re-initialize timer */
	        lastchked = time (NULL);
	        break;
 
 	    }                             /* switch (pid) */
 	  }                            /* if (difftime... */

 	  /* no error - time to check the outbox ? */
 	  if ((obchkinterval > 0) &&
 	     (difftime (time (NULL), ob_lastchked) >= obchkinterval)) {
	    if (childprocs.count < 1) {
 	      syslog ((FACILITY | LOG_INFO), "initiating OUTBOX check.");

 	      /* now fork a child to handle mailbox check */
 	      pid = fork ();
 	      switch (pid) {
 		case (-1):                       /* error */
 		  syserr ("main(): can't fork for MBCO");
 	          break;

 		case (0):
 	          /* ##########< I'm the child >######### */
 	          /*--Initialize child-specific variables */
	          context = CONTEXT_MBCO;
                  if (debug & DEBUG_PRINTF) {
                    printf ("PID <%d>, context set to <%d>\n", getpid (), context);
                  }
		  /* allocate space for the ob_symbols */
		  ob_symbols.smsc = (char *) malloc ((MAXPHNUMLEN + 1) * sizeof (char));
		  ob_symbols.destgsm = (char *) malloc ((MAXPHNUMLEN + 1) * sizeof (char));
		  ob_symbols.message = (char *) malloc ((MAXMSGLEN + 1) * sizeof (char));
		  ob_symbols.pdu = NULL; /* allocated only as needed */
		  ob_symbols.user = (char *) malloc ((MAXUIDLEN + 1) * sizeof (char));

		  /* set those ob_symbols to their default values */
		  strcpy (ob_symbols.smsc, DEFAULTSMSC);
		  ob_symbols.destgsm[0] = '\0';
		  ob_symbols.message[0] = '\0';
		  ob_symbols.user[0] = '\0';
		  ob_symbols.mode = SMS_MODE_DEFAULT;
		  ob_symbols.priority = 0;

 		  /*----------------------------Now do it */
 		  outboxcheck_wrapper ();

                  /*------------------------------Cleanup */
		  /* free the symbols struct members */
		  if (ob_symbols.smsc) {
		    free (ob_symbols.smsc);
		  }
		  if (ob_symbols.destgsm) {
		    free (ob_symbols.destgsm);
		  }
		  if (ob_symbols.message) {
		    free (ob_symbols.message);
		  }
		  if (ob_symbols.pdu) {
		    free (ob_symbols.pdu);
		  }
		  if (ob_symbols.user) {
		    free (ob_symbols.user);
		  }

 		  /*--------------------------Child exits */
        	  if (debug & DEBUG_PRINTF) {
        	    printf ("PID <%d>, context <%d> -- child exits.\n", getpid (), context);
        	  }
 		  exit (0);
 	          break;         /* Just to be consistent */

 		default:
 	          /* #########< I'm the father >######### */
		  /* store pid of child in obc_list */
		  obc_list_insert (&childprocs, pid);
 		  /* re-initialize timer */
 	          ob_lastchked = time (NULL);
	          break;

	      }                           /* switch (pid) */
	    }
	    else {
	      /* previous child still running, let's skip this one */
              syslog ((FACILITY | LOG_WARNING), "OUTBOX check cancelled, <%d> child procs already.",
	             childprocs.count);
              if (debug & DEBUG_PRINTF) {
		scratch = (char *) malloc (((20 * childprocs.count) + 1) * sizeof (char));
		if (! scratch) {
		  syserr ("main(): can't malloc() scratch");
		}
		scratch[0] = '\0';
		dump_obc_list (childprocs, scratch);
		syslog ((FACILITY | LOG_WARNING), "OBC list: %s", scratch);
		free (scratch);
              }
 	      /* re-initialize timer */
 	      ob_lastchked = time (NULL);
	    }                     /* if (childprocs < ... */
	  }                            /* if (difftime... */
	  break;

	default:
	  /* there was a "real" error */
          syserr ("main(): can't accept incoming connection");
	  break;
      }                                 /* switch (errno) */
    }
    else { /* we have a connection - handle it */
      /* first check it against the acl (INA) */
      if (! check_acl (&c_addr.sin_addr, gsm_acl)) {
        syslog ((FACILITY | LOG_WARNING), "illegal connect from [%s]", inet_ntoa (c_addr.sin_addr));
        /* close the socket */
        if (close (csfd) == -1)
          syserr ("main(): can't close client socket");
        continue;
      }

      /* for logging purposes */
      syslog ((FACILITY | LOG_NOTICE), "connect from [%s]", inet_ntoa (c_addr.sin_addr));
    
      /* Now fork a child to handle each individ. request & go on listening */
      pid = fork ();
      switch (pid) {
        case (-1):                               /* error */
          syserr ("main(): can't fork for user conn.");
	  break;
	
        case (0):
	  /* ###############< I'm the child >############ */
	  /*----------Initialize child-specific variables */
	  context = CONTEXT_SEND;
          if (debug & DEBUG_PRINTF) {
            printf ("PID <%d>, context set to <%d>\n", getpid (), context);
          }
	  /* allocate space for the symbols */
	  symbols.smsc = (char *) malloc ((MAXPHNUMLEN + 1) * sizeof (char));
	  symbols.destgsm = (char *) malloc ((MAXPHNUMLEN + 1) * sizeof (char));
	  symbols.message = (char *) malloc ((MAXMSGLEN + 1) * sizeof (char));
	  symbols.pdu = NULL; /* allocated only as needed */
	  symbols.user = (char *) malloc ((MAXUIDLEN + 1) * sizeof (char));

	  /* set those symbols to their default values */
	  strcpy (symbols.smsc, DEFAULTSMSC);
	  symbols.destgsm[0] = '\0';
	  symbols.message[0] = '\0';
	  symbols.user[0] = '\0';
	  symbols.mode = SMS_MODE_DEFAULT;
	  symbols.priority = SMS_DEF_Q_PRI;

	  /* scratch space for communication with the client */
	  buffer = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
	  buffer[0] = '\0';  

	  /*------------------------------Server announce */
	  sprintf (buffer, "SMS-Link for Linux, ver %s (%s) on %s.%s\r\n", SMS_SERV_VERSION,
          	 SMS_SERV_DATE, localhost, defaultdomain);
	  tellsock (csfd, buffer);
	  sprintf (buffer, "(c) Les Ateliers du Heron, 1998-2006 for Scitex Europe, S.A.\r\n"
	          "(c) Internet Access AG, 1999\r\n\r\n");
	  tellsock (csfd, buffer);

#ifdef FREEBSD
          if (fcntl (csfd, F_SETFL, (~O_NONBLOCK & sockflags)) == -1)
            syserr ("main(): can't set socket descriptor attributes");
#endif                                   /* ifdef FREEBSD */

	  /* enter the parser module - handle dialogue with client */
	  while (TRUE) {
            /* display prompt */
            sprintf (buffer, PROMPT);
            tellsock (csfd, buffer);
	    /* get client data */
	    buffer[0] = '\0';
	    nread = read (csfd, buffer, BUFFSIZE);
	    if (nread == -1) {
	      syserr ("main(): error while reading from socket");
	    }
	    buffer[nread] = '\0';     /* may not be needed, but can't harm */
            /* prepare variables to be used by flex in the new YYINPUT */
	    myinputptr = buffer;
	    myinputlim = buffer + strlen (buffer);
	    /* parse client data */
	    parser ();
	  }                               /* while (TRUE) */

	  /*--------------------------Close client socket */
	  if (close (csfd) == -1) {
	    syserr ("main(): can't close client socket");
	  }

	  /* free the symbols struct members */
	  if (symbols.smsc) {
	    free (symbols.smsc);
	  }
	  if (symbols.destgsm) {
	    free (symbols.destgsm);
	  }
	  if (symbols.message) {
	    free (symbols.message);
	  }
	  if (symbols.pdu) {
	    free (symbols.pdu);
	  }
	  if (symbols.user) {
	    free (symbols.user);
	  }
	  
	  /* the rest */
	  if (buffer) {
	    free (buffer);
	  }

	  /*---------------------------------Child exists */
          if (debug & DEBUG_PRINTF) {
            printf ("PID <%d>, context <%d> -- child exits.\n", getpid (), context);
          }
	  exit (0);
	  break; /* just to be consistent */
      
        default:
  	  /* #############< I'm the father >############# */
	  /*--------------------------Close client socket */
	  if (close (csfd) == -1) {
	    syserr ("main(): can't close client socket");
	  }
	  break;
	
      }                                   /* switch (pid) */
    }                           /* if accept() .. else .. */
    /* give the CPU some rest */
    sleep (1);
  }                                       /* while (TRUE) */
  /*------------------------------------------Conclusions */
  /* close server socket */
  if (close (ssfd) == -1) {
    syserr ("main(): can't close server socket");
  }
    
  /* let's clean what's allocated on the heap */
  free (buffer);
  free (destgsm);
  free (message);
  free (localhost);
  free (mailhost);
  free (defaultdomain);
  free (bounce_addr);
  /*------------------------------------------End program */
  exit (0);
}                                              /* main () */

/*==========================================================
 * EOF : sms_serv.c
 *===================*/
