/*==========================================================
 * Program : smtp.c                        Project : smslink
 * Authors : Philippe Andersson.
 * Date    : 09/11/05
 * Version : 0.07b
 * Notice  : (c) Les Ateliers du Heron, 2003
 * Comment : Library of SMTP-related functions for the smslink servers.
 *
 * Modification History :
 * - 0.01a (26/06/03) : Initial release.
 * - 0.02a (03/10/03) : Fix broken support for FreeBSD 4.8. Thanks
 *   to Nicki de Wet (<klitsgras@ananzi.co.za>) for the patch.
 * - 0.03a (23/02/04) : Corrected bug in dequalify().
 * ++++ Switch to Beta ++++
 * - 0.04b (27/10/04) : Fixed a bug preventing compilation on FreeBSD 5
 *   (thanks to Piet Ruyssinck <piet.ruyssinck@ugent.be> for the fix).
 * - 0.05b (25/10/05) : Improved error checking in resolve() (+ some
 *   readability improvements).
 * - 0.06b (08/11/05) : Fixed a bug introduced with the previous mod.
 *   Fixed a major bug in resolve() (missing {}'s).
 * - 0.07b (09/11/05) : Fixed the bug that caused a segfault in resolve()
 *   on x86_64 platform when called with a hostname in debug mode.
 *========================================================*/

#include <unistd.h>
#include <stdio.h>                         /* for fprintf */
#include <stdlib.h>                  /* for errno & stuff */
#include <string.h>
#include <netinet/in.h>                /* for inet_ntoa() */
#include <arpa/inet.h>                 /* for inet_ntoa() */
#include <netdb.h>                 /* for gethostbyname() */
#if (defined (FREEBSD) || defined (FREEBSD5))
#  include <sys/socket.h>
#  include <sys/time.h>
#endif
#include <sys/ioctl.h>            /* for the ioctl() call */

#include "sms_serv.h"
#include "smtp.h"

/*========================================================*/
/* For debugging purposes only - comment out for normal compile */
/* #define INCL_DEBUG_CODE */

/*========================================================*/
/**********           GLOBAL VARIABLES             ********/
/*========================================================*/
extern int debug;                    /* debug level/flags */

/*========================================================*/
/**********               FUNCTIONS                ********/
/*========================================================*/
int dequalify (char *hostname)
/* alters the input string to terminate it at the first dot */
{
  char *p;
  
  if ((p = strchr (hostname, '.')) != NULL) {
    p[0] = '\0';
  }
  
  return (strlen (hostname));
}                                         /* dequalify () */
/*========================================================*/
int is_dotted_quad (char *string)
{
  int is_dq;
  char *s;
  int ntok = 0;
  char *ptok = NULL;
  long int ip_byte;
  char **endptr;
  char *dummyp;
  
  /*--------------------------------------Initializations */
  s = (char *) malloc ((strlen (string) + 1) * sizeof (char));
  if (! s) {
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("is_dotted_quad(): can't malloc() s");
  }
  strcpy (s, string);
  /*--------------------------------------------Main Loop */
  /* no point going any further if the 1st char. is not a digit */
  is_dq = isdigit (string[0]);
  if (is_dq) {
    ptok = strtok (s, ".");
    while (ptok != NULL) {
      ntok++;
      if (debug & DEBUG_PRINTF) {
        fprintf (stderr, "got token #%d : <%s>\n", ntok, ptok);
      }
      dummyp = ptok + strlen (ptok);
      endptr = &dummyp;
      ip_byte = strtol (ptok, endptr, 10);
      if (strlen (*endptr) != 0) {
        if (debug & DEBUG_PRINTF) {
	  fprintf (stderr, "error on token #%d : can't convert <%s>\n", ntok, *endptr);
        }
	is_dq = FALSE;
      }
      ptok = strtok (NULL, ".");
    }
    if (ntok != 4) {
      is_dq = FALSE;
    }
  }                                         /* if (is_dq) */
  /*------------------------------------------Conclusions */
  free (s);
  /*-------------------------------------------------Exit */
  return (is_dq);
}                                    /* is_dotted_quad () */
/*========================================================*/
unsigned long int resolve (char *host)
/* return the host ip address in network format */
{
  struct hostent *h_ent;
  struct in_addr tiploc;
  struct in_addr *target_ip;
  char **addrs;
  char *scratch;
  
  if (is_dotted_quad (host)) {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "I think [%s] is an IP address\n", host);
    }
    target_ip = &tiploc;
    if (inet_aton (host, target_ip) == 0) {
      syslog ((FACILITY | LOG_ERR), "invalid server IP address (%s).", host);
      scratch = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
      sprintf (scratch, "sms2mailgw: invalid server IP address (%s)", host);
      herror (scratch);
      free (scratch);
      unlink (MBOX_LOCKF);
      exit (1);
    }
  }
  else {
    if (debug & DEBUG_PRINTF) {
      fprintf (stderr, "I think [%s] is a host name\n", host);
    }
    if ((h_ent = gethostbyname (host)) == NULL) {
      syslog ((FACILITY | LOG_ERR), "can't resolve hostname (%s).", host);
      scratch = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
      sprintf (scratch, "sms2mailgw: can't resolve hostname (%s)", host);
      herror (scratch);
      free (scratch);
      unlink (MBOX_LOCKF);
      exit (1);
    }
    /* lines below cf. "Beginning Linux Progr.", pp. 468-473 */
    addrs = h_ent->h_addr_list;
    if (*addrs) {
      target_ip = (struct in_addr *)*addrs;
      if (debug & DEBUG_PRINTF) {
	fprintf (stderr, "server IP address is: [%s]\n", inet_ntoa (*target_ip));
      }
    }
    else {
      syslog ((FACILITY | LOG_ERR), "empty address list for hostname (%s).", host);
      scratch = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
      sprintf (scratch, "sms2mailgw: empty address list for hostname (%s)", host);
      herror (scratch);
      free (scratch);
      unlink (MBOX_LOCKF);
      exit (1);
    }
  }                         /* if (is_dotted_quad (host)) */
  return (target_ip->s_addr);
}                                           /* resolve () */
/*========================================================*/
int slurp_n_catch (int fd, int resptime, char *catch)
/* slurps n_lines lines from the input stream, and return a bool
value telling whether catch was found in any of those lines. */
{
  int nread, retval, previous, found;
  fd_set inputs;
  struct timeval timeout;
  char *buffer;
  
  /*--------------------------------------Initializations */
  nread = previous = 0;
  found = FALSE;
  
  FD_ZERO (&inputs);
  FD_SET (fd, &inputs);
  
  timeout.tv_sec = 10;
  timeout.tv_usec = 0;
  
  /* wait for data to arrive on the line */
  retval = select (FD_SETSIZE, &inputs, NULL, NULL, &timeout);
  switch (retval) {
    case 0:
      syslog ((FACILITY | LOG_WARNING), "timeout while waiting for server reply.");
      break;
    
    case -1:
      syslog ((FACILITY | LOG_ERR), "call to select() failed.");
      unlink (MBOX_LOCKF);
      syserr ("slurp_n_catch(): call to select() failed");
      break;
      
    default:
      if (FD_ISSET (fd, &inputs)) {
        /* first wait for all data to be ready */
	ioctl (fd, FIONREAD, &nread);
	while (nread != previous) {
	  sleep (resptime);
	  previous = nread;
	  ioctl (fd, FIONREAD, &nread);
	}                    /* while (nread != previous) */
	/* we know what's the data size - alloc space for it */
	buffer = (char *) malloc ((nread + 1) * sizeof (char));
	if (!buffer) {
          syslog ((FACILITY | LOG_ERR), "can't allocate buffer space.");
          unlink (MBOX_LOCKF);
	  syserr ("slurp_n_catch(): can't allocate buffer space");
	}
	/* now we can finally read this data */
	nread = read (fd, buffer, nread);
	switch (nread) {
	  case 0:
	    /* EOF */
	    buffer[nread] = '\0';
	    syslog ((FACILITY | LOG_WARNING), "no data from server waiting for [%s].",
	            catch);
	    break;
	    
	  case -1:
            syslog ((FACILITY | LOG_ERR), "error while reading answer from server.");
            unlink (MBOX_LOCKF);
	    syserr ("slurp_n_catch(): error while reading answer from server");
	    break;
	    
	  default:
	    buffer[nread] = '\0';
            if (debug & DEBUG_PRINTF) {
	      fprintf (stderr, "pid<%d> Got : [%s] (%d char)\n", getpid (), buffer, nread);
            }
	    /* here we could pre-process it (remove Ctrl-Z) */
	    /* look for catch */
            found = (strstr (buffer, catch) != NULL);
	    break;
	}                               /* switch (nread) */
      }                                /* if (FD_ISSET... */
      free (buffer);
      break;
  }                                    /* switch (retval) */
  /*------------------------------------------------------*/
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "catch found: [%s]\n", found ? "yes" : "no");
  }
  /*----------------------------------Conclusion and exit */
  return (found);
}                                     /* slurp_n_catch () */
/*========================================================*/

/*========================================================*/
/*############## Struct email_msg Handling ###############*/
/*========================================================*/
void rcpt_list_init (rcpt_list *list)
{
  list->head = NULL;
  list->tail = NULL;
}                                    /* rcpt_list_init () */
/*========================================================*/
int empty_rcpt_list (rcpt_list list)
{
  return (list.head == NULL);
}                                   /* empty_rcpt_list () */
/*========================================================*/
void body_list_init (body_list *list)
{
  list->head = NULL;
  list->tail = NULL;
}                                    /* body_list_init () */
/*========================================================*/
int empty_body_list (body_list list)
{
  return (list.head == NULL);
}                                   /* empty_body_list () */
/*========================================================*/
void rcpt_list_insert (rcpt_list *list, char *recepient)
{
  /* WARNING : the order of the elements might be relevent - let's
   * store them as they were in the header => avoid inserting at
   * list head. Do it at the tail. */
  
  int l;
  struct rcpt_item *element;

  /* alloc memory for new element */
  element = (struct rcpt_item *) malloc (sizeof (struct rcpt_item));
  if (!element) {
    syslog ((FACILITY | LOG_ERR), "can't malloc() for new RCPT entry.");
    unlink (MBOX_LOCKF);
    syserr ("rcpt_list_insert(): can't malloc() for new RCPT entry");
  }
  
  /* initialize fields for new element */
  l = strlen (recepient);
  element->rcpt = (char *) malloc ((l + 1) * sizeof (char));
  if (!element->rcpt) {
    syslog ((FACILITY | LOG_ERR), "can't malloc() for new recepient string.");
    unlink (MBOX_LOCKF);
    syserr ("rcpt_list_insert(): can't malloc() for new recepient string");
  }
  strcpy (element->rcpt, recepient);
  
  /* chain it in the list */
  if (empty_rcpt_list (*list)) {
    list->head = element;
    list->tail = element;
    element->next = NULL;
    element->prev = NULL;
  }
  else {
    element->next = NULL;
    element->prev = list->tail;
    list->tail->next = element;
    list->tail = element;
  }
}                                  /* rcpt_list_insert () */
/*========================================================*/
void body_list_insert (body_list *list, char *line)
{
  /* WARNING : the order of the elements might be relevent - let's
   * store them as they were in the header => avoid inserting at
   * list head. Do it at the tail. */
  
  int l;
  struct body_line *element;

  /* alloc memory for new element */
  element = (struct body_line *) malloc (sizeof (struct body_line));
  if (!element) {
    syslog ((FACILITY | LOG_ERR), "can't malloc() for new body_line entry.");
    /* FIXME: test on context -- only when sms2mailgw */
    unlink (MBOX_LOCKF);
    syserr ("body_list_insert(): can't malloc() for new body_line entry");
  }
  
  /* initialize fields for new element */
  l = strlen (line);
  element->line = (char *) malloc ((l + 1) * sizeof (char));
  if (!element->line) {
    syslog ((FACILITY | LOG_ERR), "can't malloc() for new body line string.");
    /* FIXME: test on context -- only when sms2mailgw */
    unlink (MBOX_LOCKF);
    syserr ("body_list_insert(): can't malloc() for new body line string");
  }
  strcpy (element->line, line);
  
  /* chain it in the list */
  if (empty_body_list (*list)) {
    list->head = element;
    list->tail = element;
    element->next = NULL;
    element->prev = NULL;
  }
  else {
    element->next = NULL;
    element->prev = list->tail;
    list->tail->next = element;
    list->tail = element;
  }
}                                  /* body_list_insert () */
/*========================================================*/
void free_rcpt_list (rcpt_list *list)
{
  struct rcpt_item *cursor;

  if (!empty_rcpt_list (*list)) {
    /* hop to element before last */
    cursor = list->tail->prev;
    /* now go back and clean behind */
    while (cursor != NULL) {
      free (cursor->next->rcpt);
      free (cursor->next);
      cursor->next = NULL;
      list->tail = cursor;
      cursor = cursor->prev;
    }                           /* while (cursor != NULL) */
  }                          /* if (!empty_rcpt_list (... */
  /* now clean last element and reset header */
  free (list->head->rcpt);
  free (list->head);
  list->head = NULL;
  list->tail = NULL;
}                                    /* free_rcpt_list () */
/*========================================================*/
void free_body_list (body_list *list)
{
  struct body_line *cursor;

  if (!empty_body_list (*list)) {
    /* hop to element before last */
    cursor = list->tail->prev;
    /* now go back and clean behind */
    while (cursor != NULL) {
      free (cursor->next->line);
      free (cursor->next);
      cursor->next = NULL;
      list->tail = cursor;
      cursor = cursor->prev;
    }                           /* while (cursor != NULL) */
  }                          /* if (!empty_rcpt_list (... */
  /* now clean last element and reset header */
  free (list->head->line);
  free (list->head);
  list->head = NULL;
  list->tail = NULL;
}                                    /* free_rcpt_list () */
/*========================================================*/
void print_rcpt_list (rcpt_list list)
{
  struct rcpt_item *cursor;

  if (!empty_rcpt_list (list)) {
    cursor = list.head;
    fprintf (stderr, "<%s>\n", cursor->rcpt);
    while (cursor->next != NULL) {
      cursor = cursor->next;
      fprintf (stderr, "<%s>\n", cursor->rcpt);
    }
  }
  else {
    fprintf (stderr, "sms2mailgw: empty recepient list.\n");
  }
}                                   /* print_rcpt_list () */
/*========================================================*/
void print_body_list (body_list list)
{
  struct body_line *cursor;

  if (!empty_body_list (list)) {
    cursor = list.head;
    fprintf (stderr, "<%s>\n", cursor->line);
    while (cursor->next != NULL) {
      cursor = cursor->next;
      fprintf (stderr, "<%s>\n", cursor->line);
    }
  }
  else {
    fprintf (stderr, "sms2mailgw: empty body list.\n");
  }
}                                   /* print_body_list () */
/*========================================================*/
void reset_mail_struct (struct email_msg *m)
{
  if (m->from) {
    free (m->from);
    m->from = NULL;
  }
  
  if (!empty_rcpt_list (m->to))
    free_rcpt_list (&(m->to));
  if (!empty_rcpt_list (m->cc))
    free_rcpt_list (&(m->cc));
  if (!empty_rcpt_list (m->bcc))
    free_rcpt_list (&(m->bcc));
  
  if (m->reply_to) {
    free (m->reply_to);
    m->reply_to = NULL;
  }
  if (m->subject) {
    free (m->subject);
    m->subject = NULL;
  }
  if (!empty_body_list (m->body))
    free_body_list (&(m->body));
}                                 /* reset_mail_struct () */
/*========================================================*/
void mail_struct_init (struct email_msg *m)
{

  m->from = NULL;

  rcpt_list_init (&(m->to));
  rcpt_list_init (&(m->cc));
  rcpt_list_init (&(m->bcc));
  
  m->reply_to = NULL;
  m->subject = NULL;

  body_list_init (&(m->body));

}                                  /* mail_struct_init () */
/*========================================================*/

/*========================================================*/
/*################ SMTP Protocol Support #################*/
/*========================================================*/
int send_mail (struct email_msg *m, char *mailhost, char *localhost, 
               char *defdom)
{
  struct servent *sent;
  struct in_addr server_ip;
  struct sockaddr_in sockaddr;
  struct rcpt_item *r_cursor;
  struct body_line *b_cursor;
  int sockfd;
  int addrlen;
  char *cmdline;

  /*--------------------------------------Initializations */
  cmdline = (char *) malloc ((BUFFSIZE + 1) * sizeof (char));
  if (!cmdline) {    
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    unlink (MBOX_LOCKF);
    syserr ("send_mail(): can't malloc()");
  }
  cmdline[0] = '\0';

  /* first resolve server name */
  server_ip.s_addr = resolve (mailhost);
  
  /* get the port number we should connect to */
  if ((sent = getservbyname ("smtp", "tcp")) == NULL) {
    syslog ((FACILITY | LOG_ERR), "can't get service port info.");
    unlink (MBOX_LOCKF);
    syserr ("send_mail(): can't get service port info");
  }
  if (debug & DEBUG_PRINTF) {
    fprintf (stderr, "found port <%d> for service smtp\n", ntohs (sent->s_port));
  }
    
  /* create the socket */
  if ((sockfd = socket (AF_INET, SOCK_STREAM, 0)) == -1) {
    syslog ((FACILITY | LOG_ERR), "can't create socket.");
    unlink (MBOX_LOCKF);
    syserr ("send_mail(): can't create socket");
  }
  
  /* build the server socket address parameters */
  sockaddr.sin_family = AF_INET;
  sockaddr.sin_port = sent->s_port;
  sockaddr.sin_addr.s_addr = server_ip.s_addr;
  addrlen = sizeof (sockaddr);
  
  /* now connect to the server */
  if (connect (sockfd, (struct sockaddr *)&sockaddr, addrlen) == -1) {
    syslog ((FACILITY | LOG_ERR), "can't connect to SMTP server - may be down.");
    close (sockfd);
    free (cmdline);
    return (-1);
  }
    
  /*......................................Start of dialog */
  /* slurp server announce and catch prompt */
  if (!slurp_n_catch (sockfd, 1, "220")) {
    syslog ((FACILITY | LOG_ERR), "can't get server announce.");
    close (sockfd);
    free (cmdline);
    return (-1);
  }
    
  /* set sender ID - get ok */
  sprintf (cmdline, "HELO %s.%s\r\n", localhost, defdom);
  tellsock (sockfd, cmdline);
  if (debug & DEBUG_SMTP) {
    fprintf (stderr, "[%s]\n", cmdline);
  }

  if (!slurp_n_catch (sockfd, 1, "250")) {
    syslog ((FACILITY | LOG_ERR), "failed on HELO command (%s.%s).",
           localhost, defdom);
    close (sockfd);
    free (cmdline);
    return (-1);
  }
    
  /* send from: field */
  sprintf (cmdline, "MAIL FROM: <%s>\r\n", m->from);
  tellsock (sockfd, cmdline);
  if (debug & DEBUG_SMTP) {
    fprintf (stderr, "[%s]\n", cmdline);
  }

  if (!slurp_n_catch (sockfd, 1, "250")) {
    syslog ((FACILITY | LOG_ERR), "failed on MAIL command (%s).", m->from);
    close (sockfd);
    free (cmdline);
    return (-1);
  }
  
  /*. . . . . . . . . . . . . Loop through all recepients */
  /* To: field */
  if (!empty_rcpt_list (m->to)) {
    r_cursor = m->to.head;
    sprintf (cmdline, "RCPT TO: <%s>\r\n", r_cursor->rcpt);
    tellsock (sockfd, cmdline);
    if (debug & DEBUG_SMTP) {
      fprintf (stderr, "[%s]\n", cmdline);
    }

    if (!slurp_n_catch (sockfd, 1, "250")) {
      syslog ((FACILITY | LOG_ERR), "failed on (to) RCPT command (%s).", r_cursor->rcpt);
      close (sockfd);
      free (cmdline);
      return (-1);
    }
    while ((r_cursor = r_cursor->next) != NULL) {
      sprintf (cmdline, "RCPT TO: <%s>\r\n", r_cursor->rcpt);
      tellsock (sockfd, cmdline);
      if (debug & DEBUG_SMTP) {
	fprintf (stderr, "[%s]\n", cmdline);
      }

      if (!slurp_n_catch (sockfd, 1, "250")) {
	syslog ((FACILITY | LOG_ERR), "failed on (to) RCPT command (%s).", r_cursor->rcpt);
        close (sockfd);
        free (cmdline);
        return (-1);
      }
    }                                      /* while (...) */
  }                             /* if (not empty to-list) */
  
  /* CC: field */
  if (!empty_rcpt_list (m->cc)) {
    r_cursor = m->cc.head;
    sprintf (cmdline, "RCPT TO: <%s>\r\n", r_cursor->rcpt);
    tellsock (sockfd, cmdline);
    if (debug & DEBUG_SMTP) {
      fprintf (stderr, "[%s]\n", cmdline);
    }

    if (!slurp_n_catch (sockfd, 1, "250")) {
      syslog ((FACILITY | LOG_ERR), "failed on (cc) RCPT command (%s).", r_cursor->rcpt);
      close (sockfd);
      free (cmdline);
      return (-1);
    }
    while ((r_cursor = r_cursor->next) != NULL) {
      sprintf (cmdline, "RCPT TO: <%s>\r\n", r_cursor->rcpt);
      tellsock (sockfd, cmdline);
      if (debug & DEBUG_SMTP) {
	fprintf (stderr, "[%s]\n", cmdline);
      }

      if (!slurp_n_catch (sockfd, 1, "250")) {
	syslog ((FACILITY | LOG_ERR), "failed on (cc) RCPT command (%s).", r_cursor->rcpt);
        close (sockfd);
        free (cmdline);
        return (-1);
      }
    }                                      /* while (...) */
  }                             /* if (not empty cc-list) */

  /* BCC: field */
  if (!empty_rcpt_list (m->bcc)) {
    r_cursor = m->bcc.head;
    sprintf (cmdline, "RCPT TO: <%s>\r\n", r_cursor->rcpt);
    tellsock (sockfd, cmdline);
    if (debug & DEBUG_SMTP) {
      fprintf (stderr, "[%s]\n", cmdline);
    }

    if (!slurp_n_catch (sockfd, 1, "250")) {
      syslog ((FACILITY | LOG_ERR), "failed on (bcc) RCPT command (%s).", r_cursor->rcpt);
      close (sockfd);
      free (cmdline);
      return (-1);
    }
    while ((r_cursor = r_cursor->next) != NULL) {
      sprintf (cmdline, "RCPT TO: <%s>\r\n", r_cursor->rcpt);
      tellsock (sockfd, cmdline);
      if (debug & DEBUG_SMTP) {
	fprintf (stderr, "[%s]\n", cmdline);
      }

      if (!slurp_n_catch (sockfd, 1, "250")) {
	syslog ((FACILITY | LOG_ERR), "failed on (bcc) RCPT command (%s).", r_cursor->rcpt);
        close (sockfd);
        free (cmdline);
        return (-1);
      }
    }                                      /* while (...) */
  }                            /* if (not empty bcc-list) */

  /*. . . . . . . . . . . . . . . . . . . . . . . . . . . */
  /* now send mail data */
  sprintf (cmdline, "DATA\r\n");
  tellsock (sockfd, cmdline);
  if (debug & DEBUG_SMTP) {
    fprintf (stderr, "[%s]\n", cmdline);
  }

  if (!slurp_n_catch (sockfd, 1, "354")) {
    syslog ((FACILITY | LOG_ERR), "failed on DATA command.");
    close (sockfd);
    free (cmdline);
    return (-1);
  }
  
  /* From: field as "header" */
  if (m->from) {
    sprintf (cmdline, "From: %s\r\n", m->from);
    tellsock (sockfd, cmdline);
    if (debug & DEBUG_SMTP) {
      fprintf (stderr, "[%s]\n", cmdline);
    }
  }
  /* To: field as "header" */
  if (!empty_rcpt_list (m->to)) {
    r_cursor = m->to.head;
    sprintf (cmdline, "To: %s", r_cursor->rcpt);

    while ((r_cursor = r_cursor->next) != NULL) {
      strcat (cmdline, ", ");
      strcat (cmdline, r_cursor->rcpt);
    }                                      /* while (...) */

    strcat (cmdline, "\r\n");
    tellsock (sockfd, cmdline);
    if (debug & DEBUG_SMTP) {
      fprintf (stderr, "[%s]\n", cmdline);
    }
  }                             /* if (not empty to-list) */
  /* CC: field as "header" */
  if (!empty_rcpt_list (m->cc)) {
    r_cursor = m->cc.head;
    sprintf (cmdline, "Cc: %s", r_cursor->rcpt);

    while ((r_cursor = r_cursor->next) != NULL) {
      strcat (cmdline, ", ");
      strcat (cmdline, r_cursor->rcpt);
    }                                      /* while (...) */

    strcat (cmdline, "\r\n");
    tellsock (sockfd, cmdline);
    if (debug & DEBUG_SMTP) {
      fprintf (stderr, "[%s]\n", cmdline);
    }
  }                             /* if (not empty to-list) */
  /* BCC: field as "header" */
  if (!empty_rcpt_list (m->bcc)) {
    r_cursor = m->bcc.head;
    sprintf (cmdline, "Bcc: %s", r_cursor->rcpt);

    while ((r_cursor = r_cursor->next) != NULL) {
      strcat (cmdline, ", ");
      strcat (cmdline, r_cursor->rcpt);
    }                                      /* while (...) */

    strcat (cmdline, "\r\n");
    tellsock (sockfd, cmdline);
    if (debug & DEBUG_SMTP) {
      fprintf (stderr, "[%s]\n", cmdline);
    }
  }                             /* if (not empty to-list) */
  /* Reply-to: field */
  if (m->reply_to) {
    sprintf (cmdline, "Reply-to: %s\r\n", m->reply_to);
    tellsock (sockfd, cmdline);
    if (debug & DEBUG_SMTP) {
      fprintf (stderr, "[%s]\n", cmdline);
    }
  }
  /* Subject: field */
  if (m->subject) {
    sprintf (cmdline, "Subject: %s\r\n", m->subject);
    tellsock (sockfd, cmdline);
    if (debug & DEBUG_SMTP) {
      fprintf (stderr, "[%s]\n", cmdline);
    }
  }
  /* Message Body */
  if (!empty_body_list (m->body)) {
    b_cursor = m->body.head;
    /* skip 1 line before last header and body */
    sprintf (cmdline, "\r\n%s\r\n", b_cursor->line);
    tellsock (sockfd, cmdline);
    if (debug & DEBUG_SMTP) {
      fprintf (stderr, "[%s]\n", cmdline);
    }

    while ((b_cursor = b_cursor->next) != NULL) {
      sprintf (cmdline, "%s\r\n", b_cursor->line);
      tellsock (sockfd, cmdline);
      if (debug & DEBUG_SMTP) {
	fprintf (stderr, "[%s]\n", cmdline);
      }
    }                                      /* while (...) */
  }                           /* if (not empty body-list) */
  
  /* now close the DATA section: one dot only */
  sprintf (cmdline, "\r\n.\r\n");
  tellsock (sockfd, cmdline);
  if (debug & DEBUG_SMTP) {
    fprintf (stderr, "[%s]\n", cmdline);
  }

  if (!slurp_n_catch (sockfd, 1, "250")) {
    syslog ((FACILITY | LOG_ERR), "failed on DATA transfer.");
    close (sockfd);
    free (cmdline);
    return (-1);
  }
  
  /* send the QUIT command */
  sprintf (cmdline, "QUIT\r\n");
  tellsock (sockfd, cmdline);
  if (debug & DEBUG_SMTP) {
    fprintf (stderr, "[%s]\n", cmdline);
  }

  if (!slurp_n_catch (sockfd, 1, "221")) {
    syslog ((FACILITY | LOG_ERR), "failed on QUIT command.");
    close (sockfd);
    free (cmdline);
    return (-1);
  }
  
  /*------------------------------------------Conclusions */
  /* close socket */
  if (close (sockfd) == -1) {
    syslog ((FACILITY | LOG_ERR), "can't close socket.");
    unlink (MBOX_LOCKF);
    syserr ("send_mail(): can't close socket");
  }
  
  /* free what's need to be */
  free (cmdline);
  
  /* leave - success */
  return (0);
}                                         /* send_mail () */
/*========================================================*/

/*==========================================================
 * EOF : smtp.c
 *===================*/
