/*==========================================================
 * Program : stuff.c                       Project : smslink
 * Author  : Philippe Andersson.
 * Date    : 22/03/06
 * Version : 0.30b
 * Notice  : (c) Les Ateliers du Heron, 1998 for Scitex Europe, S.A.
 * Comment : Library of functions for the smslink servers.
 *
 * Modification History :
 * - 0.01a (25/08/98) : Initial release.
 * - 0.02a (27/08/98) : Added tellsock().
 * - 0.03a (01/09/98) : Added semaphore deletion to daemons_death().
 * - 0.04a (03/09/98) : Added shared memory release to daemons_death().
 * - 0.05a (06/09/98) : Inserted #ifdef's to have it compile under
 *   RedHat (GNU Libc 6).
 * - 0.06a (19/10/98) : Added gsmdevcpy ().
 * - 0.07a (20/10/98) : Included GSM communication functions.
 * - 0.08b (13/12/98) : Adapted gsmdevcpy() for struct gsms_def
 *   new member "owner".
 * - 0.09b (14/02/99) : Included code in daemon's death to
 *   clean ACL linked list.
 * - 0.10b (29/06/99) : Moved 3 functions from server.y to here
 *   (misc. string handling functions).
 * - 0.11b (03/07/99) : Added function trim(). Cosmetics.
 * - 0.12b (17/08/99) : Included code in deamons_death() to
 *   remove the checkpoint file before quitting.
 * - 0.13b (18/08/99) : Added an exit function specific for
 *   sms2mailgw.
 * - 0.14b (19/08/99) : Moved the server-specific functions to
 *   serv_stuff.c and the gateway-specific ones to gw-stuff.c.
 * - 0.15b (21/10/99) : Moved tellsock() over from serv_stuff.c.
 * - 0.16b (09/11/99) : Added nicedate() function.
 * - 0.17b (10/04/00) : Added str_insert() function.
 * - 0.18b (11/09/00) : Added unixify() function.
 * - 0.19b (28/09/00) : Added conditional compile for FreeBSD.
 * - 0.20b (18/02/01) : Added function trChar(). Contributed  
 *   by Andrew Worsley <epaanwo@asac.ericsson.se>.
 * - 0.21b (30/03/01) : Moved dump_string() from mbchecker.c to
 *   here.
 * - 0.22b (03/04/01) : Added function spchar_pp().
 * - 0.23b (24/03/01) : In tellsock(), added early exit in case
 *   socket is not present (when called through backend_send_sms()).
 *   Contributed by Ing. Andrea Vettori <a.vettori@inetronics.com>.
 * - 0.24b (30/12/01) : Added function file_exist().
 * - 0.25b (23/01/02) : Added function strupper().
 * - 0.26b (21/02/03) : Added local version of function stpcpy() (since
 *   this one doesn't belong to the POSIX API). Added dequalify().
 * - 0.27b (11/06/03) : Added support for FreeBSD 5.0 (contributed by
 *   Greg May <greg@ult.co.za>).
 * - 0.28b (08/04/04) : Expanded trim() to also deal with '\t'.
 * - 0.29b (16/06/04) : Improved the looks and features of the
 *   dump_string() function.
 * - 0.30b (22/03/06) : Added lay_pid_file() function.
 *========================================================*/

#include <unistd.h>
#include <stdio.h>                         /* for fprintf */
#include <stdlib.h>                  /* for errno & stuff */
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <syslog.h>
#include <sys/time.h>           /* for the struct timeval */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>                      /* file open mode */
#include <sys/ipc.h>
#include <sys/sem.h>                        /* semaphores */
#include <sys/shm.h>                     /* shared memory */
#include <sys/ioctl.h>            /* for the ioctl() call */
#include <dial/modems.h>           /* requires 'libmodem' */

#include "sms_serv.h"

/*========================================================*/
/* For debugging purposes only - comment out for normal compile */
/* #define INCL_DEBUG_CODE */

/*========================================================*/
/*#################### misc. file handling ###############*/
/*========================================================*/
int file_exist (char *filename)
{
  extern int errno;
  struct stat sstat;
  int exist;
  
  if (stat (filename, &sstat) == -1) {
    switch (errno) {
      case ENOENT:
        /* file (or path) doesn't exist */
	exist = FALSE;
	break;
	
      default:
        /* stat() failed due to other error (access denied ?) */
	/* file existance is unknown -- abort */
	syslog ((FACILITY | LOG_ERR), "can't stat() <%s>, error was %d.",
               filename, errno);
        syserr ("file_exist(): can't stat()");
    }                                   /* switch (errno) */
  }
  else {
    exist = TRUE;
  }
  
  return (exist);
}                                        /* file_exist () */
/*========================================================*/
int lay_pid_file (char *filename)
{
  pid_t pid;
  int pfd;
  int wl;
  char *s;
  
  pid = getpid ();
  
  s = (char *) malloc ((MINIBUFF + 1) * sizeof (char));
  if (! s) {
    syserr ("lay_pid_file(): can't malloc() s");
  }
  s[0] = '\0';
  sprintf (s, "%d\n", pid);

  if ((pfd = open (filename, O_RDWR | O_CREAT | O_EXCL, 0644)) == -1) {
    syserr ("lay_pid_file(): can't create PID file");
  }
  if ((wl = write (pfd, s, strlen (s))) != strlen (s)) {
    syserr ("lay_pid_file(): can't write PID to file");
  }

  close (pfd);
}                                      /* lay_pid_file () */
/*========================================================*/
/*################### misc. string handling ##############*/
/*========================================================*/
int shiftleft (char *s, int i)
{
  int l;
  int j, k;
  
  l = strlen (s);
  if (i >= l) {
    s[0] = '\0';
    return (0);
  }
  
  for (j = 0, k = i; k < l; k++, j++) {
    s[j] = s[k];
  }
  
  s[j] = '\0';
  
  /* exit */
  return (strlen (s));
}                                         /* shiftleft () */
/*========================================================*/
int l_basename (char *path)

/* Truncates path to its last component. Returns new length
 * of buffer.
 */
{
  char *p;
  
  if ((! path) || (strlen (path) == 0)) {
    return (0);
  }
  if ((p = strrchr (path, '/')) != NULL) {
    shiftleft (path, ((p - path) + 1));
  }

  return (strlen (path));
}                                        /* l_basename () */
/*========================================================*/
char *l_stpcpy (char *dest, char *src)

/* Copies the string pointed to by src (including the
 * terminating '\0') to the array pointed to by dest.
 * The strings may not overlap, and dest must be large
 * enough to accomodate the copy.
 *
 * Local re-implementation, since this function is not
 * part of the POSIX or ANSI API (said to come from DOS). */
{
  char *p, *q;
  
  if ((! dest) || (! src)) {
    return ((char *) NULL);
  }
  
  p = dest + strlen (dest);
  q = src;
  while (*q) {
    p[0] = q[0];
    p++;
    q++;
  }
  
  /* put the terminating null */
  p[0] = '\0';
  /* return a pointer to the end of the resulting string */
  return (p);
}                                          /* l_stpcpy () */
/*========================================================*/
int str_insert (char *s, int pos, char c)

/* Inserts char c in string s at position pos. It's the res-
 * ponsibility of the calling program to make sure that the
 * allocated space for s is large enough to hold the additional
 * character. In case the value of pos is inconsistant (< 0,
 * or > strlen), exits without modifying s, and returns -1.
 * On success, returns the new strlen. */
{
  int len;
  int i;
  
  len = strlen (s);
  if ((pos < 0) || (pos > len)) {
    return (-1);
  }
  
  for (i = len; i >= pos; i--) {
    s[i + 1] = s[i];
  }
  s[pos] = c;
  
  return (strlen (s));
}                                        /* str_insert () */
/*========================================================*/
int dequote (s)
char *s;
{
  int l;
  
  /* check for front quote and remove it */
  if (s[0] == '"') {
    l = shiftleft (s, 1);
  }
  else {
    l = strlen (s);
  }
  /* check for back quote and remove it (l = new strlen) */
  if (s[l - 1] == '"')
    s[l - 1] = '\0';
    
  /* exit */
  return (strlen (s));
}                                           /* dequote () */
/*========================================================*/
int strupper (char *string)
/* converts string to uppercase */
{
  char *p;
  
  p = string;
  while (*p) {
    *p = toupper (*p);
    p++;
  }
  
  return (p - string);
}                                          /* strupper () */
/*========================================================*/
int trim (char *string)
{
  char *p;
  int WS = TRUE;

  p = string;
  while (*p) {
    switch (*p) {
      case '\n' : {
        if (WS)
	  shiftleft (p, 1);
	else {
	  *p = ' ';
	  WS = TRUE;
	  p++;
	}
	break;
      }

      case '\r' : {
        if (WS)
	  shiftleft (p, 1);
	else {
	  *p = ' ';
	  WS = TRUE;
	  p++;
	}
	break;
      }

      case '\t' : {
        if (WS)
	  shiftleft (p, 1);
 	else {
	  *p = ' ';
	  WS = TRUE;
	  p++;
	}
	break;
      }
      
      case ' ' : {
        if (WS)
	  shiftleft (p, 1);
 	else {
	  WS = TRUE;
	  p++;
	}
	break;
      }
      
      default : {
        WS = FALSE;
	p++;
	break;
      }

    }                                      /* switch (*p) */
  }                                         /* while (*p) */
  return (strlen (string));
}                                              /* trim () */
/*========================================================*/
int unixify (char *s)
{
  char *p;
  int BR = FALSE;
  
  p = s;
  while (*p) {
    switch (*p) {
      case '\n': {
        if (BR)
	  shiftleft (p, 1);
	else {
	  BR = TRUE;
	  p++;
	}
	break;
      }
      
      case '\r': {
        if (BR)
	  shiftleft (p, 1);
	else {
	  /* replace it by a '\n' */
	  *p = '\n';
	  BR = TRUE;
	  p++;
	}
	break;
      }
      
      default: {
        BR = FALSE;
	p++;
	break;
      }
    }                                      /* switch (*p) */
  }                                         /* while (*p) */
  return (strlen (s));
}                                           /* unixify () */
/*========================================================*/
int spchar_pp (char *s)
/* Special characters preprocessor */
/* Support still missing for: \ddd and \xhhh */
{
  char *ptr;
  
  ptr = s;
  while (ptr[0]) {
    if ((ptr[0] == '\\') && ptr[1]) {
      switch (ptr[1]) {
	case 'a': {                       /* audible bell */
	  ptr[0] = '\a';
	  shiftleft ((ptr + 1), 1);
          break;
	}

	case 'b': {                          /* backspace */
	  ptr[0] = '\b';
	  shiftleft ((ptr + 1), 1);
          break;
	}

	case 'f': {                           /* formfeed */
	  ptr[0] = '\f';
	  shiftleft ((ptr + 1), 1);
          break;
	}

	case 'n': {                           /* linefeed */
	  ptr[0] = '\n';
	  shiftleft ((ptr + 1), 1);
          break;
	}

	case 'r': {                    /* carriage return */
	  ptr[0] = '\r';
	  shiftleft ((ptr + 1), 1);
          break;
	}

	case 't': {                     /* horizontal tab */
	  ptr[0] = '\t';
	  shiftleft ((ptr + 1), 1);
          break;
	}

	case 'v': {                       /* vertical tab */
	  ptr[0] = '\v';
	  shiftleft ((ptr + 1), 1);
          break;
	}

	case '\\': {                         /* backslash */
	  shiftleft ((ptr + 1), 1);
          break;
	}

	case '\'': {                      /* single quote */
	  ptr[0] = '\'';
	  shiftleft ((ptr + 1), 1);
          break;
	}

	case '\"': {                      /* double quote */
	  ptr[0] = '\"';
	  shiftleft ((ptr + 1), 1);
          break;
	}

	case '\?': {                     /* question mark */
	  ptr[0] = '\?';
	  shiftleft ((ptr + 1), 1);
          break;
	}

	default: {
          break;
	}
      }                                /* switch (ptr[0]) */
    }                                      /* if (ptr[1]) */
    ptr++;
  }                                     /* while (ptr[0]) */
  
  return (strlen (s));
}                                         /* spchar_pp () */
/*========================================================*/
void dump_string (const char *s)
{
  int addr = 0;
  int l, m;
  int i, j;
  unsigned char *ptr;
  char equiv[17];
  char hexd[9];
  char ce[2];
  char ch[3];
  char c;
  
  ptr = (char *) s;
  equiv[0] = '\0';
  hexd[0] = '\0';
  
  fprintf (stderr, "%06x | ", addr);
  while (ptr[0]) {
    sprintf (ch, "%02x", ptr[0]);
    strcat (hexd, ch);
    if (((ptr[0] >= 32) && (ptr[0] < 127)) ||
       ((ptr[0] >= 128) && (ptr[0] <= 254))) {
      c = ptr[0];
    }
    else {
      c = '.';
    }
    sprintf (ce, "%c", c);
    strcat (equiv, ce);
    ptr++;
    addr++;
    if ((addr % 4) == 0) {
      fprintf (stderr, "%s ", hexd);
      hexd[0] = '\0';
    }
    if ((addr % 16) == 0) {
      fprintf (stderr, "|%s|\n", equiv);
      equiv[0] = '\0';
      if ((char *)ptr != (s + strlen (s))) {
        fprintf (stderr, "%06x | ", addr);
      }
    }
  }
  /* pad and flush any leftover */
  if ((l = strlen (equiv)) > 0) {
    /* pad current hexd */
    m = strlen (hexd);
    for (i = 0; i < (8 - m); i++) {
      strcat (hexd, "_");
    }
    fprintf (stderr, "%s ", hexd);
    
    /* provide blank hexd's to fill the line */
    j = (int) (trunc ((16 - l) / 4));
    for (i = 0; i < j; i++) {
      fprintf (stderr, "________ ");
    }
    
    /* pad current equiv */
    for (i = 0; i < (16 - l); i++) {
      strcat (equiv, "_");
    }
    fprintf (stderr, "|%s|\n", equiv);
  }
}                                       /* dump_string () */
/*========================================================*/
char *trChar (char *s, int old, int new)
/*
 * process string and translate occurances of the character old into new
 * returns the pointer given to it initially. Return is immeadiate if given
 * a null pointer.
 */
{
    char *p = s;

    if (s == 0)
      return (s);

    while (p = strchr (p, old))
      *p++ = new;

    return (s);
}					     /* trChar () */
/*========================================================*/
int deslashdot (char *s)

/* Cleans phone numbers */
{
  char *ptr;
  
  ptr = s;
  while (*ptr) {
    if ((!isdigit (ptr[0])) && (ptr[0] != '+')) {
      shiftleft (ptr, 1);
    }
    else {
      ptr++;
    }
  }                                       /* while (*ptr) */
  return (strlen (s));
}                                        /* deslashdot () */
/*========================================================*/
char *nicedate (char *s)

/* takes a date in YYYYMMDD format and returns a string
   formatted as Mmm D, YYYY                               */
{
  char *out;
  char *ptr;
  char y[5];
  char m[3];
  char d[3];
  char mn[4];
  int i;
  char monthlist[37] = "JanFebMarAprMayJunJulAugSepOctNovDec";

  /*---------------------------------------Initialization */
  out = (char *) malloc (20 * sizeof (char));
  if (! out) {
    syslog ((FACILITY | LOG_ERR), "can't malloc().");
    syserr ("nicedate(): can't malloc() out");
  }
  out[0] = '\0';
  y[0] = '\0';
  m[0] = '\0';
  d[0] = '\0';
  mn[0] = '\0';

  /*-------------------------------------------Parse date */
  strncpy (y, s, 4);
  y[4] = '\0';
  ptr = s + 4;
  strncpy (m, ptr, 2);
  m[2] = '\0';
  ptr += 2;
  strncpy (d, ptr, 2);
  d[2] = '\0';
  
  /*--------------------------------------Generate string */
  i = atoi (m) - 1;
  ptr = monthlist + (3 * i);
  strncpy (mn, ptr, 3); 
  mn[3] = '\0';

  sprintf (out, "%s %d, %s", mn, atoi (d), y);

  /*-------------------------------------------Conclusion */
  return (out);
}                                          /* nicedate () */
/*========================================================*/
/*========================================================*/
/*################# Socket communication #################*/
/*========================================================*/
void tellsock (int sock, char *string)
{
  int length;
  
  if (sock == 0)
    /* Not an error ! When called through backend_send_sms(), */
    /* we don't have a client-side socket ! */
    return;
    
  length = strlen (string);
  if (write (sock, string, length) != length)
    syserr ("tellsock(): error while writing to socket");

}                                          /* tellsock () */
/*========================================================*/
/*========================================================*/
/*#################### Error Handling ####################*/
/*========================================================*/
void syserr (char *msg)
{
  extern int errno;
  extern char *progname;

  fprintf (stderr, "ERROR %s: %s", progname, msg);
  if (errno > 0)
    fprintf (stderr, " (%d ; %s)\n", errno, strerror (errno));
  else
    fprintf (stderr, "\n");
  exit (1);
}                                            /* syserr () */
/*========================================================*/
void fatal (char *msg)
{
  extern char *progname;

  fprintf (stderr, "FATAL ERROR %s: %s\n", progname, msg);
  exit (2);
}                                             /* fatal () */
/*==========================================================
 * EOF : stuff.c
 *===================*/
