/*---------------------------------------------------------------------------*\

    FILE....: ALAW.C
    TYPE....: C Functions
    AUTHOR..: David Rowe
    DATE....: 16/10/97

    A-law compression functions.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include "../contypes/contypes.h"
#include "../comp/comp.h"
#include <assert.h>

/*-------------------------------------------------------------------------*\

				DEFINES

\*-------------------------------------------------------------------------*/

/* A-law constants */

#define	LSIGN	0x8000		/* linear sign bit 15 = 1 if negative	   */
#define	ASIGN	0x80		/* A-law sign bit 7 = 1 if negative	   */
#define	AMAX	0x0fff		/* max A-law value in linear form	   */
#define	AMAG	0x7f		/* A-law magnitude bits 		   */
#define	BIAS	33		/* converts seg end points to powers of 2  */
#define	LINEAR	0xf		/* masks of linear portion of magnitude	   */

/*-------------------------------------------------------------------------*\

			    FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: alaw_encode
	AUTHOR...: David Rowe
	DATE.....: 14/10/97

	Encodes a vector of linear samples to A-law equivalents.  The linear
	samples are assumed to be in Q15 format.

	Reference: TI databook
		   "Theory, Alogorithms, and Implementations Vol 1", p171

	A-law format: psssqqqq

		       p is the sign
		       s is the segment
		       q if the quantisation bin number

\*--------------------------------------------------------------------------*/

void alaw_encode(ushort alaw[], short linear[], ushort sz)
/*  word   alaw[];	/* A-law encoded samples in lower 8 bits	*/
/*  word   linear[];	/* 13 bit magnitude samples in LSB		*/
/*  ushort sz;		/* number of sample in buffers			*/
{
    ushort y,mag;
    ushort p,s,q,b;
    ulong  acc;
    int    i;

    for(i=0; i<sz; i++) {
	acc = (linear[i] >> ALAW_SCALE) & 0xffff;

	/* separate sign */

	if (acc & LSIGN) {
	    p = ASIGN;
	    mag = -acc;
	}
	else {
	    p = 0;
	    mag = acc;
	}

	/* encode magnitude */

	if (mag > AMAX)
	    y = p + AMAG;
	else {

	    /* determine left most bit and therefore segment */

	    b = 0;
	    acc = mag >> 1;
	    while(acc) {
		b++;
		acc >>= 1;
	    }

	    if (b > 4) {
		s = b - 4;

		q = mag >> (b-4);
		q &= LINEAR;
	    }
	    else {
		s = 0;
		q = mag >> 1;
	    }

	    y = p + (s<<4) + q;
	}

	alaw[i] = y^0x55;		/* even bit inversion */
    }
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: alaw_decode
	AUTHOR...: David Rowe
	DATE.....: 14/10/97

	Decodes a vector of A-law samples to linear equivalents.  The linear
	samples are in Q15 format.

	Reference: TI databook
		   "Theory, Alogorithms, and Implementations Vol 1", p171

\*--------------------------------------------------------------------------*/

void alaw_decode(short linear[], ushort alaw[], ushort sz)
/*  word   linear[];	/* 13 bit magnitude samples in LSB		*/
/*  word   alaw[];	/* A-law encoded samples in lower 8 bits	*/
/*  ushort sz;		/* number of sample in buffers			*/
{
    ushort mag;
    ulong  acc,p,s,q;
    int    i;

    for(i=0; i<sz; i++) {
	acc = alaw[i] ^ 0x55;	/* even bit inversion */

	/* extract sign */

	if (acc & ASIGN)
	    p = LSIGN;
	else
	    p = 0;
	acc &= AMAG;

	/* extract q and s */

	q = acc & 0xf;
	s = acc >> 4;

	/* form linear sample */

	if (s != 0) {
	    acc = 2*q+BIAS;
	    acc <<= (s-1);
	}
	else
	    acc = 2*q+1;

	acc <<= ALAW_SCALE;
	if (p)
	    linear[i] = -acc;
	else
	    linear[i] = acc;
    }

}

