/*--------------------------------------------------------------------------*\

    FILE....: DSPFIFO.CPP
    TYPE....: C++ Functions
    AUTHOR..: David Rowe
    DATE....: 19/11/97

    Functions used to read and write to First In First Out (FIFO) queues in
    the DSP, thus facilitating PC to DSP data transfer.  The word length of
    the FIFOs is 16 bits.

	Modified 7/10/98 to support mark 2 kernalmode device driver that
	buffers DSP fifo in PC fifo.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include "../comm/dspfifo.h"
#include "../hip/hip.h"
#include "../message/mess.h"
#include "../wobbly/wobbly.h"
#include "../generic/generic.h"
#include "../hip/vpbioctl.h"

/*--------------------------------------------------------------------------*\

									DEFINES

\*--------------------------------------------------------------------------*/

// offsets of each param from base of DSP FIFO structure in DSP memory 

#define	PSTART		0
#define	PEND		1
#define	PWR			2
#define PRD			3
#define	SIZE		4

#define	FIFO_SIZE	5	// size of structure to upload

// error code for CheckDspFifo 

#define	FAIL		1

/*-------------------------------------------------------------------------*\

								CLASS

\*-------------------------------------------------------------------------*/

// pure virtual base class implementation

class baseDspFifoData {
public:	
	baseDspFifoData() {}
	virtual ~baseDspFifoData() {}
	virtual int Write(word *buf, ushort size) = 0;
	virtual int Read(word *buf, ushort size) = 0;
	virtual int HowFull(void) = 0;
	virtual int DspHowFull(void) = 0;
	virtual int HowEmpty(void) = 0;
};

// DspFifo implementation for original device driver

#pragma pack(1)		// force member alignment same as DSP

class DspFifoData : public baseDspFifoData {
    // copy of FIFO parameters in DSP memory 

    word	pstart;		// first word in FIFO			
    word	pend;		// one after last word in FIFO		
    word	pwr;		// write pointer			
    word	prd; 		// read pointer				
    ushort	size;		// total storage in FIFO		

    // extra params for PC only 

    Hip		*hip;		// hip object used to talk to DSP
	ushort	board;		// VPB board number			
    ushort	adspfifo;	// address of structure in DSP memory	
	int		direction;	// direction of DSP FIFO

public:

	DspFifoData(Hip *hip, ushort board, ushort fifo_addr, int direction, int relay);
	~DspFifoData();
	int Write(word *buf, ushort size);
	int Read(word *buf, ushort size);
	int HowFull(void);
	int HowEmpty(void);
	int DspHowFull(void);
	int CheckDspFifo();
};

#pragma pack()	// normal member alignment

class DspFifoDataDD : public baseDspFifoData {
	void    *hndFile;	// handle to device driver
	ushort	*pfifobuf;	// user mode PC fifo buffer
	ushort	index;		// id for this fifo
	int		direction;	// direction of relay fifo
	int		size;		// available storgae in PC side of relay fifo
	int		relay;		// switch for enabling relay

public:

	DspFifoDataDD(Hip *hip, ushort board, ushort fifo_addr, int direction, int relay);
	~DspFifoDataDD();
	int Write(word *buf, ushort size);
	int Read(word *buf, ushort size);
	int HowFull(void);
	int DspHowFull(void);
	int HowEmpty(void);
};

/*
   NOTE on DSP FIFO implementation:

   These FIFOs use the position of pwr and prd to determine how full the
   FIFO is.

   One way to implement a FIFO is to maintain a counter of how many words are
   in the FIFO.  The write side increments the counter, the read side
   decrements the counter.  This is a problem in our case, as the read and
   write operations are performed by unsynhcronised processes (DSP and PC),
   thus the possibility exists of both sides writing to the counter at the
   same time.

   The alternative used here is to use the position of pwr and prd to
   determine how full the FIFO is.  Only the write side modifies pwr and only
   the read side modifies prd, so no contentions will occur.

   One problem with using pwr and prd is the case of pwr = prd.  This could
   occur when the FIFO is empty or when the FIFO is full.  This is resolved
   by only allowing pwr = prd when the FIFO is empty.  When writing to the
   FIFO the pwr ptr is only allowed to get up to prd-1.  Thus the actual
   capacity of the FIFO is size-1.
*/

// used to make access to DSP thread safe

static GENERIC_CRITICAL_SECTION	DspFifoSect;

// used to give each DSP fifo a unique id

static int index_count;

// size of PC side of ALL relay (signal and message) fifos

static unsigned long szRelayBuf;

/*--------------------------------------------------------------------------*\

								FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: open_dspfifo
	AUTHOR...: David Rowe
	DATE.....: 23/3/98

	Initialises the critical section for the DSP fifo module.  This function
	should be called before any DSP fifos objects are created.

\*--------------------------------------------------------------------------*/

void dspfifo_open(unsigned long szrb)
{
	GenericInitializeCriticalSection(&DspFifoSect);
	index_count = 0;
	szRelayBuf = szrb;

	// check device driver direction codes match those in header

	assert(DSPFIFO_UP == RELAY_UP);
	assert(DSPFIFO_DOWN == RELAY_DOWN);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: close_dspfifo
	AUTHOR...: David Rowe
	DATE.....: 23/3/98

	Deletes the critical section for the DSP fifo module.  This function
	should be called all DSP fifo objects have been deleted.

\*--------------------------------------------------------------------------*/

void dspfifo_close()
{
	GenericDeleteCriticalSection(&DspFifoSect);
}

/*--------------------------------------------------------------------------*\

						DspFifo MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::DspFifo
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Opens a link to a DSP FIFO.  Assumes DSP has been booted with 
	appropriate code.

\*--------------------------------------------------------------------------*/

DspFifo::DspFifo(Hip *h, ushort bd, ushort dspfifo, int direction, int relay)
//  Hip		*h;			HIP port object 
//  ushort  bd;			VPB board number for FIFO
//  ushort  dspfido;	addr of FIFO structure in DSP		 
//	int		direction;	direction of FIFO (DSPFIFO_UP or DSPFIFO_DOWN)
//	int		relay		non-zero to enable relay, zero for normal DSP FIFO
{
	if (h->GetDeviceDriverModel() == 0) {
		// original driver

		d = new DspFifoData(h, bd, dspfifo, direction, relay);
		if (d == NULL)
			throw Wobbly(DSPFIFO_CANT_ALLOCATE_MEMORY);
	}
	else {
		// driver that contains HIP and DSPFIFO

		d = new DspFifoDataDD(h, bd, dspfifo, direction, relay);
		if (d == NULL)
			throw Wobbly(DSPFIFO_CANT_ALLOCATE_MEMORY);
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::~DspFifo()
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Closes the connection between the DSP FIFO and the PC.

\*--------------------------------------------------------------------------*/

DspFifo::~DspFifo()
{
	delete d;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::Write
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Writes a block of words from the PC to a DSP FIFO.  Returns OK
	for success.

\*--------------------------------------------------------------------------*/

int DspFifo::Write(word *buf, ushort sz)
//  word    *buf;		buf of words to write to DSP FIFO 	
//  ushort  sz;			size of buf				
{	
    return(d->Write(buf,sz));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::Read
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Reads a block of words from a DSP FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int DspFifo::Read(word *buf, ushort sz)
//  word    *buf;		buffer of words to read from DSP FIFO 	
//  ushort  sz;			size of buf				
{

//	int mal = d->Read(buf,sz);
//	return(mal);
//   Horse 15/8/98
   return(d->Read(buf, sz));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::HowFull
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Returns the number of words used in the DSP FIFO.

\*--------------------------------------------------------------------------*/

int DspFifo::HowFull(void)
{
    return(d->HowFull());
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::DspHowFull
	AUTHOR...: David Rowe
	DATE.....: 15/10/98

	Returns the number of words used in the actual DSP FIFO, not the
	relay fifo if enabled.  With old dd. reduces to HowFull

\*--------------------------------------------------------------------------*/

int DspFifo::DspHowFull(void)
{
    return(d->DspHowFull());
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::HowEmpty
	AUTHOR...: David Rowe
	DATE.....: 9/10/98

	Returns the number of words free in the DSP FIFO.

\*--------------------------------------------------------------------------*/

int DspFifo::HowEmpty(void)
{
    return(d->HowEmpty());
}

/*--------------------------------------------------------------------------*\

						DspFifoData MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::DspFifoData
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Opens a link to a DSP FIFO.  Assumes DSP has been booted with 
	appropriate code.

\*--------------------------------------------------------------------------*/

DspFifoData::DspFifoData(Hip *h, ushort bd, ushort dspfifo, int dir, int relay)
//  Hip		*h;			HIP port object 
//  ushort  bd;			VPB board number for FIFO
//  ushort  dspfido;	addr of FIFO structure in DSP		 
//  int		dir			unused in this implementation (apart from error checking)
//	int		relay		unused in this implementation (apart from error checking)
{
    // validate arguments 

    assert(h != NULL);
	assert(bd < MAX_VPB);
	assert((dir == DSPFIFO_UP) || (dir == DSPFIFO_DOWN));
	assert((relay == RELAY_ON) || (relay == RELAY_OFF));

    // init DSP fifo member variables 

    hip = h;
    board = bd;
    adspfifo = dspfifo;
	direction = dir;
	
    // upload DSP FIFO parameter structure 

	GenericEnterCriticalSection(&DspFifoSect);
    hip->ReadDspSram(board, adspfifo, FIFO_SIZE, &pstart);
	GenericLeaveCriticalSection(&DspFifoSect);
    assert(CheckDspFifo() == OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::~DspFifoData()
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Closes the connection between the DSP FIFO and the PC.

\*--------------------------------------------------------------------------*/

DspFifoData::~DspFifoData()
{
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::Write
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Writes a block of words from the PC to a DSP FIFO.  Returns OK
	for success.

\*--------------------------------------------------------------------------*/

int DspFifoData::Write(word *buf, ushort sz)
//  word    *buf;		buf of words to write to DSP FIFO 	
//  ushort  sz;			size of buf				
{	
    ushort  words_used;	// used space in FIFO			
    ushort  words_free;	// free space in FIFO			
    ushort  copy_first;	// size of first block move		
    ushort  new_pwr;	// modified pwr	 	   		

    // validate arguments 

    assert(buf != NULL);
	assert(direction == DSPFIFO_DOWN);

    // update PC's copy of DSP FIFO state variables 

	GenericEnterCriticalSection(&DspFifoSect);
    hip->ReadDspSram(board, adspfifo+PWR, 1, &pwr);
    hip->ReadDspSram(board, adspfifo+PRD, 1, &prd);

	assert(CheckDspFifo() == OK);

    // determine if there is enough room in FIFO for buf 

    if (pwr >= prd)
		words_used = pwr - prd;
    if (prd > pwr)
		words_used = size - (prd - pwr);
    words_free = size - words_used - 1;

    if (words_free < sz) {
		GenericLeaveCriticalSection(&DspFifoSect);
		return(DSPFIFO_FULL);
	}

    // If buf overlaps end of linear array split into two block moves 

    if ((pwr + sz) > pend) {
		copy_first = (pend-pwr);

		hip->WriteDspSram(board, pwr, copy_first, buf);
		hip->WriteDspSram(board, pstart, sz-copy_first,&buf[copy_first]);
    }
    else
		hip->WriteDspSram(board, pwr, sz, buf);

    // increment pwr and wrap around if required 

    new_pwr = pwr + sz;
    if (new_pwr >= pend)
		pwr = pstart + (new_pwr - pend);
    else
		pwr = new_pwr;

    // copy pwr back to DSP 

    hip->WriteDspSram(board, adspfifo+PWR, 1, &pwr);
    assert(CheckDspFifo() == OK);
	GenericLeaveCriticalSection(&DspFifoSect);

    return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::Read
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Reads a block of words from a DSP FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int DspFifoData::Read(word *buf, ushort sz)
//  word    *buf;		buffer of words to read from DSP FIFO 	
//  ushort  sz;			size of buf				
{
    ushort  words_used;	// used space in FIFO			
    ushort  copy_first;	// size of first block move		
    ushort  new_prd;	// modified prd	    		

    // validate arguments 

    assert(buf != NULL);
	assert(direction == DSPFIFO_UP);

    // update PC's copy of DSP FIFO state variables 

	GenericEnterCriticalSection(&DspFifoSect);
    hip->ReadDspSram(board, adspfifo+PWR, 1, &pwr);
    hip->ReadDspSram(board, adspfifo+PRD, 1, &prd);
    assert(CheckDspFifo() == OK);

    // determine if there is data in FIFO to fill buf 

    if (pwr >= prd)
		words_used = pwr - prd;
    if (prd > pwr)
		words_used = size - (prd - pwr);
    if (words_used < sz) {
		GenericLeaveCriticalSection(&DspFifoSect);
		return(DSPFIFO_EMPTY);
	}

    // If buf overlaps end of linear array split into two block moves 

    if ((prd + sz) > pend) {
		copy_first = (pend-prd);

		hip->ReadDspSram(board, prd, copy_first, buf);
		hip->ReadDspSram(board, pstart, sz-copy_first,&buf[copy_first]);
    }
    else
		hip->ReadDspSram(board, prd, sz, buf);

    // increment prd and wrap around if required 

    new_prd = prd + sz;
    if (new_prd >= pend)
		prd = pstart + (new_prd - pend);
    else
		prd = new_prd;

    // copy prd back to DSP 

    hip->WriteDspSram(board, adspfifo+PRD, 1l, &prd);
    assert(CheckDspFifo() == OK);

	GenericLeaveCriticalSection(&DspFifoSect);
    return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::HowFull
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Returns the number of words used in the DSP FIFO.

\*--------------------------------------------------------------------------*/

int DspFifoData::HowFull(void)
{
    ushort words;

	GenericEnterCriticalSection(&DspFifoSect);
    hip->ReadDspSram(board, adspfifo+PWR, 1, &pwr);
    hip->ReadDspSram(board, adspfifo+PRD, 1, &prd);
	
    assert(CheckDspFifo() == OK);
    if (pwr >= prd)
		words = pwr - prd;
    if (prd > pwr)
		words = size - (prd - pwr);

	GenericLeaveCriticalSection(&DspFifoSect);
	return(words);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::DspHowFull
	AUTHOR...: David Rowe
	DATE.....: 15/10/98

	Returns the number of words used in the actual DSP FIFO, not the
	relay fifo if enabled.  With old dd. reduces to HowFull

\*--------------------------------------------------------------------------*/

int DspFifoData::DspHowFull(void)
{
    return(HowFull());
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::HowEmpty
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Returns the number of words free in the DSP FIFO.

\*--------------------------------------------------------------------------*/

int DspFifoData::HowEmpty(void)
{
	return(size-1-HowFull());
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: DspFifoData::CheckDspFifo
    AUTHOR...: David Rowe
	DATE.....: 20/11/97

    Performs a few sanity checks on DSP FIFO structure.  Returns OK is DSP
    FIFO checks out returns.

\*-------------------------------------------------------------------------*/

int DspFifoData::CheckDspFifo()
{
    if (pend < pstart) assert(0);
    if (pwr < pstart) assert(0);
    if (pwr > pend) assert(0);
    if (prd < pstart) assert(0);
    if (prd > pend) assert(0);

    return(OK);
}

/*--------------------------------------------------------------------------*\

						DspFifoDataDD MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::DspFifoDataDD
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Opens a link to a relay FIFO.  Assumes DSP has been booted with 
	appropriate code.  A relay fifo is a large PC fifo that buffers
	communications to the underlying DSP fifo.  A device driver thread
	then transfers data from the PC to the DSP fifo (or vice-versa
	depending on direction of relay).

	Optionally, (relay flag == RELAY_OFF), we can just talk to the DSP
	FIFO direct, without the PC relay.  This would found to be necessary
	for the up and downmessage queues as the large buffers were causing
	messages to be segmented (length separated from body of mess), which
	caused all sorts of nasty problems.  Big buffers are only required
	for high bandwidth signal data anayway.

\*--------------------------------------------------------------------------*/

DspFifoDataDD::DspFifoDataDD(Hip *h, ushort bd, ushort dspfifo, int dir, int rel)
//  Hip		*h;			HIP port object 
//  ushort  bd;			VPB board number for FIFO
//  ushort  dspfido;	addr of FIFO structure in DSP		 
//  int		dir;		direction of relay (DSPFIFO_UP, DSPFIFO_DOWN)
//	int		rel			relay flag
{
	long			IoctlCode;
    unsigned long	ReturnedLength; 
    int				IoctlResult;
	FIFO_PARAMS		Fifo;

	GenericEnterCriticalSection(&DspFifoSect);

    // validate arguments 

    assert(h != NULL);
	assert(bd < MAX_VPB);
	assert((dir == DSPFIFO_UP) || (dir == DSPFIFO_DOWN));
	assert((rel == RELAY_ON) || (rel == RELAY_OFF));

	// get handle to device driver from hip

	hndFile = h->GethndFile();
	index = index_count++;
	direction = dir;
	size = szRelayBuf;
	relay = rel;

	if (relay == RELAY_ON) {

		// allocate user mode buffer for PC side of buffer
	
		pfifobuf = new ushort[szRelayBuf];
		assert(pfifobuf != NULL);
	
		// set up DeviceIoControl() call

		Fifo.ppc = (void*)pfifobuf;
		Fifo.length = szRelayBuf;
		Fifo.index = index;
		Fifo.board = bd;
		Fifo.pdsp = dspfifo;
		Fifo.direction = direction;

		IoctlCode = IOCTL_VPB_RELAY_FIFO_CREATE;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            NULL,		        // Buffer from driver.
                            0,			        // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}
	else {
		Fifo.index = index;
		Fifo.pdsp = dspfifo;
		Fifo.board = bd;

		IoctlCode = IOCTL_VPB_DSP_FIFO_CREATE;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            NULL,		        // Buffer from driver.
                            0,			        // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}

	GenericLeaveCriticalSection(&DspFifoSect);
	if (!IoctlResult)                            // Did the IOCTL succeed?
        throw Wobbly(DSPFIFO_OPEN_FAIL);	
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::~DspFifoDataDD()
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Closes the connection between the relay FIFO and the PC.

\*--------------------------------------------------------------------------*/

DspFifoDataDD::~DspFifoDataDD()
{
	long			IoctlCode;
    unsigned long	ReturnedLength; 
    int				IoctlResult;
	FIFO_PARAMS		Fifo;
	
	GenericEnterCriticalSection(&DspFifoSect);
	Fifo.index = index;
	
	if (relay == RELAY_ON) {

		// set up DeviceIoControl() call

		IoctlCode = IOCTL_VPB_RELAY_FIFO_DESTROY;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(Fifo),		// Length of buffer in bytes.
                            NULL,		        // Buffer from driver.
                            0,			        // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );

		delete [] pfifobuf;
	}
	else {
		// set up DeviceIoControl() call

		IoctlCode = IOCTL_VPB_DSP_FIFO_DESTROY;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(Fifo),		// Length of buffer in bytes.
                            NULL,		        // Buffer from driver.
                            0,			        // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}

	GenericLeaveCriticalSection(&DspFifoSect);
	if (!IoctlResult)                            // Did the IOCTL succeed?
        throw Wobbly(DSPFIFO_CLOSE_FAIL);	
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::Write
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Writes a block of words from the PC to a relay FIFO.  Returns OK
	for success.

\*--------------------------------------------------------------------------*/

int DspFifoDataDD::Write(word *buf, ushort sz)
//  word    *buf;		buf of words to write to DSP FIFO 	
//  ushort  sz;			size of buf				
{	
	long			IoctlCode;
    unsigned long	ReturnedLength; 
    int				IoctlResult;
	FIFO_PARAMS		Fifo;
	
	assert(direction == DSPFIFO_DOWN);

	GenericEnterCriticalSection(&DspFifoSect);
	if (relay == RELAY_ON) {

		Fifo.ppc = buf;
		Fifo.length = sz;
		Fifo.index = index;
	
		// set up DeviceIoControl() call

		IoctlCode = IOCTL_VPB_FIFO_WRITE;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(Fifo),		// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(Fifo),       // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	} 
	else {
		Fifo.index = index;
		Fifo.ppc = buf;
		Fifo.length = sz;

		IoctlCode = IOCTL_VPB_DSP_FIFO_WRITE;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}

	GenericLeaveCriticalSection(&DspFifoSect);
	if (!IoctlResult)                            // Did the IOCTL succeed?
        throw Wobbly(DSPFIFO_WRITE_FAIL);	
	if (ReturnedLength != sizeof(Fifo)) 
        throw Wobbly(DSPFIFO_WRITE_FAIL);	

	return(Fifo.ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::Read
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Reads a block of words from a relay FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int DspFifoDataDD::Read(word *buf, ushort sz)
//  word    *buf;		buffer of words to read from DSP FIFO 	
//  ushort  sz;			size of buf				
{
	long			IoctlCode;
    unsigned long	ReturnedLength; 
    int				IoctlResult;
	FIFO_PARAMS		Fifo;
	
	assert(direction == DSPFIFO_UP);

	GenericEnterCriticalSection(&DspFifoSect);

	if (relay == RELAY_ON) {

		Fifo.ppc = buf;
		Fifo.length = sz;
		Fifo.index = index;
	
		// set up DeviceIoControl() call

		IoctlCode = IOCTL_VPB_FIFO_READ;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(Fifo),		// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(Fifo),       // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}
	else {
		Fifo.index = index;
		Fifo.ppc = buf;
		Fifo.length = sz;

	    IoctlCode = IOCTL_VPB_DSP_FIFO_READ;
	    IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}

	GenericLeaveCriticalSection(&DspFifoSect);
	if (!IoctlResult)                            // Did the IOCTL succeed?
        throw Wobbly(DSPFIFO_READ_FAIL);	
	if (ReturnedLength != sizeof(Fifo)) 
        throw Wobbly(DSPFIFO_READ_FAIL);	

	return(Fifo.ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::HowFull
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Returns the number of words used in the PC side of the relay FIFO.

\*--------------------------------------------------------------------------*/

int DspFifoDataDD::HowFull(void)
{
	long			IoctlCode;
    unsigned long	ReturnedLength; 
    int				IoctlResult;
	FIFO_PARAMS		Fifo;
	
	GenericEnterCriticalSection(&DspFifoSect);

	if (relay == RELAY_ON) {
		Fifo.index = index;
	
		// set up DeviceIoControl() call

		IoctlCode = IOCTL_VPB_FIFO_HOW_FULL;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(Fifo),		// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(Fifo),       // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	
	}
	else {
		Fifo.index = index;

		IoctlCode = IOCTL_VPB_DSP_FIFO_HOW_FULL;
		IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );
	}

	GenericLeaveCriticalSection(&DspFifoSect);
	if (!IoctlResult)                           // Did the IOCTL succeed?
        throw Wobbly(DSPFIFO_HOW_FULL_FAIL);	
	if (ReturnedLength != sizeof(Fifo)) 
        throw Wobbly(DSPFIFO_HOW_FULL_FAIL);	

	return(Fifo.ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::DspHowFull
	AUTHOR...: David Rowe
	DATE.....: 15/10/98

	Returns the number of words used in the actual DSP FIFO.  Used by
	play to determine when DSP has really finished.

\*--------------------------------------------------------------------------*/

int DspFifoDataDD::DspHowFull(void)
{
	long			IoctlCode;
    unsigned long	ReturnedLength; 
    int				IoctlResult;
	FIFO_PARAMS		Fifo;
	
	GenericEnterCriticalSection(&DspFifoSect);

	Fifo.index = index;

	IoctlCode = IOCTL_VPB_DSP_FIFO_HOW_FULL;
	IoctlResult = GenericDeviceIoControl(
                            hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Fifo,		        // Buffer to driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &Fifo,		        // Buffer from driver.
                            sizeof(FIFO_PARAMS),// Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );

	GenericLeaveCriticalSection(&DspFifoSect);
	if (!IoctlResult)                           // Did the IOCTL succeed?
        throw Wobbly(DSPFIFO_HOW_FULL_FAIL);	
	if (ReturnedLength != sizeof(Fifo)) 
        throw Wobbly(DSPFIFO_HOW_FULL_FAIL);	

	return(Fifo.ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoDataDD::HowEmpty
	AUTHOR...: David Rowe
	DATE.....: 9/10/98

	Returns the number of words available in the PC side of the relay FIFO.

\*--------------------------------------------------------------------------*/

int DspFifoDataDD::HowEmpty(void)
{
	return((size-1) - HowFull());
}
