/*---------------------------------------------------------------------------*\

    FILE....: GenericLinux.cpp
    TYPE....: C Functions
    AUTHOR..: David Rowe
    DATE....: 14/9/99

	Linux implementation of Generic functions.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include "../generic/generic.h"
#include "../hip/ioctllinux.h"

void GenericInitializeCriticalSection(GENERIC_CRITICAL_SECTION *cs) {
	int ret;

	cs->v = (void*)new pthread_mutex_t;
	assert(cs->v != NULL);
	ret = pthread_mutex_init((pthread_mutex_t*)cs->v, NULL);
	assert(ret == 0);
}

void GenericDeleteCriticalSection(GENERIC_CRITICAL_SECTION *cs) {
	int ret;

	// destroy mutex
	ret = pthread_mutex_destroy((pthread_mutex_t*)cs->v);
	assert(ret == 0);
	delete cs->v;
}

void GenericEnterCriticalSection(GENERIC_CRITICAL_SECTION *cs)  {
	int ret;
	ret = pthread_mutex_lock((pthread_mutex_t*)cs->v);
	assert(ret == 0);
}

void GenericLeaveCriticalSection(GENERIC_CRITICAL_SECTION *cs)  {
	int ret;
	ret = pthread_mutex_unlock((pthread_mutex_t*)cs->v);
	assert(ret == 0);
}

// do nothing in Linux
int GenericDeviceIoControl(
	void			*hndFile,	    // Handle to device
    long			IoctlCode,      // IO Control code for Read
    void			*InBuffer,		// Buffer to driver.
	long			InBufferSize,      
    void			*OutBuffer,     // Buffer from driver
    long			OutBufferSize,     
    unsigned long	*BytesReturned,	// no. bytes returned from driver
	void			*Overlapped		// not used, included for Win32 compatibility
)
{
	return 0;
}

// do nothing in Linux
void *GenericOpenMk0Driver()
{
	return NULL;
}

// do nothing in Linux
void *GenericOpenMk1Driver()
{
	return NULL;
}

// do nothing in Linux
int GenericCloseHandle(void *hndFile) {
	return 0;
}

// return time in ms since system startup
long GenerictimeGetTime() {
	struct timeval tval;
	gettimeofday(&tval, NULL);
	return (tval.tv_sec*1000 + tval.tv_usec/1000);
}

// determies OS, returns -1 for fail

int GenericGetOS() {
	return GET_OS_LINUX;
}

// sleep for ms milliseconds
void GenericSleep(unsigned int ms)
{
	long ms_l = ms;
	struct timespec ts;
	ts.tv_sec = ms_l/1000;
	ts.tv_nsec = (ms_l-ts.tv_sec*1000)*1000000l;
	nanosleep(&ts, NULL);
}

void Generic_beginthread(void(*func)(void*), int stack, void *data) {
	pthread_t	thread;
	pthread_create(&thread, NULL, (void*(*)(void*))func, data);
}

// do nothing for initial implementation
void GenericSetThreadPriorityHigh() {
}

// do nothing for initial implementation
void GenericSetThreadPriorityNormal() {
}

// used for Linux device driver

int Generic_add_board(int fd, void *data) {
	return ioctl(fd, VPB_IOC_ADD_BOARD, data);
}

int Generic_remove_board(int fd, void *data) {
	return ioctl(fd, VPB_IOC_REMOVE_BOARD, data);
}

int Generic_block_write(int fd, void *data) {
	return ioctl(fd, VPB_IOC_BLOCK_WRITE, data);
}

int Generic_block_read(int fd, void *data) {
	return ioctl(fd, VPB_IOC_BLOCK_READ, data);
}

int Generic_open() {
	return open("/dev/vpb0",O_RDWR);
}

int	Generic_close(int fd) {
	return close(fd);
}

