/*--------------------------------------------------------------------------*\

    FILE....: DSPFIFO.C
    TYPE....: C Module
    AUTHOR..: David Rowe
    DATE....: 6/10/98

    Functions used to read and write to First In First Out (FIFO) queues in
    the DSP, thus facilitating PC to DSP data transfer.  The word length of
    the FIFOs is 16 bits.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <stdlib.h>
#include <ntddk.h>
#include "dspfifo.h"
#include "hip.h"

/*--------------------------------------------------------------------------*\

									DEFINES

\*--------------------------------------------------------------------------*/

#define	assert(x)	ASSERT(x)

// offsets of each param from base of DSP FIFO structure in DSP memory 

#define	PSTART		0
#define	PEND		1
#define	PWR			2
#define PRD			3
#define	SIZE		4

#define	FIFO_SIZE	5	// size of structure to upload

// error code for CheckDspFifo 

#define	FAIL		1

/*
   NOTE on DSP FIFO implementation:

   These FIFOs use the position of df->pwr and df->prd to determine how full the
   FIFO is.

   One way to implement a FIFO is to maintain a counter of how many words are
   in the FIFO.  The write side increments the counter, the read side
   decrements the counter.  This is a problem in our case, as the read and
   write operations are performed by unsynhcronised processes (DSP and PC),
   thus the possibility exists of both sides writing to the counter at the
   same time.

   The alternative used here is to use the position of df->pwr and df->prd to
   determine how full the FIFO is.  Only the write side modifies df->pwr and only
   the read side modifies df->prd, so no contentions will occur.

   One problem with using df->pwr and df->prd is the case of df->pwr = df->prd.  This could
   occur when the FIFO is empty or when the FIFO is full.  This is resolved
   by only allowing df->pwr = df->prd when the FIFO is empty.  When writing to the
   FIFO the df->pwr ptr is only allowed to get up to df->prd-1.  Thus the actual
   capacity of the FIFO is size-1.
*/

static int CheckDspFifo(PDSP_FIFO df);

/*--------------------------------------------------------------------------*\

								FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoOpen
	AUTHOR...: David Rowe
	DATE.....: 6/10/98

	Opens a link to a DSP FIFO.  Assumes DSP has been booted with 
	appropriate code.

\*--------------------------------------------------------------------------*/

void DspFifoOpen(PDSP_FIFO df, ushort bd, ushort dspfifo)
//	PDSP_FIFO df;			ptr to state variables for this dsp fifo
//  ushort    bd;			VPB df->board number for FIFO
//  ushort    dspfido;		addr of FIFO structure in DSP		 
{

    // validate arguments 

	assert(bd < MAX_VPB);
	assert(df != NULL);

    // init DSP fifo member variables 

    df->board = bd;
    df->adspfifo = dspfifo;
	
    // upload DSP FIFO parameter structure 

    HipReadDspSram(df->board, df->adspfifo, FIFO_SIZE, &df->pstart);
    assert(CheckDspFifo(df) == OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoClose()
	AUTHOR...: David Rowe
	DATE.....: 6/10/98

	Closes the connection between the DSP FIFO and the PC.

\*--------------------------------------------------------------------------*/

void DspFifoClose(PDSP_FIFO df)
{
	// do nothing !
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoWrite
	AUTHOR...: David Rowe
	DATE.....: 6/10/98

	Writes a block of words from the PC to a DSP FIFO.  Returns OK
	for success.

\*--------------------------------------------------------------------------*/

int DspFifoWrite(PDSP_FIFO df, word *buf, ushort sz)
//	PDSP_FIFO	df;		ptr to state variables for this dsp fifo
//  word		*buf;	buf of words to write to DSP FIFO 	
//  ushort		sz;		size of buf				
{	
    ushort  words_used;	// used space in FIFO			
    ushort  words_free;	// free space in FIFO			
    ushort  copy_first;	// size of first block move		
    ushort  new_pwr;	// modified df->pwr	 	   		

    // validate arguments 

	assert(df != NULL);
    assert(buf != NULL);

    // update PC's copy of DSP FIFO state variables 

    HipReadDspSram(df->board, df->adspfifo+PWR, 1, &df->pwr);
    HipReadDspSram(df->board, df->adspfifo+PRD, 1, &df->prd);

	assert(CheckDspFifo(df) == OK);

    // determine if there is enough room in FIFO for buf 

    if (df->pwr >= df->prd)
		words_used = df->pwr - df->prd;
    if (df->prd > df->pwr)
		words_used = df->size - (df->prd - df->pwr);
    words_free = df->size - words_used - 1;

    if (words_free < sz) {
		return(DSPFIFO_FULL);
	}

    // If buf overlaps end of linear array split into two block moves 

    if ((df->pwr + sz) > df->pend) {
		copy_first = (df->pend-df->pwr);

		HipWriteDspSram(df->board, df->pwr, copy_first, buf);
		HipWriteDspSram(df->board, df->pstart, sz-copy_first,&buf[copy_first]);
    }
    else
		HipWriteDspSram(df->board, df->pwr, sz, buf);

    // increment df->pwr and wrap around if required 

    new_pwr = df->pwr + sz;
    if (new_pwr >= df->pend)
		df->pwr = df->pstart + (new_pwr - df->pend);
    else
		df->pwr = new_pwr;

    // copy df->pwr back to DSP 

    HipWriteDspSram(df->board, df->adspfifo+PWR, 1, &df->pwr);
    assert(CheckDspFifo(df) == OK);

    return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoRead
	AUTHOR...: David Rowe
	DATE.....: 6/10/98

	Reads a block of words from a DSP FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int DspFifoRead(PDSP_FIFO df, word *buf, ushort sz)
//  word    *buf;		buffer of words to read from DSP FIFO 	
//  ushort  sz;			size of buf				
{
    ushort  words_used;	// used space in FIFO			
    ushort  copy_first;	// size of first block move		
    ushort  new_prd;	// modified df->prd	    		

    // validate arguments 

	assert(df != NULL);
    assert(buf != NULL);

    // update PC's copy of DSP FIFO state variables 

    HipReadDspSram(df->board, df->adspfifo+PWR, 1, &df->pwr);
    HipReadDspSram(df->board, df->adspfifo+PRD, 1, &df->prd);
    assert(CheckDspFifo(df) == OK);

    // determine if there is data in FIFO to fill buf 

    if (df->pwr >= df->prd)
		words_used = df->pwr - df->prd;
    if (df->prd > df->pwr)
		words_used = df->size - (df->prd - df->pwr);
    if (words_used < sz) {
		return(DSPFIFO_EMPTY);
	}

    // If buf overlaps end of linear array split into two block moves 

    if ((df->prd + sz) > df->pend) {
		copy_first = (df->pend-df->prd);

		HipReadDspSram(df->board, df->prd, copy_first, buf);
		HipReadDspSram(df->board, df->pstart, sz-copy_first,&buf[copy_first]);
    }
    else
		HipReadDspSram(df->board, df->prd, sz, buf);

    // increment df->prd and wrap around if required 

    new_prd = df->prd + sz;
    if (new_prd >= df->pend)
		df->prd = df->pstart + (new_prd - df->pend);
    else
		df->prd = new_prd;

    // copy df->prd back to DSP 

    HipWriteDspSram(df->board, df->adspfifo+PRD, 1l, &df->prd);
    assert(CheckDspFifo(df) == OK);

    return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoHowFull
	AUTHOR...: David Rowe
	DATE.....: 6/10/98

	Returns the number of words used in the DSP FIFO.

\*--------------------------------------------------------------------------*/

ushort DspFifoHowFull(PDSP_FIFO df)
{
    ushort words;

	assert(df != NULL);

    HipReadDspSram(df->board, df->adspfifo+PWR, 1, &df->pwr);
    HipReadDspSram(df->board, df->adspfifo+PRD, 1, &df->prd);
	
    assert(CheckDspFifo(df) == OK);
    if (df->pwr >= df->prd)
		words = df->pwr - df->prd;
    if (df->prd > df->pwr)
		words = df->size - (df->prd - df->pwr);

	return(words);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoHowEmpty
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Returns the number of words used in the DSP FIFO.

\*--------------------------------------------------------------------------*/

ushort DspFifoHowEmpty(PDSP_FIFO df)
{
    ushort words;

	assert(df != NULL);

    HipReadDspSram(df->board, df->adspfifo+PWR, 1, &df->pwr);
    HipReadDspSram(df->board, df->adspfifo+PRD, 1, &df->prd);
	
    assert(CheckDspFifo(df) == OK);
    if (df->pwr >= df->prd)
		words = df->pwr - df->prd;
    if (df->prd > df->pwr)
		words = df->size - (df->prd - df->pwr);

	return((df->size-1) - words);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: CheckDspFifo
    AUTHOR...: David Rowe
	DATE.....: 6/10/98

    Performs a few sanity checks on DSP FIFO structure.  Returns OK is DSP
    FIFO checks out returns.

\*-------------------------------------------------------------------------*/

static int CheckDspFifo(PDSP_FIFO df)
{
 	assert(df != NULL);
    
	if (df->pend < df->pstart) assert(0);
    if (df->pwr < df->pstart) assert(0);
    if (df->pwr > df->pend) assert(0);
    if (df->prd < df->pstart) assert(0);
    if (df->prd > df->pend) assert(0);

    return(OK);
}
