/*--------------------------------------------------------------------------*\

    FILE....: FIFO.C
    TYPE....: C Functions
    AUTHOR..: David Rowe
    DATE....: 5/10/98

    Functions used to implement First In First Out (FIFO) queues of 16 bit
    words for NT DD.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/


#include "fifo.h"
#include <ntddk.h>
#define	assert(x)	ASSERT(x)

/*--------------------------------------------------------------------------*\

				DEFINES

\*--------------------------------------------------------------------------*/

// error code for check_dspfifo 

#define	FAIL		1

#if DBG 
    #define DBGPRINT( args ) DbgPrint args 
#else 
    #define DBGPRINT( args ) 
#endif 

/*--------------------------------------------------------------------------*\

						FUNCTION HEADERS

\*--------------------------------------------------------------------------*/

static int check_dspfifo(PC_FIFO *df);

/*--------------------------------------------------------------------------*\

							FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoCreate
	AUTHOR...: David Rowe
	DATE.....: 5/10/98

	Creates a FIFO, given a pointer to a user mode buffer.  Must be
	called in user mode context (in other words called by device
	drvier dispatch routine).

\*--------------------------------------------------------------------------*/

int FifoCreate(PC_FIFO *fifo, PVOID *buf, ulong size)
//  PC_FIFO pfifo;	ptr to structure containing FIFO state variables 
//  PVOID	*buf;	ptr to user mode buffer (user mode virtual address)
//  ulong   size;	size of FIFO in 16 bit words				 
{
	NTSTATUS	Status;

    assert(fifo != NULL);
    assert(buf != NULL);

	// Create Memory Descriptor List for FIFO buffer

	fifo->pMdl = IoAllocateMdl(
					buf,			// user mode (virtual) address of fifo buffer
					sizeof(short)*size,
					FALSE,
					FALSE,
					NULL
				 );

	if (fifo->pMdl == NULL)
		return (FIFO_ERROR);

	// Lock FIFO buffer into physical memory

    try { 
		// MmProbeAndLockPages generates an exception if it fails

		MmProbeAndLockPages(
			fifo->pMdl,
			KernelMode,
			IoModifyAccess
		);
	}
	
	except( EXCEPTION_EXECUTE_HANDLER ) { 
        Status = GetExceptionCode(); 
        DBGPRINT(( "VPB.SYS: Exception %08X after MmProbeandLockPages for Fifo buffer\n", Status )); 
		return FIFO_ERROR;
	} 

	// map fifo states to system address

	fifo->pstart = MmGetSystemAddressForMdl(fifo->pMdl);
    assert(fifo->pstart != NULL);
	fifo->locked = 1;

    // init rest of fifo structure

	fifo->pend = fifo->pstart + size;
    fifo->pwr = fifo->pstart;
    fifo->prd = fifo->pstart;
    fifo->size = size;
    assert(check_dspfifo(fifo) == FIFO_OK);

	return FIFO_OK;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoDestroy
	AUTHOR...: David Rowe
	DATE.....: 5/10/98

	Destroys a previously created FIFO.  If this fifo was never created,
	then the locked parameter causes this function to do nothing.

\*--------------------------------------------------------------------------*/

void FifoDestroy(PC_FIFO *fifo)
//  void   *pvfifo;	ptr to structure containing FIFO state variables 
{
    assert(fifo != NULL);

	if (fifo->locked) {
		assert(check_dspfifo(fifo) == FIFO_OK);
		MmUnlockPages(fifo->pMdl);
		IoFreeMdl(fifo->pMdl);
		fifo->locked = 0;
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoWrite
	AUTHOR...: David Rowe
	DATE.....: 5/10/98

	Writes a block of words to a FIFO.  Returns FIFO_OK if successful.

\*--------------------------------------------------------------------------*/

int FifoWrite(PC_FIFO *fifo, word *buf, ushort size)
//  void   *pvfifo;	ptr to structure containing FIFO state variables 
//  word   *buf;	buffer of words to write to FIFO		 
//  ushort size;	size of FIFO in words				 
{
    ulong  words_used;	// used space in FIFO				 
    ulong  words_free;	// free space in FIFO				 
    ulong  copy_first;	// size of first block move			 
    word   *new_pwr;	// modified pwr				 

    // validate arguments 

    assert(fifo != NULL);
    assert(buf != NULL);
    assert(check_dspfifo(fifo) == FIFO_OK);

    // determine if there is data in FIFO to fill buf 

    if (fifo->pwr >= fifo->prd)
		words_used = fifo->pwr - fifo->prd;
    if (fifo->prd > fifo->pwr)
		words_used = fifo->size - (fifo->prd - fifo->pwr);
    words_free = fifo->size - words_used - 1;
    if (words_free < size)
		return(FIFO_FULL);

    // If buf overlaps end of linear array split into two block moves 

    if ((fifo->pwr + size) > fifo->pend) {
		copy_first = (fifo->pend-fifo->pwr);

		memcpy(fifo->pwr, buf, copy_first*sizeof(word));
		memcpy(fifo->pstart, &buf[copy_first], (size-copy_first)*sizeof(word));
    }
    else
		memcpy(fifo->pwr, buf, size*sizeof(word));

    // increment pwr and wrap around if required 

    new_pwr = fifo->pwr + size;
    if (new_pwr >= fifo->pend)
		fifo->pwr = fifo->pstart + (new_pwr - fifo->pend);
    else
		fifo->pwr = new_pwr;

    assert(check_dspfifo(fifo) == FIFO_OK);
    return(FIFO_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoRead
	AUTHOR...: David Rowe
	DATE.....: 5/10/98

	Reads a block of words from a FIFO.  Returns FIFO_OK if successful.

\*--------------------------------------------------------------------------*/

int FifoRead(PC_FIFO *fifo, word *buf, ushort size)
//  void   *pvfifo;	ptr to structure containing FIFO state variables 
//  word   *buf;	buffer of words read from FIFO			 
//  ushort size;	size of FIFO in words				 
{
    ulong  copy_first;	// size of first copy				 
    ulong  words_used;	// used space in FIFO				 
    word   *new_prd;	// modified prd					 

    // validate arguments 

    assert(fifo != NULL);
    assert(buf != NULL);
    assert(check_dspfifo(fifo) == FIFO_OK);

    // determine if there is data in FIFO for buf 

    if (fifo->pwr >= fifo->prd)
		words_used = fifo->pwr - fifo->prd;
    if (fifo->prd > fifo->pwr)
		words_used = fifo->size - (fifo->prd - fifo->pwr);
    if (words_used < size)
		return(FIFO_EMPTY);

    // If buf overlaps end of linear array split into two block moves 

	if ((fifo->prd + size) > fifo->pend) {
		copy_first = (fifo->pend-fifo->prd);

		memcpy(buf, fifo->prd, copy_first*sizeof(word));
		memcpy(&buf[copy_first], fifo->pstart, (size-copy_first)*sizeof(word));
    }
    else
		memcpy(buf, fifo->prd, size*sizeof(word));

    // increment prd and wrap around if required 

    new_prd = fifo->prd + size;
    if (new_prd >= fifo->pend)
		fifo->prd = fifo->pstart + (new_prd - fifo->pend);
    else
		fifo->prd = new_prd;

    assert(check_dspfifo(fifo) == FIFO_OK);
    return(FIFO_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoHowFull
	AUTHOR...: David Rowe
	DATE.....: 5/10/98

	Determines number of used words in FIFO.

\*--------------------------------------------------------------------------*/

ulong FifoHowFull(PC_FIFO *fifo)
//  void   *pvfifo;	ptr to structure containing FIFO state variables 
{
	ulong	words;

    assert(fifo != NULL);

    assert(check_dspfifo(fifo) == FIFO_OK);
    if (fifo->pwr >= fifo->prd)
		words = fifo->pwr - fifo->prd;
    if (fifo->prd > fifo->pwr)
		words = fifo->size - (fifo->prd - fifo->pwr);

	return words;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoHowEmpty
	AUTHOR...: David Rowe
	DATE.....: 8/10/98

	Determines number of used words in FIFO.

\*--------------------------------------------------------------------------*/

ulong FifoHowEmpty(PC_FIFO *fifo)
//  void   *pvfifo;	ptr to structure containing FIFO state variables 
{
	ulong	words;

    assert(fifo != NULL);

    assert(check_dspfifo(fifo) == FIFO_OK);
    if (fifo->pwr >= fifo->prd)
		words = fifo->pwr - fifo->prd;
    if (fifo->prd > fifo->pwr)
		words = fifo->size - (fifo->prd - fifo->pwr);

	return ((fifo->size-1)-words);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: check_dspfifo
    AUTHOR...: David Rowe
    DATE.....: 5/10/98

    Performs a few sanity checks on DSP FIFO structure.  Returns FIFO_OK is DSP
    FIFO checks out returns.

\*-------------------------------------------------------------------------*/

static int check_dspfifo(PC_FIFO *fifo)
//  FIFO  *fifo;		ptr to DSP FIFO information	
{
    if (fifo->pend < fifo->pstart) return(FAIL);
    if (fifo->pwr < fifo->pstart) return(FAIL);
    if (fifo->pwr > fifo->pend) return(FAIL);
    if (fifo->prd < fifo->pstart) return(FAIL);
    if (fifo->prd > fifo->pend) return(FAIL);

    return(FIFO_OK);
}
