/*---------------------------------------------------------------------------*\

    FILE....: Hip.c
    TYPE....: C Module
    AUTHOR..: David Rowe
    DATE....: 2/10/98

    VPB access functions for VPB device driver.  Written to be portable
	across operating systems by using OS specific PORT and ASSERT modules.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include "hip.h"
#include "port.h"		// OS specific port module
#include "assert.h"		// OS specific assert macro
#include <ntddk.h>

/*-------------------------------------------------------------------------*\

							DEFINES

\*-------------------------------------------------------------------------*/

// general

#define	SPAN		4		// number of I/O ports used 	
#define	NO_VPB		0xffff	// indicates no valid VPB		

// HIP registers 

#define	DATA		0x0		// offset of data register		
#define	CNTRL		0x1		// offset of control register	
#define	ADDRLO		0x2		// low byte of address register	
#define	ADDRHI		0x3		// hi byte of address register	

// Control register bits 

#define	RESET		0x1		// assert to place DSP in reset	
#define	CNTEN		0x2		// assert to enable addr cntr	
#define UP			0x4		// ==1 for up count, else down	

/*-------------------------------------------------------------------------*\

							STATICS

\*-------------------------------------------------------------------------*/

static ushort	num_vpb;			// number of VPB cards init-ed		
static ushort	cntrl[MAX_VPB];		// mirror of VPB control ports		
static ushort	pipmask[MAX_VPB];	// current pip enable mask

static KSPIN_LOCK	LockHip;		// guards access to HIP
static KIRQL		IrqlnHip;		// state variable for HIP spin lock

/*-------------------------------------------------------------------------*\

							FUNCTIONS

\*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

	FUNCTION.: HipOpen
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Initialises Hip states and lower level device driver software.

\*-------------------------------------------------------------------------*/

void HipOpen(void *Context)
{
	port_open(Context);
	num_vpb = 0;
	KeInitializeSpinLock(&LockHip);
}

/*-------------------------------------------------------------------------*\

	FUNCTION.: HipClose
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Closes down the HIP software and lower level drivers.

\*-------------------------------------------------------------------------*/

void HipClose(void *Context)
//	void	*Context		Context specific info passed to port module
{
	int i;

    for(i=0; i<num_vpb; i++) {
		cntrl[i] = RESET;
		port_write(i, CNTRL, cntrl[i]);
    }

	port_close(Context);
	num_vpb = 0;
}

/*-------------------------------------------------------------------------*\

	FUNCTION.: InitVpb
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Inits a HIP for a specific VPB card. Must be called before any operations are 
	performed on the VPB.  Returns VPB_OK if OK.

\*-------------------------------------------------------------------------*/

int HipInitVpb(void *Context, ushort base)
//	void   *PortContext;	Context specific info passed to port module
//  ushort base;			base address of card
{
	int	status;
	
	// check range of base address

    assert((base >= START_BASE) && (base < END_BASE));

    // now tell port about new board and reset board

    status = port_add_board(Context,base, SPAN);
    if (status == PORT_OK) {
		cntrl[num_vpb] = 0;
		pipmask[num_vpb] = 0;
		port_write(num_vpb, CNTRL, cntrl[num_vpb]);
		num_vpb++;
	}

	if (status == PORT_OK)
		return HIP_OK;
	else
		return HIP_FAIL;
}

/*-------------------------------------------------------------------------*\

	FUNCTION.: DspReset
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    asserts the reset line of the DSP, freezing code execution on the DSP.

\*-------------------------------------------------------------------------*/

void HipDspReset(ushort board)
//  ushort board;	VPB board number
{
	assert(board < num_vpb);
	KeAcquireSpinLock(&LockHip, &IrqlnHip);
    cntrl[board] |= RESET;
    port_write(board, CNTRL, cntrl[board]);
	KeReleaseSpinLock(&LockHip, IrqlnHip);
}

/*-------------------------------------------------------------------------*\

	FUNCTION.: DspRun
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    De-asserts the reset line of the DSP, starting code execution on the DSP.

\*-------------------------------------------------------------------------*/

void HipDspRun(ushort board)
//  ushort board;	VPB board number
{
 	assert(board < num_vpb);
	KeAcquireSpinLock(&LockHip, &IrqlnHip);
    cntrl[board] &= ~RESET;
    port_write(board, CNTRL, cntrl[board]);
	KeReleaseSpinLock(&LockHip, IrqlnHip);
}

/*-------------------------------------------------------------------------*\
	
	FUNCTION.: WriteDspSram
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Writes a block of 16bit words to the DSPs SRAM.  The length and address
    values are both specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void HipWriteDspSram(ushort board, ushort addr, ushort length, word *buf)
//  ushort board;	VPB board number
//  ushort addr;	start of block in DSP SRAM	
//  ushort length;	length of block		
//  word   *buf;	ptr to block in PC	
{
    // validate arguments 

	assert(board < num_vpb);
    assert(addr < MAX_LENGTH);
    assert(length < MAX_LENGTH);
    assert(buf != NULL);

    // set up for transfer 

 	KeAcquireSpinLock(&LockHip, &IrqlnHip);
    cntrl[board] |= CNTEN;
    cntrl[board] |= UP;
    port_write(board, CNTRL, cntrl[board]);
    port_write(board, ADDRLO, pipmask[board] + (addr & 0xff));
    port_write(board, ADDRHI, (addr>>8) & 0xff);

    // perform transfer 

    port_block_write(board, DATA, length, buf);
	KeReleaseSpinLock(&LockHip, IrqlnHip);
}

/*-------------------------------------------------------------------------*\
	
	FUNCTION.: Hip::ReadDspSram
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Reads a block of 16bit words from the DSPs SRAM.  The length and address
    values are both specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void HipReadDspSram(ushort board, ushort addr, ushort length, word *buf)
//  ushort board;	VPB board number
//  ushort addr;	start of block in DSP SRAM	
//  ushort length;	length of block		
//  word   *buf;	ptr to block in PC	
{
    // validate arguments

	assert(board < num_vpb);
    assert(addr < MAX_LENGTH);
    assert(length < MAX_LENGTH);
    assert(buf != NULL);

    // set up for transfer 

	KeAcquireSpinLock(&LockHip, &IrqlnHip);
    cntrl[board] |= CNTEN;
    cntrl[board] |= UP;
    port_write(board, CNTRL, cntrl[board]);
    port_write(board, ADDRLO, pipmask[board] + (addr & 0xff));
    port_write(board, ADDRHI, (addr>>8) & 0xff);

    // perform transfer

    port_block_read(board, DATA, length, buf);
	KeReleaseSpinLock(&LockHip, IrqlnHip);
}

/*-------------------------------------------------------------------------*\
	
	FUNCTION.: SetPip
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Sets the pip enable mask for a given board.

\*-------------------------------------------------------------------------*/

void HipSetPip(ushort board, char enables)
//  ushort board;	VPB board number
//  ushort enables;	eight bit mask defining which pip channels are enabled
{
	assert(board < num_vpb);
	KeAcquireSpinLock(&LockHip, &IrqlnHip);
    pipmask[board] = enables<<8;
	KeReleaseSpinLock(&LockHip, IrqlnHip);
}

