/*--------------------------------------------------------------------------*\

    FILE....: PORT.C
    TYPE....: C Module
    AUTHOR..: David Rowe
    DATE....: 2/10/98

	Win NT device driver port module.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <ntddk.h>
#include <string.h>
#include "port.h"

// Statics used to maintain data about allocated blocks of ports

static USHORT	Base[MAX_VPB];			// base port address
static ULONG	Span[MAX_VPB];			// Count of I/O addresses used.
static LONG		blocks;					// number of blocks allocated

static PVOID	MappedBase[MAX_VPB];	// base port address mapped to NT
static BOOLEAN  PortWasMapped[MAX_VPB];	// If TRUE, we have to unmap on unload
static ULONG    PortMemoryType[MAX_VPB];// HalTranslateBusAddress MemoryType

/*-------------------------------------------------------------------------*\

						 FUNCTION HEADERS

\*-------------------------------------------------------------------------*/

PCM_PARTIAL_RESOURCE_DESCRIPTOR
BuildPartialDescriptors(
	IN PCM_PARTIAL_RESOURCE_DESCRIPTOR	Prd,	// current PRD
	IN USHORT							base,
	IN USHORT							span
	);

NTSTATUS AllocateResources(IN PDRIVER_OBJECT DriverObject);

/*-------------------------------------------------------------------------*\

						 FUNCTIONS

\*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

    FUNCTION.: port_open
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

	Opens the port module.

\*-------------------------------------------------------------------------*/

void port_open(void *Context) {
	blocks = 0;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: port_close
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

	Closes the port module, releasing any allocated resources.

\*-------------------------------------------------------------------------*/

void port_close(void *Context)
{
    CM_RESOURCE_LIST	NullResourceList;
    BOOLEAN				ResourceConflict;
	ULONG				i;
	PDRIVER_OBJECT		DriverObject = (PDRIVER_OBJECT)Context;

    // Unmap the ports

    for(i=0; i<blocks; i++) 
		if (PortWasMapped[i])
			MmUnmapIoSpace(MappedBase[i], Span[i]);
	
	// Report we're not using any hardware.  If we don't do this
	// then we'll conflict with ourselves (!) on the next load

	RtlZeroMemory((PVOID)&NullResourceList, sizeof(NullResourceList));

	IoReportResourceUsage(
		NULL,
		DriverObject,
		&NullResourceList,
		sizeof(ULONG),
		NULL,
		NULL,
		0,
		FALSE,
		&ResourceConflict );

	blocks = 0;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: port_add_board
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Allocates the operating resources required to access a VPB, in NT
    this means asking the OS to give us exclusive access to the
    I/O ports required.

\*-------------------------------------------------------------------------*/

int port_add_board(void *Context, ushort base, ushort span)
//  ushort port;	port address
//  word   value;	value to write
{
	PDRIVER_OBJECT		DriverObject = (PDRIVER_OBJECT)Context;
	NTSTATUS			status;

	ASSERT((base >= START_BASE) && (base < END_BASE));

	// add new block info to block list

	Base[blocks] = base;
	Span[blocks] = span;
	blocks++;
	
	// allocate resources from NT

	status = AllocateResources(DriverObject);
	if (NT_SUCCESS(status))
		return PORT_OK;
	else
		return PORT_FAIL;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: port_write
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Writes a single 16-bit word to a 16-bit I/O port.

\*-------------------------------------------------------------------------*/

void port_write(ushort board, ushort offset, USHORT value)
//  ushort	board;	VPB number
//  ushort  offset;	Offset of port from base address of board
//  USHORT	value;	value to write
{
	PUSHORT	pPort;	// address of port to write to

	ASSERT(board < blocks);
	ASSERT(offset < Span[board]);

    pPort = (PUSHORT)((ULONG)MappedBase[board]+offset);
	WRITE_PORT_BUFFER_USHORT( 
				pPort,
				&value,
				1
			);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: port_block_write
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Writes a block of 16-bit words to a 16-bit I/O port.  The length and
    value is specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void port_block_write(ushort board, ushort offset, ushort length, word *buf)
//  ushort	board;	VPB number
//  ushort  offset;	Offset of port from base address of board
//  ushort  length;	number of words (not butes) in buf
//  word	buf[];	buffer to write to VPB
{
	PUSHORT	pPort;	// address of port to write to
	ulong	i;

    // validate arguments 

	ASSERT(board < blocks);
	ASSERT(offset < Span[board]);
	ASSERT(buf != NULL);
    ASSERT(length < MAX_LENGTH);

    // perform block write 

    pPort = (PUSHORT)((ULONG)MappedBase[board]+offset);
	WRITE_PORT_BUFFER_USHORT( 
				pPort,
				buf,
				(ULONG)length
			);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: port_block_read
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Reads a block of 16-bit words from a 16-bit I/O port.  The length and
    value is specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void port_block_read(ushort board, ushort offset, ushort length, word *buf)
//  ushort	board;	VPB number
//  ushort  offset;	Offset of port from base address of board
//  ushort  length;	number of words (not butes) in buf
//  word	buf[];	buffer read from VPB
{
	PUSHORT	pPort;	// address of port to write to
	ulong   i;

    // validate arguments 

	ASSERT(board < blocks);
	ASSERT(offset < Span[board]);
	ASSERT(buf != NULL);
    ASSERT(length < MAX_LENGTH);

    // perform block read 

    pPort = (PUSHORT)((ULONG)MappedBase[board]+offset);
	READ_PORT_BUFFER_USHORT( 
				pPort,
				buf,
				(ULONG)length
			);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: AllocateResources
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Allocates the port resources required by all of the VPB cards supported
	by this driver.

\*-------------------------------------------------------------------------*/

NTSTATUS AllocateResources(IN PDRIVER_OBJECT	DriverObject)
{
	ULONG							ListSize;		
	PCM_RESOURCE_LIST				ResourceList;	
	PCM_FULL_RESOURCE_DESCRIPTOR	Frd;
	PCM_PARTIAL_RESOURCE_DESCRIPTOR Prd;
	BOOLEAN							bConflictDetected;
	NTSTATUS						status;
	ULONG							i;
    PHYSICAL_ADDRESS				MappedAddress;
    ULONG							MemType;
   	PHYSICAL_ADDRESS				PortAddress;

	if ((blocks < 1) || (blocks > MAX_VPB))
		return(STATUS_INVALID_PARAMETER);

	// Allocate memory for resource list -------------------------------------

	// Calculate the size of the resource list 

	ListSize = FIELD_OFFSET( CM_RESOURCE_LIST, List[0] );
	ListSize +=	sizeof( CM_FULL_RESOURCE_DESCRIPTOR );
	ListSize +=	blocks * sizeof( CM_PARTIAL_RESOURCE_DESCRIPTOR );

	// Try and allocate paged memory for the resource list. If it works, 
	// zero out the list.
	
	ResourceList = ExAllocatePool( PagedPool, ListSize );
	if( ResourceList == NULL )
	{
		return STATUS_INSUFFICIENT_RESOURCES;
	}
	
	RtlZeroMemory( ResourceList, ListSize );

	// Enter info into resource list -----------------------------------------
	
	ResourceList->Count = 1;
	Frd = &ResourceList->List[0];
	Frd->InterfaceType = Isa;
	Frd->BusNumber = 0;
	Frd->PartialResourceList.Count = blocks;
	
	Prd = &Frd->PartialResourceList.PartialDescriptors[0];

	// Set up on partial resource descriptor for each block

	for( i=0; i < blocks; i++ ) {
	    if ((Base[i] < START_BASE) || (Base[i] > END_BASE)) {
			ExFreePool( ResourceList );
			return(STATUS_INVALID_PARAMETER);
		}
		Prd = BuildPartialDescriptors(Prd, Base[i], Span[i]);
	}
	
	// Request ownership of resources by DEVICE --------------------------

	status = IoReportResourceUsage(
				NULL,				// Optional class name
				DriverObject,
				ResourceList,
				ListSize,
				NULL,
				NULL,
				0,
				FALSE,				// Don't override
				&bConflictDetected );

	ExFreePool( ResourceList );

	if( !NT_SUCCESS( status ) || bConflictDetected )
		return STATUS_INSUFFICIENT_RESOURCES;
    
	// Now translate address to form NT likes ----------------------

	for(i=0; i<blocks; i++) {

		PortAddress.LowPart = Base[i];
		PortAddress.HighPart = 0;

	    MemType = 1;                        // located in IO space
        HalTranslateBusAddress( Isa,
		                        0,
			    			    PortAddress,
				    			&MemType,
					    		&MappedAddress );

        // Initialize the local driver info for each device object.
    
        if (MemType == 0) {
            // Port is accessed through memory space - so get a virtual address

            PortWasMapped[i] = TRUE;            
            MappedBase[i] = MmMapIoSpace(MappedAddress, Span[i], FALSE);
		    if (MappedBase[i] == NULL) 
			    return STATUS_INSUFFICIENT_RESOURCES;
        }
        else {
            PortWasMapped[i] = FALSE;
            MappedBase[i] = (PVOID)MappedAddress.LowPart;
        }

        PortMemoryType[i] = MemType;
	}


	return STATUS_SUCCESS;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: BuildPartialDescriptors
    AUTHOR...: David Rowe
    DATE.....: 15/11/97

    Sets up a partial descriptor for the current block of I/O ports we wish
	to allocate to the device driver. 

\*-------------------------------------------------------------------------*/

PCM_PARTIAL_RESOURCE_DESCRIPTOR
BuildPartialDescriptors(
	IN PCM_PARTIAL_RESOURCE_DESCRIPTOR	Prd,		// current PRD
	IN USHORT							base,
	IN USHORT							span
	)
{
	PHYSICAL_ADDRESS	PortAddress;

	PortAddress.LowPart = base;
	PortAddress.HighPart = 0;

	Prd->Type = CmResourceTypePort;
	Prd->ShareDisposition = CmResourceShareDriverExclusive;
	Prd->Flags = CM_RESOURCE_PORT_IO;
	Prd->u.Port.Start = PortAddress;
	Prd->u.Port.Length = span;

	// return pointer to next PRD in list

	Prd++;
	return Prd;
}


