/*---------------------------------------------------------------------------*\

    FILE....: RELAYFIFO.C
    TYPE....: C Header File
    AUTHOR..: David Rowe
    DATE....: 7/10/98

    Functions used to implement Kernal mode buffering of the DSP Fifos.
	These fifos relay data from the DSP fifos to a large buffer in PC
	memory that mimics a large "virtual" dsp fifo for the user mode 
	driver software.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <stdlib.h>
#include <ntddk.h>
#include "relayfifo.h"
#include "fifo.h"
#include "dspfifo.h"

/*--------------------------------------------------------------------------*\

									DEFINES

\*--------------------------------------------------------------------------*/

#define	assert(x)	ASSERT(x)		// use NT DDK asserts
#define	MAX_DSPBUF	2000			// max size of DSP FIFO		

/*--------------------------------------------------------------------------*\

								FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: RelayFifoOpen
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Opens a DSP FIFO to PC FIFO link.  Assumes DSP has been booted with 
	appropriate code.

\*--------------------------------------------------------------------------*/

int RelayFifoOpen(PFIFO f, PFIFO_PARAMS fp)
//	PFIFO			f;		ptr to state variables for this relay fifo
//	PFIFO_PARAMS	fp;		ptr to arguments passed from PC
{
	int				ret;
		
	// validate arguments 

	assert(f != NULL);
	assert(fp != NULL);

	// init PC fifo

	assert(fp->ppc != NULL);
	assert(fp->length);

	ret = FifoCreate(&f->PcFifo, fp->ppc, fp->length);
	if (ret != FIFO_OK)
		return(RELAY_FIFO_FAIL);

	// init DSP fifo

	assert(fp->board < MAX_VPB);
	assert(fp->pdsp != 0);

	DspFifoOpen(&f->DspFifo, fp->board, fp->pdsp);

	assert((fp->direction == RELAY_UP) || (fp->direction == RELAY_DOWN));
	f->direction = 	fp->direction;

	return(RELAY_FIFO_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: RelayFifoClose()
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Closes the connection between the DSP FIFO and the PC.

\*--------------------------------------------------------------------------*/

void RelayFifoClose(PFIFO f)
//	PFIFO			f;		ptr to state variables for this relay fifo
{
	assert(f != NULL);
	assert(f->dsponly == 0);

	FifoDestroy(&f->PcFifo);
	DspFifoClose(&f->DspFifo);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: RelayFifoHowFull
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Returns the number of words used in the DSP side of the relay.

\*--------------------------------------------------------------------------*/

int RelayFifoHowFull(PFIFO f)
//	PFIFO			f;		ptr to state variables for this relay fifo
{
    int words;

	assert(f != NULL);

	words = FifoHowFull(&f->DspFifo);
	return(words);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: RelayFifoRelay
    AUTHOR...: David Rowe
	DATE.....: 7/10/98

    Performs the realy of data from a DSP fifo to a PC fifo, depending
	on desired direction of relay.

\*-------------------------------------------------------------------------*/

void RelayFifoRelay(PFIFO f)
{
	ushort		dspbuf[MAX_DSPBUF];
	ushort		dsp_words;
	int			pc_words;
	int			words;

 	assert(f != NULL);

	// DSP only FIFOs dont need relay, user writes to DSP FIFO using
	// DSP FIFO Ioctls

	if (f->dsponly)
		return;
	
	assert((f->direction == RELAY_UP) || (f->direction == RELAY_DOWN));

    if (f->direction == RELAY_UP) {
		// data transferred from the DSP to the PC fifo-----------------

		// First determine size of block to transfer
		// this is the minimum of how full the DSP fifo is and
		// how empty the PC fifo is

		dsp_words = DspFifoHowFull(&f->DspFifo);
		pc_words = FifoHowEmpty(&f->PcFifo);
		words = min(dsp_words, pc_words);

		// relay from DSP to PC fifos
		
		if (dsp_words) {
			DspFifoRead(&f->DspFifo, dspbuf, words);
			FifoWrite(&f->PcFifo, dspbuf, words);
		}
	}
	else {
		// data transferred from the PC to the DSP fifo-----------------

		// First determine size of block to transfer
		// this is the minimum of how full the PC fifo is and
		// how empty the DSP fifo is

		dsp_words = DspFifoHowEmpty(&f->DspFifo);
		pc_words = FifoHowFull(&f->PcFifo);
		words = min(dsp_words, pc_words);

		// relay from PC to DSP fifos
		
		if (dsp_words) {
			FifoRead(&f->PcFifo, dspbuf, words);
			DspFifoWrite(&f->DspFifo, dspbuf, words);
		}
	}

}
