/*---------------------------------------------------------------------------*\

    FILE....: VPB.C
    TYPE....: C Module
    AUTHOR..: David Rowe
    DATE....: 1/10/98

    Windows NT Device Driver for Voicetronix VPB cards.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include "vpb.h"
#include "stdlib.h"
#include "tport.h"
#include "hip.h"
#include "relayfifo.h"

NTSTATUS
DriverEntry(
    IN PDRIVER_OBJECT  DriverObject,
    IN PUNICODE_STRING RegistryPath
    )

/*++

Routine Description:
    This routine is the entry point for the driver.  It is responsible
    for setting the dispatch entry points in the driver object and creating
    the device object.  Any resources such as ports, interrupts and DMA
    channels used must be reported.  A symbolic link must be created between
    the device name and an entry in \DosDevices in order to allow Win32
    applications to open the device.

Arguments:
    
    DriverObject - Pointer to driver object created by the system.

Return Value:

    STATUS_SUCCESS if the driver initialized correctly, otherwise an error
    indicating the reason for failure.

--*/

{
    ULONG PortBase;                 // Port location, in NT's address form.
    ULONG PortCount;                // Count of contiguous I/O ports
    PHYSICAL_ADDRESS PortAddress;

    PLOCAL_DEVICE_INFO pLocalInfo;  // Device extension:
                                    //      local information for each device.
    NTSTATUS Status;
    PDEVICE_OBJECT DeviceObject;

    CM_RESOURCE_LIST ResourceList;  // Resource usage list to report to system
    BOOLEAN ResourceConflict;       // This is set true if our I/O ports
                                    //      conflict with another driver

    // Initialize the driver object dispatch table.
    // NT sends requests to these routines.

    DriverObject->MajorFunction[IRP_MJ_CREATE]          = VpbDispatch;
    DriverObject->MajorFunction[IRP_MJ_CLOSE]           = VpbDispatch;
    DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL]  = VpbDispatch;
    DriverObject->DriverUnload                          = VpbUnload;

    // Create our device.
    Status = VpbCreateDevice(
                    VPB_DEVICE_NAME,
                    VPB_TYPE,
                    DriverObject,
                    &DeviceObject
                    );
    
    // Initialize the local driver info for each device object.
	
	pLocalInfo = (PLOCAL_DEVICE_INFO)DeviceObject->DeviceExtension;
	pLocalInfo->DeviceObject    = DeviceObject;
	pLocalInfo->DeviceType      = VPB_TYPE;
	KeInitializeSpinLock(&pLocalInfo->LocknFifo);
	KeInitializeEvent(&pLocalInfo->DeadThread,
					  NotificationEvent,
					  FALSE);
    
	return Status;
}

NTSTATUS
VpbCreateDevice(
    IN   PWSTR              PrototypeName,
    IN   DEVICE_TYPE        DeviceType,
    IN   PDRIVER_OBJECT     DriverObject,
    OUT  PDEVICE_OBJECT     *ppDevObj
    )

/*++

Routine Description:
    This routine creates the device object and the symbolic link in
    \DosDevices.
    
    Ideally a name derived from a "Prototype", with a number appended at
    the end should be used.  For simplicity, just use the fixed name defined
    in the include file.  This means that only one device can be created.
    
    A symbolic link must be created between the device name and an entry
    in \DosDevices in order to allow Win32 applications to open the device.

Arguments:

    PrototypeName - Name base, # WOULD be appended to this.

    DeviceType - Type of device to create

    DriverObject - Pointer to driver object created by the system.

    ppDevObj - Pointer to place to store pointer to created device object

Return Value:

    STATUS_SUCCESS if the device and link are created correctly, otherwise
    an error indicating the reason for failure.

--*/


{
    NTSTATUS Status;                        // Status of utility calls
    UNICODE_STRING NtDeviceName;
    UNICODE_STRING Win32DeviceName;
    PLOCAL_DEVICE_INFO pLDI;				// Device extension:


    // Get UNICODE name for device.

    RtlInitUnicodeString(&NtDeviceName, PrototypeName);

    Status = IoCreateDevice(					// Create it.
                    DriverObject,
                    sizeof(LOCAL_DEVICE_INFO),
                    &NtDeviceName,
                    DeviceType,
                    0,
                    TRUE,						// Only one thread can call at a time
                    ppDevObj
                    );

    if (!NT_SUCCESS(Status))
        return Status;             // Give up if create failed.

    // Clear local device info memory

    RtlZeroMemory((*ppDevObj)->DeviceExtension, sizeof(LOCAL_DEVICE_INFO));

    //
    // Set up the rest of the device info
    //  These are used for IRP_MJ_READ and IRP_MJ_WRITE which we don't use
    //    
    //  (*ppDevObj)->Flags |= DO_BUFFERED_IO;
    //  (*ppDevObj)->AlignmentRequirement = FILE_BYTE_ALIGNMENT;
    //

    RtlInitUnicodeString(&Win32DeviceName, DOS_DEVICE_NAME);

    Status = IoCreateSymbolicLink( &Win32DeviceName, &NtDeviceName );

    if (!NT_SUCCESS(Status))    // If we we couldn't create the link then
    {                           //  abort installation.
		IoDeleteSymbolicLink(&Win32DeviceName);
        IoDeleteDevice(*ppDevObj);
    }

	return Status;
}

   
NTSTATUS
VpbDispatch(
    IN    PDEVICE_OBJECT pDO,
    IN    PIRP pIrp             
    )

/*++

Routine Description:
    This routine is the dispatch handler for the driver.  It is responsible
    for processing the IRPs.

Arguments:
    
    pDO - Pointer to device object.

    pIrp - Pointer to the current IRP.

Return Value:

    STATUS_SUCCESS if the IRP was processed successfully, otherwise an error
    indicating the reason for failure.

--*/

{
    PLOCAL_DEVICE_INFO	pLDI;
    PIO_STACK_LOCATION	pIrpStack;
    NTSTATUS			Status;
	ULONG				IoControlCode;
	int					i;

    //  Initialize the irp info field.
    //      This is used to return the number of bytes transfered.

    pIrp->IoStatus.Information = 0;
    pLDI = (PLOCAL_DEVICE_INFO)pDO->DeviceExtension;    // Get local info struct

    pIrpStack = IoGetCurrentIrpStackLocation(pIrp);
	IoControlCode = pIrpStack->Parameters.DeviceIoControl.IoControlCode;            //  Dispatch on IOCTL

    //  Set default return status
    Status = STATUS_NOT_IMPLEMENTED;

    // Dispatch based on major fcn code.

    switch (pIrpStack->MajorFunction)
    {
        case IRP_MJ_CREATE:
			// initialise

			VpbSetnFifo(pLDI, 0);
			for(i=0; i<MAX_FIFO; i++)
				pLDI->Fifo[i].PcFifo.locked = 0;

			pLDI->ThreadHandle = 0;
			KeClearEvent(&pLDI->DeadThread);

            Status = STATUS_SUCCESS;
		break;

        case IRP_MJ_CLOSE:
			// kill thread (if we havent already)
			
			KdPrint(("IRP_MJ_CLOSE..."));
			if (pLDI->ThreadHandle != 0) {
				
				// get timeout into large int format

				LARGE_INTEGER 		Interval;
				Interval = RtlConvertLongToLargeInteger(-100l*TIME_SCALE);
				
				pLDI->KillThread = TRUE;

				// wait until we get signal that thread is really dead.  This
				// ensures that we are not trying to kill it at the same time
				// as thread is accessing it

				Status = KeWaitForSingleObject(
							&pLDI->DeadThread,
							Executive,
							KernelMode,
							FALSE,
							&Interval);	
				KeClearEvent(&pLDI->DeadThread);
				KdPrint(("IRP_MJ_CLOSE: Status = %d\n",Status));

			}
			// kill all FIFOs

			for(i=0; i<pLDI->nFifo; i++) {
				FifoDestroy(&pLDI->Fifo[i].PcFifo);		// if DSP fifo "locked" param will cause this func to do nothin
				DspFifoClose(&pLDI->Fifo[i].DspFifo);	// does nothing in this implementation
			}
			VpbSetnFifo(pLDI, 0);

			// kill all port allocations

			HipClose(pDO->DriverObject);

			KdPrint(("IRP_MJ_CLOSE: Ending..."));
            Status = STATUS_SUCCESS;
        break;

        case IRP_MJ_DEVICE_CONTROL:
			switch(IoControlCode) {
				case IOCTL_VPB_TPORT:
					Status = tport_tport(pDO->DriverObject);
				break;
				case IOCTL_VPB_HIP_OPEN:
				case IOCTL_VPB_HIP_CLOSE:
				case IOCTL_VPB_HIP_INIT_VPB:
				case IOCTL_VPB_HIP_DSP_RESET:
				case IOCTL_VPB_HIP_DSP_RUN:
				case IOCTL_VPB_HIP_WRITE_DSP_SRAM:
				case IOCTL_VPB_HIP_READ_DSP_SRAM:
				case IOCTL_VPB_HIP_SET_PIP:
				case IOCTL_VPB_FIFO_CREATE:
				case IOCTL_VPB_FIFO_DESTROY:
				case IOCTL_VPB_FIFO_WRITE:
				case IOCTL_VPB_FIFO_READ:
				case IOCTL_VPB_FIFO_HOW_FULL:
				case IOCTL_VPB_DSP_FIFO_CREATE:
				case IOCTL_VPB_DSP_FIFO_DESTROY:
				case IOCTL_VPB_DSP_FIFO_WRITE:
				case IOCTL_VPB_DSP_FIFO_READ:
				case IOCTL_VPB_DSP_FIFO_HOW_FULL:
				case IOCTL_VPB_RELAY_FIFO_CREATE:
				case IOCTL_VPB_RELAY_FIFO_DESTROY:
				case IOCTL_VPB_RELAY_FIFO_HOW_FULL:
				case IOCTL_VPB_CREATE_THREAD:
				case IOCTL_VPB_TERMINATE_THREAD:
		            Status = VpbProcessIoctl(
                            pLDI,
                            pIrp,
                            pIrpStack,
                            pIrpStack->Parameters.DeviceIoControl.IoControlCode,
							pDO->DriverObject
                            );
				break;
			}

		break;
    }

    // We're done with I/O request.  Record the status of the I/O action.
    pIrp->IoStatus.Status = Status;

    // Don't boost priority when returning since this took little time.
    IoCompleteRequest(pIrp, IO_NO_INCREMENT );

    return Status;
}

NTSTATUS
VpbProcessIoctl(
    IN PLOCAL_DEVICE_INFO pLDI,
    IN PIRP pIrp,
    IN PIO_STACK_LOCATION IrpStack,
    IN ULONG IoctlCode,
	IN PDRIVER_OBJECT DriverObject
)
/*

Return Value:
    STATUS_SUCCESS           -- OK

    STATUS_INVALID_PARAMETER -- The buffer sizes were incorrectfor
								the specified IOCTL
							 --	(INIT_VPB) The ports could not be 
								 allocated

*/
{
    PUCHAR		pIOBuffer;		// IOCTL I/O buffer
    ULONG		InBufferSize;	// size of buffer coming from application
    ULONG		OutBufferSize;	// size of buffer sent to application
	PHIP_PARAMS	php;			// parameters for Hip call
	PFIFO_PARAMS pfifo;			// parameters for FIFO call
	NTSTATUS	status;
	int			ret;
	
    InBufferSize  = IrpStack->Parameters.DeviceIoControl.InputBufferLength;
    OutBufferSize = IrpStack->Parameters.DeviceIoControl.OutputBufferLength;

    // NT copies inbuf here before entry and copies this to outbuf after
    // return, for METHOD_BUFFERED IOCTL's.
    
	pIOBuffer = (PSHORT)pIrp->AssociatedIrp.SystemBuffer;
	php = (PHIP_PARAMS)pIOBuffer;
	pfifo = (PFIFO_PARAMS)pIOBuffer;
	status = STATUS_SUCCESS;
	pIrp->IoStatus.Information = 0;

	switch(IoctlCode) {

		//------------------- HIP ----------------------------------

		case IOCTL_VPB_HIP_OPEN:
			if (InBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			HipOpen(DriverObject);
		break;

		case IOCTL_VPB_HIP_CLOSE:
			if (InBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			HipClose(DriverObject);
		break;

		case IOCTL_VPB_HIP_INIT_VPB:
			if (InBufferSize != sizeof(HIP_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			if (HipInitVpb(DriverObject, php->base) == HIP_OK)
				status = STATUS_SUCCESS;
			else
				status = STATUS_INVALID_PARAMETER;
		break;

		case IOCTL_VPB_HIP_DSP_RESET:
			if (InBufferSize != sizeof(HIP_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			HipDspReset(php->board);
		break;
		
		case IOCTL_VPB_HIP_DSP_RUN:
			if (InBufferSize != sizeof(HIP_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			HipDspRun(php->board);
		break;
		
		case IOCTL_VPB_HIP_WRITE_DSP_SRAM:
			if (InBufferSize != (php->length*sizeof(ushort)+sizeof(HIP_PARAMS)) )
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			HipWriteDspSram(php->board, php->addr, php->length, (word*)(pIOBuffer+sizeof(HIP_PARAMS)));
		break;
		
		case IOCTL_VPB_HIP_READ_DSP_SRAM:
			if (InBufferSize != sizeof(HIP_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != sizeof(ushort)*php->length)
	            return STATUS_INVALID_PARAMETER;
			HipReadDspSram(php->board, php->addr, php->length, (word*)php);
			pIrp->IoStatus.Information = OutBufferSize;
		break;
		
		case IOCTL_VPB_HIP_SET_PIP:
			if (InBufferSize != sizeof(HIP_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			HipSetPip(php->board, php->enables);
		break;

		//------------------- FIFO ----------------------------------

		case IOCTL_VPB_FIFO_CREATE:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;

			ret = FifoCreate(&pLDI->Fifo[pfifo->index].PcFifo, pfifo->ppc, pfifo->length);
			if (ret != FIFO_OK)
				return STATUS_INVALID_PARAMETER;
			VpbSetnFifo(pLDI, VpbGetnFifo(pLDI)+1);
		break;

		case IOCTL_VPB_FIFO_DESTROY:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;
			
			// does nothing, fifos get shut down in IRP_MJ_CLOSE
		break;

		case IOCTL_VPB_FIFO_WRITE:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;

			ret = FifoWrite(&pLDI->Fifo[pfifo->index].PcFifo, (USHORT*)pfifo->ppc, pfifo->length);
			pfifo->ret = ret;
			pIrp->IoStatus.Information = OutBufferSize;
		break;

		case IOCTL_VPB_FIFO_READ:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;

			ret = FifoRead(&pLDI->Fifo[pfifo->index].PcFifo, (USHORT*)pfifo->ppc, pfifo->length);
			pfifo->ret = ret;
			pIrp->IoStatus.Information = OutBufferSize;
		break;

		case IOCTL_VPB_FIFO_HOW_FULL:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;

			ret = FifoHowFull(&pLDI->Fifo[pfifo->index].PcFifo);
			pfifo->ret = ret;
			pIrp->IoStatus.Information = OutBufferSize;
		break;

		//------------------- DSP FIFO ----------------------------------

		case IOCTL_VPB_DSP_FIFO_CREATE:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->board > MAX_VPB)
	            return STATUS_INVALID_PARAMETER;

			DspFifoOpen(&pLDI->Fifo[pfifo->index].DspFifo, pfifo->board, pfifo->pdsp);
			pLDI->Fifo[pfifo->index].dsponly = 1;
			VpbSetnFifo(pLDI, VpbGetnFifo(pLDI)+1);
			KdPrint(("Dsp Fifo Create: index = %d nFifo = %d\n",pfifo->index, pLDI->nFifo));
		break;

		case IOCTL_VPB_DSP_FIFO_DESTROY:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;
			ASSERT(pLDI->Fifo[pfifo->index].dsponly);
			
			// does nothing, dsp fifos get shut down in IRP_MJ_CLOSE
		break;

		case IOCTL_VPB_DSP_FIFO_WRITE:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;
			ASSERT(pLDI->Fifo[pfifo->index].dsponly);
			
			ret = DspFifoWrite(&pLDI->Fifo[pfifo->index].DspFifo, (USHORT*)pfifo->ppc, pfifo->length);
			pfifo->ret = ret;
			pIrp->IoStatus.Information = OutBufferSize;
		break;

		case IOCTL_VPB_DSP_FIFO_READ:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;
			ASSERT(pLDI->Fifo[pfifo->index].dsponly);

			ret = DspFifoRead(&pLDI->Fifo[pfifo->index].DspFifo, (USHORT*)pfifo->ppc, pfifo->length);
			pfifo->ret = ret;
			pIrp->IoStatus.Information = OutBufferSize;
		break;

		case IOCTL_VPB_DSP_FIFO_HOW_FULL:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;
			ASSERT(pLDI->Fifo[pfifo->index].dsponly);

			ret = DspFifoHowFull(&pLDI->Fifo[pfifo->index].DspFifo);
			pfifo->ret = ret;
			pIrp->IoStatus.Information = OutBufferSize;
		break;

		//------------------- RELAY FIFO --------------------------------

		case IOCTL_VPB_RELAY_FIFO_CREATE:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;
			ret = RelayFifoOpen(&pLDI->Fifo[pfifo->index], pfifo);

			if (ret != RELAY_FIFO_OK)
				return STATUS_INVALID_PARAMETER;
			pLDI->Fifo[pfifo->index].dsponly = 0;
			VpbSetnFifo(pLDI, VpbGetnFifo(pLDI)+1);
			KdPrint(("Realy Fifo Create: index = %d nFifo = %d\n",pfifo->index, pLDI->nFifo));
		break;

		case IOCTL_VPB_RELAY_FIFO_DESTROY:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;

			// does nothing, dsp fifos get shut down in IRP_MJ_CLOSE
		break;

		case IOCTL_VPB_RELAY_FIFO_HOW_FULL:
			if (InBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != sizeof(FIFO_PARAMS))
	            return STATUS_INVALID_PARAMETER;
			if (pfifo->index > pLDI->nFifo)
	            return STATUS_INVALID_PARAMETER;

			ret = RelayFifoHowFull(&pLDI->Fifo[pfifo->index]);
			pfifo->ret = ret;
			pIrp->IoStatus.Information = OutBufferSize;
		break;

		// -------------------- Kernel thread -------------------------

		case IOCTL_VPB_CREATE_THREAD:
			if (InBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;

			status = PsCreateSystemThread(
					&pLDI->ThreadHandle,
					(ACCESS_MASK)0,
					NULL,
					(HANDLE)0,
					NULL,
					VpbThreadMain,
					pLDI );
		break;

		case IOCTL_VPB_TERMINATE_THREAD:
			if (InBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;
			if (OutBufferSize != 0)
	            return STATUS_INVALID_PARAMETER;

			pLDI->KillThread = TRUE;
		break;

	}

    return status;
}

VOID
VpbUnload(
    PDRIVER_OBJECT DriverObject
    )

/*++

Routine Description:
    This routine prepares our driver to be unloaded.  It is responsible
    for freeing all resources allocated by DriverEntry as well as any 
    allocated while the driver was running.  The symbolic link must be
    deleted as well.

Arguments:
    
    DriverObject - Pointer to driver object created by the system.

Return Value:

    None

--*/

{
    PLOCAL_DEVICE_INFO pLDI;
    UNICODE_STRING Win32DeviceName;
	int i;

    // Find our global data

    pLDI = (PLOCAL_DEVICE_INFO)DriverObject->DeviceObject->DeviceExtension;

	// Assume all handles are closed down.
	// Delete the things we allocated - devices, symbolic links

	RtlInitUnicodeString(&Win32DeviceName, DOS_DEVICE_NAME);
	IoDeleteSymbolicLink(&Win32DeviceName);
    IoDeleteDevice(pLDI->DeviceObject);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: VpbThreadMain
    AUTHOR...: David Rowe
    DATE.....: 1/10/98

	System thread function, just delays itself and measure tardiness
	until terminated.

\*-------------------------------------------------------------------------*/

VOID VpbThreadMain(
	IN PVOID	Context
	)
{
    PLOCAL_DEVICE_INFO	pLDI = (PLOCAL_DEVICE_INFO) Context;
	LARGE_INTEGER 		Interval,StartTime,FinishTime,ActualInterval;
	TIME_FIELDS			TimeFields;
	int					i;
	ushort				nFifo;
	
	pLDI->KillThread = FALSE;
	
	// construct interval that is 20ms relative (hence -) from when started
	
	Interval = RtlConvertLongToLargeInteger(-20*TIME_SCALE);

	// set thread priority low

	KeSetPriorityThread(
		KeGetCurrentThread(),
		LOW_REALTIME_PRIORITY );

	while(TRUE) {

		// sample system time before and after delay

		KeQuerySystemTime(&StartTime);
		KeDelayExecutionThread(
			KernelMode,
			FALSE,
			&Interval
		);
		KeQuerySystemTime(&FinishTime);
		
		// determine actual delay time

		ActualInterval = RtlLargeIntegerSubtract(FinishTime,StartTime);
		RtlTimeToTimeFields(&ActualInterval,&TimeFields);

		// preserve largest so far

		if (TimeFields.Milliseconds > pLDI->MaxTardiness)
			pLDI->MaxTardiness = TimeFields.Milliseconds;

		// check for kill command (only way out of loop)

		if (pLDI->KillThread) {
			pLDI->KillThread = 0;

			// signal that thread is dead, this way we can be sure two
			// threads arent pissing about with the FIFOs at the same time

			KeSetEvent(&pLDI->DeadThread, 
					   0,					// no priority boost
					   FALSE);				// dont raise IRQL
			PsTerminateSystemThread(STATUS_SUCCESS);			
		}

		nFifo = VpbGetnFifo(pLDI);
		for(i=0; i<nFifo; i++)
			RelayFifoRelay(&pLDI->Fifo[i]);
	}
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: VpbSetnFifo
    AUTHOR...: David Rowe
    DATE.....: 7/10/98

	Sets the nFifo value, guarded by a spin lock.

\*-------------------------------------------------------------------------*/

void VpbSetnFifo(PLOCAL_DEVICE_INFO pLDI, ushort val)
{
	KIRQL	IrqlnFifo;					// state variable for spin lock

	KeAcquireSpinLock(&pLDI->LocknFifo, &IrqlnFifo);
	pLDI->nFifo = val;
	KeReleaseSpinLock(&pLDI->LocknFifo, IrqlnFifo);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: VpbGetnFifo
    AUTHOR...: David Rowe
    DATE.....: 7/10/98

	Gets the nFifo value, guarded by a spin lock.

\*-------------------------------------------------------------------------*/

ushort VpbGetnFifo(PLOCAL_DEVICE_INFO pLDI)
{
	KIRQL	IrqlnFifo;					// state variable for spin lock
	ushort	val;

	KeAcquireSpinLock(&pLDI->LocknFifo, &IrqlnFifo);
	val = pLDI->nFifo;
	KeReleaseSpinLock(&pLDI->LocknFifo, IrqlnFifo);

	return(val);
}
