/*--------------------------------------------------------------------------*\

    FILE....: PORTLINUX.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 22/9/99

    Linux version of low level port I/O module. Interfaces to kernel mode
	driver.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

				 INCLUDES

\*-------------------------------------------------------------------------*/

#include <assert.h>
#include <stdio.h>

#include "port.h"
#include "../generic/generic.h"        
#include "../wobbly/wobbly.h"
#include "ioctllinux.h"

typedef struct {
	ushort	base;		// base address
	ushort	span;		// number of ports
} BLOCK,*PBLOCK;

// Structure that holds a list of blocks.  This structure is used to
// communicate with the device driver via DeviceIoControl.

typedef struct {
	long	blocks;
	BLOCK	block[MAX_VPB];
} BLOCKLIST, *PBLOCKLIST;

/*-------------------------------------------------------------------------*\

				 CLASS

\*-------------------------------------------------------------------------*/

class PortDataLinux {
public:
    static int	exists;
    BLOCKLIST   blockList;  // list of port blocks allocated to driver
  int         fd;         // file handle for driver
 
    PortDataLinux();
    ~PortDataLinux();
    void addBoard(ushort base, ushort span);
};

int PortDataLinux::exists = 0;

/*-------------------------------------------------------------------------*\

				 FUNCTIONS

\*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortLinux::PortLinux
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Initialises PortDataLinux object, performs device driver initialisation 
    (not required for DOS).

\*-------------------------------------------------------------------------*/

PortLinux::PortLinux() {
   if (PortDataLinux::exists)
	    throw Wobbly(PORT_ALREADY_INITIALISED);
    d = new PortDataLinux;
	if (d == 0)
		throw Wobbly(PORT_CANT_ALLOCATE_MEMORY);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: ~PortLinux::PortLinux
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Frees PortDataLinux, closes down device driver if required.

\*-------------------------------------------------------------------------*/

PortLinux::~PortLinux() {
    delete d;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortLinux::addBoard
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Allocates the operating resources required to access a VPB, in NT
    and Win 95 this means asking the OS to give us exclusive access to the
    I/O ports required.

\*-------------------------------------------------------------------------*/

void PortLinux::addBoard(ushort base, ushort span)
//  ushort base;	base address of port block
//  ushort span;	number of ports in block
{
	d->addBoard(base, span);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortLinux::write
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Writes a single 16-bit word to a 16-bit I/O port.

\*-------------------------------------------------------------------------*/

void PortLinux::write(ushort board, ushort offset, word value)
//  ushort	board;	VPB number
//  ushort  offset;	Offset of port from base address of board
//  word	value;	value to write
{
	VPB_DATA vpb_data;
    int ret;

	assert(board < d->blockList.blocks);
	assert(offset < d->blockList.block[board].span);

	vpb_data.port = d->blockList.block[board].base + offset;
	vpb_data.length = 1;
	vpb_data.data = &value;
	ret = Generic_block_write(d->fd, (void*)&vpb_data);
	if (ret == -1)
        throw Wobbly(PORT_WRITE_FAIL);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortLinux::blockWrite
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Writes a block of 16-bit words to a 16-bit I/O port.  The length and
    value is specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void PortLinux::blockWrite(ushort board, ushort offset, ushort length, word *buf)
//  ushort	board;	VPB number
//  ushort  offset;	Offset of port from base address of board
//  ushort  length;	number of words (not butes) in buf
//  word	buf[];	buffer to write to VPB
{
	VPB_DATA vpb_data;
	int ret;

    /* validate arguments */

	assert(board < d->blockList.blocks);
	assert(offset < d->blockList.block[board].span);
	assert(buf != NULL);
    assert(length < MAX_LENGTH);

    /* perform block write */

	ushort port = d->blockList.block[board].base + offset;
	vpb_data.port = d->blockList.block[board].base + offset;
	vpb_data.length = length;
	vpb_data.data = buf;
	ret = Generic_block_write(d->fd, &vpb_data);
	if (ret == -1)
        throw Wobbly(PORT_WRITE_FAIL);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortLinux:blockRead
    AUTHOR...: David Rowe
    DATE.....: 30/9/97

    Reads a block of 16-bit words from a 16-bit I/O port.  The length and
    value is specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void PortLinux::blockRead(ushort board, ushort offset, ushort length, word *buf)
//  ushort	board;	VPB number
//  ushort  offset;	Offset of port from base address of board
//  ushort  length;	number of words (not butes) in buf
//  word	buf[];	buffer read from VPB
{
	VPB_DATA vpb_data;
	int ret;

    /* validate arguments */

	assert(board < d->blockList.blocks);
	assert(offset < d->blockList.block[board].span);
	assert(buf != NULL);
    assert(length < MAX_LENGTH);

    /* perform block read */

	vpb_data.port = d->blockList.block[board].base + offset;
	vpb_data.length = length;
	vpb_data.data = buf;
	ret = Generic_block_read(d->fd, &vpb_data);
	if (ret == -1)
        throw Wobbly(PORT_READ_FAIL);
}


/*-------------------------------------------------------------------------*\

    FUNCTION.: PortDataLinux::PortDataLinux
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    PortDataLinux constructor.

\*-------------------------------------------------------------------------*/

PortDataLinux::PortDataLinux() {
    exists = 1;
	blockList.blocks = 0;
    fd = Generic_open();
    if (fd == -1)
        throw Wobbly(PORT_CANT_OPEN_DEVICE);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: ~PortDataLinux::PortDataLinux
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    PortDataLinux destuctor.

\*-------------------------------------------------------------------------*/

PortDataLinux::~PortDataLinux()
{
	// release port regions
	int i,ret;
	VPB_DATA vpb_data;

	for(i=0; i<blockList.blocks; i++) {
		vpb_data.port = blockList.block[i].base;
		vpb_data.length = blockList.block[i].span;
		vpb_data.data = NULL;
		ret = Generic_remove_board(fd, &vpb_data);
		assert(ret != -1);
	}

	ret = Generic_close(fd);
    if (fd == -1)
        throw Wobbly(PORT_CANT_CLOSE_DEVICE);
    exists = 0;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortDataLinux::addBoard
    AUTHOR...: David Rowe
    DATE.....: 15/11/97

    This version just stores the block information (base and span)
	in the local structure to allow port to operate using board
	and offset specifiers.  Under NT and eventually Win 9x this
	function also calls the low level device driver and allocate
	ports to this appliaction.

\*-------------------------------------------------------------------------*/

void PortDataLinux::addBoard(ushort base, ushort span)
//  ushort port;	port address
//  word   value;	value to write
{
  int ret;
  VPB_DATA vpb_data = {base, NULL, span};

	// add new block info to block list

	assert((base >= START_BASE) && (base < END_BASE));
	blockList.block[blockList.blocks].base = base;
	blockList.block[blockList.blocks].span = span;
	blockList.blocks++;
	assert(blockList.blocks < MAX_VPB);
	ret = Generic_add_board(fd, &vpb_data);
	if (ret == -1)
		throw Wobbly(PORT_ALLOCATE_FAIL);
}

