/*--------------------------------------------------------------------------*\

    FILE....: PORTWIN95.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 14/11/97

    Microsoft C++ DOS/Win 95 version of low level port I/O module. If
	compiled for Win 95, then port allocation is not performed.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

				 INCLUDES

\*-------------------------------------------------------------------------*/

#include <assert.h>
#ifdef WIN32
#include <conio.h>
#else
#include <asm/io.h>
#define	_outp(port, value) outb(value, port)
#define	_outpw(port, value)	outw(value, port)
#define	_inp(port) inb(port) 
#define	_inpw(port) inw(port)
#endif
#include <stdio.h>

#include "port.h"
#include "../generic/generic.h"        
#include "../wobbly/wobbly.h"
#include "gpioctl.h"

/*-------------------------------------------------------------------------*\

				 CLASS

\*-------------------------------------------------------------------------*/

class PortDataWin95 {
    public:
    static int	exists;
	BLOCKLIST	blockList;			// list of port blocks allocated to driver

	PortDataWin95();
    ~PortDataWin95();
	void addBoard(ushort base, ushort span);
};

int PortDataWin95::exists = 0;

/*-------------------------------------------------------------------------*\

				 FUNCTIONS

\*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortWin95::PortWin95
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Initialises PortDataWin95 object, performs device driver initialisation (not
    required for DOS).

\*-------------------------------------------------------------------------*/

PortWin95::PortWin95() {
   if (PortDataWin95::exists)
	    throw Wobbly(PORT_ALREADY_INITIALISED);
    d = new PortDataWin95;
	if (d == NULL)
		throw Wobbly(PORT_CANT_ALLOCATE_MEMORY);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: ~PortWin95::PortWin95
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Frees PortDataWin95, closes down device driver if required.

\*-------------------------------------------------------------------------*/

PortWin95::~PortWin95() {
    delete d;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortWin95::addBoard
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Allocates the operating resources required to access a VPB, in NT
    and Win 95 this means asking the OS to give us exclusive access to the
    I/O ports required.

\*-------------------------------------------------------------------------*/

void PortWin95::addBoard(ushort base, ushort span)
//  ushort base;	base address of port block
//  ushort span;	number of ports in block
{
#ifndef WIN32
	ioperm(base, span, 1);
#endif
	d->addBoard(base, span);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortWin95::write
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Writes a single 16-bit word to a 16-bit I/O port.

\*-------------------------------------------------------------------------*/

void PortWin95::write(ushort board, ushort offset, word value)
//  ushort	board;	VPB number
//  ushort  offset;	Offset of port from base address of board
//  word	value;	value to write
{
	assert(board < d->blockList.blocks);
	assert(offset < d->blockList.block[board].span);

	ushort port = d->blockList.block[board].base + offset;
    if (port % 2)
		_outp(port,value);
	else
		_outpw(port,value);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortWin95::blockWrite
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Writes a block of 16-bit words to a 16-bit I/O port.  The length and
    value is specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void PortWin95::blockWrite(ushort board, ushort offset, ushort length, word *buf)
//  ushort	board;	VPB number
//  ushort  offset;	Offset of port from base address of board
//  ushort  length;	number of words (not butes) in buf
//  word	buf[];	buffer to write to VPB
{
	ulong  i;

    /* validate arguments */

	assert(board < d->blockList.blocks);
	assert(offset < d->blockList.block[board].span);
	assert(buf != NULL);
    assert(length < MAX_LENGTH);

    /* perform block write */

	ushort port = d->blockList.block[board].base + offset;
	for(i=0; i<length; i++)
		_outpw(port,*buf++);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortWin95:blockRead
    AUTHOR...: David Rowe
    DATE.....: 30/9/97

    Reads a block of 16-bit words from a 16-bit I/O port.  The length and
    value is specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void PortWin95::blockRead(ushort board, ushort offset, ushort length, word *buf)
//  ushort	board;	VPB number
//  ushort  offset;	Offset of port from base address of board
//  ushort  length;	number of words (not butes) in buf
//  word	buf[];	buffer read from VPB
{
	ulong  i;

    /* validate arguments */

	assert(board < d->blockList.blocks);
	assert(offset < d->blockList.block[board].span);
	assert(buf != NULL);
    assert(length < MAX_LENGTH);

    /* perform block read */

	ushort port = d->blockList.block[board].base + offset;
	for(i=0; i<length; i++)
		*buf++ = _inpw(port);
}


/*-------------------------------------------------------------------------*\

    FUNCTION.: PortDataWin95::PortDataWin95
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    PortDataWin95 constructor.

\*-------------------------------------------------------------------------*/

PortDataWin95::PortDataWin95() {
    exists = 1;
	blockList.blocks = 0;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: ~PortDataWin95::PortDataWin95
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    PortDataWin95 destuctor.

\*-------------------------------------------------------------------------*/

PortDataWin95::~PortDataWin95()
{
    exists = 0;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortDataWin95::addBoard
    AUTHOR...: David Rowe
    DATE.....: 15/11/97

    This version just stores the block information (base and span)
	in the local structure to allow port to operate using board
	and offset specifiers.  Under NT and eventually Win 9x this
	function also calls the low level device driver and allocate
	ports to this appliaction.

\*-------------------------------------------------------------------------*/

void PortDataWin95::addBoard(ushort base, ushort span)
//  ushort port;	port address
//  word   value;	value to write
{
	// add new block info to block list

	assert((base >= START_BASE) && (base < END_BASE));
	blockList.block[blockList.blocks].base = base;
	blockList.block[blockList.blocks].span = span;
	blockList.blocks++;
	assert(blockList.blocks < MAX_VPB);
}
