/*--------------------------------------------------------------------------*\

    FILE....: PORTWINNT.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 14/11/97

    Mircosoft C++ Win32 version of low level port I/O module.  This
	module is used under windows NT.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

									INCLUDES

\*-------------------------------------------------------------------------*/

#include <assert.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#include "port.h"
#include "../wobbly/wobbly.h"
#include "gpioctl.h"        
#include "../generic/generic.h"        

/*-------------------------------------------------------------------------*\

									 CLASS

\*-------------------------------------------------------------------------*/

class PortDataWinNT {
    public:
    static int	exists;				// asserted if Port/PortDataWinNT object exists
    void		*hndFile;			// Handle to device opened with CreateFile
	BLOCKLIST	blockList;			// list of port blocks allocated to driver

	PortDataWinNT();
    ~PortDataWinNT();
	void addBoard(ushort base, ushort span);
};

int PortDataWinNT::exists = 0;

/*-------------------------------------------------------------------------*\

								 FUNCTIONS

\*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortWinNT::PortWinNT
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Initialises PortDataWinNT object, performs device driver initialisation (not
    required for DOS).

\*-------------------------------------------------------------------------*/

PortWinNT::PortWinNT() {
    if (PortDataWinNT::exists)
	    throw Wobbly(PORT_ALREADY_INITIALISED);
    d = new PortDataWinNT;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: ~PortWinNT::PortWinNT
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Frees PortDataWinNT, closes down device driver if required.

\*-------------------------------------------------------------------------*/

PortWinNT::~PortWinNT() {
    delete d;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortWinNT::addBoard
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Allocates the operating resources required to access a VPB, in NT
    and Win 95 this means asking the OS to give us exclusive access to the
    I/O ports required.

\*-------------------------------------------------------------------------*/

void PortWinNT::addBoard(ushort base, ushort span)
//  ushort port;	port address
//  word   value;	value to write
{
	d->addBoard(base, span);
}


/*-------------------------------------------------------------------------*\

    FUNCTION.: PortWinNT::write
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Writes a single 16-bit word to a 16-bit I/O port.

\*-------------------------------------------------------------------------*/

void PortWinNT::write(ushort board, ushort offset, word value)
//  ushort	board;	VPB number
//  ushort  offset;	Offset of port from base address of board
//  word	value;	value to write
{
	blockWrite(board, offset, 1, &value);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortWinNT::blockWrite
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Writes a block of 16-bit words to a 16-bit I/O port.  The length and
    value is specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void PortWinNT::blockWrite(ushort board, ushort offset, ushort length, word *buf)
//  ushort	board;			VPB number
//  ushort  offset;			Offset of port from base address of board
//  ulong	length;			length of block		
//  word	*buf;			ptr to block in PC	
{
	long    IoctlCode;
    ulong   ReturnedLength; 
	ulong   LengthBytes;	// length of buffer to write including header
    int	IoctlResult;

	// construct buffer concatenating Port header and buf

	ushort *writeBuf = new ushort[PORT+length];
	if (writeBuf == NULL) 
		throw Wobbly(PORT_NEW_FAIL);
	writeBuf[0] = board;
	writeBuf[1] = offset;
	writeBuf[2] = (ushort)length;
	memcpy(&writeBuf[PORT], buf, length*sizeof(ushort));

	// set up for DeviceIoControl() call

    IoctlCode = IOCTL_GPD_WRITE_PORT_USHORT;
    LengthBytes = (PORT+length)*sizeof(ushort);
    IoctlResult = GenericDeviceIoControl(
                            d->hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            writeBuf,	        // Buffer to driver.
                            LengthBytes,		// Length of buffer in bytes.
                            NULL,		        // Buffer from driver.
                            0,			        // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );

    delete [] writeBuf;

	if (!IoctlResult)                            // Did the IOCTL succeed?
        throw Wobbly(PORT_WRITE_FAIL);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortWinNT:blockRead
    AUTHOR...: David Rowe
    DATE.....: 30/9/97

    Reads a block of 16-bit words from a 16-bit I/O port.  The length and
    value is specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void PortWinNT::blockRead(ushort board, ushort offset, ushort length, word *buf)
//  ushort	board;			VPB number
//  ushort  offset;			Offset of port from base address of board
//  ulong  length;			length of block		
//  word   *buf;			ptr to block in PC	
{
	long    IoctlCode;
    ulong   LengthBytes;
    ulong   ReturnedLength; 
    int		IoctlResult;
	ushort	Port[PORT];

	// construct Port descriptor

	Port[0] = board;
	Port[1] = offset;
	Port[2] = length;

	// set up for DeviceIoControl() call

    IoctlCode = IOCTL_GPD_READ_PORT_USHORT;
    LengthBytes = length*sizeof(ushort);
    IoctlResult = GenericDeviceIoControl(
                            d->hndFile,	        // Handle to device
                            IoctlCode,          // IO Control code for Read
                            &Port,		        // Buffer to driver.
                            PORT*sizeof(ushort),// Length of buffer in bytes.
                            buf,		        // Buffer from driver.
                            LengthBytes,        // Length of buffer in bytes.
                            &ReturnedLength,    // Bytes placed in DataBuffer.
                            NULL                // NULL means wait till op. completes.
                            );

	if (!IoctlResult)                            // Did the IOCTL succeed?
        throw Wobbly(PORT_READ_FAIL);
    if (ReturnedLength != LengthBytes)
        throw Wobbly(PORT_READ_FAIL);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortDataWinNT::PortDataWinNT
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    PortDataWinNT constructor.

\*-------------------------------------------------------------------------*/

PortDataWinNT::PortDataWinNT() {
    exists = 1;
	blockList.blocks = 0;
    
	// Open device driver for write access

	hndFile = GenericOpenMk0Driver();

    if (hndFile == INVALID_HANDLE_VALUE)        // Was the device opened?
        throw Wobbly(PORT_CANT_OPEN_DEVICE);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: ~PortDataWinNT::PortDataWinNT
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    PortDataWinNT destuctor.

\*-------------------------------------------------------------------------*/

PortDataWinNT::~PortDataWinNT()
{
    if (!GenericCloseHandle(hndFile))                  
        throw Wobbly(PORT_CANT_CLOSE_DEVICE);
    exists = 0;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortDataWinNT::addBoard
    AUTHOR...: David Rowe
    DATE.....: 15/11/97

    Allocates the operating resources required to access a VPB card, in NT
    and Win 95 this means asking the OS to give us exclusive access to the
    I/O ports required.

\*-------------------------------------------------------------------------*/

void PortDataWinNT::addBoard(ushort base, ushort span)
//  ushort port;	port address
//  word   value;	value to write
{
	// add new block info to block list

	assert((base >= START_BASE) && (base < END_BASE));
	blockList.block[blockList.blocks].base = base;
	blockList.block[blockList.blocks].span = span;
	blockList.blocks++;
	assert(blockList.blocks < MAX_VPB);
    
	// give device driver the new block list

	long                IoctlCode;
    ulong               DataLength;
    ulong               ReturnedLength; 
    int				IoctlResult;

	// set up for DeviceIoControl() call

	IoctlCode = IOCTL_GPD_ALLOCATE_PORTS;
    DataLength = sizeof(BLOCKLIST);
    IoctlResult = GenericDeviceIoControl(
                        hndFile,		        // Handle to device
                        IoctlCode,              // IO Control code for Write
                        &blockList,		        // Buffer to driver.  Holds port & data.
                        DataLength,             // Length of buffer in bytes.
                        NULL,                   // Buffer from driver.   Not used.
                        0,                      // Length of buffer in bytes.
                        &ReturnedLength,        // Bytes placed in outbuf.  Should be 0.
                        NULL                    // NULL means wait till I/O completes.
                        );

    if (!IoctlResult)                            // Did the IOCTL succeed?
        throw Wobbly(PORT_ALLOCATE_FAIL);
}
