/*--------------------------------------------------------------------------*\

    FILE....: VPB.C
    TYPE....: Linux device driver
    AUTHOR..: David Rowe
    DATE....: 22/9/99

    Linux kernel mode device driver for VPB.  Interfaces to PortLinux
    module.  Version 2.2.12 Linux Kernel.  Compile with:

    gcc vpb.c -c -O

    Install with:

    insmod vpb.o (driver wil print out major node number, eg 254)
    mknod /dev/vpb0 c 254 0 (subs 254 for your major node number)

\*--------------------------------------------------------------------------*/

#define	NAME	"vpb"

#define MODULE
#define __KERNEL__

#include <linux/ioport.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include "ioctllinux.h"
#include <asm/segment.h>
#include <asm/uaccess.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <asm/system.h>

/* making these global symbols allows one to override the default values
   in either conf.modules with an "options" statement or when manually
   loading the driver with insmod.  The major, by default, is "0"
   to assure a "dynamic" major number is picked, but can have a fixed
   value specified.
*/

unsigned int major = 0;

static int vpb_ioctl(struct inode *, struct file *, unsigned int, unsigned long);

static struct file_operations vpb_fops={
	NULL,
	NULL,
	NULL, 
	NULL, 
	NULL,
	vpb_ioctl,
};

int init_module(void) {
	major = register_chrdev(major, NAME, &vpb_fops);
	printk("<1>vpb: major = %d\n",major);
	return 0;
}

void cleanup_module(void) {
	int ret;

	ret = unregister_chrdev(major, NAME);
	if (ret < 0)
		printk(KERN_WARNING "unregister_chrdev() failed!\n");
}

static int vpb_ioctl(struct inode *inode,
			  struct file *filp,
			  unsigned int cmd,
			  unsigned long arg)
{
	int ret;
	VPB_DATA vpb_data;
	int port;
	int length;
	short *data;
	int i;

	ret = copy_from_user(&vpb_data, (void*)arg, sizeof(VPB_DATA));
	if (ret != 0) {
	  printk(KERN_CRIT "verify area VPB_DATA failed\n");
	  return -ENODEV;
	}

	port = vpb_data.port;
	length = vpb_data.length;
	data = vpb_data.data;

	switch(cmd) {
		case VPB_IOC_ADD_BOARD:
			ret = check_region(port,length);
			if (ret != 0) {
				return -ENODEV;
			}
		break;
		case VPB_IOC_REMOVE_BOARD:
			release_region(port,length);
		break;
		case VPB_IOC_BLOCK_WRITE:
		        ret = access_ok(VERIFY_READ, data, length);
			if (ret != 1)
				return ret;
	
			if (port % 2) {
				// single byte transfers to odd addresses
				char b;
				__get_user(b,(char*)data);
				outb(b, port);
			}
			else {
			        short w;
				// even address transfers
				for(i=0; i<length; i++) {
				  __get_user(w,&data[i]);
				  outw(w, port);
				}
			}
		break;
		case VPB_IOC_BLOCK_READ:
			if (ret != 0)
				return ret;
	
			for(i=0; i<length; i++)
			  __put_user(inw(port),&data[i]);
		break;
	}
	return 0;
}













