/*---------------------------------------------------------------------------*\

    FILE....: TCTAPI.CPP
    TYPE....: Win32 C++ Console program
    AUTHOR..: David Rowe
    DATE....: 18/6/98

    This program tests the CT API functions using several test modules
	that put each API function through the wringer.

    Link this program with the actual API modules, rather than the DLL
	form of the API to make debugging easier.
  	 
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#ifdef WIN32
#include <crtdbg.h>
#include <conio.h>
#endif
#include <math.h>

#include "../vpbapi/apifunc.h"
#include "../coff/coff.h"
#include "../vpbapi/vpbapi.h"
#include "../vpbapi/verbose.h"
#include "../message/mess.h"
#include "../generic/generic.h"
extern "C" {
#include "../../c50/comp/comp.h"
}

#define	FRAMES	300
#define	N		160
		
// "hidden" API functions useful for VT debug

int WINAPI vpb_comp_load(unsigned short board);
int WINAPI vpb_echo_canc_force_adapt_on();
int WINAPI vpb_echo_canc_force_adapt_off();
int WINAPI vpb_echo_canc_disable();
int WINAPI vpb_echo_canc_enable();
void playrec_diag(unsigned short *play, unsigned short *rec);
void playrec_diag_reset();
void playtone_sync_terminate(void* d);

// local functions

void studio308_off_hook(int h, int max_tries);

#ifdef WIN32
void test_comp_load(void)
{
	int	h;
	verbose(1);

	h = vpb_open(2,4);
	
	vpb_echo_canc_force_adapt_on();
	while(!kbhit()) {
		vpb_comp_load(0);
		GenericSleep(1000);
	}
	vpb_echo_canc_force_adapt_off();

	vpb_close(h);

	if(_CrtDumpMemoryLeaks())
		printf("memory leak");

}

void test_wave(void)
{
	int				i,j;
	void			*win, *wout;
	short			linear[N],pcm[N];
	char			pcm_pack[N];
		
	vpb_wave_open_read(&win,"linear.wav");
	vpb_wave_open_write(&wout, "linearmu.wav", VPB_MULAW);
	vpb_seterrormode(VPB_DEVELOPMENT);

	i = 0;
	while(vpb_wave_read(win, (char*)linear, N*sizeof(short)) == N*sizeof(short)) {
		mulaw_encode((ushort*)pcm, linear, N);
		for(j=0; j<N; j++)
			pcm_pack[j] = (char)(pcm[j]);
		vpb_wave_write(wout, pcm_pack, N);
		printf("frames: %d\r",i++);
	}

	vpb_wave_close_read(win);
	vpb_wave_close_write(wout);

	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
}

// DR 13/11/98
// test routine for tracking down HIP error on VPB4

#define	HIPCH	2

void test_hip_bug(void)
{
	int			h[HIPCH+1];
	VPB_EVENT	e;
	int			finish,ret;
	char		s[VPB_MAX_STR];
	int			i;

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);	

	do {
		for(i=1; i<=HIPCH; i++) {
			h[i] = vpb_open(1,i);
			vpb_play_file_async(h[i], "rx.wav", 0);
		}
		finish = 0;

		// wait for finish or key pressed

		do {
			ret = vpb_get_event_sync(&e, 1000);
			if (ret == VPB_OK) {
				vpb_translate_event(&e, s);
				printf("%s",s);
				if (e.type == VPB_PLAYEND)
					finish = 1;
			}
		} while(!kbhit() && !finish);
	
		for(i=1; i<=HIPCH; i++) {
			vpb_play_terminate(h[i]);
			vpb_close(h[i]);
		}

		// exit test on key pressed

	} while(!kbhit());

	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
}

void test_play_file_async(void)
{
	int			h;
	VPB_EVENT	e;
	int			finish,ret;
	char		s[VPB_MAX_STR];

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);

	h = vpb_open(1,1);
	
	vpb_sethook_sync(h,VPB_OFFHOOK);
	
	do {
		//vpb_play_file_async(h, "linear.wav", 0);
		vpb_play_file_async(h, "c:\\argyle\\main.wav", 0);
		finish = 0;
		do {
			ret = vpb_get_event_sync(&e, 1000);
			if (ret == VPB_OK) {
				vpb_translate_event(&e, s);
				printf("%s",s);
				if (e.type == VPB_PLAYEND)
					finish = 1;
			}
		} while(!kbhit() && !finish);
	} while(!kbhit());

	vpb_play_terminate(h);
	vpb_sethook_sync(h, VPB_ONHOOK);
	vpb_sleep(100);

	vpb_close(h);

	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
}

void test_play_voxfile_async(void)
{
	int			h;
	VPB_EVENT	e;
	int			ret;
	char		s[VPB_MAX_STR];
	int			finish;

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);

	h = vpb_open(1,1);
	
	vpb_sethook_sync(h,VPB_OFFHOOK);
	
	do {
		vpb_play_voxfile_async(h, "k:\\dlgcdev\\dialogic\\samples\\d42\\sample1.vox",VPB_OKIADPCM, 0);
		finish = 0;
		do {
			ret = vpb_get_event_sync(&e, 1000);
			if (ret == VPB_OK) {
				vpb_translate_event(&e, s);
				printf("%s",s);
				if (e.type == VPB_PLAYEND)
					finish = 1;
			}

		} while(!finish && !kbhit());
	} while(!kbhit());

	vpb_play_terminate(h);
	vpb_sethook_sync(h, VPB_ONHOOK);
	vpb_sleep(100);

	vpb_close(h);

	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
}

#endif

void test_play_file_sync(void)
{
	int			h;

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);

	h = vpb_open(1,1);

	printf("after vpb open\n");
	
	vpb_sethook_sync(h,VPB_OFFHOOK);
	printf("after sethook\n");
#ifdef WIN32
	while(!kbhit()) {
#endif
		//vpb_play_file_sync(h, "linear.wav");
		vpb_play_file_sync(h, "alaw.wav");
		//vpb_play_file_sync(h, "mulaw.wav");
		//vpb_play_file_sync(h, "test.wav");
		printf("play\n");
#ifdef WIN32
	}
#endif
	vpb_sethook_sync(h, VPB_ONHOOK);

	vpb_close(h);
#ifdef WIN32
	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
#endif
}

#ifdef WIN32

void test_play_voxfile_sync(void)
{
	int			h;

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);

	h = vpb_open(1,1);
	
	vpb_sethook_sync(h,VPB_OFFHOOK);
	while(!kbhit()) {
		vpb_play_voxfile_sync(h, "adpcm1.vox"/*"k:\\dlgcdev\\dialogic\\samples\\d42\\sample1.vox"*/,VPB_OKIADPCM);
		vpb_sleep(1000);
		printf("finished!\n");
	}
	vpb_sethook_sync(h, VPB_ONHOOK);

	vpb_close(h);
	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
}

void test_record_file_async(void)
{
	int			h;
	VPB_EVENT	e;
	VPB_RECORD	record;
	char		s[VPB_MAX_STR];

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);

	h = vpb_open(1,1);
	record.term_digits = "1";
	record.time_out = 10000;
	vpb_record_set(h, &record);

	// record until key pressed

//	vpb_sethook_sync(h,VPB_OFFHOOK);	 
	vpb_record_file_async(h, "k:\\ivr\\pc\\apps\\vpbapi\\kos.wav", VPB_OKIADPCM);

	// wait for terminate

	do {
		int ret = vpb_get_event_sync(&e, 60000);
		if (ret == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s",s);

			if (e.type == VPB_RECORD_OVERFLOW) {
				vpb_reset_record_fifo_alarm(e.handle);
				printf("reseting ch %d alarm\n",e.handle);
			}
		}


		/*if (kbhit())
			vpb_record_terminate(h);*/

	} while(e.type != VPB_RECORDEND);

	// play back

	vpb_play_file_sync(h, "kos.wav");

	// hangup

//	vpb_sethook_sync(h, VPB_ONHOOK);

	vpb_close(h);

	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
}

void test_record_voxfile_async(void)
{
	int			h;
	VPB_EVENT	e;
	VPB_RECORD	record;
	char		s[VPB_MAX_STR];

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);

	h = vpb_open(1,1);
	
	record.term_digits = "";
	record.time_out = 4000;
	vpb_record_set(h, &record);

	// record for 2 sec

	vpb_sethook_sync(h,VPB_OFFHOOK);
//	vpb_record_voxfile_async(h, "linear.vox", VPB_LINEAR);
//	vpb_record_voxfile_async(h, "alaw.vox", VPB_ALAW);
//	vpb_record_voxfile_async(h, "mulaw.vox", VPB_MULAW);
	vpb_record_voxfile_async(h, "adpcm.vox", VPB_OKIADPCM);
	
	do {
		int ret = vpb_get_event_sync(&e, 1000);
		if (ret == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s",s);
		}

		if (kbhit())
			vpb_record_terminate(h);

	} while(e.type != VPB_RECORDEND);

	// play back

//	vpb_play_voxfile_sync(h, "linear.vox",VPB_LINEAR);
//	vpb_play_voxfile_sync(h, "alaw.vox",VPB_ALAW);
//	vpb_play_voxfile_sync(h, "mulaw.vox",VPB_MULAW);
	vpb_play_voxfile_sync(h, "adpcm.vox",VPB_OKIADPCM);

	// hangup

	vpb_sethook_sync(h, VPB_ONHOOK);
	vpb_sleep(100);

	vpb_close(h);

	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
}

void test_adpcm(void)
{
	int				h;
	int				i;
	FILE			*f;
	char			adpcm[N/2];

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1, 1);
	vpb_sethook_sync(h,VPB_OFFHOOK);

//#define  REC
#ifdef REC
	// record file

	f = fopen("adpcm.bin","wb");
	assert(f);
	vpb_record_buf_start(h,VPB_OKIADPCM);
	
	for(i=0; i<FRAMES; i++) {
		vpb_record_buf_sync(h, adpcm, N/2);
		fwrite(adpcm, sizeof(char), N/2, f);
		printf("frames: %d\r",i);
	}

	vpb_record_buf_finish(h);
	fclose(f);
#endif

	// play back

	short			linear[N];
	unsigned short	nlinear;
	void			*states;

	f = fopen("adpcm.bin","rb");
	vpb_play_buf_start(h,VPB_LINEAR);
	vpb_adpcm_open(&states);
	
	i = 0;
	while(fread(adpcm, sizeof(char), N/2, f) == N/2) {
		vpb_adpcm_decode(states, linear, &nlinear, adpcm, N/2);
		vpb_play_buf_sync(h, (char*)linear, sizeof(short)*N);
		printf("frames: %d\r",i++);
	}

	vpb_play_buf_finish(h);
	vpb_adpcm_close(states);
	fclose(f);

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_close(h);

	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
}

void test_linear(void)
{
	int				h;
	int				i;
	FILE			*f;
	char			linear[N];
		
	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1, 1);
	vpb_sethook_sync(h,VPB_OFFHOOK);

#define  REC
#ifdef REC
	// record file

	f = fopen("linear.wav","wb");
	assert(f);
	vpb_record_buf_start(h,VPB_LINEAR);
	
	for(i=0; i<FRAMES; i++) {
		vpb_record_buf_sync(h, linear, 2*N);
		fwrite(linear, sizeof(char), 2*N, f);
		printf("frames: %d\r",i);
	}

	vpb_record_buf_finish(h);
	fclose(f);
#endif
	// play back

	f = fopen("linear.wav","rb");
	vpb_play_buf_start(h,VPB_LINEAR);
	
	for(i=0; i<FRAMES; i++) {
		fread(linear, sizeof(char), 2*N, f);
		vpb_play_buf_sync(h, linear, 2*N);
		printf("frames: %d\r",i);
	}

	vpb_play_buf_finish(h);
	fclose(f);

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);

	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
}

// DR trying to track down clicks when multiple files played

void test_play_buf_click(void)
{
	int				h;
	int				i;
	char			linear[N];
	void			*wave;
		
	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1, 1);
	vpb_sethook_sync(h,VPB_OFFHOOK);

	// play back

	short *pshort = (short*)linear;
	for(i=0; i<N/2; i++)
		pshort[i] = 0;

	do {
		vpb_play_buf_start(h,VPB_LINEAR);
		vpb_wave_open_read(&wave, "mal.wav");

		while(vpb_wave_read(wave, linear, N) == N) {
			vpb_play_buf_sync(h, linear, N);
		}

		vpb_play_buf_finish(h);
		vpb_wave_close_read(wave);
	} while(!kbhit());

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);

	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
}

void test_vox(void)
{
	int				h;
	VPB_EVENT		e;
	char			s[VPB_MAX_STR];
	
	h = vpb_open(1,1);
	vpb_sethook_sync(h, VPB_OFFHOOK);
 	while(!kbhit()) {
		vpb_get_event_sync(&e, 0);
		vpb_translate_event(&e, s);
		printf("%s",s);
	}
	vpb_sethook_sync(h, VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);	
}

// Try setting and getting lots of tones 

void test_toned() {
	int			h;
	VPB_DETECT	d;
	int			tones = 0;
	VPB_EVENT	e;
	char		s[VPB_MAX_STR];

	vpb_seterrormode(VPB_DEVELOPMENT);
	verbose(1);
	h = vpb_open(1,1);

	while(!kbhit()) {
		vpb_gettonedet(h, 0, &d);
		vpb_settonedet(h, &d);
		printf("tones set: %d\r",tones++);
		if (vpb_get_event_async(&e) == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s",s);
		}
	}

	vpb_close(h);
}

void test_open() {
	int			h;
	int			c=0;

	vpb_seterrormode(VPB_DEVELOPMENT);
	while(!kbhit()) {
		h = vpb_open(1,1);
		vpb_close(h);
		printf("count = %d\r",c++);
	}
}

// Give the DSP lots of messages via the down message q to try
// and upset it!

void test_thrash_down_q() {
	int			h;
	int			c=0;

	vpb_seterrormode(VPB_DEVELOPMENT);
	verbose(1);
	h = vpb_open(1,1);
	
	while(!kbhit()) {
		vpb_sethook_sync(h, VPB_OFFHOOK);
		vpb_sethook_sync(h, VPB_ONHOOK);
		printf("count = %d\r",c++);
	}
	
	vpb_close(h);
}

void horses_vlc_test(void)
{
	int			h1,h2;
	VPB_EVENT	e;
	char		s[VPB_MAX_STR];

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);

	h1 = vpb_open(1,1);
	h2 = vpb_open(1,2);
	vpb_sethook_sync(h1,VPB_ONHOOK);
	vpb_sleep(500);
	studio308_off_hook(h1, 10);
	vpb_dial_async(h1, ",11");

	// wait for ring event on channel 2

	do {
		vpb_get_event_sync(&e, 0);
		vpb_translate_event(&e, s);
		printf("%s",s);
	}
	while((e.type != VPB_RING) && (e.handle != h2)  && !kbhit());

	// take channel 2 off hook

	vpb_sethook_sync(h2,VPB_OFFHOOK);
	printf("Channel 2 off hook and playing test file!");
	vpb_play_file_sync(h1, "mal.wav");
	
	vpb_sethook_sync(h1, VPB_ONHOOK);
	vpb_sethook_sync(h2, VPB_ONHOOK);
	vpb_sleep(100);

	vpb_close(h1);
	vpb_close(h2);

	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
}

// software fix for Nortel Studio 308 seizure problem
// checks for dial tone after attempted off hook, retires if
// no dial tone after a period
// DR 15/7/98

void studio308_off_hook(int h, int max_tries)
{
	VPB_EVENT	e;
	char		s[VPB_MAX_STR];
	int			success;
	int			tries = 0;
	void		*t;
	int			id, time_out;

	id = vpb_timer_get_unique_timer_id();
	vpb_timer_open(&t, h, id, 4000);
	vpb_timer_start(t);
	
	do {
		vpb_sethook_sync(h,VPB_OFFHOOK);

		// wait for dial tone on channel 2

		vpb_get_event_sync(&e, 0);
		vpb_translate_event(&e, s);
		printf("%s",s);

		success = (e.type == VPB_TONEDETECT) && (e.handle == h) && (e.data == VPB_DIAL);
		time_out = (e.type == VPB_TIMEREXP) && (e.handle == h) && (e.data == id);
		
		if (time_out) {
			vpb_sethook_sync(h,VPB_ONHOOK);
			vpb_timer_start(t);
			vpb_sleep(500);
			tries++;
		}

	}  while(!success && (tries < max_tries));

	vpb_timer_close(t);
	if (success)
		printf("studio 308 off hook success after %d tries\n", tries);
}

void test_studio308_off_hook() {
	int			h;
	int			c=0;

	vpb_seterrormode(VPB_DEVELOPMENT);
	verbose(1);
	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(500);
	
	while(!kbhit()) {
		studio308_off_hook(h, 10);
		vpb_sethook_sync(h, VPB_ONHOOK);
		vpb_sleep(3000);
		printf("count = %d\r",c++);
	}
	
	vpb_close(h);
}

// rings a number then echos rx speech to tx
// this has nothing to do with testing the echo canceller!

void test_echo() {
	int			h;
	VPB_EVENT	e;
	char		s[VPB_MAX_STR];
	char		linear[N];

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);

	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(500);
	studio308_off_hook(h, 10);
	vpb_dial_async(h, ",14");

	// wait for dialling to finish

	do {
		vpb_get_event_sync(&e, 0);
		vpb_translate_event(&e, s);
		printf("%s",s);
	}  while((e.type != VPB_DIALEND) && (e.handle != h)  && !kbhit());

	// ok, start record and playback

	vpb_record_buf_start(h,VPB_LINEAR);
	vpb_play_buf_start(h,VPB_LINEAR);

	while(!kbhit()) {
		vpb_record_buf_sync(h, linear, N);
		vpb_play_buf_sync(h, linear, N);
	}

	vpb_record_buf_finish(h);
	vpb_play_buf_finish(h);

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);
}

void WINAPI callback(VPB_EVENT *e, void *context) {
	char		s[VPB_MAX_STR];

	vpb_translate_event(e, s);
	printf("%s",s);
}

void test_event_callback() {
	int			h;

	vpb_seterrormode(VPB_DEVELOPMENT);
	verbose(1);
	h = vpb_open(1,1);
	vpb_set_event_callback(h, callback, (void*)1);

	while(!kbhit()) {
		vpb_sleep(20);
	}
	
	vpb_close(h);
}

void test_rings() 
{
	int				h;
	VPB_EVENT		e;
	char			s[VPB_MAX_STR];
	
	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(2,1);
	vpb_set_ring(h, 1, 500);
	
	while(!kbhit()) {
		vpb_get_event_sync(&e, 0);
		vpb_translate_event(&e, s);
		printf("%s",s);
	}
	vpb_close(h);	
}

void test_get_digits_async() {
	int				h;
	VPB_EVENT		e;
	char			s[VPB_MAX_STR];
	VPB_DIGITS		vd;
	char			buf[VPB_MAX_STR];

	verbose(1);
	vd.term_digits = "#*";
	vd.max_digits = 3;
	vd.digit_time_out = 20000;
	vd.inter_digit_time_out = 10000;

	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_OFFHOOK);
	
	// first test pre-buffering

	printf("press some digits, then press any key\n");
	while(!kbhit());
	getch();

	vpb_get_digits_async(h, &vd, buf);
	
	// now test normal operation until key pressed

	while(!kbhit()) {
		vpb_get_event_sync(&e, 0);
		vpb_translate_event(&e, s);
		printf("%s",s);
		if (e.type == VPB_DIGIT) {
			printf("\nbuf = %s\n",buf);
			vpb_get_digits_async(h, &vd, buf);
		}
	}

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);	
}

static char *term_str[] = {
	"Terminating Digit",
	"Maximum Digits",
	"Time Out",
	"Inter-Digit Time Out",
};

void test_get_digits_sync() {
	int				h;
	VPB_DIGITS		vd;
	char			buf[VPB_MAX_STR];
	int				ret;

	vd.term_digits = "";
	vd.max_digits = 1;
	vd.digit_time_out = 20000;
	vd.inter_digit_time_out = 10000;

	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_OFFHOOK);
	
	// first test pre-buffering

	printf("press some digits, then press any key\n");
	while(!kbhit());
	getch();
	
	// now test normal operation until key pressed

	while(!kbhit()) {
		ret = vpb_get_digits_sync(h, &vd, buf);
		printf("ret code: %s , digit str: '%s'\n",term_str[ret],buf);
	}

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);	
}

void test_terminate_play() {
	int				h;
	VPB_PLAY		play;
	VPB_DIGITS		vd;
	char			buf[VPB_MAX_STR];
	int				ret;

	vd.term_digits = "";
	vd.max_digits = 1;
	vd.digit_time_out = 20000;
	vd.inter_digit_time_out = 10000;

	play.term_digits = "123";

	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sethook_sync(h,VPB_OFFHOOK);
	vpb_play_set(h, &play);
	
	while(!kbhit()) {
		vpb_flush_digits(h);
		vpb_play_file_sync(h,"k:\\ivr\\vpb8ltests\\og.wav");
		ret = vpb_get_digits_sync(h, &vd, buf);
		printf("ret code: %s , digit str: '%s'\n",term_str[ret],buf);
	}
 
	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);	
}

void test_terminate_record_and_play() {
	int				h;
	VPB_PLAY		play;
	VPB_RECORD		record;
	VPB_DIGITS		vd;

	vd.term_digits = "#*";
	vd.max_digits = 1;
	vd.digit_time_out = 20000;
	vd.inter_digit_time_out = 10000;

	play.term_digits = "123";
	record.term_digits = "456";
	record.time_out = 5000;

	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_OFFHOOK);
	vpb_play_set(h, &play);
	vpb_record_set(h, &record);
	
	while(!kbhit()) {
		printf("record, dial 4,5 or 6 to terminate\n");
		vpb_record_file_sync(h,"digterm.wav", VPB_LINEAR);
		printf("play, dial 1,2 or 3 to terminate\n");
		vpb_play_file_sync(h,"digterm.wav");
	}

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);	
}

void test_dial_async() {
	int				h,ret;
	VPB_EVENT		e;
	char			s[VPB_MAX_STR];

	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);
	studio308_off_hook(h, 10);
	vpb_dial_async(h, "0,42272174");
	
	while(!kbhit()) {
		ret = vpb_get_event_sync(&e, 1);
		if (ret == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s",s);
		}
	}
	getch();

	vpb_play_terminate(h);
	printf("Press and key to end\n");
	while(!kbhit());

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);	
}

void test_dial_sync() {
	int				h,ret;
	VPB_EVENT		e;
	char			s[VPB_MAX_STR];

	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);
	studio308_off_hook(h, 10);
	Generic_beginthread(playtone_sync_terminate, 0, (void*)&h);
	vpb_dial_sync(h, "0,42272174");
	
	while(!kbhit()) {
		ret = vpb_get_event_sync(&e, 1);
		if (ret == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s",s);
		}
	}

	vpb_play_terminate(h);
	printf("Press and key to end\n");
	while(!kbhit());

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);	
}

void test_call_async() {
	int				h;
	VPB_EVENT		e;
	char			s[VPB_MAX_STR];
	int				finished = 0;

	vpb_seterrormode(VPB_DEVELOPMENT);
	verbose(1);
	h = vpb_open(1,1);
	//studio308_off_hook(h, 10);
	vpb_sethook_sync(h,VPB_OFFHOOK);
	
	while(!kbhit()) {
		vpb_call_async(h, "11");
		
		do {
			vpb_get_event_sync(&e, 0);
			vpb_translate_event(&e, s);
			printf("%s",s);

			// try again !

			if (e.type == VPB_CALLEND) {
				vpb_sethook_sync(h,VPB_ONHOOK);
				vpb_sleep(500);
				vpb_sethook_sync(h,VPB_OFFHOOK);
				printf("------------------------------------------\n");
			}	
		} while(e.type != VPB_CALLEND);
	}

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);	
}

static char *call_str[] = {
"Connected",
"No Dial Tone",
"No Ring Back",
"Busy",
"No Answer",
"Disconnected",
"Invalid Data"
};

void test_call_sync() {
	int				h,ret;
	VPB_EVENT		e;
	char			s[VPB_MAX_STR];

	//verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_OFFHOOK);
	
	while(!kbhit())  {
		
		// sync call and print result

		ret = vpb_call_sync(h, "11");
		printf("\n->>>>>>>>>>>>Call result sync: %s\n",call_str[ret]);
		
		// list events during call

		printf("Press any key for EVENTS during sync call:\n");
		while(!kbhit());
		getch();
		
		do {
			ret = vpb_get_event_async(&e);
			if (ret == VPB_OK) {
				vpb_translate_event(&e, s);
				printf("\t%s",s);
			}
		} while( ret == VPB_OK);

		// try again !

		printf("------------------------------------------\n");
		vpb_sethook_sync(h,VPB_ONHOOK);
		vpb_sleep(500);
		vpb_sethook_sync(h,VPB_OFFHOOK);
	}

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);	
}

void test_debug_tonedet() {
	int				handle;
	VPB_EVENT		e;
	char			s[VPB_MAX_STR];

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	handle = vpb_open(2,1);
	
	// start logging tone detector data for ringback tone

	studio308_off_hook(handle, 10);
	vpb_debug_tonedet(handle, VPB_BUSY, "logbusy.txt", 30);
	vpb_dial_sync(handle, "13");

	printf("Press any key to hangup\n");
   	while(!kbhit()) {
		vpb_get_event_sync(&e, 0);
		vpb_translate_event(&e, s);
		printf("%s",s);
	}

	getch();

	vpb_sethook_sync(handle, VPB_ONHOOK);

	printf("Press any key to finish\n");
	while(!kbhit());

	vpb_close(handle);	
}

void start_play(void *data) {
	int h = *((int*)data);
	vpb_play_file_async(h,"og.wav", 0);
}

void test_terminate_play_async() {
	int				h;
	int				its = 0;

	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_OFFHOOK);
	
	while(!kbhit()) {
		Generic_beginthread(start_play, 0, (void*)&h);
		vpb_sleep(1000);
		vpb_play_terminate(h);
		its++;
		printf("its: %d\n",its);
	}
 
	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_sleep(100);
	vpb_close(h);	
}

void test_queues_async() {
	int				h1,h2,ret;
	VPB_EVENT		e1,e2;
	char			s[VPB_MAX_STR];

	vpb_seterrormode(VPB_DEVELOPMENT);
	h1 = vpb_open(1,1);
	h2 = vpb_open(1,2);
	vpb_set_event_mask(h1,VPB_MRING);
	vpb_set_event_mask(h2,VPB_MRING);

	while(!kbhit()) {

		// read Q for ch 1

		ret = vpb_get_event_ch_async(h1,&e1);
		if (ret == VPB_OK) {
			vpb_translate_event(&e1, s);
			printf("Ch1: %s",s);
		}

		// read Q for ch 2

		ret = vpb_get_event_ch_async(h2,&e2);
		if (ret == VPB_OK) {
			vpb_translate_event(&e2, s);
			printf("Ch2: %s",s);
		}

		vpb_sleep(100);
	}
 
	vpb_close(h1);	
	vpb_close(h2);	
}

int stop;

void events(void *data) {
	int h = *(int *)data;
	VPB_EVENT		e;
	char			s[VPB_MAX_STR];

	while(!stop) {
		vpb_get_event_ch_sync(h,&e, 0);
		vpb_translate_event(&e, s);
		printf("h[%d] %s",h,s);
	}
}

void test_queues_sync() {
	int				h1,h2;
	VPB_EVENT		e;

	vpb_seterrormode(VPB_DEVELOPMENT);
	h1 = vpb_open(1,1);
	h2 = vpb_open(1,2);
	vpb_set_event_mask(h1,VPB_MRING);
	vpb_set_event_mask(h2,VPB_MRING);

	stop = 0;
	Generic_beginthread(events, 0, (void*)&h1);
	Generic_beginthread(events, 0, (void*)&h2);

	while(!kbhit());
	
	// send dummy events to make sync get evt functions exit

	stop = 1;
	e.data = 0; e.type = VPB_RING;
	e.handle = h1;
	vpb_put_event(&e);
	e.handle = h2;
	vpb_put_event(&e);
	
	vpb_sleep(500);		// ensure threads have exited
	
	vpb_close(h1);	
	vpb_close(h2);	
}

void test_set_call() {
	int				h;
	VPB_CALL		call;

	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);

	vpb_get_call(h, &call);
	vpb_set_call(h, &call);

	vpb_close(h);	
}

#define	LENGTH	1000			// length of buffers to play
#define	AMP		2000
#define	F		1000
#define	FS		8000
#define	TWO_PI	6.283185

void test_play_buf(void)
{
	int				h;
	int				i;
	char			*buf;		// buffer of samples to play
	int				buflen;		// length of buffer in chars
		
	// construct buffer

	buflen = LENGTH*sizeof(short);
	buf = new char[buflen];
	short *sbuf = (short*)buf;
	for(i=0; i<LENGTH; i++)
		sbuf[i] = (short)(AMP*cos(i*TWO_PI*F/FS));
	
	// play until key pressed

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1, 1);
	vpb_sethook_sync(h,VPB_OFFHOOK);

	vpb_play_buf_start(h, VPB_LINEAR);
	while(!kbhit())
		vpb_play_buf_sync(h, buf, buflen);
	vpb_play_buf_finish(h);

	vpb_sethook_sync(h, VPB_ONHOOK);
	vpb_close(h);

	delete [] buf;
}

void test_play_gain() {
	int				h;
	float			play_gain;

	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);

	vpb_sethook_async(h, VPB_OFFHOOK);
	vpb_play_get_gain(h,&play_gain);
	vpb_play_set_gain(h, (float)(-12.0));
	vpb_play_get_gain(h,&play_gain);
	vpb_play_file_async(h, "og.wav", 0);
	while(!kbhit())
		vpb_sleep(20);
	vpb_play_terminate(h);
	vpb_sethook_async(h, VPB_ONHOOK);

	vpb_close(h);	
}

void test_record_gain() {
	int				h;
	float			rec_gain;

	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);

	vpb_sethook_async(h, VPB_OFFHOOK);
	
	// first record a file

	vpb_record_set_gain(h, (float)(12.0));
	vpb_record_get_gain(h,&rec_gain);
	vpb_record_file_async(h, "mal.wav", VPB_LINEAR);
	while(!kbhit())
		vpb_sleep(20);
	getch();
	vpb_record_terminate(h);

	// now play the file back

	vpb_play_file_async(h, "mal.wav", 0);
	while(!kbhit())
		vpb_sleep(20);
	vpb_play_terminate(h);
	vpb_sethook_async(h, VPB_ONHOOK);

	vpb_close(h);	
}

void test_tone_reg() {
	int			h;
	VPB_DETECT	det;

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);
	vpb_gettonedet(h, VPB_DIAL, &det);
	vpb_tonedet_make_default(&det);
	vpb_close(h);
}

void test_play_all_modes(void)
{
	int			h;

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);

	h = vpb_open(1,1);
	
	vpb_sethook_sync(h,VPB_OFFHOOK);
	while(!kbhit()) {
		vpb_play_voxfile_sync(h, "adpcm1.vox",VPB_OKIADPCM);
		vpb_play_file_sync(h, "alaw.wav");
		vpb_play_file_sync(h, "mulaw.wav");
		vpb_play_file_sync(h, "lin.wav");
		printf("finished!\n");
	}
	vpb_sethook_sync(h, VPB_ONHOOK);

	vpb_close(h);
	if(_CrtDumpMemoryLeaks())
		printf("memory leak");
}

// plays some near end speech while recording, replay the recorded 
// file and check that the near end speech has been attenuated 
// EC can be disabled to check difference, wihich isnt a real lot 
// with a good hybrid/line.

void test_echo_canc() 
{
	int				h;
	int				ret;
	VPB_EVENT		e;
	char			s[VPB_MAX_STR];
	VPB_RECORD		rec;

	rec.time_out = 3000;
	rec.term_digits = "";

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_OFFHOOK);
	vpb_record_set(h,&rec);
	
	//vpb_echo_canc_disable();
	vpb_play_file_async(h,"oga.wav", 0);
	vpb_record_file_async(h,"echo.wav",VPB_MULAW);
//	vpb_record_voxfile_async(h,"echo.vox",VPB_OKIADPCM);

	printf("press any key to end....\n");
	int stop = 0;
	while(!kbhit() && !stop) {
		ret = vpb_get_event_ch_async(h,&e);
		if (ret == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s",s);

			if ((e.type == VPB_RECORDEND) && (e.handle == h))
				stop = 1;
		}
		vpb_sleep(20);
	}
	
	vpb_play_terminate(h);
	vpb_record_terminate(h);
	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_close(h);	
}

void test_load() 
{
	int				h;
	unsigned short	play,rec;

	verbose(1);
	vpb_seterrormode(VPB_DEVELOPMENT);
	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_OFFHOOK);
	
	vpb_play_file_async(h,"og.wav", 0);

	printf("press any key to end....\n");
	while(!kbhit()) {
		playrec_diag(&play, &rec);
		//playrec_diag_reset();
		printf("play = %d  rec = %d\n",play, rec);
	}

	vpb_record_terminate(h);

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_close(h);	
}

// test function for playtone

void playtone_sync_terminate(void* d);

void test_playtone_sync() 
{
	int				h;
	VPB_TONE		tone;

	tone.freq1 = 1000;
	tone.freq2 = 1000;
	tone.freq3 = 1000;
	tone.level1 = -6;
	tone.level2 = -100;
	tone.level3 = -100;
	tone.ton = 10000;
	tone.toff = 1000;

	verbose(1);
	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_OFFHOOK);
	
	Generic_beginthread(playtone_sync_terminate, 0, (void*)&h);
	vpb_playtone_sync(h, &tone);

	//printf("Press any key to finish\n");
	//while(!kbhit());

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_close(h);	
}

void playtone_sync_terminate(void* d) {
	int h = *(int*)d;
	printf("Press any key to terminate\n");
	while(!kbhit());
	getch();
	vpb_play_terminate(h);
}

// test playtone, and early termination

void test_playtone_async() {
	int				h,ret;
	VPB_TONE		tone;
	VPB_EVENT		e;
	char			s[VPB_MAX_STR];

	verbose(1);
	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_OFFHOOK);
	
	tone.freq1 = 1000;
	tone.freq2 = 1000;
	tone.freq3 = 1000;
	tone.level1 = -1;
	tone.level2 = -100;
	tone.level3 = -100;
	tone.ton = 40000;
	tone.toff = 1000;

	vpb_playtone_async(h, &tone);

	int stop = 0;
	while(!kbhit() && !stop) {
		ret = vpb_get_event_ch_async(h,&e);
		if (ret == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s",s);

			if ((e.type == VPB_DIALEND) && (e.handle == h))
				vpb_playtone_async(h, &tone);
	}
		vpb_sleep(20);
	}
	getch();
	
	vpb_play_terminate(h);

	// now see if terminate event posted

	printf("Terminate command given, press any key to finish\n");
	stop = 0;
	while(!kbhit() && !stop) {
		ret = vpb_get_event_ch_async(h,&e);
		if (ret == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s",s);

			if ((e.type == VPB_DIALEND) && (e.handle == h))
				stop = 1;
		}
		vpb_sleep(20);
	}

	while(!kbhit());

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_close(h);	
}

// tests rapid restarting after terimation, should restart immediately

void test_playtone_async_restart() 
{
	int				h,ret;
	VPB_TONE		tone;
	VPB_EVENT		e;
	char			s[VPB_MAX_STR];

	tone.freq1 = 1000;
	tone.freq2 = 1000;
	tone.freq3 = 1000;
	tone.level1 = -20;
	tone.level2 = -100;
	tone.level3 = -100;
	tone.ton = 5000;
	tone.toff = 1000;

	verbose(1);
	h = vpb_open(1,1);
	vpb_sethook_sync(h,VPB_OFFHOOK);
	
	while(!kbhit()) {
		vpb_playtone_async(h, &tone);
		vpb_sleep(500);
		ret = vpb_get_event_ch_async(h,&e);
		if (ret == VPB_OK) {
			vpb_translate_event(&e, s);
			printf("%s",s);
		}
		vpb_play_terminate(h);
	}

	vpb_sethook_sync(h,VPB_ONHOOK);
	vpb_close(h);	
}

void test_get_model() {
	char	s[VPB_MAX_STR];

	vpb_get_model(s);
	printf("model: %s\n",s);
	while(!kbhit());
}
void test_verbose() {
	int h;
	//fp = fopen("happy.txt","wt");
	verbose_file(1,"happy.txt");
	//mprintf("Happy to be with you");
	h=vpb_open(1,1);
	vpb_close(h);
	mess_mprintf_off();
}
#endif

void main() {
	test_play_file_sync();
//	test_play_file_async();
//	test_play_voxfile_sync();
//	test_play_voxfile_async();
//	test_record_file_async();
//	test_adpcm();
//	test_comp_load();
//	test_vox();
//	test_toned();
//	test_linear();
//	test_open();
//	test_thrash_down_q();
//	horses_vlc_test();
//	test_studio308_off_hook();
//	test_echo();
//	test_event_callback();
//	test_rings();
//	test_get_digits_async();
//	test_get_digits_sync();
//	test_terminate_play();
//	test_terminate_record_and_play();
//	test_dial_async();
//	test_dial_sync();
//	test_call_async();
//	test_call_sync();
//	test_debug_tonedet();
//	test_terminate_play_async();
//	test_queues_async();
//	test_queues_sync();
//	test_set_call();
//	test_play_buf();
//	test_play_gain();
//	test_record_gain();	
//	test_tone_reg();
//	test_play_buf_click();
//	test_play_all_modes();
//	test_wave();
//	test_echo_canc();
//	test_load();
//	test_playtone_sync();
//	test_playtone_async();
//	test_playtone_async_restart();
//	test_get_model();
//	test_record_voxfile_async();
//	test_verbose();
//	test_hip_bug();

#ifdef WIN32
	if(_CrtDumpMemoryLeaks())
		printf("\nBummer - memory leak\n");
#endif
}
