/*---------------------------------------------------------------------------*\

    FILE....: DIGBUF.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 22/4/98

	DTMF digit buffer module.
	 
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>

#include "../voicemail/digbuf.h"
#include "../fifo/fifo.h"
#include "../voicemail/vmerrors.h"

typedef struct {
	ushort	numch;
	ushort	size;
	Fifo	**fifos;
} DIGBUF, *PDIGBUF;

typedef Fifo* pFifo;

#define	MAX_CH		256	// maximum number of channels
#define	MAX_SIZE	200	// maximum size of each digit buffer in digits

/*--------------------------------------------------------------------------*\

	FUNCTION.: digbuf_open
	AUTHOR...: David Rowe
	DATE.....: 25/4/98

	Opens the digit buffers.

\*--------------------------------------------------------------------------*/

void *digbuf_open(ushort newnumch, ushort newsize)
{
	int		i;
	PDIGBUF	p;
	
	assert(newnumch < MAX_CH);
	assert(newsize < MAX_SIZE);

	p = new DIGBUF;
	if (p == NULL)
		throw DIGBUF_OUT_OF_MEMORY;

	p->numch = newnumch;
	p->size = newsize;

	p->fifos = new pFifo[p->numch];
	if (p->fifos == NULL)
		throw DIGBUF_OUT_OF_MEMORY;

	for(i=0; i<p->numch; i++)
		p->fifos[i] = new Fifo(p->size);

	return((void*)p);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: digbuf_close
	AUTHOR...: David Rowe
	DATE.....: 25/4/98

	Closes the digit buffers.

\*--------------------------------------------------------------------------*/

void digbuf_close(void *digbuf)
{
	int		i;
	PDIGBUF	p = (PDIGBUF)digbuf;
	
	assert(digbuf != NULL);

	for(i=0; i<p->numch; i++)
		delete p->fifos[i];

	delete [] p->fifos;
	delete p;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: digbuf_clear
	AUTHOR...: David Rowe
	DATE.....: 25/4/98

	Clears a digit buffer.

\*--------------------------------------------------------------------------*/

void digbuf_clear(void *digbuf, ushort chdev)
{
	PDIGBUF	p = (PDIGBUF)digbuf;
	word	digit;
	
	assert(digbuf != NULL);
	assert(chdev < MAX_CH);

	while(p->fifos[chdev]->Read(&digit, 1) == OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: digbuf_write
	AUTHOR...: David Rowe
	DATE.....: 25/4/98

	Writes a digit to a digit buffer.  Does nothing if digit buffer full
	(looses digits).

\*--------------------------------------------------------------------------*/

void digbuf_write(void *digbuf, ushort chdev, ushort digit)
{
	PDIGBUF	p = (PDIGBUF)digbuf;
	
	assert(digbuf != NULL);
	assert(chdev < MAX_CH);
	
	p->fifos[chdev]->Write(&digit, 1);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: digbuf_read
	AUTHOR...: David Rowe
	DATE.....: 25/4/98

	Reads a digit from a digit buffer.  Returns OK if digit obtained,
	DIGBUF_EMPTY otherwise.

\*--------------------------------------------------------------------------*/

int digbuf_read(void *digbuf, ushort chdev, ushort *digit)
{
	PDIGBUF	p = (PDIGBUF)digbuf;
	
	assert(digbuf != NULL);
	assert(chdev < MAX_CH);
	assert(digit != NULL);
	
	if (p->fifos[chdev]->Read(digit, 1) == OK) 
		return(OK);
	else
		return(DIGBUF_EMPTY);
}

