/*--------------------------------------------------------------------------*\

    FILE....: FIFO.CPP
    TYPE....: C++ Functions
    AUTHOR..: David Rowe
    DATE....: 13/5/98

	This version of FIFO doesnt use wobblies as it is designed for use
	above the API layer in th blocks module.  Pity we need another version
	of this, I guess error handling is not portable enough.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include "..\voicemail\fifo.h"
#include <stdlib.h>
#include <string.h>

/*--------------------------------------------------------------------------*\

								DEFINES

\*--------------------------------------------------------------------------*/

#define	FAIL	1		// returned by CheckFifo()

/*--------------------------------------------------------------------------*\

								CLASS

\*--------------------------------------------------------------------------*/

class FifoData {
    word   *pstart;		// first word in FIFO			
    word   *pend;		// one after last word in FIFO		
    word   *pwr;		// write pointer			
    word   *prd; 		// read pointer				
    ushort size;		// total storage in FIFO
public:
	FifoData(ushort size);
	~FifoData();
	int Write(word *buf, ushort size);
	int Read(word *buf, ushort size);
	void HowFull(ushort *words);
	int CheckFifo();
};

/*--------------------------------------------------------------------------*\

							FIFO MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: Fifo::Fifo
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Creates a FIFO.

\*--------------------------------------------------------------------------*/

Fifo::Fifo(ushort sz)
//  ushort sz;		size of FIFO in words				 
{
    d = new FifoData(sz);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Fifo::~Fifo
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Destorys a FIFO.

\*--------------------------------------------------------------------------*/

Fifo::~Fifo()
{
    delete d;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Fifo::Write
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Writes a block of words to a FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int Fifo::Write(word *buf, ushort size)
//  word   *buf;	buffer of words to write to FIFO	
//  ushort size;	size of FIFO in words				 
{
    return(d->Write(buf, size));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Fifo::Read
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Reads a block of words from a FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int Fifo::Read(word *buf, ushort size)
//  word   *buf;	buffer of words read from FIFO		
//  ushort size;	size of FIFO in words				 
{
    return(d->Read(buf, size));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Fifo::HowFull
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Determines number of used words in FIFO.

\*--------------------------------------------------------------------------*/

void Fifo::HowFull(ushort *words)
//  word   *words;	number used words in FIFO			 
{
    d->HowFull(words);
}

/*--------------------------------------------------------------------------*\

						  FIFODATA MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoData::FifoData
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Creates a FIFO.

\*--------------------------------------------------------------------------*/

FifoData::FifoData(ushort sz)
//  ushort sz;		size of FIFO in words				 
{
    pstart = new word[size];
    pend = pstart + size;

    pwr = pstart;
    prd = pstart;
    size = sz;
    assert(CheckFifo() == OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoData::~FifoData
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Destorys a FIFO.

\*--------------------------------------------------------------------------*/

FifoData::~FifoData()
{
    delete [] pstart;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoData::Write
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Writes a block of words to a FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int FifoData::Write(word *buf, ushort sz)
//  word   *buf;	buffer of words to write to FIFO	
//  ushort sz;		size of FIFO in words				 
{
    ushort words_used;	// used space in FIFO		
    ushort words_free;	// free space in FIFO		
    ushort copy_first;	// size of first block move	
    word   *new_pwr;	// modified pwr				 

    // validate arguments

    assert(buf != NULL);
    assert(CheckFifo() == OK);

    // determine if there is data in FIFO to fill buf

    if (pwr >= prd)
		words_used = pwr - prd;
    if (prd > pwr)
		words_used = size - (prd - pwr);
	words_free = size - words_used - 1;
    if (words_free < sz)
		return(FIFO_FULL);

    // If buf overlaps end of linear array split into two block moves 

    if ((pwr + sz) > pend) {
		copy_first = (pend-pwr);

		memcpy(pwr, buf, copy_first*sizeof(word));
		memcpy(pstart, &buf[copy_first], (sz-copy_first)*sizeof(word));
    }
    else
		memcpy(pwr, buf, sz*sizeof(word));

    // increment pwr and wrap around if required 

    new_pwr = pwr + sz;
    if (new_pwr >= pend)
		pwr = pstart + (new_pwr - pend);
    else
		pwr = new_pwr;

    assert(CheckFifo() == OK);
    return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoData::Read
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Reads a block of words from a FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int FifoData::Read(word *buf, ushort sz)
//  word   *buf;	buffer of words read from FIFO		
//  ushort sz;		size of FIFO in words				 
{
    ushort copy_first;	// size of first copy				 
    ushort words_used;	// used space in FIFO				 
    word   *new_prd;	// modified prd					 

    /* validate arguments */

    assert(buf != NULL);
    assert(CheckFifo() == OK);

    // determine if there is enough data in FIFO to fill buf 

    if (pwr >= prd)
		words_used = pwr - prd;
    if (prd > pwr)
		words_used = size - (prd - pwr);
    if (words_used < sz)
		return(FIFO_EMPTY);

    // If buf overlaps end of linear array split into two block moves 

    if ((prd + sz) > pend) {
		copy_first = (pend-prd);

		memcpy(buf, prd, copy_first*sizeof(word));
		memcpy(&buf[copy_first], pstart, (sz-copy_first)*sizeof(word));
    }
    else
		memcpy(buf, prd, sz*sizeof(word));

    /* increment prd and wrap around if required */

    new_prd = prd + sz;
    if (new_prd >= pend)
		prd = pstart + (new_prd - pend);
    else
		prd = new_prd;

    assert(CheckFifo() == OK);
    return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: FifoData::HowFull
	AUTHOR...: David Rowe
	DATE.....: 19/11/97

	Determines number of used words in FIFO.

\*--------------------------------------------------------------------------*/

void FifoData::HowFull(ushort *words)
//  word   *words;	number used words in FIFO			 
{
    assert(words != NULL);

    assert(CheckFifo() == OK);
    if (pwr >= prd)
		*words = pwr - prd;
    if (prd > pwr)
		*words = size - (prd - pwr);
}

/*-------------------------------------------------------------------------*\

	FUNCTION.: FifoData::CheckFifo
    AUTHOR...: David Rowe
	DATE.....: 19/11/97

    Performs a few sanity checks on DSP FIFO structure.  Returns OK is DSP
    FIFO checks out returns.

\*-------------------------------------------------------------------------*/

int FifoData::CheckFifo()
{
    if (pend < pstart) return(FAIL);
    if (pwr < pstart) return(FAIL);
    if (pwr > pend) return(FAIL);
    if (prd < pstart) return(FAIL);
    if (prd > pend) return(FAIL);

    return(OK);
}
