/*---------------------------------------------------------------------------*\

    FILE....: APIFUNC.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 24/6/98

    Some selected helper functions used to implement the API, does
	not correpond one to one with apifunc.h.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include "../vpbapi/apifunc.h"
#include "../vpbapi/vpbapi.h"
#include "../wobbly/wobbly.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static int ErrorMode = VPB_DEVELOPMENT;	// VPB_DEVELOPMENT as default

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_seterrormode()
	AUTHOR...: David Rowe
	DATE.....: 4/3/98

	Helper function that determines the way errors are handled in the
	RunTimeError function below.  See RunTimeError() for more info.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_seterrormode(int mode)
// int	mode	see defines in vpbapi.h
{
	try {
		// validate argument

		if ((mode != VPB_DEVELOPMENT) && (mode != VPB_ERROR_CODE) &&
			(mode != VPB_EXCEPTION)) {
			throw Wobbly(VPBAPI_SETERRORMODE_MODE_INVALID);
		}

		// everything OK

		ErrorMode = mode;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_seterrormode"));
	}
	
	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: VpbException::VpbException()
	AUTHOR...: David Rowe
	DATE.....: 5/3/98

	Constructor for API run time error class.

\*--------------------------------------------------------------------------*/

VpbException::VpbException(int c, char trans[], char api_func[])
// int	code;			error code
// char	trans[];		translated string explaining code
// char	api_function[]	api function where exception occured
{
	code = c;
	strcpy(s, trans);
	strcpy(api_function, api_func);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: RunTimeError()
	AUTHOR...: David Rowe
	DATE.....: 5/3/98

	Helper function that deals with run time errors generated in API
	function call.  There are three error handling modes:
	
	  1. Development: Used by VT engineers when developing.  This mode
		 prints the error message to the screen using mprintf and kills
		 the application.
	  2. Error code: Causes API function to return an error code which
	     application can then check and interpret.
	  3. Exception: causes API function to throw another exception that
	     must be handled by application.  This enables application to
		 write exception handler and handle run time errors more efficiently
		 then checking error codes after every API function call.

\*--------------------------------------------------------------------------*/

int RunTimeError(Wobbly w, char api_function[])
// Wobbly	w				exception object thrown by run time error
// char		api_function[]	api function where exception occured
{
	char s[MAX_STR];
	w.translate(s);
		
	if (ErrorMode == VPB_DEVELOPMENT) {
		printf("exception caught: %s, file: %s line:%d\n",s,w.file,w.line);
        exit(0);
	}
	if (ErrorMode == VPB_ERROR_CODE) {
		return(-w.code);
	}
	if (ErrorMode == VPB_EXCEPTION) {
		throw VpbException(w.code, s, api_function);
	}

	// should never get to here

	assert(0);
	return(0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CheckNew()
	AUTHOR...: David Rowe
	DATE.....: 4/3/98

	Helper function that throws a wobbly if new fails.  Use this instead
	of new in API functions to automatically handle out memory errors.

\*--------------------------------------------------------------------------*/

void CheckNew(void *p)
// void *p	ptr returned by new function
{
	if(p == NULL)
		throw Wobbly(VPBAPI_CANT_ALLOCATE_MEMORY);
}
			
/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_throw_exception()
	AUTHOR...: David Rowe
	DATE.....: 4/8/98

	Helper function that throws a VpbException.

\*--------------------------------------------------------------------------*/

void WINAPI vpb_throw_exception(int c, char trans[], char api_function[])
{
	throw VpbException(c, trans, api_function);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: apifunc_trace()
	AUTHOR...: David Rowe
	DATE.....: 10/10/98

	Prints trace messages to a file.

\*--------------------------------------------------------------------------*/

static FILE *f = NULL;

void WINAPI apifunc_trace(char s[])
{
	// first time around open trace file

	if (f == NULL) {
		f = fopen("trace.txt","wt");
		assert(f != NULL);
	}

	fprintf(f, s);
	fflush(f);
	printf(s);
}

