/*---------------------------------------------------------------------------*\

	FILE....: PIP.CPP
	TYPE....: C++ Module
	AUTHOR..: Davo and Kos
	DATE....: 28/7/98

	Pip generation module for VPB API.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>

#include "../vpbapi/apifunc.h"
#include "../comm/comm.h"
#include "../hip/hip.h"
#include "../vpbapi/pip.h"
#include "../wobbly/wobbly.h"

/*---------------------------------------------------------------------------*\

								DEFINES

\*--------------------------------------------------------------------------*/

#define	VPB8L_CH	8
#define	DEF_WIDTH	425
#define	DEF_PERIOD	15000

// pip generator states

#define	SILENCE		0
#define	PIPON		1

/*---------------------------------------------------------------------------*\

								TYPEDEFS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

								STATICS

\*--------------------------------------------------------------------------*/

static unsigned int *width;	// width of pip pulse for each channel
static unsigned int *period;// period of pip pulse for each channel
static int		*state;		// pip cadence state machine state
static ushort	numch;		// number of channels
static ushort	numbd;		// number of boards
static char		*en;		// enable masks for each board
static char		*pipreg;	// pip register mirrors for each board
static unsigned int *time;	// current time in pip cycle in ms
			
/*---------------------------------------------------------------------------*\

								FUNCTIONS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: pip_open
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Initialises the pip module.

\*--------------------------------------------------------------------------*/

void pip_open(ushort new_numch) {
	int i;

	numch = new_numch;
	numbd = numch / VPB8L_CH;

	width = new unsigned int[new_numch];
	period = new unsigned int[new_numch];
	state = new int[new_numch];
	time = new unsigned int[new_numch];
	for(i=0; i<numch; i++) {
		width[i] = DEF_WIDTH;
		period[i] = DEF_PERIOD;
		state[i] = SILENCE;
		time[i] = 0;
	}

	en = new char[numbd];
	pipreg = new char[numbd];
	for(i=0; i<numbd; i++) {
		en[i] = 0;
		pipreg[i] = 0;
	}
}

/*---------------------------------------------------------------------------*\

	FUNCTION: pip_close
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Closes the pip module.

\*--------------------------------------------------------------------------*/

void pip_close() {
	delete [] en;
	delete [] pipreg;
	delete [] width;
	delete [] period;
	delete [] state;
	delete [] time;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_set_pip
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Sets the pip parameters for all channels.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_set_pip(VPB_PIP *vpb_pip)
{
	int i;

	try {
		for(i=0; i<numch; i++) {
			width[i] = vpb_pip->width;
			period[i] = vpb_pip->period;
		}
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_set_pip"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_set_pip_ch
	AUTHOR..: David Rowe
	DATE....: 9/6/99

	Sets the pip parameters for a specifed channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_set_pip_ch(int handle, VPB_PIP *vpb_pip)
{
	try {
		ValidHandleCheck(handle);
		width[handle] = vpb_pip->width;
		period[handle] = vpb_pip->period;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_set_pip"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_pip
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Gets the pip parameters for the first channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_pip(VPB_PIP *vpb_pip)
{
	try {
		vpb_pip->width = width[0];
		vpb_pip->period = period[0];
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_get_pip"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_pip_ch
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Gets the pip parameters for the specified channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_pip_ch(int handle, VPB_PIP *vpb_pip)
{
	try {
		ValidHandleCheck(handle);
		vpb_pip->width = width[handle];
		vpb_pip->period = period[handle];
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_get_pip"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_pip_on
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Enables pip generation for a channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_pip_on(int handle)
{
	try {
		ValidHandleCheck(handle);
		
		int board = handle / VPB8L_CH;
		int bit = handle % VPB8L_CH;
		en[board] |= 1<<bit;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_pip_on"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_pip_off
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Disables pip generation for a channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_pip_off(int handle)
{
	try {
		ValidHandleCheck(handle);

		int board = handle / VPB8L_CH;
		int bit = handle % VPB8L_CH;
		en[board] &= ~(1<<bit);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_pip_off"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: pip_pip
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Called by the MMQ periodically to manage the pip.

\*--------------------------------------------------------------------------*/

void pip_pip(ushort poll_period)
// ushort	period		interval since this function was last called
{
	int		next_state;
	int		i;
	Hip		*h;
	char	mask;
	int		bd;

	for(i=0; i<numch; i++) {
		time[i] += poll_period;
		next_state = state[i];
		
		switch(state[i]) {
			case SILENCE:
				if (time[i] > (period[i]-width[i])) {
					next_state = PIPON;

					// resolve board from handle
					bd = i / VPB8L_CH;
					h = vpb_c->hip(bd);

					// set pipreg bit if enabled
					mask = 1 << (i % VPB8L_CH);
					mask &= en[bd];
					pipreg[bd] |= mask;
					h->SetPip(bd, pipreg[bd]);
				}
			break;
			case PIPON:
				if (time[i] > period[i]) {
					next_state = SILENCE;

					// resolve board from handle
					bd = i / VPB8L_CH;
					h = vpb_c->hip(bd);

					// reset pipreg bit
					mask = ~(1 << (i % VPB8L_CH));
					pipreg[bd] &= mask;
					h->SetPip(bd, pipreg[bd]);
					time[i] = 0;
				}
			break;
			default:
				assert(0);
		}

		state[i] = next_state;
	}

}
