/*---------------------------------------------------------------------------*\

    FILE....: PLAYREC.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 9/2/98

    This module implements the play and record functions for the VPB API.
	 
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include "../vpbapi/apifunc.h"
#include "../vpbapi/vpbapi.h"
#include "../vpbapi/vpbfileman.h"
#include "../comm/comm.h"
#include  "../coff/coff.h"
#include "../vpbapi2/comp.h"
#include "../vpbapi2/arbch.h"
#include "../vpbapi2/objtrack.h"
#include "../config/config.h"
#include "../vpbapi/playrec.h"
#include "../vpbapi/mapdev.h"
#include "../unittest/wavecpp.h"
#include "../message/mess.h"
#include "../generic/generic.h"

#include <assert.h>
#include <memory>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <ctype.h>
						
/*---------------------------------------------------------------------------*\

							DEFINES

\*--------------------------------------------------------------------------*/

// play and record buf states

#define	IDLE						0
#define	PREPARED					1	// channnel configured but not playing/rec yet
#define	PLAYING						2	// channnel playing from a buffer
#define	RECORDING					3	// channnel recording to a buffer
#define	WAITING_FOR_DISABLE_FIFO	4	// waiting for record FIFO to be disabled
#define	FLUSH_FIFO					5	// waiting for record FIFO to be disabled

// terminate digits

#define	MAX_TERM					31		// maximum number of terminate digits
#define	NUM_VALID_DIG				16		// number of valid digits

// used to pass arguments to play_file_async_thread via void ptr

typedef struct {
	int		handle;
	void	*wout;
	char	*buf;
	ushort	bytesread;
	ushort	bytes;
	int		data;
} PLAY_FILE_ASYNC_ARGS;

// used to pass arguments to record_file_async_thread via void ptr

typedef struct {
	int		handle;
	void	*win;
	char	*buf;
	ushort	bytes;
	int		data;
} RECORD_FILE_ASYNC_ARGS;

// used to pass arguments to record_file_async_thread via void ptr

typedef struct {
	int		handle;
	ulong	size;
	char	*masterbuf;
	char	*buf;
	ushort	bytes;
	int		data;
} RECORD_BUF_ASYNC_ARGS;

typedef struct {
	char			term_digits[MAX_TERM+1];	// string of digits to terminate collection
	ushort			mode;						// current play mode
	ushort			buf_underflow;				// non-zero if buf has underflowed
	ushort			underflow_valid;			// non-zero if underflow valid (in some conditions we expect underflows)
} PLAY;

typedef struct {
	char			term_digits[MAX_TERM+1];	// string of digits to terminate collection
	unsigned int	time_out;
	unsigned int	time_start;
	ushort			buf_overflow;				// non-zero if buf has overlowed
} RECORD;

/*---------------------------------------------------------------------------*\

							STATICS

\*--------------------------------------------------------------------------*/

// states of play and record buf play functions

static int *play_ch;
static int *record_ch;
static float *play_gain;
static float *record_gain;
static Compression **play_comp;
static Compression **record_comp;
static PLAY *play;		// play params
static RECORD *record;	// record params
static int sleepms=20;	// length of time in ms to sleep between polling DSP

// valid terminate digits

static char valid_dig[] = "0123456789*#ABCD";

// diagnostics

static ushort	fplay_min;	// minimum words in DSP play (write) buffer
static ushort	frec_max;	// maximum words in DSP record (read) buffer

/*---------------------------------------------------------------------------*\

							FUNCTION HEADERS

\*--------------------------------------------------------------------------*/

void play_file(int handle, char file_name[], unsigned short mode);
void record_file(int handle, char file_name[], unsigned short mode);
void play_file_async_thread(void *args);
void record_file_async_thread(void *args);
void record_configure_VPB4(int handle, ushort b, ushort mode);
void record_configure_VPB8L(int handle, ushort b, ushort mode);
static void	validate_digits(char *digits);
static int digit_match(char digit, char *term_digits);
void play_voxfile_async_thread(void *pvargs);
void play_voxfile_async_thread(void *pvargs);
void play_voxfile(int handle, char file_name[], ushort mode);
void record_voxfile_async_thread(void *pvargs);
void record_buf_async_thread(void *pvargs);
void record_voxfile(int handle, char file_name[], unsigned short mode);
int time_out(int handle);
void start_time_out(int handle);
void play_buf_enable(int handle, unsigned short mode);
int play_rec_check_buffer(int handle);
/*---------------------------------------------------------------------------*\

								FUNCTIONS

\*--------------------------------------------------------------------------*/

GENERIC_CRITICAL_SECTION PlayRecSect;
GENERIC_CRITICAL_SECTION PlayRecReleaseSect;

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_open
	AUTHOR..: David Rowe
	DATE....: 9/3/98

	Initialises the playrec module.

\*--------------------------------------------------------------------------*/

void playrec_open(ushort numch) {
	play_ch = new int[numch];
	record_ch = new int[numch];
	play_comp = new Compression*[numch];
	record_comp = new Compression*[numch];
	play = new PLAY[numch];	
	record = new RECORD[numch];
	play_gain = new float[numch];
	record_gain = new float[numch];
	
	int i; 
	for(i=0; i<numch; i++) {
		play_ch[i] = 0;
		record_ch[i] = 0;
		play_comp[i] = NULL;
		record_comp[i] = NULL;
		play[i].term_digits[0] = 0;
		play[i].buf_underflow = 0;
		play[i].underflow_valid = 0;
		record[i].term_digits[0] = 0;
		record[i].time_out = 0;
		record[i].time_start = 0;
		record[i].buf_overflow = 0;
		play_gain[i] = (float)0.0;
		record_gain[i] = (float)0.0;
		
	}

	fplay_min = 64000;
	frec_max = 0;
	GenericInitializeCriticalSection(&PlayRecSect);
 	GenericInitializeCriticalSection(&PlayRecReleaseSect);

}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_close
	AUTHOR..: David Rowe
	DATE....: 9/3/98

	Closes the playrec module.

\*--------------------------------------------------------------------------*/

void playrec_close() {
	delete [] play_ch;
	delete [] record_ch;
	delete [] play_comp;
	delete [] record_comp;
	delete [] play;
	delete [] record;
	delete [] play_gain;
	delete [] record_gain;
	
	GenericDeleteCriticalSection(&PlayRecSect);
 	GenericDeleteCriticalSection(&PlayRecReleaseSect);

}

/*---------------------------------------------------------------------------*\

							PLAY FUNCTIONS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_set
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Sets play parameters or this channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_set(int handle, VPB_PLAY *vpb_play)
{
	try {
		ValidHandleCheck(handle);
		validate_digits(vpb_play->term_digits);
		strcpy(play[handle].term_digits,vpb_play->term_digits);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_set"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_new_digit_play
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Called by the MMQ when DSP detects a digit event.  Causes the play
	operation on the current channel to be terminated.

\*--------------------------------------------------------------------------*/

void playrec_new_digit_play(int h, ushort digit) {

	if ((play_ch[h] == PLAYING) && digit_match((char)digit, play[h].term_digits)) {
		arbch_grab_play_channel(h);
		arbch_release_play_channel(h);
	}

}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_buf_start
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	API callable version of play_buf_start.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_buf_start(int handle, unsigned short mode)
//	int				handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{

	try {
		play_buf_start(handle, mode);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_buf_start"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_buf_sync
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	API version of play_buf_sync().  Returns VPB_FINISH if playing
	should finish, else returns VPB_OK.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_buf_sync(int handle, char *buf, unsigned short length)
//	int				handle		handle of destination device
//	unsigned short	mode		specifies compression mode and async/sync
//	char			*buf		ptr to buffer to send
//	unsigned short	length		length of buffer in bytes
{
	int		ret;

	try {
		ret = play_buf_sync(handle, buf, length);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_buf_sync"));
	}
	
	return(ret);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_buf_finish
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Completes the playing of a channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_buf_finish(int handle)
//	int				handle		handle of destination device
{
	try {
		play_buf_finish(handle);	
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_buf_finish"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_buf_start
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Prepares a channel for playing a buffer by configuring the channel for
	the appropriate compression rate.  Internal API version.

\*--------------------------------------------------------------------------*/

void play_buf_start(int handle, unsigned short mode)
//	int				handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{
	ushort			b,ch;		// board and channel for this handle
	ushort			stobj;		// first sig proc obj for this device

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	arbch_grab_play_channel(handle);
	assert(play_ch[handle] == IDLE);
	play_ch[handle] = PREPARED;
	play[handle].mode = mode;

	// Set compression mode ------------------------------------------

	stobj = objtrack_handle_to_id(DNOBJ, handle);

	// disable all objects except for D/A
		
	config_disable_object(vpb_c, b, stobj+0);	// FIFO
	config_disable_object(vpb_c, b, stobj+1);	// UNPACK
	config_disable_object(vpb_c, b, stobj+2);	// TONEG
	config_disable_object(vpb_c, b, stobj+3);	// ALAW2LIN
	config_disable_object(vpb_c, b, stobj+4);	// MULAW2LIN
	config_disable_object(vpb_c, b, stobj+5);	// ADPCM2LIN
	config_disable_object(vpb_c, b, stobj+6);	// LIN2ALAW
	config_disable_object(vpb_c, b, stobj+8);	// DELAY

	// clear wires that may change
		
	config_clear_wire(vpb_c, b, stobj+6);
	config_clear_wire(vpb_c, b, stobj+8);

	// rewire and enable objects depending on compression mode
	// dont enable FIFO yet as we haven't filled it

	mode &= 0x7;
	switch (mode) {
		case (VPB_LINEAR):
			config_create_wire(vpb_c, b, stobj+0, stobj+6);
			config_create_wire(vpb_c, b, stobj+0, stobj+8);
			config_packrate(vpb_c, b, stobj, 1);
			play_comp[handle] = new LinearCompression;
		break;
		case (VPB_ALAW):
			config_create_wire(vpb_c, b, stobj+3, stobj+6);
			config_create_wire(vpb_c, b, stobj+3, stobj+8);
        	config_packrate(vpb_c, b, stobj, 2);
			config_packrate(vpb_c, b, stobj+1, 2);
			play_comp[handle] = new LinearCompression;
		break;
		case (VPB_MULAW):
			config_create_wire(vpb_c, b, stobj+4, stobj+6);
			config_create_wire(vpb_c, b, stobj+4, stobj+8);
        	config_packrate(vpb_c, b, stobj, 2);
			config_packrate(vpb_c, b, stobj+1, 2);
			play_comp[handle] = new LinearCompression;
		break;
		case (VPB_OKIADPCM):
			config_create_wire(vpb_c, b, stobj+5, stobj+6);
			config_create_wire(vpb_c, b, stobj+5, stobj+8);
        	config_packrate(vpb_c, b, stobj, 4);
			config_packrate(vpb_c, b, stobj+1, 4);
			config_adpcm_reset(vpb_c, b, stobj+5);
			play_comp[handle] = new LinearCompression;
		break;
		default:
			throw Wobbly(VPBAPI_PLAY_NOT_SUPPORTED);
	} // switch

}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_buf_enable
	AUTHOR..: David Rowe
	DATE....: 8/9/98

	Enables the config manager objects to start playing.  Should
	only be called after valid data is in FIFO.

\*--------------------------------------------------------------------------*/

void play_buf_enable(int handle)
//	int				handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{
	ushort			b,ch;						// board and channel for this handle
	ushort			stobj;						// first sig proc obj for this device
	ushort			objlist[CONFIG_MAX_LIST];	// list of config manager objects

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	stobj = objtrack_handle_to_id(DNOBJ, handle);
		
	// wait for DSP FIFO to fill (this may take a while (20ms) with relay driver)

	VPBREG			*v;			// ptr to vpbreg for this card
	v = vpb_c->vpbreg(b);
	int f;
	while ((f=v->txdf[ch]->DspHowFull()) < v->lsf) {
		GenericSleep(sleepms);
	}

	int cnt = 0;
	objlist[cnt++] = stobj+0;		// FIFO
	objlist[cnt++] = stobj+1;		// UNPACK

	ushort mode = play[handle].mode;
	mode &= 0x7;
	switch (mode) {
		case (VPB_LINEAR):
		break;
		case (VPB_ALAW):
			objlist[cnt++] = stobj+3;		// ALAW
		break;
		case (VPB_MULAW):
			objlist[cnt++] = stobj+4;		// MULAW
		break;
		case (VPB_OKIADPCM):
			objlist[cnt++] = stobj+5;		// ADPCM
		break;
		default:
			throw Wobbly(VPBAPI_PLAY_NOT_SUPPORTED);
	} // switch

	objlist[cnt++] = stobj+6;				// LIN2ALAW
	objlist[cnt++] = stobj+8;				// DELAY

	// all objects enabled at once to ensure one object doesnt start
	// processing before another (important for ADPCM)

	config_enable_object_list(vpb_c, b, objlist, cnt);

	// start checking for underflows

	play[handle].underflow_valid = 1;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_buf_sync
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Plays a buffer of speech samples to a VPB channel.  Returns when all of
	the buffer has been sent to the VPB's DSP FIFO, ie operates 
	synchronously.  Note that when this function exits the DSP is still 
	playing info, the finish function below must wait for the FIFO to empty
	before disabling the FIFO, and releasing the channel.

	However, if another buffer is to be sent, then when this function exits
	there is plenty of time to prepare the next buffer before the DSP fifo
	empties.

\*--------------------------------------------------------------------------*/

int play_buf_sync(int handle, char *buf, unsigned short length)
//	int				handle		handle of destination device
//	char			*buf		ptr to buffer to send
//	unsigned short	length		length of buffer in bytes
{
	ushort			b,ch;		// board and channel for this handle
	Compression		*comp;
	word			*wordbuf;	// buffer of wrods to download
	ushort			words;		// number of samples (words) to download
	ushort			bytes;		// number of bytes to read from buf
	VPBREG			*v;			// ptr to vpbreg for this card

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	comp = play_comp[handle];
	v = vpb_c->vpbreg(b);
	vpb_c->CheckForAssert(b);
	
	// make sure device not idle

	if (play_ch[handle] == IDLE)
		throw Wobbly(VPBAPI_DEVICE_NOT_PREPARED);
		
	// download half DSP FIFO size buffers at a time

	words = v->sztxdf[ch]/2;
	wordbuf = new word[words];
	
	int f;
	while(length && arbch_keep_play_channel(handle)) {

		// wait for buffer to half empty to send next buffer

		if (play_ch[handle] == PLAYING) {
			f = v->txdf[ch]->HowEmpty();
			while (f < words) {				
				f = v->txdf[ch]->HowEmpty();
				GenericSleep(sleepms);
			}
			// determine how empty buf gets for diagnostic purposes

			f = v->txdf[ch]->HowFull();
			if (f < fplay_min) {
				fplay_min = f;
			}
		}

		// obtain a buffer of unpacked words and send to DSP

		bytes = comp->words2bytes(words);
		if (bytes > length)  {
			bytes = length;
			words = comp->bytes2words(bytes);
		}
		comp->unpack(wordbuf, buf, bytes);
		length -= bytes;
		buf += bytes;

		// send to DSP tx FIFO

		if (v->txdf[ch]->Write(wordbuf, words) != 0)
			assert(0);

		// if this is the first buffer enable objects

		if (play_ch[handle] == PREPARED) {
			play_ch[handle] = PLAYING;
			play_buf_enable(handle);
		}
	}

	delete [] wordbuf;

	if (arbch_keep_play_channel(handle) == 0)
		return(VPB_FINISH);
	else
		return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_buf_finish
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Completes the playing of a channel.

\*--------------------------------------------------------------------------*/

void play_buf_finish(int handle)
//	int				handle		handle of destination device
{
	ushort			b,ch;		// board and channel for this handle
	VPBREG			*v;			// ptr to vpbreg for this card
	ushort			stobj;		// first sig proc obj for this device

	// validate

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	v = vpb_c->vpbreg(b);
		
	// stop checking for underflows !

	play[handle].underflow_valid = 0;

	// wait for DSP FIFO to empty
	
	int f;
	while ((f=v->txdf[ch]->DspHowFull()) >= v->lsf) {
		GenericSleep(sleepms);
	}
	// disable all objects except D/A

	stobj = objtrack_handle_to_id(DNOBJ, handle);
	config_disable_object(vpb_c, b, stobj+0);	// FIFO
	config_disable_object(vpb_c, b, stobj+1);	// UNPACK
	config_disable_object(vpb_c, b, stobj+2);	// TONEG
	config_disable_object(vpb_c, b, stobj+3);	// ALAW2LIN
	config_disable_object(vpb_c, b, stobj+4);	// MULAW2LIN
	config_disable_object(vpb_c, b, stobj+5);	// ADPCM2LIN
	config_disable_object(vpb_c, b, stobj+6);	// DELAY
	config_disable_object(vpb_c, b, stobj+8);	// LIN2ALAW

	// flush remaining

	config_flush_fifo(vpb_c, b, stobj+0);
	// make sure fifo has been flushed
	while ((f=v->txdf[ch]->DspHowFull()) != 0) {
		GenericSleep(sleepms);
	}

	// connect DELAY and LIN2ALAW to zero source

	config_clear_wire(vpb_c, b, stobj+6);
	config_clear_wire(vpb_c, b, stobj+8);
	config_connect_to_zerobuf(vpb_c, b, stobj+6);
	config_connect_to_zerobuf(vpb_c, b, stobj+8);
	config_enable_object(vpb_c, b, stobj+6);	
	config_enable_object(vpb_c, b, stobj+8);	
	
	// clean up!

	play_ch[handle] = IDLE;
	delete play_comp[handle];
	arbch_release_play_channel(handle);	
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_file
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to play a file to a channel.  This function is used
	to create the aync and sync play functions.

\*--------------------------------------------------------------------------*/

void play_file(int handle, char file_name[])
//	int				handle		handle of destination device
//	char			file_name[]	name of file to play
{
	void			*wout;		// open wave file
	ushort			words;		// words to download to FIFO
	ushort			bytes;		// num bytes to try to read from file
	ushort			bytesread;	// num bytes read from file
	char			*buf;		// buffer of bytes read from file
	ushort			b,ch;		// board and channel for this handle
	ushort			mode;		// mode of wave file

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	wave_open_read(&wout, file_name);
	wave_get_mode(wout, &mode);
	play_buf_start(handle, mode);

	GenericSetThreadPriorityHigh();

	// calculate buffer size to fit DSP FIFO

	words = vpb_c->vpbreg(b)->sztxdf[ch]/2;
	bytes = play_comp[handle]->words2bytes(words);
	buf = new char[bytes];
		
	// initial read

	bytesread = vpb_wave_read(wout, buf, bytes);

	// now loop until end of file

	while(bytesread && arbch_keep_play_channel(handle)) {
		play_buf_sync(handle, buf, bytesread);
		bytesread = vpb_wave_read(wout, buf, bytes);
	}

	delete [] buf;

	play_buf_finish(handle);
	vpb_wave_close_read(wout);

	GenericSetThreadPriorityNormal();
}
/*---------------------------------------------------------------------------*\

	FUNCTION: play_voxfile
	AUTHOR..: David Rowe & John Kostogiannis
	DATE....: 20/8/98

	Utility function to play a vox file to a channel.  This function is used
	to create the aync and sync play functions.

\*--------------------------------------------------------------------------*/

void play_voxfile(int handle, char file_name[], ushort mode)
//	int				handle		handle of destination device
//	char			file_name[]	name of file to play
//	ushort			mode		compression mode
{
	FILE			*vout;		// open vox file
	ushort			words;		// words to download to FIFO
	ushort			bytes;		// num bytes to try to read from file
	ushort			bytesread;	// num bytes read from file
	char			*buf;		// buffer of bytes read from file
	ushort			b,ch;		// board and channel for this handle

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	vout = fopen(file_name,"rb");
	if (vout == NULL) 
		throw Wobbly(VPBAPI_RECORD_CANT_OPEN_VOXFILE);
	
	GenericSetThreadPriorityHigh();

	play_buf_start(handle, mode);

	// calculate buffer size to fit DSP FIFO

	words = vpb_c->vpbreg(b)->sztxdf[ch]/2;
	bytes = play_comp[handle]->words2bytes(words);
	buf = new char[bytes];
		
	// initial read

	bytesread = fread( buf,sizeof(char), bytes,vout);
	// now loop until end of file

	while(bytesread && arbch_keep_play_channel(handle)) {
		play_buf_sync(handle, buf, bytesread);
		bytesread = fread( buf,sizeof(char), bytes,vout);
	}

	delete [] buf;

	play_buf_finish(handle);
	fclose(vout);

	GenericSetThreadPriorityNormal();
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_file_sync
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to play a file to a channel.  Function returns when
	playing is finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_file_sync(int handle, char file_name[])
//	int				handle		handle of destination device
//	char			file_name[]	name of file to play
{

	try {
		play_file(handle, file_name);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_file_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_voxfile_sync
	AUTHOR..: David Rowe & john kostogiannis
	DATE....: 20/8/98

	Utility function to play a vox file to a channel.  Function returns when
	playing is finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_voxfile_sync(int handle, char file_name[],unsigned short mode)
//	int				handle		handle of destination device
//	char			file_name[]	name of file to play
//	unsined short	mode		compression mode
{

	try {
		play_voxfile(handle, file_name, mode);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_voxfile_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_file_async
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to play a file to a channel.  Function returns as soon
	as playig has started, and places an event on the API Q when playing
	has finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_file_async(int handle, char file_name[], int data)
//	int				handle		handle of destination device
//	char			file_name[]	name of file to play
//	int				data		VPB_PLAYEND event data
{
	void			*wout;		// open wave file
	ushort			words;		// words to download to FIFO
	ushort			bytes;		// num bytes to try to read from file
	ushort			bytesread;	// num bytes read from file
	char			*buf;		// buffer of bytes read from file
	ushort			b,ch;		// board and channel for this handle
	ushort			mode;

	try {
		// Validation ----------------------------------------------------

		ValidHandleCheck(handle);
		maphndletodev(handle, &b, &ch);
		wave_open_read(&wout, file_name);
		wave_get_mode(wout, &mode);
		play_buf_start(handle, mode);

		// calculate buffer size to fit DSP FIFO

		words = vpb_c->vpbreg(b)->sztxdf[ch]/2;
		bytes = play_comp[handle]->words2bytes(words);
		buf = new char[bytes];
				
		// initial read

		bytesread = vpb_wave_read(wout, buf, bytes);

		// now start thread

		PLAY_FILE_ASYNC_ARGS *args = new PLAY_FILE_ASYNC_ARGS;
		args->handle = handle;
		args->wout = wout;
		args->buf = buf;
		args->bytesread = bytesread;
		args->bytes = bytes;
		args->data = data;
		Generic_beginthread(play_file_async_thread, 0, (void*)args);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_file_async"));
	}
	
	return(VPB_OK);
}
		
/*---------------------------------------------------------------------------*\

	FUNCTION: play_file_async_thread
	AUTHOR..: David Rowe
	DATE....: 23/3/98

	Thread used to implement play file function in async mode.  Posts an 
	event on the API Q when it has finished.

	If error occurs posts wobbly event code in data of termination event,
	otherwise posts event on successful termination.

\*--------------------------------------------------------------------------*/

void play_file_async_thread(void *pvargs)
//	void	*args;		arguments structure
{
	PLAY_FILE_ASYNC_ARGS *args = (PLAY_FILE_ASYNC_ARGS*)pvargs;
	int		handle = args->handle;
	void	*wout = args->wout;
	char	*buf = args->buf;
	ushort	bytesread = args->bytesread;
	ushort	bytes = args->bytes;
	int		data = args->data;
	VPB_EVENT	e;

	delete args;
	
	GenericSetThreadPriorityHigh();

	try {
		while(bytesread && arbch_keep_play_channel(handle)) {
			play_buf_sync(handle, buf, bytesread);
			bytesread = vpb_wave_read(wout, buf, bytes);
		}

		delete [] buf;

		vpb_wave_close_read(wout);

		e.type = VPB_PLAYEND;
		e.handle = handle;
		e.data = data;

		play_buf_finish(handle);
		putevt(&e, 0);		
	}

	catch(Wobbly w){
		vpb_wave_close_read(wout);

		e.type = VPB_PLAYEND;
		e.handle = handle;
		e.data = w.code;
		try {
			play_buf_finish(handle);
		}
		catch(Wobbly w){w;}

		putevt(&e, 0);		
	}

}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_voxfile_async
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to play a vox file to a channel.  Function returns as soon
	as playig has started, and places an event on the API Q when playing
	has finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_voxfile_async(int handle, char file_name[], unsigned short mode,int data)
//	int				handle		handle of destination device
//	char			file_name[]	name of file to play
//	ushort			mode		compression mode
//	int				data		VPB_PLAYEND event data
{
	FILE			*vout;		// open vox file
	ushort			words;		// words to download to FIFO
	ushort			bytes;		// num bytes to try to read from file
	ushort			bytesread;	// num bytes read from file
	char			*buf;		// buffer of bytes read from file
	ushort			b,ch;		// board and channel for this handle

	try {
		// Validation ----------------------------------------------------

		ValidHandleCheck(handle);
		maphndletodev(handle, &b, &ch);
		vout = fopen(file_name,"rb");
		if (vout == NULL) 
			throw Wobbly(VPBAPI_PLAY_CANT_OPEN_VOXFILE);
	
		play_buf_start(handle, mode);

		// calculate buffer size to fit DSP FIFO

		words = vpb_c->vpbreg(b)->sztxdf[ch]/2;
		bytes = play_comp[handle]->words2bytes(words);
		buf = new char[bytes];
		
		// initial read

		bytesread = fread( buf,sizeof(char), bytes,vout);

		// now loop until end of file in separate thread
		
		PLAY_FILE_ASYNC_ARGS *args = new PLAY_FILE_ASYNC_ARGS;
		args->handle = handle;
		args->wout = vout;
		args->buf = buf;
		args->bytesread = bytesread;
		args->bytes = bytes;
		args->data = data;
		Generic_beginthread(play_voxfile_async_thread, 0, (void*)args);

	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_voxfile_async"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_voxfile_async_thread
	AUTHOR..: David Rowe & John Kostogiannis
	DATE....: 20/8/98

	Thread used to implement play voxfile function in async mode.  Posts an 
	event on the API Q when it has finished.

	If error occurs posts wobbly event code in data of termination event,
	otherwise posts event on successful termination.

\*--------------------------------------------------------------------------*/

void play_voxfile_async_thread(void *pvargs)
//	void	*args;		arguments structure
{
	PLAY_FILE_ASYNC_ARGS *args = (PLAY_FILE_ASYNC_ARGS*)pvargs;
	int		handle = args->handle;
	FILE	*vout = (FILE *) args->wout;
	char	*buf = args->buf;
	ushort	bytesread = args->bytesread;
	ushort	bytes = args->bytes;
	int		data = args->data;
	VPB_EVENT	e;

	delete args;
	
	GenericSetThreadPriorityHigh();

	try {
		while(bytesread && arbch_keep_play_channel(handle)) {
			play_buf_sync(handle, buf, bytesread);
			bytesread = fread(buf,sizeof(char), bytes, vout);
		}

		delete [] buf;

		fclose(vout);

		e.type = VPB_PLAYEND;
		e.handle = handle;
		e.data = data;

		play_buf_finish(handle);
		putevt(&e, 0);		
	}

	catch(Wobbly w){
		fclose(vout);

		e.type = VPB_PLAYEND;
		e.handle = handle;
		e.data = w.code;
		try {
			play_buf_finish(handle);
		}
		catch(Wobbly w){w;}

		putevt(&e, 0);		
	}

}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_terminate
	AUTHOR..: David Rowe
	DATE....: 24/3/98

	Stops playing immediately. Does nothing if we are not playing.
	
\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_terminate(int handle)
//	int	handle	handle of channel 
{
	try {
		arbch_grab_play_channel(handle);
		arbch_release_play_channel(handle);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_terminate"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

							RECORD FUNCTIONS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_set
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Sets record parameters or this channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_set(int handle, VPB_RECORD *vpb_record)
{
	try {
		ValidHandleCheck(handle);
		validate_digits(vpb_record->term_digits);
		strcpy(record[handle].term_digits,vpb_record->term_digits);
		record[handle].time_out = vpb_record->time_out;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_set"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_new_digit_record
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Called by the MMQ when DSP detects a digit event.  Causes the record
	operation on the current channel to be terminated.

\*--------------------------------------------------------------------------*/

void playrec_new_digit_record(int h, ushort digit) {
	if ((record_ch[h] == RECORDING) && digit_match((char)digit, record[h].term_digits)) {
		arbch_grab_record_channel(h);
		arbch_release_record_channel(h);
	}
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_buf_start
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	API callable version of record_buf_start.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_buf_start(int handle, unsigned short mode)
//	int				handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{

	try {
		record_buf_start(handle, mode);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_buf_start"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_buf_sync
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	API version of record_buf_sync().

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_buf_sync(int handle, char *buf, unsigned short length)
//	int				handle		handle of destination device
//	char			*buf		ptr to buffer to receive
//	unsigned short	length		length of buffer in bytes
{
	int	ret;

	try {
		ret = record_buf_sync(handle, buf, length);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_buf_sync"));
	}
	
	return(ret);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_buf_finish
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Completes the recording of a channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_buf_finish(int handle)
//	int				handle		handle of destination device
{
	try {
		record_buf_finish(handle);	
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_buf_finish"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_buf_start
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Prepares a channel for recording a buffer by configuring the channel for
	the appropriate compression rate.  Internal API version.

\*--------------------------------------------------------------------------*/

void record_buf_start(int handle, unsigned short mode)
//	int				handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{
	ushort			b,ch;		// board and channel for this handle

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	arbch_grab_record_channel(handle);
	assert(record_ch[handle] == IDLE);
	record_ch[handle] = PREPARED;
		
	// Set compression mode ------------------------------------------

	switch (vpb_c->vpbreg(b)->model) {
		case VPB4:
			record_configure_VPB4(handle, b,  mode);
		break;
		case VPB8L:
			record_configure_VPB8L(handle, b, mode);
		break;
		default:
			throw Wobbly(VPBAPI_MODEL_NOT_SUPPORTED);
		break;
	}

	start_time_out(handle);
	record_ch[handle] = RECORDING;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_configure_VPB4
	AUTHOR..: David Rowe
	DATE....: 20/6/98

	Configures a VPB4 channel for recording.

\*--------------------------------------------------------------------------*/

void record_configure_VPB4(int handle, ushort b, ushort mode)
//	int				handle		handle of destination device
//	ushort			b			board number
//	unsigned short	mode		specifies compression mode
{
	ushort			stobj;		// first sig proc obj for this device
		
	stobj = objtrack_handle_to_id(UPOBJ, handle);
	config_disable_object(vpb_c, b, stobj+5);	// ALAW
	config_disable_object(vpb_c, b, stobj+6);	// MULAW
	config_disable_object(vpb_c, b, stobj+7);	// ADPCM
	config_disable_object(vpb_c, b, stobj+9);	// PACK
	config_disable_object(vpb_c, b, stobj+10);	// PACKED_FIFO_UP

	// clear wiring

	config_clear_wire(vpb_c, b, stobj+9);		// PACKED_FIFO_UP
	config_clear_wire(vpb_c, b, stobj+10);		// PACK

	// clear FIFO

#ifdef TEMP
	VPBREG *v = vpb_c->vpbreg(b);
	
	// wait until all messages are read by the DSP, which means there
	// is no way it will be writing to the FIFO, as it will have rec
	// the disable message for the fifo.  This is espec important for
	// ADPCM sync.

	int ret;
	do {
		ret = v->dnmess->HowFull();
	} while(ret);

	ushort ch,f;
	maphndletodev(handle, &b, &ch);
	f = v->rxdf[ch]->HowFull();
	word *wordbuf = new word[f];
	v->rxdf[ch]->Read(wordbuf, f);
	delete [] wordbuf;
#endif

	// rewire connection to FIFO but dont enable just yet

	mode &= 0x7;
	switch (mode) {
		case (VPB_LINEAR):
			config_create_wire(vpb_c, b, stobj+2, stobj+10);
			config_packrate(vpb_c, b, stobj+10, 1);
			config_enable_object(vpb_c, b, stobj+10);		// FIFO
			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_ALAW):
			config_create_wire(vpb_c, b, stobj+5, stobj+9);
			config_create_wire(vpb_c, b, stobj+9, stobj+10);
			config_packrate(vpb_c, b, stobj+9, 2);
			config_packrate(vpb_c, b, stobj+10, 2);
			config_enable_object(vpb_c, b, stobj+5);
			config_enable_object(vpb_c, b, stobj+9);
			config_enable_object(vpb_c, b, stobj+10);		// FIFO
			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_MULAW):
			config_create_wire(vpb_c, b, stobj+6, stobj+9);
			config_create_wire(vpb_c, b, stobj+9, stobj+10);
			config_packrate(vpb_c, b, stobj+9, 2);
			config_packrate(vpb_c, b, stobj+10, 2);
			config_enable_object(vpb_c, b, stobj+6);
			config_enable_object(vpb_c, b, stobj+9);
			config_enable_object(vpb_c, b, stobj+10);		// FIFO
			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_OKIADPCM):
			config_create_wire(vpb_c, b, stobj+7, stobj+9);
			config_create_wire(vpb_c, b, stobj+9, stobj+10);
			config_packrate(vpb_c, b, stobj+9, 4);
			config_packrate(vpb_c, b, stobj+10, 4);
			config_adpcm_reset(vpb_c, b, stobj+7);
			config_enable_object(vpb_c, b, stobj+9);
			config_enable_two_objects(vpb_c, b, stobj+10,stobj+7);		// FIFO and ADPCM
			record_comp[handle] = new LinearCompression;
		break;
		default:
			throw Wobbly(VPBAPI_REC_NOT_SUPPORTED);
	} // switch

}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_configure_VPB8L
	AUTHOR..: David Rowe
	DATE....: 20/6/98

	Configures a VPB4 channel for recording.

\*--------------------------------------------------------------------------*/

void record_configure_VPB8L(int handle, ushort b, ushort mode)
//	int				handle		handle of destination device
//	ushort			b			board number
//	unsigned short	mode		specifies compression mode
{
	ushort			stobj;		// first sig proc obj for this device

	stobj = objtrack_handle_to_id(UPOBJ, handle);

	// Note, we assume objects are disabled....

	// clear wiring

	config_clear_wire(vpb_c, b, stobj+6);		// PACKED_FIFO_UP
	config_clear_wire(vpb_c, b, stobj+5);		// PACK

	// configure objects and make appropriate connections
	// enable objects to start sampling

	mode &= 0x7;
	switch (mode) {
		case (VPB_LINEAR):
			config_rate8k(vpb_c, b, stobj+0);
			config_rate8k(vpb_c, b, stobj+1);
			config_rate8k(vpb_c, b, stobj+2);
			config_rate8k(vpb_c, b, stobj+4);
			config_rate8k(vpb_c, b, stobj+5);
			config_rate8k(vpb_c, b, stobj+6);	

			// wiring: FIRD->PACKED_UP_FIFO

			config_create_wire(vpb_c, b, stobj+0, stobj+6);

			// packrate: FIFO.packrate = 1

			config_packrate(vpb_c, b, stobj+6, 1);

			// enable objects

			config_enable_object(vpb_c, b, stobj+6);
			
			// firmware already packs so use linear compression 
			// packer func at PC side

			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_ALAW):
			config_rate8k(vpb_c, b, stobj+0);
			config_rate8k(vpb_c, b, stobj+1);
			config_rate8k(vpb_c, b, stobj+2);
			config_rate8k(vpb_c, b, stobj+4);
			config_rate8k(vpb_c, b, stobj+5);
			config_rate8k(vpb_c, b, stobj+6);	

			// wiring: ALAW->PACK, PACK->PACKED_UP_FIFO

			config_create_wire(vpb_c, b, stobj+8, stobj+5);
			config_create_wire(vpb_c, b, stobj+5, stobj+6);

			// packrate: FIFO.packrate = 2, PACK.packrate = 2

			config_packrate(vpb_c, b, stobj+6, 2);
			config_packrate(vpb_c, b, stobj+5, 2);

			// enable objects

			config_enable_object(vpb_c, b, stobj+5);	// PACK
			config_enable_object(vpb_c, b, stobj+8);	// ALAW
			config_enable_object(vpb_c, b, stobj+6);	// FIFO
	
			// firmware already packs so use linear compression 
			// packer func at PC side

			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_MULAW):
			config_rate8k(vpb_c, b, stobj+0);
			config_rate8k(vpb_c, b, stobj+1);
			config_rate8k(vpb_c, b, stobj+2);
			config_rate8k(vpb_c, b, stobj+4);
			config_rate8k(vpb_c, b, stobj+5);
			config_rate8k(vpb_c, b, stobj+6);	

			// wiring: MULAW->PACK, PACK->PACKED_UP_FIFO

			config_create_wire(vpb_c, b, stobj+7, stobj+5);
			config_create_wire(vpb_c, b, stobj+5, stobj+6);

			// packrate: FIFO.packrate = 2, PACK.packrate = 2

			config_packrate(vpb_c, b, stobj+6, 2);
			config_packrate(vpb_c, b, stobj+5, 2);

			// enable objects

			config_enable_object(vpb_c, b, stobj+5);	// PACK
			config_enable_object(vpb_c, b, stobj+7);	// MULAW
			config_enable_object(vpb_c, b, stobj+6);	// FIFO

			// firmware already packs so use linear compression 
			// packer func at PC side

			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_OKIADPCM):
			config_rate8k(vpb_c, b, stobj+0);
			config_rate8k(vpb_c, b, stobj+1);
			config_rate8k(vpb_c, b, stobj+2);
			config_rate8k(vpb_c, b, stobj+4);
			config_rate8k(vpb_c, b, stobj+5);
			config_rate8k(vpb_c, b, stobj+6);	
			
			// wiring: ADPCM->PACK, PACK->PACKED_UP_FIFO

			config_create_wire(vpb_c, b, stobj+4, stobj+5);
			config_create_wire(vpb_c, b, stobj+5, stobj+6);
			config_adpcm_reset(vpb_c, b, stobj+4);

			// packrate: FIFO.packrate = 4, PACK.packrate = 4

			config_packrate(vpb_c, b, stobj+6, 4);
			config_packrate(vpb_c, b, stobj+5, 4);

			// enable objects

			config_enable_object(vpb_c, b, stobj+5);	// PACK
			config_enable_two_objects(vpb_c, b, stobj+4, stobj+6); // ADPCM and FIFO

			// firmware already packs so use linear compression 
			// packer func at PC side

			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_OKIADPCM24):

			config_rate6k(vpb_c, b, stobj+0);
			config_rate6k(vpb_c, b, stobj+1);
			config_rate6k(vpb_c, b, stobj+2);
			config_rate6k(vpb_c, b, stobj+4);
			config_rate6k(vpb_c, b, stobj+5);
			config_rate6k(vpb_c, b, stobj+6);	

			// wiring: ADPCM->PACK, PACK->PACKED_UP_FIFO

			config_create_wire(vpb_c, b, stobj+4, stobj+5);
			config_create_wire(vpb_c, b, stobj+5, stobj+6);
			config_adpcm_reset(vpb_c, b, stobj+4);

			// packrate: FIFO.packrate = 4, PACK.packrate = 4

			config_packrate(vpb_c, b, stobj+6, 4);
			config_packrate(vpb_c, b, stobj+5, 4);

			// enable objects

			config_enable_object(vpb_c, b, stobj+5);	// PACK
			config_enable_two_objects(vpb_c, b, stobj+4, stobj+6);	// ADPCM and FIFO

			// firmware already packs so use linear compression 
			// packer func at PC side

			record_comp[handle] = new LinearCompression;
		break;
		default:
			throw Wobbly(VPBAPI_REC_NOT_SUPPORTED);
	} // switch
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_buf_sync
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Records a buffer of speech samples from a VPB channel.  Returns when the
	buffer if filled from the VPB's DSP FIFO, ie operates synchronously.  
	Note that when this function exits the DSP is still recording info, the
	finish function must be called to disable recording.

	If another buffer is to be obtained, then when this function exits
	there is plenty of time to prepare for the next buffer before the DSP
	fifo fills.

\*--------------------------------------------------------------------------*/

int record_buf_sync(int handle, char *buf, unsigned short length)
//	int				handle		handle of destination device
//	unsigned short	mode		specifies compression mode and async/sync
//	char			*buf		ptr to buffer to receive
//	unsigned short	length		length of buffer in bytes
{
	ushort			b,ch;		// board and channel for this handle
	Compression		*comp;
	word			*wordbuf;	// buffer of wrods to download
	ushort			words;		// number of samples (words) to download
	ushort			bytes;		// number of bytes to read from buf
	VPBREG			*v;			// ptr to vpbreg for this card

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	comp = record_comp[handle];
	v = vpb_c->vpbreg(b);
		
	// make sure device not idle

	if (record_ch[handle] == IDLE)
		throw Wobbly(VPBAPI_DEVICE_NOT_PREPARED);
		
	// upload half DSP FIFO size buffers at a time

	words = v->sztxdf[ch]/2;
	wordbuf = new word[words];

	ushort f;
	while(length && arbch_keep_record_channel(handle) && !time_out(handle)) {

		// wait for buffer to half fill

		if (record_ch[handle] == RECORDING) {
			while ((f = v->rxdf[ch]->HowFull()) < words) {
				GenericSleep(sleepms);
			}
			if (f > frec_max)
				frec_max = f;
		}

		// obtain a buffer of unpacked words from DSP
		
		bytes = comp->words2bytes(words);
		if (bytes > length)  {
			bytes = length;
			words = comp->bytes2words(bytes);
		}
		if (v->rxdf[ch]->Read(wordbuf, words) != 0)
			assert(0);		

		// pack words for PC

		comp->pack(buf, wordbuf, words);
		length -= bytes;
		buf += bytes;
	}

	delete [] wordbuf;

	if ((arbch_keep_record_channel(handle) == 0) || time_out(handle))
		return(VPB_FINISH);
	else
		return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_buf_finish
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Completes the recording of a channel.

\*--------------------------------------------------------------------------*/

void record_buf_finish(int handle)
//	int				handle		handle of source device
{
	// validate

	ValidHandleCheck(handle);
		
	// disable objects

	ushort b,ch,stobj;
	maphndletodev(handle, &b, &ch);

	switch (vpb_c->vpbreg(b)->model) {
		case VPB4:
			stobj = objtrack_handle_to_id(UPOBJ, handle);
			config_disable_object(vpb_c, b, stobj+5);	// ALAW
			config_disable_object(vpb_c, b, stobj+6);	// MULAW
			config_disable_object(vpb_c, b, stobj+7);	// ADPCM
			config_disable_object(vpb_c, b, stobj+9);	// PACK
			config_disable_object(vpb_c, b, stobj+10);	// PACKED_FIFO_UP
		break;
		case VPB8L:
			stobj = objtrack_handle_to_id(UPOBJ, handle);
			config_disable_object(vpb_c, b, stobj+6);	// PACKED_FIFO_UP
			config_disable_object(vpb_c, b, stobj+5);	// PACK
			config_disable_object(vpb_c, b, stobj+4);	// LIN2ADPCM
			config_disable_object(vpb_c, b, stobj+7);	// LIN2MULAW
			config_disable_object(vpb_c, b, stobj+8);	// LIN2ALAW
		break;
		default:
			throw Wobbly(VPBAPI_MODEL_NOT_SUPPORTED);
		break;
	}

	// wait for FIFO to be disabled

	record_ch[handle] = WAITING_FOR_DISABLE_FIFO;
	int waiting = 0;
	while((record_ch[handle] == WAITING_FOR_DISABLE_FIFO) && (waiting < 1000)) {
		GenericSleep(sleepms);
		waiting += sleepms;
	}
/*	mess_mprintf_on("waittime.txt","a");
	mprintf("%d\n",waiting);
	mess_mprintf_off();*/

	// Now we can safely flush the FIFO, knowing that there is no way
	// the DSP will be writing to it!

	// wait for PC to read DSP fifo into relay buf (relay buf enabled only)

	VPBREG *v = vpb_c->vpbreg(b);
	int f;
	waiting = 0;
	if (v->ddmodel == 1) {
		while(f = v->rxdf[ch]->DspHowFull())
			GenericSleep(sleepms);
			waiting +=sleepms;
	}

/*	mess_mprintf_on("dwaittime.txt","a");
	mprintf("%d\n",waiting);
	mess_mprintf_off();
*/
	// now read (relay) fifo to flush

	f = v->rxdf[ch]->HowFull();
	word *wordbuf = new word[f];
	v->rxdf[ch]->Read(wordbuf, f);
	delete [] wordbuf;

	// clean up!

	delete record_comp[handle];
	record_ch[handle] = IDLE;
	arbch_release_record_channel(handle);	
}

void playrec_fifo_disabled(int h) {
	if (record_ch[h] == WAITING_FOR_DISABLE_FIFO)
		record_ch[h] = FLUSH_FIFO;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_file
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to record a file from a channel.  This function is 
	used to create the async and sync record functions.  The function 
	continues until the channel is "grabbed" by another thread (see arbch
	module).

\*--------------------------------------------------------------------------*/

void record_file(int handle, char file_name[], unsigned short mode)
//	int				handle		handle of source device
//	char			file_name[]	name of file to record
//	unsigned short	mode		specifies compression mode
{
	void			*win;
	ushort			words;		// words to upload from FIFO
	ushort			bytes;		// num bytes to try to read from file
	ushort			byteswrite;	// num bytes written to file
	char			*buf;		// buffer of bytes read from file
	ushort			b,ch;		// board and channel for this handle
	
	GenericSetThreadPriorityHigh();

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	wave_open_write(&win, file_name, mode);
	record_buf_start(handle, mode);

	// calculate buffer size to fit DSP FIFO

	words = vpb_c->vpbreg(b)->szrxdf[ch]/2;
	bytes = record_comp[handle]->words2bytes(words);
	buf = new char[bytes];
		
	// now loop until channel "grabbed"

	while(arbch_keep_record_channel(handle) && !time_out(handle)) {
		record_buf_sync(handle, buf, bytes);
		if (arbch_keep_record_channel(handle)) {
			byteswrite = vpb_wave_write(win, buf, bytes);
			if (byteswrite != bytes)
				throw Wobbly(PLAYREC_DISK_FULL);
		}
	}

	delete [] buf;

	record_buf_finish(handle);
	vpb_wave_close_write(win);

	GenericSetThreadPriorityNormal();
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_file_sync
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to record a file to a channel.  Function returns when
	recording is finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_file_sync(int handle, char file_name[], unsigned short mode)
//	int				handle		handle of destination device
//	char			file_name[]	name of file to record
//	unsigned short	mode		specifies compression mode
{

	try {
		record_file(handle, file_name, mode);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_file_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_voxfile
	AUTHOR..: David Rowe & John Kostogiannis
	DATE....: 20/8/98

	Utility function to record a vox file from a channel.  This function is 
	used to create the async and sync record functions.  The function 
	continues until the channel is "grabbed" by another thread (see arbch
	module).

\*--------------------------------------------------------------------------*/

void record_voxfile(int handle, char file_name[], unsigned short mode)
//	int				handle		handle of source device
//	char			file_name[]	name of file to record
//	unsigned short	mode		specifies compression mode
{
	FILE			*vin;
	ushort			words;		// words to upload from FIFO
	ushort			bytes;		// num bytes to try to read from file
	ushort			byteswrite;	// num bytes written to file
	char			*buf;		// buffer of bytes read from file
	ushort			b,ch;		// board and channel for this handle

	GenericSetThreadPriorityHigh();

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	vin = fopen(file_name, "wb");
	if(vin == NULL)
		throw Wobbly(VPBAPI_PLAY_CANT_OPEN_VOXFILE);
	record_buf_start(handle, mode);

	// calculate buffer size to fit DSP FIFO

	words = vpb_c->vpbreg(b)->szrxdf[ch]/2;
	bytes = record_comp[handle]->words2bytes(words);
	buf = new char[bytes];
		
	// now loop until channel "grabbed"

	while(arbch_keep_record_channel(handle) && !time_out(handle)) {
		record_buf_sync(handle, buf, bytes);
		if (arbch_keep_record_channel(handle)) {
			byteswrite = fwrite(buf, sizeof(char), bytes, vin);
			if (byteswrite != bytes)
				throw Wobbly(PLAYREC_DISK_FULL);
		}	
	}

	delete [] buf;

	record_buf_finish(handle);
	fclose(vin);

	GenericSetThreadPriorityNormal();
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_voxfile_sync
	AUTHOR..: David Rowe & John Kostogiannis
	DATE....: 20/8/98

	Utility function to record a vox file to a channel.  Function returns when
	recording is finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_voxfile_sync(int handle, char file_name[], unsigned short mode)
//	int				handle		handle of destination device
//	char			file_name[]	name of file to record
//	unsigned short	mode		specifies compression mode
{

	try {
		record_voxfile(handle, file_name, mode);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_voxfile_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_file_async
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to record a file from a channel.  Function returns as 
	soon as recording has started, and places an event on the API Q when 
	recording has finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_file_async(int handle, char file_name[], unsigned short mode)
//	int				handle		handle of destination device
//	char			file_name[]	name of file to record
//	unsigned short	mode		specifies compression mode
//	int				data		VPB_RECORDEND event data
{
	void			*win;
	ushort			words;		// words to download to FIFO
	ushort			bytes;		// num bytes to try to read from file
	char			*buf;		// buffer of bytes read from file
	ushort			b,ch;		// board and channel for this handle

	try {
		// Validation ----------------------------------------------------

		ValidHandleCheck(handle);
		maphndletodev(handle, &b, &ch);
		wave_open_write(&win, file_name, mode);
		record_buf_start(handle, mode);

		// calculate buffer size to fit DSP FIFO

		words = vpb_c->vpbreg(b)->sztxdf[ch]/2;
		bytes = record_comp[handle]->words2bytes(words);
		buf = new char[bytes];
		
		// now loop until end of file in separate thread
		
		RECORD_FILE_ASYNC_ARGS *args = new RECORD_FILE_ASYNC_ARGS;
		args->handle = handle;
		args->win = win;
		args->buf = buf;
		args->bytes = bytes;
		Generic_beginthread(record_file_async_thread, 0, (void*)args);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_file_async"));
	}
	
	return(VPB_OK);
}
		
/*---------------------------------------------------------------------------*\

	FUNCTION: record_file_async_thread
	AUTHOR..: David Rowe
	DATE....: 23/3/98

	Thread used to implement record file function in async mode.  Posts an 
	event on the API Q when it has finished.

	If error occurs posts wobbly event code in data of termination event,
	otherwise posts event on successful termination.

\*--------------------------------------------------------------------------*/

void record_file_async_thread(void *pvargs)
//	void	*args;		arguments structure
{
	RECORD_FILE_ASYNC_ARGS *args = (RECORD_FILE_ASYNC_ARGS*)pvargs;
	int		handle = args->handle;
	void	*win = args->win;
	char	*buf = args->buf;
	ushort	byteswrite;
	ushort	bytes = args->bytes;
	VPB_EVENT	e;

	delete args;
	
	GenericSetThreadPriorityHigh();

	// start recording.....

	try {
		while(arbch_keep_record_channel(handle) && !time_out(handle)) {
			record_buf_sync(handle, buf, bytes);
			if (arbch_keep_record_channel(handle)) {
				byteswrite = vpb_wave_write(win, buf, bytes);
				if (byteswrite != bytes)
					throw Wobbly(PLAYREC_DISK_FULL);
			}
		}
		if(time_out(handle))
			e.data = VPB_RECORD_TIMEOUT;
		delete [] buf;

		vpb_wave_close_write(win);

		e.type = VPB_RECORDEND;
		e.handle = handle;

		record_buf_finish(handle);
		putevt(&e, 0);		
	}

	catch(Wobbly w){
		vpb_wave_close_write(win);
		e.type = VPB_RECORDEND;
		e.handle = handle;
		e.data = w.code;
		try {
			record_buf_finish(handle);
		}
		catch(Wobbly w){w;}

		putevt(&e, 0);		
	}

}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_voxfile_async
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to record a vox file from a channel.  Function returns as 
	soon as recording has started, and places an event on the API Q when 
	recording has finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_voxfile_async(int handle, char file_name[], unsigned short mode)
//	int				handle		handle of destination device
//	char			file_name[]	name of file to record
//	unsigned short	mode		specifies compression mode
//	int				data		VPB_RECORDEND event data
{
	FILE			*vin;
	ushort			words;		// words to download to FIFO
	ushort			bytes;		// num bytes to try to read from file
	char			*buf;		// buffer of bytes read from file
	ushort			b,ch;		// board and channel for this handle

	try {
		// Validation ----------------------------------------------------

		ValidHandleCheck(handle);
		maphndletodev(handle, &b, &ch);
		vin = fopen(file_name, "wb");
		if(vin == NULL)
			throw Wobbly(VPBAPI_RECORD_CANT_OPEN_VOXFILE);
		record_buf_start(handle, mode);

		// calculate buffer size to fit DSP FIFO

		words = vpb_c->vpbreg(b)->sztxdf[ch]/2;
		bytes = record_comp[handle]->words2bytes(words);
		buf = new char[bytes];
		
		// now loop until end of file in separate thread
		
		RECORD_FILE_ASYNC_ARGS *args = new RECORD_FILE_ASYNC_ARGS;
		args->handle = handle;
		args->win = vin;
		args->buf = buf;
		args->bytes = bytes;
		Generic_beginthread(record_voxfile_async_thread, 0, (void*)args);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_voxfile_async"));
	}
	
	return(VPB_OK);
}
		
/*---------------------------------------------------------------------------*\

	FUNCTION: record_voxfile_async_thread
	AUTHOR..: David Rowe & John Kostogiannis
	DATE....: 20/8/98

	Thread used to implement record vox file function in async mode.  Posts an 
	event on the API Q when it has finished.

	If error occurs posts wobbly event code in data of termination event,
	otherwise posts event on successful termination.

\*--------------------------------------------------------------------------*/

void record_voxfile_async_thread(void *pvargs)
//	void	*args;		arguments structure
{
	RECORD_FILE_ASYNC_ARGS *args = (RECORD_FILE_ASYNC_ARGS*)pvargs;
	int		handle = args->handle;
	FILE	*vin = (FILE *)args->win;
	char	*buf = args->buf;
	ushort	byteswrite;
	ushort	bytes = args->bytes;
	VPB_EVENT	e;

	delete args;
	
	GenericSetThreadPriorityHigh();

	// start recording...

	try {
		while(arbch_keep_record_channel(handle) && !time_out(handle)) {
			record_buf_sync(handle, buf, bytes);
			if (arbch_keep_record_channel(handle)) {
				byteswrite = fwrite(buf, sizeof(char), bytes, vin);
				if (byteswrite != bytes)
					throw Wobbly(PLAYREC_DISK_FULL);
			}

		}
		if(time_out(handle))
			e.data = VPB_RECORD_TIMEOUT;
		delete [] buf;

		fclose(vin);

		e.type = VPB_RECORDEND;
		e.handle = handle;

		record_buf_finish(handle);
		putevt(&e, 0);		
	}

	catch(Wobbly w){
		fclose(vin);
		e.type = VPB_RECORDEND;
		e.handle = handle;
		e.data = w.code;
		try {
			record_buf_finish(handle);
		}
		catch(Wobbly w){w;}

		putevt(&e, 0);		
	}

}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_buf_async
	AUTHOR..: John Kostogiannis
	DATE....: 13/4/99

	Utility function to record a buffer from a channel.  Function returns as 
	soon as recording has started, and places an event on the API Q when 
	recording has finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_buf_async(int handle, unsigned short mode,char *mbuf,unsigned long size)
//	int				handle		handle of destination device
//	unsigned short	mode		specifies compression mode
//	char			mbuf[]		pointer to buffer
// unsigned long	size		length of buffer
{
	ushort			words;		// words to upload from FIFO
	ushort			bytes;		// num bytes written to mbuf
	char			*buf;		// buffer of bytes written to mbuf
	ushort			b,ch;		// board and channel for this handle

	try {
		// Validation ----------------------------------------------------

		ValidHandleCheck(handle);
		maphndletodev(handle, &b, &ch);

		record_buf_start(handle, mode);

		// calculate buffer size to fit DSP FIFO

		words = vpb_c->vpbreg(b)->sztxdf[ch]/2;
		bytes = record_comp[handle]->words2bytes(words);
		buf = new char[bytes];
		
		// now loop until end of file in separate thread
		
		RECORD_BUF_ASYNC_ARGS *args = new RECORD_BUF_ASYNC_ARGS;
		args->handle = handle;
		args->buf = buf;
		args->bytes = bytes;
		args->masterbuf = mbuf;
		args->size = size;
		Generic_beginthread(record_buf_async_thread, 0, (void*)args);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_buf_async"));
	}
	
	return(VPB_OK);
}
		
/*---------------------------------------------------------------------------*\

	FUNCTION: record_buf_async_thread
	AUTHOR..: John Kostogiannis
	DATE....: 13/4/99

	Thread used to implement record buf function in async mode.  Posts an 
	event on the API Q when it has finished.

	If error occurs posts wobbly event code in data of termination event,
	otherwise posts event on successful termination.

\*--------------------------------------------------------------------------*/

void record_buf_async_thread(void *pvargs)
//	void	*args;		arguments structure
{
	RECORD_BUF_ASYNC_ARGS *args = (RECORD_BUF_ASYNC_ARGS*)pvargs;
	int		handle = args->handle;
	ulong	size = args->size;
	char	*masterbuf =args->masterbuf;
	char	*buf = args->buf;
	ushort	bytes = args->bytes;
	VPB_EVENT	e;
	ulong length;
	
	delete args;
	
	length = size;

	GenericSetThreadPriorityHigh();

	// start recording...

	try {
		while(arbch_keep_record_channel(handle) && !time_out(handle) && (size>=bytes)) {
			record_buf_sync(handle, buf, bytes);
			if (arbch_keep_record_channel(handle)) {
				memcpy(masterbuf,buf,bytes);
				masterbuf+=bytes;
				size-=bytes;
			}

		}
		// determine the reason for termination

		if(time_out(handle))
			e.data = VPB_RECORD_TIMEOUT;
		if(size<bytes)
			e.data = VPB_RECORD_ENDOFDATA;
		e.data1 = length-size;			
		delete [] buf;
		e.type = VPB_RECORDEND;
		e.handle = handle;

		record_buf_finish(handle);
		putevt(&e, 0);		
	}

	catch(Wobbly w){
		e.type = VPB_RECORDEND;
		e.handle = handle;
		e.data = w.code;
		try {
			record_buf_finish(handle);
		}
		catch(Wobbly w){w;}

		putevt(&e, 0);		
	}

}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_terminate
	AUTHOR..: David Rowe
	DATE....: 24/3/98

	Stops recording immediately. Does nothing if we are not recording.
	
\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_terminate(int handle)
//	int	handle	handle of channel 
{
	try {
		arbch_grab_record_channel(handle);
		arbch_release_record_channel(handle);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_terminate"));
	}
	
	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

					DIGIT TERMINATING FUNCTIONS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: validate_digits
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Determines if the digit string contains valid characters and is of
	valid length.

\*--------------------------------------------------------------------------*/

static void	validate_digits(char *digits)
{
	int i,j;
	
	// validate this entries digit string

	ushort len = strlen(digits);
	if (len > MAX_TERM)
		throw Wobbly(VPBAPI_TERM_DIGITS_STRING_TOO_LONG);

	for(j=0; j<len; j++) {
		// search for digit in table

		char c = toupper(digits[j]);
		int num = -1;
		for(i=0; i<NUM_VALID_DIG; i++) {
			if (valid_dig[i] == c)
				num = i;
		}
		if (num < 0)
			throw Wobbly(VPBAPI_INVALID_TERM_DIGITS);
	}
}

/*---------------------------------------------------------------------------*\

	FUNCTION: digit_match
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Determines if the digit is a member of the term_digit string.

\*--------------------------------------------------------------------------*/

static int digit_match(char digit, char *term_digits)
{
	char *p = term_digits;

	while(*p != 0) {
		if (*p == digit)
			return(1);
		p++;
	}

	return(0);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_diag
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Returns playrec diagnostics.

\*--------------------------------------------------------------------------*/

void playrec_diag(ushort *play, ushort *rec)
{
	*play = fplay_min;
	*rec = frec_max;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_diag_clear
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Resets playrec diagnostics.

\*--------------------------------------------------------------------------*/

void playrec_diag_reset()
{
	fplay_min = 64000;
	frec_max = 0;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_set_gain
	AUTHOR..: David Rowe
	DATE....: 17/8/98

	Sets the gain of the play channel, using the hardware gain of the
	TS5070 codec on the VPB4.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_set_gain(int handle, float gain) 
{
	ushort	b,ch;					// board and channel for this handle
	word	mess[PC_LCODEC_DNGAIN];

	try {
		// validate

		ValidHandleCheck(handle);

		// gain is in range of +/- 12dB for TS5070

		if ((gain > 12.0) || (gain < -12.0))
			throw Wobbly(VPBAPI_GAIN_OUT_OF_RANGE);

		// now scale float gain to 8 bit unsigned int
		// each unit is 0.1 dB, 0dB is 0x80

		ushort scaled_gain = (ushort)(gain*10.0 + 0x80);

		maphndletodev(handle, &b, &ch);
		mess[0] = PC_LCODEC_DNGAIN;
		mess[1] = PC_CODEC_DNGAIN;
		mess[2] = ch;
		mess[3] = scaled_gain;
		vpb_c->PutMessageVPB(b,mess);
		play_gain[handle] = gain;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_set_gain"));
	}
	
	return(VPB_OK);
}
/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_get_gain
	AUTHOR..: David Rowe
	DATE....: 17/8/98

	Gets the gain of the play channel, using the hardware gain of the
	TS5070 codec on the VPB4.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_get_gain(int handle, float *gain) 
{

	try {
		// validate

		ValidHandleCheck(handle);

		*gain = play_gain[handle];	
		
	
		// gain is in range of +/- 12dB for TS5070

		if ((*gain > 12.0) || (*gain < -12.0))
			throw Wobbly(VPBAPI_GAIN_OUT_OF_RANGE);

	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_get_gain"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_set_gain
	AUTHOR..: David Rowe
	DATE....: 17/8/98

	Sets the gain of the record channel, using the hardware gain of the
	TS5070 codec on the VPB4.  I discovered by experiment that the
	gain values work backwards for the record side, ie the maximum
	positive value gives the miminum gain.  Hence I have negated the
	gain values to give the same gain convention as for play, +12 dB max,
	-12 dB min.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_set_gain(int handle, float gain) 
{
	ushort	b,ch;					// board and channel for this handle
	word	mess[PC_LCODEC_UPGAIN];

	try {
		// validate

		ValidHandleCheck(handle);

		// gain is in range of +/- 12dB for TS5070

		if ((gain > 12.0) || (gain < -12.0))
			throw Wobbly(VPBAPI_GAIN_OUT_OF_RANGE);

		// now scale float gain to 8 bit unsigned int
		// each unit is 0.1 dB, 0dB is 0x80

		ushort scaled_gain = (ushort)(-gain*10.0 + 0x80);

		maphndletodev(handle, &b, &ch);
		mess[0] = PC_LCODEC_UPGAIN;
		mess[1] = PC_CODEC_UPGAIN;
		mess[2] = ch;
		mess[3] = scaled_gain;
		vpb_c->PutMessageVPB(b,mess);
		record_gain[handle] = gain;

	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_set_gain"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_get_gain
	AUTHOR..: David Rowe
	DATE....: 17/8/98

	Gets the gain of the record channel, using the hardware gain of the
	TS5070 codec on the VPB4.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_get_gain(int handle, float *gain) 
{

	try {
		// validate

		ValidHandleCheck(handle);

		*gain = record_gain[handle];	
		
	
		// gain is in range of +/- 12dB for TS5070

		if ((*gain > 12.0) || (*gain < -12.0))
			throw Wobbly(VPBAPI_GAIN_OUT_OF_RANGE);

	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_get_gain"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: start_time_out
	AUTHOR..: David Rowe
	DATE....: 21/8/98

	Starts the time out logic for a given record session.

\*--------------------------------------------------------------------------*/

void start_time_out(int handle)
{
	record[handle].time_start = GenerictimeGetTime();
}

/*---------------------------------------------------------------------------*\

	FUNCTION: time_out
	AUTHOR..: David Rowe
	DATE....: 21/8/98

	Returns non-zero if record session has timed out.  Should be called 
	Starts the time out logic for a given record session.

\*--------------------------------------------------------------------------*/

int time_out(int handle)
{
	if (record[handle].time_out)
		return ((record[handle].time_out + record[handle].time_start) < GenerictimeGetTime());
	else
		return(0);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_reset_record_fifo_alarm
	AUTHOR..: David Rowe
	DATE....: 23/9/98

	Resets the latch that detects record DSP buffer overflows.  After
	calling this function another event may be generated if the buffer
	overflows.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_reset_record_fifo_alarm(int handle)
{
	ushort	stobj,b,ch;

	try {
		ValidHandleCheck(handle);
		maphndletodev(handle, &b, &ch);
		stobj = objtrack_handle_to_id(UPOBJ, handle);

		switch (vpb_c->vpbreg(b)->model) {
			case VPB4:
				config_reset_fifo_alarm(vpb_c, b, stobj+10);	// PACKED_FIFO_UP
			break;
			case VPB8L:					
				config_reset_fifo_alarm(vpb_c, b, stobj+6);		// PACKED_FIFO_UP
			break;
			default:
				throw Wobbly(VPBAPI_MODEL_NOT_SUPPORTED);
			break;
		}
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_reset_record_underflow"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_reset_play_fifo_alarm
	AUTHOR..: David Rowe
	DATE....: 23/9/98

	Resets the latch that detects record DSP buffer underflows.  After
	calling this function another event may be generated if the buffer
	underflows.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_reset_play_fifo_alarm(int handle)
{
	ushort	stobj,b,ch;

	try {
		ValidHandleCheck(handle);
		maphndletodev(handle, &b, &ch);
		stobj = objtrack_handle_to_id(DNOBJ, handle);

		assert(vpb_c->vpbreg(b)->model==VPB4);
		config_reset_fifo_alarm(vpb_c, b, stobj);	// PACKED_FIFO_UP
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_reset_record_underflow"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_underflow_valid
	AUTHOR..: David Rowe
	DATE....: 19/10/98

	Returns non-zero if an underflow is valid.  At certain times (like
	when we are waiting for the FIFO to empty at the end of a play
	session), underflow events from the DSP are not valid.

\*--------------------------------------------------------------------------*/

int playrec_underflow_valid(int handle) {
	return(play[handle].underflow_valid);
}
