/*---------------------------------------------------------------------------*\

	FILE....: RING.CPP
	TYPE....: C++ Module
	AUTHOR..: Davo and Kos
	DATE....: 28/7/98

	Ring detection module for VPB API.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>

#include "../vpbapi/apifunc.h"
#include "../vpbapi/ring.h"
#include "../timer/timer.h"
#include "../wobbly/wobbly.h"

/*---------------------------------------------------------------------------*\

								DEFINES

\*--------------------------------------------------------------------------*/

#define	DEF_TIME_OUT		5000
#define	DEF_RINGS_TO_FIRE	2

/*---------------------------------------------------------------------------*\

								TYPEDEFS

\*--------------------------------------------------------------------------*/

typedef struct {
	int		rings_to_fire;	// number of rings to fire VPB_RING event
	int		rings_so_far;	// number of rings so far since reset
	Timer	*timer;			// timer used to detect time out
	int		time_out;		// time out time in ms
} RING;

/*---------------------------------------------------------------------------*\

								STATICS

\*--------------------------------------------------------------------------*/

static RING *ring;			// ptr to array of RING types
static ring_numch;			// keep record of numch for shutdown

/*---------------------------------------------------------------------------*\

								FUNCTIONS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: ring_open
	AUTHOR..: David Rowe
	DATE....: 27/7/98

	Initialises the ring module.

\*--------------------------------------------------------------------------*/

void ring_open(ushort numch) {
	ring = new RING[numch];
	ring_numch = numch;

	int i;
	for(i=0; i<numch; i++) {
		ring[i].rings_to_fire = DEF_RINGS_TO_FIRE;
		ring[i].rings_so_far = 0;
		ring[i].time_out = DEF_TIME_OUT;
		ring[i].timer = new Timer();
	}

}

/*---------------------------------------------------------------------------*\

	FUNCTION: ring_close
	AUTHOR..: David Rowe
	DATE....: 27/7/98

	Closes the ring module.

\*--------------------------------------------------------------------------*/

void ring_close() {
	int i;

	for(i=0; i<ring_numch; i++)
		delete ring[i].timer;

	delete [] ring;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_set_ring
	AUTHOR..: David Rowe
	DATE....: 27/7/98

	Sets the ring parameters for a channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_set_ring(int handle, unsigned int rings_to_fire, unsigned int time_out)
{
	try {
		ValidHandleCheck(handle);
		ring[handle].rings_to_fire = rings_to_fire;
		ring[handle].time_out = time_out;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_set_ring"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_get_ring
	AUTHOR..: David Rowe
	DATE....: 27/7/98

	Gets the ring parameters for a channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_get_ring(int handle, unsigned int *rings_to_fire, unsigned int *time_out)
{
	try {
		ValidHandleCheck(handle);
		*rings_to_fire = ring[handle].rings_to_fire;
		*time_out = ring[handle].time_out;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_get_ring"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: ring_ring
	AUTHOR..: David Rowe
	DATE....: 27/7/98

	Called by the MMQ when DSP detects a ring event.

\*--------------------------------------------------------------------------*/

void ring_ring(int h) {
	VPB_EVENT	e;
	ushort		time_out;

	if (ring[h].rings_so_far == 0) {

		// first ring, start timer

		ring[h].timer->timer_start();
	} else {

		// subsequent rings, check timer

		ring[h].timer->timer_check_time_out_ms(ring[h].time_out,
											  &time_out);
		
		// if time out reset counter and timer

		if (time_out == TIME_OUT) {
			ring[h].rings_so_far = 0;
			ring[h].timer->timer_start();
		}
	}

	ring[h].rings_so_far++;

	if (ring[h].rings_so_far >= ring[h].rings_to_fire) {	
		e.type = VPB_RING;
		e.handle = h;
		e.data = 0;
		putevt(&e, VPB_MRING);
		ring[h].rings_so_far = 0;
	}
}
