/*---------------------------------------------------------------------------*\

    FILE....: VPBAGC.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 18/6/98

	This file contains the VPB API functions required to control the
	AGC DSP firmware.
	
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

							INCLUDES
							
\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "../comm/comm.h"
#include "../config/config.h"
#include "../vpbapi/apifunc.h"
#include "../vpbapi/mapdev.h"
#include "../vpbapi/vpbagc.h"
#include "../vpbapi/vpbapi.h"
#include "../vpbapi/vpbhandle.h"
#include "../vpbapi2/arbch.h"
#include "../vpbapi2/objtrack.h"
#include "../wobbly/wobbly.h"

/*---------------------------------------------------------------------------*\

								DEFINES
							
\*---------------------------------------------------------------------------*/

// defaults

#define	DEFAULT_EX				50		// energy filter output
#define	DEFAULT_SETPOINT		-20.0
#define	DEFAULT_ATTACK			0.5
#define	DEFAULT_DECAY			0.992
#define	DEFAULT_DIVISOR			1		// only used for debug

#define	AGCMAX				0.0			// 0 dB maximum
#define	AGCMAXLIN			32767		// 0 dB maximum
#define	ATTACKMAX			1.0
#define	DECAYMAX			1.0

/* These will need to be constructed for each new channel */

typedef struct {
	ushort id;			// object ID	
    ushort ex;			// input energy				
    ushort setpoint;	// energy setpoint			
    ushort attack1;		// energy filter attack 		
    ushort attack2;		// energy filter attack conjugate	
    ushort decay;		// energy filter decay			
    short divisor;		// input signal level			
						// used to scale input signal manually	
} AGCCONFIG;

/*---------------------------------------------------------------------------*\

								GLOBALS
							
\*---------------------------------------------------------------------------*/

static AGCCONFIG	*agcconfig;	// ptr to array of AGC structures
								// stores vox config for each channel

/*-------------------------------------------------------------------------*\

							FUNCTION HEADERS

\*-------------------------------------------------------------------------*/


/*-------------------------------------------------------------------------*\

								FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbagc_open()
	AUTHOR...: David Rowe
	DATE.....: 18/6/98

	Initialises the agc module.  This function should be called when the API
	is initialised.  This function loads up the default agc parameters for
	each channel.

\*--------------------------------------------------------------------------*/

void vpbagc_open(ushort numch) {
	int		i;
	word	m[PC_LAGC_UPDATE];
	ushort	b,ch;

	CheckNew(agcconfig = new AGCCONFIG[numch]);	

	// config DSP state variables and mirror in PC

	m[0] = PC_LAGC_UPDATE;
	m[1] = PC_AGC_UPDATE;

	for(i=0; i<numch; i++) {
		maphndletodev(i, &b, &ch);

		agcconfig[i].id = objtrack_handle_to_id(AGCOBJ, i);
		agcconfig[i].ex = DEFAULT_EX;
		agcconfig[i].setpoint = (ushort)(AGCMAXLIN*pow(10.0, DEFAULT_SETPOINT/20.0));
		agcconfig[i].attack1 = (ushort)(DEFAULT_ATTACK*AGCMAXLIN);
		agcconfig[i].attack2 = (ushort)((1.0-DEFAULT_ATTACK)*AGCMAXLIN);
		agcconfig[i].decay = (ushort)(DEFAULT_DECAY*AGCMAXLIN);
		agcconfig[i].divisor = DEFAULT_DIVISOR;

		m[2] = agcconfig[i].id;
		m[3] = agcconfig[i].ex;
		m[4] = agcconfig[i].setpoint;
		m[5] = agcconfig[i].attack1;
		m[6] = agcconfig[i].attack2;
		m[7] = agcconfig[i].decay;
		m[8] = agcconfig[i].divisor;
		vpb_c->PutMessageVPB(b,m);
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbagc_close()
	AUTHOR...: David Rowe
	DATE.....: 19/6/98

	Closes down vpbagc module.

\*--------------------------------------------------------------------------*/

void vpbagc_close() {
	delete [] agcconfig;	
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_setagc()
	AUTHOR...: David Rowe
	DATE.....: 16/6/98

	Sets the agc parameters for a specified channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_setagc(int handle, VPB_AGC *agc)
//	int				handle			handle of channel device
//	AGC				*agc			AGC parameters 
{
	ushort			b;				// board number
	ushort			ch;				// channel on board b
	word			m[PC_LAGC_UPDATE];

	try {
		// validate

		ValidHandleCheck(handle);
		if (agc->setpoint > AGCMAX)
			throw Wobbly(VPBAPI_AGC_SETPOINT_TOO_BIG);
		if (agc->attack > ATTACKMAX)
			throw Wobbly(VPBAPI_AGC_ATTACK_TOO_BIG);
		if (agc->attack < AGCMAX)
			throw Wobbly(VPBAPI_AGC_ATTACK_TOO_SMALL);
		if (agc->decay > ATTACKMAX)
			throw Wobbly(VPBAPI_AGC_DECAY_TOO_BIG);
		if (agc->decay < AGCMAX)
			throw Wobbly(VPBAPI_AGC_DECAY_TOO_SMALL);
		
		maphndletodev(handle, &b, &ch);

		// update PC mirror of agc config

		agcconfig[handle].setpoint = (ushort)(AGCMAXLIN*pow(10.0, agc->setpoint/20.0));
		agcconfig[handle].attack1 = (ushort)(agc->attack*AGCMAXLIN);
		agcconfig[handle].attack2 = (ushort)((1.0-agc->attack)*AGCMAXLIN);
		agcconfig[handle].decay = (ushort)(agc->decay*AGCMAXLIN);

		// send new config to DSP
		
		m[0] = PC_LAGC_UPDATE;
		m[1] = PC_AGC_UPDATE;
		m[2] = agcconfig[handle].id;
		m[3] = agcconfig[handle].ex;
		m[4] = agcconfig[handle].setpoint;
		m[5] = agcconfig[handle].attack1;
		m[6] = agcconfig[handle].attack2;
		m[7] = agcconfig[handle].decay;
		m[8] = agcconfig[handle].divisor;
		vpb_c->PutMessageVPB(b,m);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_setagc"));
	}
	
	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_getagc()
	AUTHOR...: David Rowe
	DATE.....: 16/6/98

	Gets the agc parameters for a specified channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_getagc(int handle, VPB_AGC *agc)
//	int				handle			handle of channel device
//	AGC				*agc			ptr to returned AGC parameters 
{
	VPBREG			*v;
	ushort			b,ch;

	try {
		ValidHandleCheck(handle);
		maphndletodev(handle, &b, &ch);
		v = vpb_c->vpbreg(b);

		agc->setpoint = (float)20.0*log10((float)agcconfig[handle].setpoint/AGCMAXLIN);
		agc->attack = ((float)agcconfig[handle].attack1)/AGCMAXLIN;
		agc->decay = ((float)agcconfig[handle].decay)/AGCMAXLIN;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_getagc"));
	}
	
	return(VPB_OK);
}

