/*---------------------------------------------------------------------------*\

    FILE....: VPBFILEMAN.CPP
    TYPE....: C++ Functions
    AUTHOR..: John Kostogiannis
    DATE....: 7/11/97

    Basic C File I/O functions that have error trapping shells around them.
	
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <io.h>
#include <errno.h>

#include "..\vpbapi\apifunc.h"
#include "..\vpbapi\vpbapi.h"		// funcs in this module exported to API
#include "..\vpbapi\vpbfileman.h"	// funcs in this module internal to API
#include "..\wobbly\wobbly.h"

/*-------------------------------------------------------------------------*\

								FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: fileopen()
	AUTHOR...: John Kostogiannis
	DATE.....: 28/11/97

	Creates a file handle - internal version.

\*--------------------------------------------------------------------------*/

int fileopen(const char *filep, int flags,int pmode)
{
	int filehandle;

	filehandle=_open(filep,flags,pmode);
	if(filehandle == -1){
		switch(errno){
			case EACCES:
				throw Wobbly(VPBAPI_FILEOPEN_CANT_WRITE_TO_READ_ONLY_FILE);
			break;
			case EEXIST:
				throw Wobbly(VPBAPI_FILEOPEN_FILENAME_ALREADY_EXISTS);
			break;
			case EINVAL:
				throw Wobbly(VPBAPI_FILEOPEN_INVALID_ARGUMENT);
			break;
			case EMFILE:
				throw Wobbly(VPBAPI_FILEOPEN_TOO_MANY_OPEN_FILES);
			break;
			case ENOENT:
				throw Wobbly(VPBAPI_FILEOPEN_FILE_OR_PATH_NOT_FOUND);
			break;
			default:	
				throw Wobbly(VPBAPI_FILEOPEN_CANT_CREATE_FILEHANDLE);
		}
	}

	return(filehandle);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_fileopen()
	AUTHOR...: John Kostogiannis
	DATE.....: 28/11/97

	Creates a file handle - API version.

\*--------------------------------------------------------------------------*/

int vpb_fileopen(const char *filep, int flags,int pmode)
{
	int filehandle;

	try {
		filehandle = fileopen(filep, flags, pmode);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_fileopen"));
	}

	return(filehandle);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: fileclose()
	AUTHOR...: John Kostogiannis
	DATE.....: 28/11/97

	Closes a file handle - internal version.

\*--------------------------------------------------------------------------*/

int fileclose(int filehandle)
{
	int indicator;

	indicator=_close(filehandle);
	if(indicator == -1) {
		throw Wobbly(VPBAPI_FILECLOSE_INVALID_FILEHANDLE);
	}

	return(indicator);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_fileclose()
	AUTHOR...: John Kostogiannis
	DATE.....: 28/11/97

	Closes a file handle - API version.

\*--------------------------------------------------------------------------*/

int vpb_fileclose(int filehandle)
{
	int indicator;

	try {
		indicator = fileclose(filehandle);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_fileclose"));
	}

	return(indicator);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: fileseek()
	AUTHOR...: John Kostogiannis
	DATE.....: 28/11/97

	Moves a file pointer to a specified location - internal version.

\*--------------------------------------------------------------------------*/

long fileseek(int filehandle,long offset,int origin)
{
	long offsetinbytes;
	
	offsetinbytes=_lseek(filehandle,offset,origin);
	if(offsetinbytes == -1L){
		switch(errno){
			case EBADF:
				throw Wobbly(VPBAPI_FILESEEK_INVALID_FILEHANDLE);
			break;
			case EINVAL:	
				throw Wobbly(VPBAPI_FILESEEK_INVALID_ARGUMENT);
			break;
			default:
				throw Wobbly(VPBAPI_FILESEEK_CANT_MOVE_POINTER_TO_SPECIFIED_LOCATION);
		}
	}

	return(offsetinbytes);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_fileseek()
	AUTHOR...: John Kostogiannis
	DATE.....: 28/11/97

	Moves a file pointer to a specified location - API version.

\*--------------------------------------------------------------------------*/

long vpb_fileseek(int filehandle,long offset,int origin)
{
	long offsetinbytes;

	try {
		offsetinbytes = fileseek(filehandle, offset, origin);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_fileseek"));
	}

	return(offsetinbytes);
}
	
/*--------------------------------------------------------------------------*\

	FUNCTION.: fileread()
	AUTHOR...: John Kostogiannis
	DATE.....: 28/11/97

	Returns number of bytes read from a file - internal version.

\*--------------------------------------------------------------------------*/

int fileread(int filehandle,void *buffer,unsigned int count)
{
	int countinbytes;

	countinbytes=_read(filehandle,buffer,count);
	if(countinbytes == -1){
		switch(errno){
			case EBADF:
				throw Wobbly(VPBAPI_FILEREAD_INVALID_FILEHANDLE);
			break;
			default:
				throw Wobbly(VPBAPI_FILEREAD_CANT_READ_FILE);
		}
	}

	return(countinbytes);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_fileread()
	AUTHOR...: John Kostogiannis
	DATE.....: 28/11/97

	Returns number of bytes read from a file - API version.

\*--------------------------------------------------------------------------*/

int vpb_fileread(int filehandle,void *buffer,unsigned int count)
{
	int countinbytes;

	try {
		countinbytes = fileread(filehandle, buffer, count);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_fileread"));
	}

	return(countinbytes);
}


/*--------------------------------------------------------------------------*\

	FUNCTION.: filewrite()
	AUTHOR...: John Kostogiannis
	DATE.....: 28/11/97

	Returns number of bytes written to a file - internal version.

\*--------------------------------------------------------------------------*/

int filewrite(int filehandle,void *buffer,unsigned int count)
{
	int countinbytes;

	countinbytes=_write(filehandle,buffer,count);
	if(countinbytes == -1){
		switch(errno){
			case EBADF:
				throw Wobbly(VPBAPI_FILEWRITE_INVALID_FILEHANDLE);
			break;
			case ENOSPC:
				throw Wobbly(VPBAPI_FILEWRITE_NOT_ENOUGH_MEMORY_SPACE);
			break;
			default:
				throw Wobbly(VPBAPI_FILEWRITE_CANT_WRITE_TO_FILE);
		}
	}

	return(countinbytes);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_filewrite()
	AUTHOR...: John Kostogiannis
	DATE.....: 28/11/97

	Returns number of bytes written to a file - API version.

\*--------------------------------------------------------------------------*/

int vpb_filewrite(int filehandle,void *buffer,unsigned int count)
{
	int countinbytes;

	try {
		countinbytes = filewrite(filehandle,buffer,count);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_filewrite"));
	}

	return(countinbytes);
}
