/*---------------------------------------------------------------------------*\

    FILE....: VPBVOX.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 18/6/98

	This file contains the VPB API functions required to control the
	VOX firmware.

  6/10/98 IR	Throw wobbly if user wants runon of 0
	
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

							INCLUDES
							
\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "../comm/comm.h"
#include "../config/config.h"
#include "../vpbapi/apifunc.h"
#include "../vpbapi/mapdev.h"
#include "../vpbapi/vpbvox.h"
#include "../vpbapi/vpbapi.h"
#include "../vpbapi/vpbhandle.h"
#include "../vpbapi2/arbch.h"
#include "../vpbapi2/objtrack.h"
#include "../wobbly/wobbly.h"

/*---------------------------------------------------------------------------*\

								DEFINES
							
\*---------------------------------------------------------------------------*/

// defaults

#define DEFAULT_ONLEVEL		-12.0	// on level	
#define DEFAULT_OFFLEVEL	-18.0	// off level
#define	DEFAULT_RUNON		2000	// runon in ms
#define	VOXMAX			-2.0	// -2 dB maximum
#define	VOXMIN			-36.0	// -36 dB minimum
#define	VOXRUNMAX		30000	// 30 sec max runon
#define	VOXRUNMIN		50	// 50 ms min runon
#define	VOXMAXLIN		32767	// 0 dB maximum
#define	RUNONMAX		65535   // max runon 65.5 seconds
#define	FS			8000	// sampling rate	

typedef struct {
   ushort	id;			//  object ID
   ushort	onlevel;	// signal has to exceed this level
   ushort	offlevel;	// countdown starts when signal drops below
   ushort 	runon;		// how many frames to count down		
} VOXCONFIG;

/*---------------------------------------------------------------------------*\

								GLOBALS
							
\*---------------------------------------------------------------------------*/

static VOXCONFIG	*voxconfig;	// ptr to array of VOX structures
								// stores vox config for each channel

/*-------------------------------------------------------------------------*\

							FUNCTION HEADERS

\*-------------------------------------------------------------------------*/


/*-------------------------------------------------------------------------*\

								FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbvox_open()
	AUTHOR...: David Rowe
	DATE.....: 18/6/98

	Initialises the vox module.  This function should be called when the API
	is initialised.  This function load.s up the default VOX parameters for
	each channel.

\*--------------------------------------------------------------------------*/

void vpbvox_open(ushort numch) {
	int		i;
	word	m[PC_LVOX_UPDATE];
	ushort	b,ch;
	VPBREG	*v;

	CheckNew(voxconfig = new VOXCONFIG[numch]);	

	// config DSP state variables and mirror in PC

	m[0] = PC_LVOX_UPDATE;
	m[1] = PC_VOX_UPDATE;

	for(i=0; i<numch; i++) {
		maphndletodev(i, &b, &ch);
		v = vpb_c->vpbreg(b);

		voxconfig[i].id = objtrack_handle_to_id(VOXOBJ, i);
		voxconfig[i].onlevel = (ushort)(VOXMAXLIN*pow(10.0, DEFAULT_ONLEVEL/20.0));
		voxconfig[i].offlevel = (ushort)(VOXMAXLIN*pow(10.0, DEFAULT_OFFLEVEL/20.0));
		voxconfig[i].runon = (ushort)((float)DEFAULT_RUNON/1000)*(FS/v->lsf);

		m[2] = voxconfig[i].id;
		m[3] = voxconfig[i].onlevel;
		m[4] = voxconfig[i].offlevel;
		m[5] = voxconfig[i].runon;
		vpb_c->PutMessageVPB(b,m);
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbvox_close()
	AUTHOR...: David Rowe
	DATE.....: 19/6/98

	Closes down vpbvox module.

\*--------------------------------------------------------------------------*/

void vpbvox_close() {
	delete [] voxconfig;	
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_setvox()
	AUTHOR...: David Rowe
	DATE.....: 16/6/98

	Sets the vox parameters for a specified channel.

\-------------------------------------------------------------------------*/

int WINAPI vpb_setvox(int handle, VPB_VOX *vox)
//	int				handle			handle of channel device
//	VOX				*vox			VOX parameters 
{
	ushort			b;				// board number
	ushort			ch;				// channel on board b
	word			m[PC_LVOX_UPDATE];
	VPBREG			*v;
	ushort			runon_frames;
	int				frames_sec;
	float			runon_secs;

	try {
		// validate

		ValidHandleCheck(handle);
		if (vox->onlevel > VOXMAX)
			throw Wobbly(VPBAPI_VOX_ONLEVEL_TOO_BIG);
		if (vox->onlevel < VOXMIN)
			throw Wobbly(VPBAPI_VOX_ONLEVEL_TOO_SMALL);
		if (vox->offlevel > VOXMAX)
			throw Wobbly(VPBAPI_VOX_OFFLEVEL_TOO_BIG);
		if (vox->offlevel < VOXMIN)
			throw Wobbly(VPBAPI_VOX_OFFLEVEL_TOO_SMALL);
		if (vox->runon > VOXRUNMAX)
			throw Wobbly(VPBAPI_VOX_RUNON_TOO_BIG);
		if (vox->runon < VOXRUNMIN)
			throw Wobbly(VPBAPI_VOX_RUNON_TOO_SMALL);
		maphndletodev(handle, &b, &ch);
		v = vpb_c->vpbreg(b);
		runon_secs = vox->runon/(float)1000.0;
		frames_sec =  FS/v->lsf;
		runon_frames = (ushort)(runon_secs*frames_sec);

		if (runon_frames > RUNONMAX)
			throw Wobbly(VPBAPI_VOX_RUNON_TOO_BIG);

		// update PC mirror of vox config

		voxconfig[handle].onlevel = (ushort)(VOXMAXLIN*pow(10.0, vox->onlevel/20.0));
		voxconfig[handle].offlevel = (ushort)(VOXMAXLIN*pow(10.0, vox->offlevel/20.0));
		voxconfig[handle].runon = runon_frames;
		
		// update DSPs vox params

		m[0] = PC_LVOX_UPDATE;
		m[1] = PC_VOX_UPDATE;
		m[2] = voxconfig[handle].id;
		m[3] = voxconfig[handle].onlevel;
		m[4] = voxconfig[handle].offlevel;
		m[5] = voxconfig[handle].runon;
		vpb_c->PutMessageVPB(b,m);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_setvox"));
	}
	
	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_getvox()
	AUTHOR...: David Rowe
	DATE.....: 16/6/98

	Gets the vox parameters for a specified channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_getvox(int handle, VPB_VOX *vox)
//	int				handle			handle of channel device
//	VOX				*vox			ptr to returned VOX parameters 
{
	VPBREG			*v;
	ushort			b,ch;
	long			runon;

	try {
		ValidHandleCheck(handle);
		maphndletodev(handle, &b, &ch);
		v = vpb_c->vpbreg(b);

		vox->onlevel = (float)20.0*log10((float)voxconfig[handle].onlevel/VOXMAXLIN);
		vox->offlevel = (float)20.0*log10((float)voxconfig[handle].offlevel/VOXMAXLIN);
		runon = (long)((float)voxconfig[handle].runon * 1000.0) / (FS/v->lsf);
		vox->runon = (ushort)runon;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_getvox"));
	}
	
	return(VPB_OK);
}

