/*---------------------------------------------------------------------------*\

    FILE....: ARBCH.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 19/3/98

    This module arbitrates the use of channels.  This is required because
	different threads may request the use of a given input or output channel
	at the same time.
	 
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <assert.h>

#include "../comm/comm.h"
#include "../wobbly/wobbly.h"
#include "../vpbapi2/arbch.h"
#include "../message/mess.h"
#include "../generic/generic.h"

#define	TIME_OUT	10000	// time out for grabbbing channel in ms

/*---------------------------------------------------------------------------*\

								STATICS

\*--------------------------------------------------------------------------*/

static GENERIC_CRITICAL_SECTION	*GrabPlaySect, *GrabRecordSect;
static GENERIC_CRITICAL_SECTION	PlaySect,RecordSect;
static ushort *play_channel_in_use_array;
static ushort *record_channel_in_use_array;
static ushort sleepms;

/*---------------------------------------------------------------------------*\

								FUNCTIONS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: arbch_open
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Initialises the arbch module.

\*--------------------------------------------------------------------------*/

void arbch_open(ushort numch) {
	GrabPlaySect = new GENERIC_CRITICAL_SECTION[numch];
	GrabRecordSect = new GENERIC_CRITICAL_SECTION[numch];
	play_channel_in_use_array = new ushort[numch];
	record_channel_in_use_array = new ushort[numch];
	sleepms = 20;

	int i;
	for(i=0; i<numch; i++) {
		GenericInitializeCriticalSection(&GrabPlaySect[i]);
		GenericInitializeCriticalSection(&GrabRecordSect[i]);
		play_channel_in_use_array[i] = 0;
		record_channel_in_use_array[i] = 0;
	}

	GenericInitializeCriticalSection(&PlaySect);
	GenericInitializeCriticalSection(&RecordSect);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: arbch_close
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Closes the playrec module.

\*--------------------------------------------------------------------------*/

void arbch_close(ushort numch) {
	int i;
	for(i=0; i<numch; i++) {
		GenericDeleteCriticalSection(&GrabPlaySect[i]);
		GenericDeleteCriticalSection(&GrabRecordSect[i]);
	}
	delete [] play_channel_in_use_array;
	delete [] record_channel_in_use_array;
	delete [] GrabPlaySect;
	delete [] GrabRecordSect;
	GenericDeleteCriticalSection(&PlaySect);
	GenericDeleteCriticalSection(&RecordSect);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_channel_in_use
	AUTHOR..: David Rowe
	DATE....: 24/3/98

	Thread safe access function for play_channel_in_use_array.

\*--------------------------------------------------------------------------*/

int play_channel_in_use(ushort handle)
//	ushort	handle		handle of channel
{
	GenericEnterCriticalSection(&PlaySect);
	int ret = play_channel_in_use_array[handle];
	GenericLeaveCriticalSection(&PlaySect);
	return(ret);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: set_play_channel_in_use
	AUTHOR..: David Rowe
	DATE....: 24/3/98

	Thread safe access function for play_channel_in_use_array.

\*--------------------------------------------------------------------------*/

void set_play_channel_in_use(ushort handle, int val)
//	ushort	handle		handle of channel
{
	GenericEnterCriticalSection(&PlaySect);
	play_channel_in_use_array[handle] = val;
	GenericLeaveCriticalSection(&PlaySect);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_channel_in_use
	AUTHOR..: David Rowe
	DATE....: 24/3/98

	Thread safe access function for record_channel_in_use_array.

\*--------------------------------------------------------------------------*/

int record_channel_in_use(ushort handle)
//	ushort	handle		handle of channel
{
	GenericEnterCriticalSection(&RecordSect);
	int ret = record_channel_in_use_array[handle];
	GenericLeaveCriticalSection(&RecordSect);
	return(ret);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: set_record_channel_in_use
	AUTHOR..: David Rowe
	DATE....: 24/3/98

	Thread safe access function for record_channel_in_use_array.

\*--------------------------------------------------------------------------*/

void set_record_channel_in_use(ushort handle, int val)
//	ushort	handle		handle of channel
{
	GenericEnterCriticalSection(&RecordSect);
	record_channel_in_use_array[handle] = val;
	GenericLeaveCriticalSection(&RecordSect);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: arbch_grab_play_channel
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Attempts to grab a play channel for the caller.  Returns when the channel
	has been grabbed.  If the channel is in use, then the function using the
	channel must call arbch_keep_play_channel() periodically to determine
	if the channel is being requested by another thread.  Otherwise this
	function will throw a wobbly after a certain time to avoid hanging.

\*--------------------------------------------------------------------------*/

void arbch_grab_play_channel(ushort handle)
//	ushort	handle		handle of channel to grab
{
	GenericEnterCriticalSection(&GrabPlaySect[handle]);

	if (play_channel_in_use(handle)) {
		
		// signal function using handle to give it up!
	
		set_play_channel_in_use(handle,2);

		// wait until channel availible or time out
		
		ushort timer = 0;
		while (play_channel_in_use(handle)) {
			GenericSleep(sleepms);
			timer++;
			if (timer > TIME_OUT/sleepms)
				throw Wobbly(ARBCH_PLAY_CHANNEL_IN_USE);
		}
	}

	set_play_channel_in_use(handle,1);
	
	GenericLeaveCriticalSection(&GrabPlaySect[handle]);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: arbch_release_play_channel
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Releases a play channel, making it available.

\*--------------------------------------------------------------------------*/

void arbch_release_play_channel(ushort handle)
//	ushort	handle		handle of channel to grab
{
	assert(play_channel_in_use(handle));
	set_play_channel_in_use(handle,0);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: arbch_grab_record_channel
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Attempts to grab a record channel for the caller.  Returns without 
	incident if channel grabbed, otherwise throws a wobbly.

\*--------------------------------------------------------------------------*/

void arbch_grab_record_channel(ushort handle)
//	ushort	handle		handle of channel to grab
{
	GenericEnterCriticalSection(&GrabRecordSect[handle]);

	if (record_channel_in_use(handle)) {
		
		// signal function using handle to give it up!
		set_record_channel_in_use(handle,2);

		// wait until channel availible or time out
		
		ushort timer = 0;
		while (record_channel_in_use(handle)) {
			GenericSleep(sleepms);
			timer++;
			if (timer > TIME_OUT/sleepms)
				throw Wobbly(ARBCH_RECORD_CHANNEL_IN_USE);
		}
/*		mess_mprintf_on("releasetime.txt","a");
		mprintf("%d\n",timer*sleepms);
		mess_mprintf_off();*/

	}

	set_record_channel_in_use(handle,1);
	
	GenericLeaveCriticalSection(&GrabRecordSect[handle]);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: arbch_release_record_channel
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Releases a record channel, making it available.

\*--------------------------------------------------------------------------*/

void arbch_release_record_channel(ushort handle)
//	ushort	handle		handle of channel to grab
{
	assert(record_channel_in_use(handle));
	set_record_channel_in_use(handle,0);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: arbch_keep_play_channel
	AUTHOR..: David Rowe
	DATE....: 24/3/98

	Returns non-zero if no one else has requested the use of this channel.
	This function should be called periodically by the functions using the
	channel to allow other functions to "grab" the channel.

\*--------------------------------------------------------------------------*/

int arbch_keep_play_channel(ushort handle)
//	ushort	handle		handle of channel in use
{
	return(play_channel_in_use(handle)==1);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: arbch_keep_record_channel
	AUTHOR..: David Rowe
	DATE....: 24/3/98

	Returns non-zero if no one else has requested the use of this channel.
	This function should be called periodically by the functions using the
	channel to allow other functions to "grab" the channel.

\*--------------------------------------------------------------------------*/

int arbch_keep_record_channel(ushort handle)
//	ushort	handle		handle of channel in use
{
	return(record_channel_in_use(handle)==1);
}
