
/*---------------------------------------------------------------------------*\

    FILE....: VPBREGWINNT.CPP
    TYPE....: WIN32 C++ Function
    AUTHOR..: John Kostogiannis
    DATE....: 6/11/97

    Functions used to load configuration database from the PC into the
    VPB DSP.  This database is known as the DSP Registry.  Each VPB has its
    own registry.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#define STRICT
#include <windows.h>
#include <winnt.h>
#include <winreg.h>
#include <winbase.h>
extern "C" {
#include <assert.h>
#include <stdlib.h>
#include <malloc.h>
#include <stdio.h>
#include <ctype.h>
}
#include "../Wobbly/Wobbly.h"
#include "../vpbreg/vpbreg.h"
#include "../message/mess.h"
#include "../vpbapi/vpbapi.h"
#include "../vpbapi/apifunc.h"
#include "../vpbapi/vpbtoned.h"

/*-------------------------------------------------------------------------*\

				DEFINES

\*-------------------------------------------------------------------------*/

#define	MS		80		/* maximum string size */

#define	CONFIGSTR		"ForcedConfig"
#define	NUMCHSTR		"channels"
#define	SIZEMESSSTR		"sizemess"
#define	SIZERXFIFOSTR	"sizerxfifo"
#define	SIZETXFIFOSTR	"sizetxfifo"
#define	LSF				"lsf"
#define	FIRMWARE		"firmware"
#define	MODEL			"model"
#define	BASE			"base"
#define	DDMODEL			"ddmodel"
#define	SZRELAYBUF		"szRelayBuf"

/*-------------------------------------------------------------------------*\

			   FUNCTION HEADERS

\*-------------------------------------------------------------------------*/


/*-------------------------------------------------------------------------*\

			      CLASS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: ~VPBRegister
	AUTHOR...: John Kostogiannis
	DATE.....: 18/11/97

	Closes the VPB registry.

\*--------------------------------------------------------------------------*/

VPBRegister::~VPBRegister(){
	
	// free allocated VPB registry memory 

	assert(reg != NULL);
    free(reg);
}

	
/*--------------------------------------------------------------------------*\

	FUNCTION.: VPBRegister
	AUTHOR...: John Kostogiannis
	DATE.....: 6/11/97

	Creates the VPB registry by reading the system's registry.

\*--------------------------------------------------------------------------*/

VPBRegister::VPBRegister(ushort *num)
/*  VPBREG **reg;	ptr to array of VPBREG structures						*/
/*  ushort *num;    number of VPBs											*/
{
   
	OSVERSIONINFO os;		// operating system info
	VPBREG *preg;			/* ptr to current VPB registry					*/
    ushort    i,j;
	
	HKEY hKeyparm,hKey;		/* handles for key entries in the registry		*/

	ulong *subkeybufsize,*psubkeybufsize;	/* ptr to key size buffer		*/
	char *subkeybuf,*psubkeybuf;			/* ptr to key name buffer		*/
	FILETIME	lastmodified;					
	
	char	path[MS];
	unsigned char	buffer[MS];				/* holds the parameter value	*/		
	ulong	bufdata = 0;
	ulong	size;							/* size in bytes of buffer		*/
	DWORD	lpType;
	LPBYTE	lpData;

	// determine operating system

	os.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	BOOL ret = GetVersionEx(&os);
	if (ret == 0)
		throw Wobbly(VPBREG_CANT_GET_OS_VER);
	
	// initialise appropriate port class for this OS

	switch(os.dwPlatformId) {
		case VER_PLATFORM_WIN32_WINDOWS:
			strcpy(path, WIN95_PATH);			
			mprintf("\nWin 95\n");
		break;
		case VER_PLATFORM_WIN32_NT:
			strcpy(path, WINNT_PATH);			
			mprintf("\nWin NT\n");
		break;
		default:
			throw Wobbly(VPBREG_OS_NOT_SUPPORTED);
	}

	/* validate arguments */

    //assert(reg != NULL);
    assert(num != NULL);

	subkeybuf = (char *)malloc(sizeof(char)*MAXCH*MS);
	assert(subkeybuf!=NULL);

    
	subkeybufsize = (ulong *)malloc(sizeof(ulong)*MAXCH);
	assert(subkeybufsize!=NULL);
		
	
	psubkeybuf = subkeybuf;
	psubkeybufsize = subkeybufsize;
	

	for(i=0;i<MAXCH;i++)
		*psubkeybufsize++ = MS;
		
	psubkeybufsize = subkeybufsize;
	
	/* open system's registry to the key depicted by the path */

    if(RegOpenKeyEx(HKEY_LOCAL_MACHINE,path,NULL,KEY_ALL_ACCESS,&hKeyparm)!=ERROR_SUCCESS){
		// error in opening the key entries in the registry, thus throw a Wobbly
		throw Wobbly(VPBREG_REG_KEY_ACCESS_FAIL);
	}
	
	// scan registry to determine number of VPB devices in the system	
	
	*num = 0;
	int subkeys = 0;
	while(RegEnumKeyEx(hKeyparm,subkeys,psubkeybuf+(*num)*MS,psubkeybufsize+*num,NULL,NULL,NULL,&lastmodified) !=ERROR_NO_MORE_ITEMS){

		// only count numeric keys, to distinuigh from "tones" and "enum"
		// whic are other non-card reg entries

		char *subkeyname = psubkeybuf+(*num)*MS;
		if (isdigit(*subkeyname)) {
			(*num)++;
		}
	
		subkeys++;
	}
	
	/* check if any devices detected */

	if(*num == 0)
		throw Wobbly(VPBREG_NO_VPB_DEVICES_DETECTED);
	
	/* check if more than max devices have been detected */
	
	if(*num>MAX_VPB)
		throw Wobbly(VPBREG_TOO_MANY_VPBS_ACCESSED);

	/* Allocate memory for VPB DSP registry */

	if(((reg) = (VPBREG*)malloc(sizeof(VPBREG)*(*num)))==NULL)
		throw Wobbly(VPBREG_CANT_ALLOCATE_MEMORY);

	preg = reg;

	psubkeybuf = subkeybuf;
	psubkeybufsize = subkeybufsize;
	
	// Note: (NT only) Valid subkeys must come before "Enum" subkey 
	// as following code assumes unwanted "Enum" subkey is after the
	// vaild keys.  Use names alphabetically in front of Enum to
	// ensure this, eg 0000, 00001, 0002,....

	for(i=0;i<(*num);i++)
	{
				
		/* open VPB device sub-key */   
	
		if(RegOpenKeyEx(hKeyparm,subkeybuf+(i*MS),NULL,KEY_ALL_ACCESS,&hKey)!=ERROR_SUCCESS){
		
			// error in opening the VPB device key  in the registry
			throw Wobbly(VPBREG_REG_KEY_ACCESS_FAIL);
		}

		// Need large size

		size = 40*sizeof(unsigned long);

		switch(os.dwPlatformId) {
			case VER_PLATFORM_WIN32_WINDOWS:
				/* scan registry for base address of device - inside ForcedConfig array */
		

				if(RegQueryValueEx(hKey,CONFIGSTR,NULL,NULL, buffer,&size)!=ERROR_SUCCESS){
			
					// error in determining the base address of device
					throw Wobbly(VPBREG_REG_KEY_PROPERTY_FAIL);
				}	
		
				// NC 6/4/98 base address is stored in this part of ForcedConfig array	
				
				bufdata = buffer[20];
				bufdata += buffer[21]<<8;
				preg->base = (unsigned short)bufdata;

			break;
			case VER_PLATFORM_WIN32_NT:
				if(RegQueryValueEx(hKey,BASE,NULL,NULL, buffer,&size)!=ERROR_SUCCESS){
			
					// error in determining the base address of device

					throw Wobbly(VPBREG_REG_KEY_PROPERTY_FAIL);
				}	
				preg->base = *(unsigned short*)buffer;
			break;
			default:
				throw Wobbly(VPBREG_OS_NOT_SUPPORTED);
		}
		
		/* open VPB Adapter root key */   
	
		if(RegOpenKeyEx(hKeyparm,NULL,NULL,KEY_ALL_ACCESS,&hKey)!=ERROR_SUCCESS){
		
			// error in opening the VPB device root key in the registry
			throw Wobbly(VPBREG_REG_KEY_ACCESS_FAIL);
		}
		
		/* scan registry for number of channels in the device */
		
		size = sizeof(unsigned long);
		
		if(RegQueryValueEx(hKey,NUMCHSTR,NULL,NULL,buffer,&size)!=ERROR_SUCCESS){
			// error in determining the number of channels in the device
			throw Wobbly(VPBREG_REG_KEY_PROPERTY_FAIL);
		}	
				
		bufdata = buffer[0];
		for(j=1;j<size;j++)
			bufdata += buffer[j]<<8;		
	
		preg->numch = (unsigned short)bufdata;
				
		/* scan registry for size of message queue in the device */
		size = sizeof(unsigned long);
		
		if(RegQueryValueEx(hKey,SIZEMESSSTR,NULL,NULL,buffer,&size)!=ERROR_SUCCESS){
			
		// error in determining the size of the message queue in the device
			throw Wobbly(VPBREG_REG_KEY_PROPERTY_FAIL);
		}	
				
		bufdata = buffer[0];
		for(j=1;j<size;j++)
			bufdata += buffer[j]<<8;
		preg->szmess = (unsigned short) bufdata;
				
		/* scan registry for size of receive data queue for  device */
		
		size = sizeof(unsigned long);
		
		if(RegQueryValueEx(hKey,SIZERXFIFOSTR,NULL,NULL,buffer,&size)==ERROR_SUCCESS){
			bufdata = buffer[0];
			for(j=1;j<size;j++)
				bufdata += buffer[j]<<8;
				
			for(j=0; j<preg->numch; j++)
				preg->szrxdf[j] = (unsigned short)bufdata;
		}
		else
		{
			// error in determining the size of the receive data queue

			throw Wobbly(VPBREG_REG_KEY_PROPERTY_FAIL);
		}	
			
		
		/* scan registry for size of transmit data queue for device */
		
		size = sizeof(unsigned long);	
		if(RegQueryValueEx(hKey,SIZERXFIFOSTR,NULL,NULL,buffer,&size)==ERROR_SUCCESS){
			bufdata = buffer[0];
			for(j=1;j<size;j++)
				bufdata += buffer[j]<<8;
			for(j=0; j<preg->numch; j++)
				preg->sztxdf[j] = (unsigned short)bufdata;
		}
		else
		{
			// error in determining the size of the transmit data queue

			throw Wobbly(VPBREG_REG_KEY_PROPERTY_FAIL);
		}
		
		/* scan registry for length of superframe for  device */

		size = sizeof(unsigned long);	
		if(RegQueryValueEx(hKey,LSF,NULL,NULL,buffer,&size)==ERROR_SUCCESS){
			bufdata = buffer[0];
			for(j=1;j<size;j++)
				bufdata += buffer[j]<<8;
			for(j=0; j<preg->numch; j++)
				preg->lsf = (unsigned short)bufdata;
		}
		else
		{
			// error in determining the size of the transmit data queue

			throw Wobbly(VPBREG_REG_KEY_PROPERTY_FAIL);
		}
		
		// DR 28/11/97 - NULL out uninitialised members

		preg->dnmess = NULL;		
		preg->upmess = NULL;			
		for(j=0; j<MAXCH; j++) {
			preg->rxdf[j] = NULL;		
			preg->txdf[j] = NULL;		
		}

		// end DR 28/11/97
		
		/* DR 4/3/98 -------------------------------------------------*/
		
		/* scan registry for firmware file name and path */

		size = MAX_STR-1;
		lpType = REG_SZ;
		lpData = (unsigned char*)preg->firm;
		if(RegQueryValueEx(hKey,FIRMWARE,NULL,&lpType,lpData,&size)!=ERROR_SUCCESS)
			throw Wobbly(VPBREG_REG_KEY_PROPERTY_FAIL);
		
		/* END DR 4/3/98 ----------------------------------------------*/
		
		/* DR 17/6/98 -------------------------------------------------*/
		
		/* scan registry for VPB model number */

		size = sizeof(unsigned long);	
		if(RegQueryValueEx(hKey,MODEL,NULL,NULL,buffer,&size)==ERROR_SUCCESS){
			bufdata = buffer[0];
			for(j=1;j<size;j++)
				bufdata += buffer[j]<<8;
			preg->model = (unsigned short)bufdata;
		}
		else
		{
			// error in determining the size of the transmit data queue

			throw Wobbly(VPBREG_REG_KEY_PROPERTY_FAIL);
		}
		
		/* END DR 17/6/98 ---------------------------------------------*/

		/* DR 2/10/98 -------------------------------------------------*/
		
		/* scan registry for Device Driver model number */

		size = sizeof(unsigned long);	
		if(RegQueryValueEx(hKey,DDMODEL,NULL,NULL,buffer,&size)==ERROR_SUCCESS){
			bufdata = buffer[0];
			for(j=1;j<size;j++)
				bufdata += buffer[j]<<8;
			preg->ddmodel = (unsigned short)bufdata;
		}
		else
		{
			// error in determining the size of the transmit data queue

			throw Wobbly(VPBREG_REG_KEY_PROPERTY_FAIL);
		}
		
		/* END DR 2/10/98 ---------------------------------------------*/

		/* DR 7/10/98 -------------------------------------------------*/
		
		/* scan registry for size of relay buffer */

		size = sizeof(unsigned long);	
		if(RegQueryValueEx(hKey,SZRELAYBUF,NULL,NULL,buffer,&size)==ERROR_SUCCESS){
			bufdata = buffer[0];
			for(j=1;j<size;j++)
				bufdata += buffer[j]<<8;
			preg->szRelayBuf = (unsigned short)bufdata;
		}
		else
		{
			// error in determining the size of the transmit data queue

			throw Wobbly(VPBREG_REG_KEY_PROPERTY_FAIL);
		}
		
		/* END DR 7/10/98 ---------------------------------------------*/

		preg++;					/* increment VPB registry structure */
	}
	
	
	/* free allocated memory */
	free(subkeybuf);
	free(subkeybufsize);

}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbreg_load_default_tones
	AUTHOR...: David Rowe
	DATE.....: 25/8/98

	Loads default tones (if they exist) from registry.

\*--------------------------------------------------------------------------*/

void vpbreg_load_default_tones(int numch)
{
	OSVERSIONINFO	os;				// operating system info
	char			path[VPB_MAX_STR];	// registry path to VPB key
	HKEY			hVPB;			// key for VPB data
	HKEY			hTones;			// key for tones
	DWORD			Disposition;

	mprintf("Loading programmable tone dets from reg....\n");

	// determine operating system

	os.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	BOOL ret = GetVersionEx(&os);
	if (ret == 0)
		throw Wobbly(VPBREG_CANT_GET_OS_VER);

	// initialise appropriate port class for this OS

	switch(os.dwPlatformId) {
		case VER_PLATFORM_WIN32_WINDOWS:
			strcpy(path, WIN95_PATH);			
			mprintf("\nWin 95\n");
		break;
		case VER_PLATFORM_WIN32_NT:
			strcpy(path, WINNT_PATH);			
			mprintf("\nWin NT\n");
		break;
		default:
			throw Wobbly(VPBREG_OS_NOT_SUPPORTED);
	}

	// get into VPB registry key

	if(RegOpenKeyEx(HKEY_LOCAL_MACHINE,path,NULL,KEY_ALL_ACCESS,&hVPB)!=ERROR_SUCCESS){
		throw Wobbly(VPBREG_REG_KEY_ACCESS_FAIL);
	}
	
	// attempt to open "tones" subkey if it is present, otherwise create

	ret = RegCreateKeyEx(
			hVPB,						// handle of an open key 
		    "tones",					// address of subkey name 
			(DWORD)0,					// reserved 
			NULL,						// address of class string 
			REG_OPTION_NON_VOLATILE,	// special options flag 
			KEY_ALL_ACCESS,
			NULL,						// address of key security structure 
			&hTones,
			&Disposition 				// address of disposition value buffer 
		  );	
		
	// tones subkey may not exist, so exit

	if (ret != ERROR_SUCCESS) 
		return;
 
	// OK, now scan for each tone

	int i,j;
	char tone_key[VPB_MAX_STR];
	VPB_DETECT det;
	DWORD len = sizeof(VPB_DETECT);
	DWORD Type = REG_BINARY;

	for(i=0; i<VPB_MD; i++) {
		sprintf(tone_key,"tone%d",i);
		
		// if tone stored in reg, then program VPB

		if(RegQueryValueEx(hTones,tone_key,NULL,&Type,(unsigned char*)&det,&len)==ERROR_SUCCESS){
			assert(len == sizeof(VPB_DETECT));
			mprintf("Found tone %d in reg, loading...\n",det.tone_id);
			for(j=0; j<numch; j++)
				settonedet(j, &det);
		}
	}	
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_tonedet_make_default
	AUTHOR...: David Rowe
	DATE.....: 25/8/98

	Stores a programmable tone to the registry, where it becomes the
	a default tone (for all channels) next time the VPBAPI is booted.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_tonedet_make_default(VPB_DETECT *d)
{
	OSVERSIONINFO	os;				// operating system info
	char			path[MS];		// registry path to VPB key
	HKEY			hVPB;			// key for VPB data
	HKEY			hTones;			// key for tones
	DWORD			Disposition;

	try {
		// determine operating system

		os.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
		BOOL ret = GetVersionEx(&os);
		if (ret == 0)
			throw Wobbly(VPBREG_CANT_GET_OS_VER);
	
		// initialise appropriate port class for this OS

		switch(os.dwPlatformId) {
			case VER_PLATFORM_WIN32_WINDOWS:
				strcpy(path, WIN95_PATH);			
				mprintf("\nWin 95\n");
			break;
			case VER_PLATFORM_WIN32_NT:
				strcpy(path, WINNT_PATH);			
				mprintf("\nWin NT\n");
			break;
			default:
				throw Wobbly(VPBREG_OS_NOT_SUPPORTED);
		}

		// get into VPB registry key

		if(RegOpenKeyEx(HKEY_LOCAL_MACHINE,path,NULL,KEY_ALL_ACCESS,&hVPB)!=ERROR_SUCCESS){
			throw Wobbly(VPBREG_REG_KEY_ACCESS_FAIL);
		}
	
		// attempt to open "tones" subkey if it is present, otherwise create

		ret = RegCreateKeyEx(
				hVPB,						// handle of an open key 
			    "tones",					// address of subkey name 
				(DWORD)0,					// reserved 
				NULL,						// address of class string 
				REG_OPTION_NON_VOLATILE,	// special options flag 
				KEY_ALL_ACCESS,
				NULL,						// address of key security structure 
				&hTones,
				&Disposition 				// address of disposition value buffer 
			  );	
		if (ret != ERROR_SUCCESS) 
			throw Wobbly(VPBREG_REG_KEY_ACCESS_FAIL);
 
		// OK, now write VPB_DETECT tone data to tone data key. 

		char tone_key[MS];
		if (d->tone_id >= VPB_MD)
			throw Wobbly(VPBAPI_DETECT_INVALID_TONE);
		sprintf(tone_key,"tone%d",d->tone_id);
		
		if(RegSetValueEx(hTones,tone_key,(DWORD)0,REG_BINARY,(unsigned char*)d,sizeof(VPB_DETECT))!=ERROR_SUCCESS){
			throw Wobbly(VPBREG_REG_KEY_ACCESS_FAIL);	
		}

	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_tonedet_make_default"));
	}
	
}

