/*---------------------------------------------------------------------------*\

    FILE....: COMM.CPP
    TYPE....: C+ Functions
    AUTHOR..: David Rowe
    DATE....: 20/11/97

    Functions used for PC to DSP communications, which includes passing
    messages to and from the DSP, and transfering information to and from
    FIFOs in the DSP.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include "config.h"

#include "comm.h"
#include "coff.h"
#include "hip.h"
#include "timer.h"
#include "mess.h"
#include "dspfifo.h"
#include "generic.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>

/*--------------------------------------------------------------------------*\

				    DEFINES

\*--------------------------------------------------------------------------*/

#define	WAIT	5	// time out delay in seconds		

/*--------------------------------------------------------------------------*\

				    CLASS

\*--------------------------------------------------------------------------*/

class CommData {

	static int  exists;	        // asserted if Commdata object exists
		
	ulong	    avpbreg;	        // address of registry info in DSP	
	ushort	    lvpbreg;	        // length of registry info in DSP	
	ulong	    a_asst;		// address of assert falg in DSP
	char        firm[MAX_STR];	// firmware file name

public:
	VPBRegister *v;			// object that stores system config info
	Hip	    *hip;		// object for talking to VPB hips
	ushort	    numvpb;		// number of VPB cards 			

	CommData();
	~CommData();
	int PutMessageVPB(ushort board, word *mess);
	int GetMessageVPB(ushort board, word *mess);
	void WaitForMessageVPB(ushort board, word *mess, ushort mtype, 
			       ushort wait);
	void CheckForAssert(ushort board);
	void GetStringFromDSP(ushort board, char s[], char name[]);
};

int CommData::exists = 0;

// Critical section for PutMessageVPBCriticalSect() to prevent API
// functions and timer callback function writing to DSP message Q
// at the same time

GENERIC_CRITICAL_SECTION	PutMessageSect;

/*-------------------------------------------------------------------------*\

			   COMM MEMBER FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::Comm
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Initialises the VPBs to the desired configuration, and boots the
	DSP programs in each VPB.  After this function is called,
	communications I/O can proceed between the VPB and the PC.

\*--------------------------------------------------------------------------*/

Comm::Comm()
{
	d = new CommData();
	if (d == NULL)
		assert(0); // COMM_CANT_ALLOCATE_MEMORY
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::~Comm
	AUTHOR...: David Rowe
        DATE.....: 20/11/97

	Closes down the comm link to the VPBs.

\*--------------------------------------------------------------------------*/

Comm::~Comm()
{
	delete d;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::PutMessageVPB
	AUTHOR...: David Rowe
        DATE.....: 20/11/97

	Sends a message to the DSPs message queue.  Returns OK if successful,
	or COMM_FULL if DSP message queue full.  Length is in first word
	of mess[].

\*--------------------------------------------------------------------------*/

int Comm::PutMessageVPB(ushort board, word *mess)
//  ushort  board;		VPB board number
//  word    *mess;		ptr to message				
{
	return(d->PutMessageVPB(board, mess));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::GetMessageVPB
	AUTHOR...: David Rowe
        DATE.....: 20/11/97

	Gets a message from the DSPs message queue.  Length of message is
	stored in first word of message.  Returns OK if message read,
	else COMM_EMPTY if no messages on queue.

\*--------------------------------------------------------------------------*/

int Comm::GetMessageVPB(ushort board, word *mess)
//  ushort  board;		VPB board number
//  word    *mess;		ptr to message				
{
	return(d->GetMessageVPB(board, mess));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::WaitForMessageVPB
	AUTHOR...: David Rowe
	DATE.....: 28/11/97

	Keeps checking the DSPs message for a message of a certain type,
	messages of other types are discarded.  If the desired message is
	not found after a specified time a time out wobbly is thrown.

	The caller is responsible for having enough storage in mess[] for
	the desired message.

\*--------------------------------------------------------------------------*/

void Comm::WaitForMessageVPB(ushort board, word *mess, ushort mtype, ushort wait)
//  ushort  board;		VPB board number
//  word    *mess;		ptr to message	
//  ushort  mtype;		type of message to wait for
//  ushort  wait;		timeout delay in seconds			
{
	d->WaitForMessageVPB(board, mess, mtype, wait);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::vpbreg
	AUTHOR...: David Rowe
	DATE.....: 28/11/97

	Nasty un-object oriented function to return a pointer to the 
	vpbreg strcuture for a given board.

\*--------------------------------------------------------------------------*/

VPBREG *Comm::vpbreg(ushort board)
//	ushort	board;		VPB board number
{
	return(&d->v->reg[board]);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::numboards
	AUTHOR...: David Rowe
	DATE.....: 4/3/98

	Returns the number of boards controlled by Com object.

\*--------------------------------------------------------------------------*/

ushort Comm::numboards()
{
	return(d->numvpb);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::hip
	AUTHOR...: David Rowe
	DATE.....: 28/11/97

	Nasty un-object oriented function to return a pointer to the 
	hip strucuture.

\*--------------------------------------------------------------------------*/

Hip *Comm::hip(ushort board)
//	ushort	board;		VPB board number
{
	return(d->hip);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: Comm::CheckForAssert
	AUTHOR...: David Rowe
        DATE.....: 1/12/97

	Debug function that checks the DSP to see if an assert has
	occurred, which hangs the DSP.  If an assert occurs this
	function throws a Wobbly.

\*--------------------------------------------------------------------------*/

void Comm::CheckForAssert(ushort board)
//	ushort	board;		VPB board number
{
	d->CheckForAssert(board);
}

/*-------------------------------------------------------------------------*\

			 COMMDATA MEMBER FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::CommData
	AUTHOR...: David Rowe
	DATE.....: 4/2/98

	Initialises the VPBs to the desired configuration, and boots the
	DSP programs in each VPB.  After this function is called,
	communications I/O can proceed between the VPB and the PC.

\*--------------------------------------------------------------------------*/

CommData::CommData()
{
	word     data;	      // copy of data variable from DSP mem	
	Timer    timer;	      // timer object			
	ushort   time_out;    // == TIME_OUT if time out occurred	
	VPBREG	 *vr;
	int      i;	
	ulong	 a_model;     // address of model in firmware
	word	 model;	      // model number read from firmware
	
	// bomb out if CommData already exists

	if (exists)
		assert(0); // COMM_ALREADY_INITIALISED
	exists = 1;

	// Read registry and init HIP

	v = new VPBRegister(&numvpb);
	if (v == NULL)
		assert(0); // COMM_CANT_ALLOCATE_MEMORY
	vr = v->reg;
	hip = new Hip(vr->ddmodel);
	if (hip == NULL)
		assert(0); // COMM_CANT_ALLOCATE_MEMORY

	// load a few constants 

	coff_get_address(vr->firm, "_model", &a_model);
	coff_get_address(vr->firm, "_asst", &a_asst);
	strcpy(firm, vr->firm);

	// Boot each VPB -------------------------------------------------------

	dspfifo_open(vr->szRelayBuf);
	coff_get_address(vr->firm, "_vpbreg", &avpbreg);
	lvpbreg = &vr[i].base - &vr[i].data;
	for(i=0; i<numvpb; i++) {

		// init the HIP for the current VPB 

		hip->InitVpb(vr[i].base);

		// download firmware and start DSP

		coff_load_dsp_firmware(hip, i, vr->firm);
		hip->DspRun(i);

		// check model number in registry matches firmware

		hip->ReadDspSram(i, (ushort)a_model, 1, (word*)&model);
		if (model != vr[i].model)
			assert(0); // COMM_FIRMWARE_MODEL_NUMBER

		// download config structure and start DSP
		
		vr[i].data = 0;
		hip->WriteDspSram(i, (ushort)avpbreg, lvpbreg, (word*)&vr[i]);

		// now assert "data" to signal DSP to start its config

		vr[i].data = 1;
		hip->WriteDspSram(i, (ushort)avpbreg, 1, (word*)&vr[i]);

		// DSP then fills in certain parameters in the VPBREG
		// structure which we can upload, such as the location
		// of the message fifos in DSP memory.
		// when DSP has finished it's init, it writes a 0 to "data" 

		timer.timer_start();
		do {
			hip->ReadDspSram(i, (ushort)avpbreg, 1, &data);
			timer.timer_check_time_out(WAIT, &time_out);
		} while((data != 0) && (time_out == OK));

		if (time_out == TIME_OUT) {
			// first check for DSP assert

			CheckForAssert(i);

			assert(0); // COMM_TIME_OUT_CONFIGURING_DSP
		}

		// OK, up load reg structure to fill in DSP init info	

		hip->ReadDspSram(i, (ushort)avpbreg, lvpbreg, (word*)&vr[i]);

		// now up perform PC side of message DSP FIFO initialisation

		vr[i].dnmess = new DspFifo(hip, i, vr->a_dnmess, DSPFIFO_DOWN, 
					   RELAY_OFF);
		vr[i].upmess = new DspFifo(hip, i, vr->a_upmess, DSPFIFO_UP, 
					   RELAY_OFF);

		mprintf("DSP [%02d] Message FIFOs booted OK\n",i);
	}

	GenericInitializeCriticalSection(&PutMessageSect);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::~CommData
	AUTHOR...: David Rowe
    DATE.....: 20/11/97

	Closes down the comm link to the VPBs.

\*--------------------------------------------------------------------------*/

CommData::~CommData()
{
    int      i;

    // free storage used by message FIFOs 

    for(i=0; i<numvpb; i++) {
	    delete v->reg[i].dnmess;
	    delete v->reg[i].upmess;
    }

    // check program memory unmodified
	
    for(i=0; i<numvpb; i++)
	    coff_check_dsp_firmware(hip, i, v->reg->firm);
	
    // Close down hip and VPB registry objects 

    delete hip;
    delete v;

    GenericDeleteCriticalSection(&PutMessageSect);
    dspfifo_close();

    exists = 0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::PutMessageVPB
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Sends a message to the DSPs message queue.  Returns OK if successful.
	Keeps trying if DSP message queue full.  Length is in first word
	of mess[].

\*--------------------------------------------------------------------------*/

int CommData::PutMessageVPB(ushort board, word *mess)
//  ushort  board;		VPB board number
//  word    *mess;		ptr to message				
{
	int ret;

	GenericEnterCriticalSection(&PutMessageSect);

        do {
		ret = v->reg[board].dnmess->Write(mess, mess[0]);
		if (ret == DSPFIFO_FULL) {
			GenericSleep(5);	// dont hammer DSP
		}
	} while (ret == DSPFIFO_FULL);

	CheckForAssert(board);
	GenericLeaveCriticalSection(&PutMessageSect);

	return(ret);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::GetMessageVPB
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Gets a message from the DSPs message queue.  Length of message is
	stored in first word of message.  Returns OK if message read,
	else COMM_EMPTY if no messages on queue.

\*--------------------------------------------------------------------------*/

int CommData::GetMessageVPB(ushort board, word *mess)
//  ushort  board;		VPB board number
//  word    *mess;		ptr to message				
{
	int     ret;
	VPBREG  *vr = &v->reg[board];

	assert(mess != NULL);

	// get length of message (first word in message) 

	ret = vr->upmess->Read(mess, 1);
	CheckForAssert(board);
	if (ret != OK)
		return(COMM_EMPTY);

	// get rest of message

	ret = vr->upmess->Read(&mess[1], mess[0]-1);
	CheckForAssert(board);
    
	// this should never happen unless DSP crashed

	if (ret != OK) 
		assert(0); // COMM_DSP_MESSAGE_CORRUPT
    
	return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::WaitForMessageVPB
	AUTHOR...: David Rowe
	DATE.....: 28/11/97

	Keeps checking the DSPs message for a message of a certain type,
	messages of other types are discarded.  If the desired message is
	not found after a specified time a time out wobbly is thrown.

	The caller is responsible for having enough storage in mess[] for
	the desired message.

\*--------------------------------------------------------------------------*/

void CommData::WaitForMessageVPB(ushort board, word *mess, ushort mtype, ushort wait)
//      ushort  board;		VPB board number
//      word    *mess;		ptr to message	
//      ushort  mtype;		type of message to wait for
//      ushort  wait;		timeout delay in seconds			
{
	Timer	timer;
	word	m[COMM_MAX_MESS];	// big enough to hold any mess
	ushort	found;
	ushort  ret;
	ushort  time_out;

	assert(board < MAX_VPB);
	assert(mess != NULL);
	assert(wait != 0);

	found = 0;
	timer.timer_start();
	do {
		ret = GetMessageVPB(board, m);
		if (ret == OK) {
			if (m[1] == mtype) {
				memcpy(mess, m, sizeof(word)*m[0]);
				found = 1;
			}
		}
		timer.timer_check_time_out(wait, &time_out);
	} while((found == 0) && (time_out == OK));

	if (time_out == TIME_OUT) {
		// first check for asserts in DSP

		CheckForAssert(board);

		assert(0); // COMM_TIME_OUT_WAITING_FOR_MESS
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::CheckForAssert
	AUTHOR...: David Rowe
	DATE.....: 1/12/97

	Debug function that checks the DSP to see if an assert has
	occurred, which hangs the DSP.  If an assert occurs this
	function kills the program.

\*--------------------------------------------------------------------------*/

void CommData::CheckForAssert(ushort board)
//	ushort	board;		VPB board number
{
	word	asst;		// asst value read from DSP
	char	gmsg[MAX_STR];	// assert message from DSP
	char	gcnd[MAX_STR];	// assert condition from DSP
	char	gfile[MAX_STR];	// assert file from DSP
	word	gline;		// assert line number from DSP
	ulong	a_gline;	// address of gline in DSP
	word    data;           // debug data 
        ulong   a_data;         // address of debug data
    
	assert(board < MAX_VPB);
	hip->ReadDspSram(board, (ushort)a_asst, 1, &asst);
	
	if (asst) {
	    
		// if asserted get assert params from DSP

		GetStringFromDSP(board, gmsg, "_gmsg");
		GetStringFromDSP(board, gcnd, "_gcnd");
		GetStringFromDSP(board, gfile, "_gfile");

  		coff_get_address(firm, "_gline", &a_gline);
		hip->ReadDspSram(board, (ushort)a_gline, 1, &gline);
	
		// output string length must be < MAX_STR

		assert((strlen(gfile) + 5) < MAX_STR);

		// kill program

		mprintf("DSP ASSERT: %s, line %d\n",gfile, gline);
		exit(1);
	}
    
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: CommData::GetStringFromDSP
	AUTHOR...: David Rowe
	DATE.....: 1/12/97

	Obtains a text string from the DSP, the global name passed to
	this function is assumed to point to the string required.

\*--------------------------------------------------------------------------*/

void CommData::GetStringFromDSP(ushort board, char s[], char name[])
//      ushort  board;		VPB board number
//      char    s[];		string uploaded from DSP
//      char    name[];		name of ptr to string in DSP program
{
	ulong   a_sword;	// address of string in DSP
	word    sword[MAX_STR];	// string in DSP 16 bit word per char format
	word	addr;		// address of string
	int		i;

	// validate arguments

	assert(board < MAX_VPB);
	assert(s != NULL);
	assert(name != NULL);

	// get address of ptr, load ptr and hence string

	coff_get_address(firm, name, &a_sword);
	hip->ReadDspSram(board, (ushort)a_sword, 1, &addr);
	hip->ReadDspSram(board, addr, MAX_STR, sword);

	// convert to string made of 8 bit chars

	i = 0;
	do {
		s[i] = (char)sword[i];
		i++;
	} while ( s[i-1] && (i < (MAX_STR-1)) );

	s[i] = 0;
}


