/*--------------------------------------------------------------------------*\

    FILE....: DSPFIFO.CPP
    TYPE....: C++ Functions
    AUTHOR..: David Rowe
    DATE....: 19/11/97

    Functions used to read and write to First In First Out (FIFO) queues in
    the DSP, thus facilitating PC to DSP data transfer.  The word length of
    the FIFOs is 16 bits.


\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include "config.h"

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include "dspfifo.h"
#include "hip.h"
#include "mess.h"
#include "generic.h"

/*--------------------------------------------------------------------------*\

				    DEFINES

\*--------------------------------------------------------------------------*/

// offsets of each param from base of DSP FIFO structure in DSP memory 

#define	PSTART		0
#define	PEND		1
#define	PWR		2
#define PRD		3
#define	SIZE		4

#define	FIFO_SIZE	5	// size of structure to upload

// error code for CheckDspFifo 

#define	FAIL		1

/*-------------------------------------------------------------------------*\

				     CLASS

\*-------------------------------------------------------------------------*/

// pure virtual base class implementation

class baseDspFifoData {
public:	
	baseDspFifoData() {}
	virtual ~baseDspFifoData() {}
	virtual int Write(word *buf, ushort size) = 0;
	virtual int Read(word *buf, ushort size) = 0;
	virtual int HowFull(void) = 0;
	virtual int DspHowFull(void) = 0;
	virtual int HowEmpty(void) = 0;
};

// DspFifo implementation for original device driver

#pragma pack(1)		// force member alignment same as DSP

class DspFifoData : public baseDspFifoData {
	// copy of FIFO parameters in DSP memory 

	word	pstart;		// first word in FIFO			
	word	pend;		// one after last word in FIFO		
	word	pwr;		// write pointer			
	word	prd; 		// read pointer				
	ushort	size;		// total storage in FIFO		

	// extra params for PC only 

	Hip	*hip;		// hip object used to talk to DSP
	ushort	board;		// VPB board number			
	ushort	adspfifo;	// address of structure in DSP memory	
	int	direction;	// direction of DSP FIFO

public:

	DspFifoData(Hip *hip, ushort board, ushort fifo_addr, int direction, 
		    int relay);
	~DspFifoData();
	int Write(word *buf, ushort size);
	int Read(word *buf, ushort size);
	int HowFull(void);
	int HowEmpty(void);
	int DspHowFull(void);
	int CheckDspFifo();
        void dump();
};

#pragma pack()	// normal member alignment


/*
   NOTE on DSP FIFO implementation:

   These FIFOs use the position of pwr and prd to determine how full the
   FIFO is.

   One way to implement a FIFO is to maintain a counter of how many words are
   in the FIFO.  The write side increments the counter, the read side
   decrements the counter.  This is a problem in our case, as the read and
   write operations are performed by unsynchronised processes (DSP and PC),
   thus the possibility exists of both sides writing to the counter at the
   same time.

   The alternative used here is to use the position of pwr and prd to
   determine how full the FIFO is.  Only the write side modifies pwr and only
   the read side modifies prd, so no contentions will occur.

   One problem with using pwr and prd is the case of pwr = prd.  This could
   occur when the FIFO is empty or when the FIFO is full.  This is resolved
   by only allowing pwr = prd when the FIFO is empty.  When writing to the
   FIFO the pwr ptr is only allowed to get up to prd-1.  Thus the actual
   capacity of the FIFO is size-1.
*/

// used to make access to DSP thread safe

static GENERIC_CRITICAL_SECTION	DspFifoSect;

// used to give each DSP fifo a unique id

static int index_count;

// size of PC side of ALL relay (signal and message) fifos

static unsigned long szRelayBuf;

/*--------------------------------------------------------------------------*\

			       FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: open_dspfifo
	AUTHOR...: David Rowe
	DATE.....: 23/3/98

	Initialises the critical section for the DSP fifo module.  This
	function should be called before any DSP fifos objects are created.

\*--------------------------------------------------------------------------*/

void dspfifo_open(unsigned long szrb)
{
	GenericInitializeCriticalSection(&DspFifoSect);
	index_count = 0;
	szRelayBuf = szrb;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: close_dspfifo
	AUTHOR...: David Rowe
	DATE.....: 23/3/98

	Deletes the critical section for the DSP fifo module.  This function
	should be called all DSP fifo objects have been deleted.

\*--------------------------------------------------------------------------*/

void dspfifo_close()
{
	GenericDeleteCriticalSection(&DspFifoSect);
}

/*--------------------------------------------------------------------------*\

			  DspFifo MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::DspFifo
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Opens a link to a DSP FIFO.  Assumes DSP has been booted with 
	appropriate code.

\*--------------------------------------------------------------------------*/

DspFifo::DspFifo(Hip *h, ushort bd, ushort dspfifo, int direction, int relay)
//      Hip	*h;		HIP port object 
//      ushort  bd;		VPB board number for FIFO
//      ushort  dspfido;	addr of FIFO structure in DSP		 
//	int	direction;	direction of FIFO (DSPFIFO_UP or DSPFIFO_DOWN)
//	int	relay		zero for normal DSP FIFO
{
	d = new DspFifoData(h, bd, dspfifo, direction, relay);
	if (d == NULL)
		assert(0); // DSPFIFO_CANT_ALLOCATE_MEMORY
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::~DspFifo()
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Closes the connection between the DSP FIFO and the PC.

\*--------------------------------------------------------------------------*/

DspFifo::~DspFifo()
{
	delete d;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::Write
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Writes a block of words from the PC to a DSP FIFO.  Returns OK
	for success.

\*--------------------------------------------------------------------------*/

int DspFifo::Write(word *buf, ushort sz)
//      word    *buf;		buf of words to write to DSP FIFO 	
//      ushort  sz;		size of buf				
{	
	return(d->Write(buf,sz));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::Read
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Reads a block of words from a DSP FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int DspFifo::Read(word *buf, ushort sz)
//      word    *buf;		buffer of words to read from DSP FIFO 	
//      ushort  sz;		size of buf				
{
	return(d->Read(buf, sz));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::HowFull
	AUTHOR...: David Rowe
	DATE.....: 20/11/97

	Returns the number of words used in the DSP FIFO.

\*--------------------------------------------------------------------------*/

int DspFifo::HowFull(void)
{
	return(d->HowFull());
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::DspHowFull
	AUTHOR...: David Rowe
	DATE.....: 15/10/98

	Returns the number of words used in the actual DSP FIFO, not the
	relay fifo if enabled.  With old dd. reduces to HowFull

\*--------------------------------------------------------------------------*/

int DspFifo::DspHowFull(void)
{
	return(d->DspHowFull());
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::HowEmpty
	AUTHOR...: David Rowe
	DATE.....: 9/10/98

	Returns the number of words free in the DSP FIFO.

\*--------------------------------------------------------------------------*/

int DspFifo::HowEmpty(void)
{
	return(d->HowEmpty());
}

/*--------------------------------------------------------------------------*\

			     DspFifoData MEMBER FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::DspFifoData
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Opens a link to a DSP FIFO.  Assumes DSP has been booted with 
	appropriate code.

\*--------------------------------------------------------------------------*/

DspFifoData::DspFifoData(Hip *h, ushort bd, ushort dspfifo, int dir, int relay)
//     Hip     *h;	HIP port object 
//     ushort  bd;	VPB board number for FIFO
//     ushort  dspfido;	addr of FIFO structure in DSP		 
//     int     dir	unused in this implementation (apart from error checking)
//     int     relay 	unused in this implementation (apart from error checking)
{
	// validate arguments 

	assert(h != NULL);
	assert(bd < MAX_VPB);
	assert((dir == DSPFIFO_UP) || (dir == DSPFIFO_DOWN));
	assert((relay == RELAY_ON) || (relay == RELAY_OFF));

	// init DSP fifo member variables 

	hip = h;
	board = bd;
	adspfifo = dspfifo;
	direction = dir;
	
	// upload DSP FIFO parameter structure 

	GenericEnterCriticalSection(&DspFifoSect);
	hip->ReadDspSram(board, adspfifo, FIFO_SIZE, &pstart);
	GenericLeaveCriticalSection(&DspFifoSect);
	assert(CheckDspFifo() == OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::~DspFifoData()
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Closes the connection between the DSP FIFO and the PC.

\*--------------------------------------------------------------------------*/

DspFifoData::~DspFifoData()
{
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::Write
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Writes a block of words from the PC to a DSP FIFO.  Returns OK
	for success.

\*--------------------------------------------------------------------------*/

int DspFifoData::Write(word *buf, ushort sz)
//      word    *buf;		buf of words to write to DSP FIFO 	
//      ushort  sz;		size of buf				
{	
	ushort  words_used;	// used space in FIFO			
	ushort  words_free;	// free space in FIFO			
	ushort  copy_first;	// size of first block move		
	ushort  new_pwr;	// modified pwr	 	   		

	// validate arguments 

	assert(buf != NULL);
	assert(direction == DSPFIFO_DOWN);

        // update PC's copy of DSP FIFO state variables 

	GenericEnterCriticalSection(&DspFifoSect);
	hip->ReadDspSram(board, adspfifo+PWR, 1, &pwr);
	hip->ReadDspSram(board, adspfifo+PRD, 1, &prd);

	assert(CheckDspFifo() == OK);

        // determine if there is enough room in FIFO for buf 

	if (pwr >= prd)
		words_used = pwr - prd;
	if (prd > pwr)
		words_used = size - (prd - pwr);
	words_free = size - words_used - 1;

	if (words_free < sz) {
		GenericLeaveCriticalSection(&DspFifoSect);
		return(DSPFIFO_FULL);
	}

	// If buf overlaps end of linear array split into two block moves 

	if ((pwr + sz) > pend) {
		copy_first = (pend-pwr);

		hip->WriteDspSram(board, pwr, copy_first, buf);
		hip->WriteDspSram(board, pstart, sz-copy_first,&buf[copy_first]);
	}
	else
		hip->WriteDspSram(board, pwr, sz, buf);
	// increment pwr and wrap around if required 

	new_pwr = pwr + sz;
	if (new_pwr >= pend)
		pwr = pstart + (new_pwr - pend);
	else
		pwr = new_pwr;

        // copy pwr back to DSP 

	hip->WriteDspSram(board, adspfifo+PWR, 1, &pwr);
	assert(CheckDspFifo() == OK);
	GenericLeaveCriticalSection(&DspFifoSect);

	return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::Read
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Reads a block of words from a DSP FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int DspFifoData::Read(word *buf, ushort sz)
//      word    *buf;		buffer of words to read from DSP FIFO 	
//      ushort  sz;		size of buf				
{
	ushort  words_used;	// used space in FIFO			
	ushort  copy_first;	// size of first block move		
	ushort  new_prd;	// modified prd	    		

        // validate arguments 

	assert(buf != NULL);
	assert(direction == DSPFIFO_UP);

        // update PC's copy of DSP FIFO state variables 

	GenericEnterCriticalSection(&DspFifoSect);
	hip->ReadDspSram(board, adspfifo+PWR, 1, &pwr);
	hip->ReadDspSram(board, adspfifo+PRD, 1, &prd);
	assert(CheckDspFifo() == OK);

        // determine if there is data in FIFO to fill buf 

	if (pwr >= prd)
		words_used = pwr - prd;
	if (prd > pwr)
		words_used = size - (prd - pwr);
	if (words_used < sz) {
		GenericLeaveCriticalSection(&DspFifoSect);
		return(DSPFIFO_EMPTY);
	}

	// If buf overlaps end of linear array split into two block moves 

	if ((prd + sz) > pend) {
		copy_first = (pend-prd);

		hip->ReadDspSram(board, prd, copy_first, buf);
		hip->ReadDspSram(board, pstart, sz-copy_first,&buf[copy_first]);
	}
	else
		hip->ReadDspSram(board, prd, sz, buf);

        // increment prd and wrap around if required 

	new_prd = prd + sz;
	if (new_prd >= pend)
		prd = pstart + (new_prd - pend);
	else
		prd = new_prd;

        // copy prd back to DSP 

	hip->WriteDspSram(board, adspfifo+PRD, 1l, &prd);
	assert(CheckDspFifo() == OK);

	GenericLeaveCriticalSection(&DspFifoSect);
	return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::HowFull
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Returns the number of words used in the DSP FIFO.

\*--------------------------------------------------------------------------*/

int DspFifoData::HowFull(void)
{
	ushort words;

	GenericEnterCriticalSection(&DspFifoSect);
	hip->ReadDspSram(board, adspfifo+PWR, 1, &pwr);
	hip->ReadDspSram(board, adspfifo+PRD, 1, &prd);
	
	assert(CheckDspFifo() == OK);
	if (pwr >= prd)
		words = pwr - prd;
	if (prd > pwr)
		words = size - (prd - pwr);

	GenericLeaveCriticalSection(&DspFifoSect);
	return(words);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifo::DspHowFull
	AUTHOR...: David Rowe
	DATE.....: 15/10/98

	Returns the number of words used in the actual DSP FIFO, not the
	relay fifo if enabled.  With old dd. reduces to HowFull

\*--------------------------------------------------------------------------*/

int DspFifoData::DspHowFull(void)
{
	return(HowFull());
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: DspFifoData::HowEmpty
	AUTHOR...: David Rowe
	DATE.....: 7/10/98

	Returns the number of words free in the DSP FIFO.

\*--------------------------------------------------------------------------*/

int DspFifoData::HowEmpty(void)
{
	return(size-1-HowFull());
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: DspFifoData::CheckDspFifo
    AUTHOR...: David Rowe
    DATE.....: 20/11/97

    Performs a few sanity checks on DSP FIFO structure.  Returns OK is DSP
    FIFO checks out returns.

\*-------------------------------------------------------------------------*/

int DspFifoData::CheckDspFifo()
{
	hip->ReadDspSram(board, adspfifo+PSTART, 1, &pstart);
	hip->ReadDspSram(board, adspfifo+PEND, 1, &pend);
	if (pend < pstart) assert(0);
	if (pwr < pstart) assert(0);
	if (pwr > pend) assert(0);
	if (prd < pstart) assert(0);
	if (prd > pend) assert(0);

	return(OK);
}

