#ifdef	__FreeBSD__

/*---------------------------------------------------------------------------*\

    FILE....: GenericLinux.cpp
    TYPE....: C Functions
    AUTHOR..: David Rowe
    DATE....: 14/9/99

    FreeBSD implementation of Generic functions.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include "config.h"

#include <assert.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include "generic.h"
#include "ioctllinux.h"

void GenericInitializeCriticalSection(GENERIC_CRITICAL_SECTION *cs) {
	int                 ret;
	pthread_mutex_t     *pmut;
	pthread_mutexattr_t  mutexattr;

	// create mutex
        pmut = new pthread_mutex_t;
	assert(pmut != NULL);

	pthread_mutexattr_init(&mutexattr);
	pthread_mutexattr_settype(&mutexattr, PTHREAD_MUTEX_RECURSIVE);
	assert(pthread_mutex_init(pmut,&mutexattr) == 0);
	pthread_mutexattr_destroy(&mutexattr);

	cs->v = (void*)pmut;
}

void GenericDeleteCriticalSection(GENERIC_CRITICAL_SECTION *cs) {
	int ret;

	// destroy mutex
	ret = pthread_mutex_destroy((pthread_mutex_t*)cs->v);
	assert(ret == 0);
	delete (pthread_mutex_t*)cs->v;
}

void GenericEnterCriticalSection(GENERIC_CRITICAL_SECTION *cs)  {
	int ret;
	ret = pthread_mutex_lock((pthread_mutex_t*)cs->v);
	assert(ret == 0);
}

void GenericLeaveCriticalSection(GENERIC_CRITICAL_SECTION *cs)  {
	int ret;
	ret = pthread_mutex_unlock((pthread_mutex_t*)cs->v);
	assert(ret == 0);
}

// return time in ms since system startup
long GenerictimeGetTime() {
	struct timeval tval;
	gettimeofday(&tval, NULL);
	return (tval.tv_sec*1000 + tval.tv_usec/1000);
}

// determies OS, returns -1 for fail

int GenericGetOS() {
	return GET_OS_LINUX;
}

// sleep for ms milliseconds
void GenericSleep(unsigned int ms)
{
	long ms_l = ms;
	struct timespec ts;
	ts.tv_sec = ms_l/1000;
	ts.tv_nsec = (ms_l-ts.tv_sec*1000)*1000000l;
	nanosleep(&ts, NULL);
}

void Generic_beginthread(void(*func)(void*), int stack, void *data) {
	pthread_t	thread;
	pthread_create(&thread, NULL, (void*(*)(void*))func, data);
}

// Linux device driver ISA I/O functions

int Generic_add_board(int fd, void *data) {
	return ioctl(fd, VPB_IOC_ADD_BOARD, data);
}

int Generic_remove_board(int fd, void *data) {
	return ioctl(fd, VPB_IOC_REMOVE_BOARD, data);
}

int Generic_block_write(int fd, void *data) {
	return ioctl(fd, VPB_IOC_BLOCK_WRITE, data);
}

int Generic_block_read(int fd, void *data) {
	return ioctl(fd, VPB_IOC_BLOCK_READ, data);
}

int Generic_open() {
	return open("/dev/vpb0",O_RDWR);
}

int Generic_close(int fd) {
	return close(fd);
}

// Linux device driver PCI I/O functions

int Generic_pci_dsp_reset(int fd, ushort board) {
	VPB_DATA params;
	int      ret;

	params.pci_num = board;
	ret = ioctl(fd, VPB_IOC_PCI_DSP_RESET, &params);
  
	return ret;
}

int Generic_pci_dsp_run(int fd, ushort board) {
	VPB_DATA params;
	int      ret;

	params.pci_num = board;
	ret = ioctl(fd, VPB_IOC_PCI_DSP_RUN, &params);
  
	return ret;
}

int Generic_pci_block_read(
			   int fd, 
			   unsigned short board, 
			   unsigned short addr, 
			   unsigned short length, 
			   unsigned short *buf)
{
	VPB_DATA params;
  
	// set up ioctlparams
	params.length = length;
	params.dsp_addr = addr;
	params.pci_num = board;
	params.data = buf;
  
	return ioctl(fd, VPB_IOC_PCI_BLOCK_READ, &params);
}

int Generic_pci_block_write(
			    int fd,
			    unsigned short board, 
			    unsigned short addr, 
			    unsigned short length, 
			    unsigned short *buf)
{
	VPB_DATA params;
  
	// set up ioctlparams
	params.length = length;
	params.dsp_addr = addr;
	params.pci_num = board;
	params.data = buf;
  
	return ioctl(fd, VPB_IOC_PCI_BLOCK_WRITE, &params);
}

#endif
