/*---------------------------------------------------------------------------*\

    FILE....: HIP.CPP
    TYPE....: C+ Functions
    AUTHOR..: David Rowe
    DATE....: 19/11/97

    VPB access functions that implement the PC side of the Host Interface 
    Port (HIP) - the interface between the PC and DSP.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include "config.h"

#include "hip.h"
#include "port.h"
#include "generic.h"

#include <assert.h>
#include <stdio.h>
#include <memory.h>

/*-------------------------------------------------------------------------*\

				DEFINES

\*-------------------------------------------------------------------------*/

/* general */

#define	SPAN		4	/* number of I/O ports used 	*/
#define	NO_VPB		0xffff	/* indicates no valid VPB	*/

/* HIP registers */

#define	DATA		0x0	/* offset of data register	*/
#define	CNTRL		0x1	/* offset of control register	*/
#define	ADDRLO		0x2	/* low byte of address register	*/
#define	ADDRHI		0x3	/* hi byte of address register	*/

/* Control register bits */

#define	RESET		0x1	/* assert to place DSP in reset	*/
#define	CNTEN		0x2	/* assert to enable addr cntr	*/
#define UP		0x4     /* ==1 for up count, else down	*/

/*-------------------------------------------------------------------------*\

				CLASSES

\*-------------------------------------------------------------------------*/

// pure virtual base class implementation

class baseHipData {
public:	
	baseHipData() {}
	virtual ~baseHipData() {}
	virtual void InitVpb(ushort base) = 0;
	virtual void DspReset(ushort board) = 0;
	virtual void DspRun(ushort board) = 0;
	virtual void WriteDspSram(ushort board, ushort addr, ushort length, 
				  word *buf) = 0;
	virtual void ReadDspSram(ushort board, ushort addr, ushort length,
				 word *buf) = 0;
	virtual void SetPip(ushort board, char enables) = 0;
	virtual void *GethndFile() = 0;
	virtual int GetDeviceDriverModel() = 0;
};

// Hip implementation for original device driver

class HipData : public baseHipData {
	static	int exists;		// asserted if HIP module exists	
	ushort	num_vpb;		// number of VPB cards init-ed		
	ushort	cntrl[MAX_VPB];	        // mirror of VPB control ports		
	Port	*port;			// lower level port driver
	ushort	pipmask;		// current pip enable mask
	int		DeviceDriverModel;
	int		os;		// operating system identifier
public:	
	HipData(int DeviceDriverModel);
	~HipData();
	void InitVpb(ushort base);
	void DspReset(ushort board);
	void DspRun(ushort board);
	void WriteDspSram(ushort board, ushort addr, ushort length, word *buf);
	void ReadDspSram(ushort board, ushort addr, ushort length, word *buf);
	void SetPip(ushort board, char enables);
	void *GethndFile() {return 0;}
	int  GetDeviceDriverModel() { return DeviceDriverModel; }
};

int HipData::exists = 0;

// Alternate implementation for V4PCI

class HipDataPCI : public baseHipData {

	int     hndFile;
	int	DeviceDriverModel;
public:	
	HipDataPCI(int DeviceDriverModel);
	~HipDataPCI();
	void InitVpb(ushort base);
	void DspReset(ushort board);
	void DspRun(ushort board);
	void WriteDspSram(ushort board, ushort addr, ushort length, word *buf);
	void ReadDspSram(ushort board, ushort addr, ushort length, word *buf);
	void SetPip(ushort board, char enables);
	void *GethndFile() {return (void*)hndFile;}
	int  GetDeviceDriverModel() { return DeviceDriverModel; }
};

// used to make access to DSP thread safe

static GENERIC_CRITICAL_SECTION	HipSect;

/*-------------------------------------------------------------------------*\

			     HIP MEMBER FUNCTIONS

\*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

    FUNCTION.: Hip::Hip
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    Initialises Hip states and lower level device driver software.

\*-------------------------------------------------------------------------*/

Hip::Hip(int DeviceDriverModel)
{
	if ((DeviceDriverModel == 0) || (DeviceDriverModel == 3)) {
		// original driver

		d = new HipData(DeviceDriverModel);
		if (d == NULL)
			assert(0); // HIP_CANT_ALLOCATE_MEMORY
	}

	if (DeviceDriverModel == 1) {
		assert(0); // not supported in this version of driver
	}

	if (DeviceDriverModel == 2) {
		// PCI driver (Linux only)

		d = new HipDataPCI(DeviceDriverModel);
		if (d == NULL)
			assert(0); // HIP_CANT_ALLOCATE_MEMORY
	}
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: Hip::~Hip()
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    Closes down the HIP software and lower level drivers.

\*-------------------------------------------------------------------------*/

Hip::~Hip()
{
	delete d;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: Hip:InitVpb
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    Inits a specific VPB card. Must be called before any operations are 
    performed on the VPB.

\*-------------------------------------------------------------------------*/

void Hip::InitVpb(ushort base)
//      ushort base;	base address of card
{
	d->InitVpb(base);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: Hip::DspReset
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    Asserts the reset line of the DSP, freezing code execution on the DSP.

\*-------------------------------------------------------------------------*/

void Hip::DspReset(ushort board)
//      ushort board;	VPB board number
{
	d->DspReset(board);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: Hip::DspRun
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    De-asserts the reset line of the DSP, starting code execution on the DSP.

\*-------------------------------------------------------------------------*/

void Hip::DspRun(ushort board)
//      ushort board;	VPB board number
{
	d->DspRun(board);
}

/*-------------------------------------------------------------------------*\
	
    FUNCTION.: Hip::WriteDspSram
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    Writes a block of 16bit words to the DSPs SRAM.  The length and address
    values are both specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void Hip::WriteDspSram(ushort board, ushort addr, ushort length, word *buf)
//      ushort board;	VPB board number
//      ushort addr;	start of block in DSP SRAM	
//      ushort length;	length of block		
//      word   *buf;	ptr to block in PC	
{
	d->WriteDspSram(board, addr, length, buf);
}

/*-------------------------------------------------------------------------*\
	
    FUNCTION.: Hip::ReadDspSram
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    Reads a block of 16bit words from the DSPs SRAM.  The length and address
    values are both specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void Hip::ReadDspSram(ushort board, ushort addr, ushort length, word *buf)
//      ushort board;	VPB board number
//      ushort addr;	start of block in DSP SRAM	
//      ushort length;	length of block		
//      word   *buf;	ptr to block in PC	
{
	d->ReadDspSram(board, addr, length, buf);
}

/*-------------------------------------------------------------------------*\
	
    FUNCTION.: Hip::SetPip
    AUTHOR...: David Rowe
    DATE.....: 9/8/98

    Sets the pip enable mask for a given board.

\*-------------------------------------------------------------------------*/

void Hip::SetPip(ushort board, char enables)
//      ushort board;	VPB board number
//      char   enables;	eight bit mask defining which pip channels are enabled
{
	d->SetPip(board, enables);
}

// helper functions

void *Hip::GethndFile() {return d->GethndFile();}
int  Hip::GetDeviceDriverModel() { return d->GetDeviceDriverModel(); }

/*-------------------------------------------------------------------------*\

			   HIPDATA MEMBER FUNCTIONS

\*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

    FUNCTION.: HipData::HipData
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    Initialises Hip states and lower level device driver software.

\*-------------------------------------------------------------------------*/

HipData::HipData(int Ddm)
{
	int    i;
	
	DeviceDriverModel = Ddm;
	if (exists) 
		assert(0); // HIP_ALREADY_INITIALISED
	exists = 1;

	GenericInitializeCriticalSection(&HipSect);

	for(i=0; i<MAX_VPB; i++)
		cntrl[i] = 0;
	num_vpb = 0;
	
	// determine operating system

	os = GenericGetOS();
	if (os == GET_OS_FAIL)
		assert(0); // HIP_CANT_GET_OS_VER
	
	// initialise appropriate port class for this OS

	switch(Ddm) {
	case 0:
		// Linux specific port I/O
		port = new PortLinux;
		if (port == NULL)
			assert(0); // HIP_CANT_ALLOCATE_MEMORY
		break;
	case 3:
		// Generic port I/O with user mode port access
		port = new PortUser;
		if (port == NULL)
			assert(0); // HIP_CANT_ALLOCATE_MEMORY
		break;
	default:
		assert(0);         // HIP_OS_NOT_SUPPORTED
	}

	pipmask = 0;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: HipData::~HipData()
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    Closes down the HIP software and lower level drivers.

\*-------------------------------------------------------------------------*/

HipData::~HipData()
{
	int	i;

	/* Place all DSPs in RESET */

	GenericEnterCriticalSection(&HipSect);
	for(i=0; i<num_vpb; i++) {
		cntrl[i] = RESET;
		port->write(i, CNTRL, cntrl[i]);
	}

	// delete appropriate port class for this OS

	switch(os) {
	case GET_OS_LINUX:
		delete (PortLinux*)port;
		break;
	case GET_OS_USER:
		delete (PortUser*)port;
		break;
	default:
		assert(0); // HIP_OS_NOT_SUPPORTED
	}

	exists = 0;
	GenericLeaveCriticalSection(&HipSect);

	GenericDeleteCriticalSection(&HipSect);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: HipData:InitVpb
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    Inits a specific VPB card. Must be called before any operations are 
    performed on the VPB.

\*-------------------------------------------------------------------------*/

void HipData::InitVpb(ushort base)
//      ushort base;	base address of card
{
	// check range of base address

	assert((base >= START_BASE) && (base < END_BASE));

	// now tell port about new board and reset board

	GenericEnterCriticalSection(&HipSect);
	port->addBoard(base, SPAN);
	cntrl[num_vpb] = 0;
	port->write(num_vpb, CNTRL, cntrl[num_vpb]);
	num_vpb++;
	GenericLeaveCriticalSection(&HipSect);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: HipData::DspReset
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    Asserts the reset line of the DSP, freezing code execution on the DSP.

\*-------------------------------------------------------------------------*/

void HipData::DspReset(ushort board)
//      ushort board;	VPB board number
{
	assert(board < num_vpb);
	GenericEnterCriticalSection(&HipSect);
	cntrl[board] |= RESET;
	port->write(board, CNTRL, cntrl[board]);
	GenericLeaveCriticalSection(&HipSect);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: HipData::DspRun
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    De-asserts the reset line of the DSP, starting code execution on the DSP.

\*-------------------------------------------------------------------------*/

void HipData::DspRun(ushort board)
//      ushort board;	VPB board number
{
 	assert(board < num_vpb);
	GenericEnterCriticalSection(&HipSect);
	cntrl[board] &= ~RESET;
	port->write(board, CNTRL, cntrl[board]);
	GenericLeaveCriticalSection(&HipSect);
}

/*-------------------------------------------------------------------------*\
	
    FUNCTION.: HipData::WriteDspSram
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    Writes a block of 16bit words to the DSPs SRAM.  The length and address
    values are both specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void HipData::WriteDspSram(ushort board, ushort addr, ushort length, word *buf)
//      ushort board;	VPB board number
//      ushort addr;	start of block in DSP SRAM	
//      ushort length;	length of block		
//      word   *buf;	ptr to block in PC	
{
	/* validate arguments */

	assert(board < num_vpb);
	assert(addr < MAX_LENGTH);
	assert(length < MAX_LENGTH);
	assert(buf != NULL);

	/* set up for transfer */

	GenericEnterCriticalSection(&HipSect);
	cntrl[board] |= CNTEN;
	cntrl[board] |= UP;
	port->write(board, CNTRL, cntrl[board]);
	port->write(board, ADDRLO, pipmask + (addr & 0xff));
	port->write(board, ADDRHI, (addr>>8) & 0xff);

	/* perform transfer */

	port->blockWrite(board, DATA, length, buf);
	GenericLeaveCriticalSection(&HipSect);
}

/*-------------------------------------------------------------------------*\
	
    FUNCTION.: Hip::ReadDspSram
    AUTHOR...: David Rowe
    DATE.....: 19/11/97

    Reads a block of 16bit words from the DSPs SRAM.  The length and address
    values are both specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void HipData::ReadDspSram(ushort board, ushort addr, ushort length, word *buf)
//      ushort board;	VPB board number
//      ushort addr;	start of block in DSP SRAM	
//      ushort length;	length of block		
//      word   *buf;	ptr to block in PC	
{
        /* validate arguments */

	assert(board < num_vpb);
	assert(addr < MAX_LENGTH);
	assert(length < MAX_LENGTH);
	assert(buf != NULL);

	/* set up for transfer */

	GenericEnterCriticalSection(&HipSect);
	cntrl[board] |= CNTEN;
	cntrl[board] |= UP;
	port->write(board, CNTRL, cntrl[board]);
	port->write(board, ADDRLO, pipmask + (addr & 0xff));
	port->write(board, ADDRHI, (addr>>8) & 0xff);

	/* perform transfer */

	port->blockRead(board, DATA, length, buf);
	GenericLeaveCriticalSection(&HipSect);
}

/*-------------------------------------------------------------------------*\
	
    FUNCTION.: HipData::SetPip
    AUTHOR...: David Rowe
    DATE.....: 9/8/98

    Sets the pip enable mask for a given board.

\*-------------------------------------------------------------------------*/

void HipData::SetPip(ushort board, char enables)
//  ushort board;	VPB board number
//  ushort enables;	eight bit mask defining which pip channels are enabled
{
    assert(board < num_vpb);
    pipmask = enables<<8;
}

/*-------------------------------------------------------------------------*\

			  HIPDATAPCI MEMBER FUNCTIONS

\*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

    FUNCTION.: HipDataPCI::HipDataPCI
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Initialises Hip states and lower level device driver software.

\*-------------------------------------------------------------------------*/

HipDataPCI::HipDataPCI(int Ddm)
{
	DeviceDriverModel = Ddm;

	// check this operating system supported

	int os = GenericGetOS();
	if (os == -1)
		assert(0); // HIP_CANT_GET_OS_VER
	
	// initialise appropriate port class for this OS

	switch(os) {
		case GET_OS_LINUX:
		break;
		default:
			assert(0); // HIP_OS_NOT_SUPPORTED
	}

	// Open device driver

	hndFile = Generic_open();

        if (hndFile == -1)        // Was the device opened?
		assert(0); // HIP_CANT_START_DEVICE_DRIVER

	// Tell DD to open hip module
	GenericInitializeCriticalSection(&HipSect);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: HipDataPCI::~HipDataPCI()
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Closes down the HIP software and lower level drivers.

\*-------------------------------------------------------------------------*/

HipDataPCI::~HipDataPCI()
{
	Generic_close(hndFile);
	GenericDeleteCriticalSection(&HipSect);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: HipDataPCI:InitVpb
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Inits a specific VPB card. Must be called before any operations are 
    performed on the VPB.

\*-------------------------------------------------------------------------*/

void HipDataPCI::InitVpb(ushort base)
//      ushort base;	base address of card
{
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: HipDataPCI::DspReset
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Asserts the reset line of the DSP, freezing code execution on the DSP.

\*-------------------------------------------------------------------------*/

void HipDataPCI::DspReset(ushort board)
//  ushort board;	VPB board number
{
	int ret;

	GenericEnterCriticalSection(&HipSect);

	ret = Generic_pci_dsp_reset(hndFile, board);
	if (ret == -1)                          
		assert(0); // HIP_DSP_RESET_FAIL

	GenericLeaveCriticalSection(&HipSect);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: HipDataPCI::DspRun
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    De-asserts the reset line of the DSP, starting code execution on the DSP.

\*-------------------------------------------------------------------------*/

void HipDataPCI::DspRun(ushort board)
//      ushort board;	VPB board number
{
	int ret;

	GenericEnterCriticalSection(&HipSect);

	ret = Generic_pci_dsp_run(hndFile, board);
	if (ret == -1)                          
		assert(0); // HIP_DSP_RUN_FAIL

	GenericLeaveCriticalSection(&HipSect);
}

/*-------------------------------------------------------------------------*\
	
    FUNCTION.: HipDataPCI::WriteDspSram
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Writes a block of 16bit words to the DSPs SRAM.  The length and address
    values are both specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void HipDataPCI::WriteDspSram(ushort board, ushort addr, ushort length, word *buf)
//      ushort board;	VPB board number
//      ushort addr;	start of block in DSP SRAM	
//      ushort length;	length of block		
//      word   *buf;	ptr to block in PC	
{
	// validate arguments

	assert(addr < MAX_LENGTH);
	assert(length < MAX_LENGTH);
	assert(buf != NULL);

	int ret;

	GenericEnterCriticalSection(&HipSect);

	ret = Generic_pci_block_write(hndFile, board, addr, length, buf);
	if (ret == -1)                          
		assert(0); // HIP_WRITE_DSP_SRAM_FAIL
 
	GenericLeaveCriticalSection(&HipSect);
}

/*-------------------------------------------------------------------------*\
	
    FUNCTION.: HipDataPCI::ReadDspSram
    AUTHOR...: David Rowe
    DATE.....: 2/10/98

    Reads a block of 16bit words from the DSPs SRAM.  The length and address
    values are both specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void HipDataPCI::ReadDspSram(ushort board, ushort addr, ushort length, word *buf)
//      ushort board;	VPB board number
//      ushort addr;	start of block in DSP SRAM	
//      ushort length;	length of block		
//      word   *buf;	ptr to block in PC	
{
	// validate arguments

	assert(addr < MAX_LENGTH);
	assert(length < MAX_LENGTH);
	assert(buf != NULL);

	int ret;

	GenericEnterCriticalSection(&HipSect);

	ret = Generic_pci_block_read(hndFile, board, addr, length, buf);
	if (ret == -1)                          
		assert(0); // HIP_READ_DSP_SRAM_FAIL

	GenericLeaveCriticalSection(&HipSect);
}

/*-------------------------------------------------------------------------*\
	
    FUNCTION.: HipDataPCI::SetPip
    AUTHOR...: David Rowe
    DATE.....: 9/8/98

    Sets the pip enable mask for a given (VPB8L) board.  Does nothing as
    no PCI VPB8L exists (yet!).

\*-------------------------------------------------------------------------*/

void HipDataPCI::SetPip(ushort board, char enables)
//      ushort board;	VPB board number
//      ushort enables;	eight bit mask defining which pip channels are enabled
{
}



