/*---------------------------------------------------------------------------*\

    FILE....: PLAYREC.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 9/2/98

    This module implements the play and record functions for the VPB API.
	 
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include "config.h"

#include "apifunc.h"
#include "vpbapi.h"
#include "comm.h"
#include "coff.h"
#include "comp.h"
#include "arbch.h"
#include "objtrack.h"
#include "vpbconfig.h"
#include "playrec.h"
#include "mapdev.h"
#include "mess.h"
#include "generic.h"

#include <assert.h>
#include <memory>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <ctype.h>
						
/*--------------------------------------------------------------------------*\

				    DEFINES

\*--------------------------------------------------------------------------*/

// play and record buf states

#define	IDLE	                  0
#define	PREPARED                  1  // channnel configured but active yet
#define	PLAYING	                  2  // channnel playing from a buffer
#define	RECORDING                 3  // channnel recording to a buffer
#define	WAITING_FOR_DISABLE_FIFO  4  // waiting for record FIFO to be disabled
#define	FLUSH_FIFO		  5  // waiting for record FIFO to be disabled

#define SLEEPMS                   5  // how long to sleep in polling loop

typedef struct {
	ushort	 mode;	          // current play mode
	ushort	 buf_underflow;   // non-zero if buf has underflowed
	ushort	 underflow_valid; // non-zero if underflow valid
	                          // (in some conditions we expect underflows)
} PLAY;

typedef struct {
	ushort	 buf_overflow;	  // non-zero if buf has overlowed
} RECORD;

/*---------------------------------------------------------------------------*\

							STATICS

\*--------------------------------------------------------------------------*/

// states of play and record buf play functions

static int         *play_ch;
static int         *record_ch;
static float       *play_gain;
static float       *record_gain;
static Compression **play_comp;
static Compression **record_comp;
static PLAY        *play;		// play params
static RECORD      *record;	        // record params

// diagnostics

static ushort	fplay_min;	// minimum words in DSP play (write) buffer
static ushort	frec_max;	// maximum words in DSP record (read) buffer

/*---------------------------------------------------------------------------*\

							FUNCTION HEADERS

\*--------------------------------------------------------------------------*/

void play_file(int handle, char file_name[], unsigned short mode);
void record_file(int handle, char file_name[], unsigned short mode);
void play_file_async_thread(void *args);
void record_file_async_thread(void *args);
void record_configure_VPB4(int handle, ushort b, ushort mode);
void record_configure_VPB8L(int handle, ushort b, ushort mode);
static void	validate_digits(char *digits);
static int digit_match(char digit, char *term_digits);
void play_voxfile_async_thread(void *pvargs);
void play_voxfile_async_thread(void *pvargs);
void play_voxfile(int handle, char file_name[], ushort mode);
void record_voxfile_async_thread(void *pvargs);
void record_buf_async_thread(void *pvargs);
void record_voxfile(int handle, char file_name[], unsigned short mode);
int time_out(int handle);
void start_time_out(int handle);
void play_buf_enable(int handle, unsigned short mode);
int play_rec_check_buffer(int handle);

/*---------------------------------------------------------------------------*\

								FUNCTIONS

\*--------------------------------------------------------------------------*/

GENERIC_CRITICAL_SECTION PlayRecSect;
GENERIC_CRITICAL_SECTION PlayRecReleaseSect;

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_open
	AUTHOR..: David Rowe
	DATE....: 9/3/98

	Initialises the playrec module.

\*--------------------------------------------------------------------------*/

void playrec_open(ushort numch) {
	play_ch = new int[numch];
	record_ch = new int[numch];
	play_comp = new Compression*[numch];
	record_comp = new Compression*[numch];
	play = new PLAY[numch];	
	record = new RECORD[numch];
	play_gain = new float[numch];
	record_gain = new float[numch];
	
	int i; 
	for(i=0; i<numch; i++) {
		play_ch[i] = 0;
		record_ch[i] = 0;
		play_comp[i] = NULL;
		record_comp[i] = NULL;
		play[i].buf_underflow = 0;
		play[i].underflow_valid = 0;
		record[i].buf_overflow = 0;
		play_gain[i] = (float)0.0;
		record_gain[i] = (float)0.0;
		
	}

	fplay_min = 64000;
	frec_max = 0;
	GenericInitializeCriticalSection(&PlayRecSect);
 	GenericInitializeCriticalSection(&PlayRecReleaseSect);

}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_close
	AUTHOR..: David Rowe
	DATE....: 9/3/98

	Closes the playrec module.

\*--------------------------------------------------------------------------*/

void playrec_close() {
	delete [] play_ch;
	delete [] record_ch;
	delete [] play_comp;
	delete [] record_comp;
	delete [] play;
	delete [] record;
	delete [] play_gain;
	delete [] record_gain;
	
	GenericDeleteCriticalSection(&PlayRecSect);
 	GenericDeleteCriticalSection(&PlayRecReleaseSect);

}

/*---------------------------------------------------------------------------*\

							PLAY FUNCTIONS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_set
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Sets play parameters or this channel.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_buf_start
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	API callable version of play_buf_start.

\*--------------------------------------------------------------------------*/

int vpb_play_buf_start(int handle, unsigned short mode)
//	int		handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{
	play_buf_start(handle, mode);
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_buf_sync
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	API version of play_buf_sync().  Returns VPB_FINISH if playing
	should finish, else returns VPB_OK.

\*--------------------------------------------------------------------------*/

int vpb_play_buf_sync(int handle, char *buf, unsigned short length)
//      int		handle  handle of destination device
//	unsigned short	mode	specifies compression mode and async/sync
//	char		*buf	ptr to buffer to send
//	unsigned short	length	length of buffer in bytes
{
	int		ret;

	ret = play_buf_sync(handle, buf, length);
	return(ret);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_buf_finish
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Completes the playing of a channel.

\*--------------------------------------------------------------------------*/

int vpb_play_buf_finish(int handle)
//	int   handle		handle of destination device
{
	play_buf_finish(handle);	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_buf_start
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Prepares a channel for playing a buffer by configuring the channel for
	the appropriate compression rate.  Internal API version.

\*--------------------------------------------------------------------------*/

void play_buf_start(int handle, unsigned short mode)
//	int             handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{
	ushort		b,ch;		// board and channel for this handle
	ushort		stobj;		// first sig proc obj for this device

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	arbch_grab_play_channel(handle);
	assert(play_ch[handle] == IDLE);
	play_ch[handle] = PREPARED;
	play[handle].mode = mode;

	// Set compression mode ------------------------------------------

	stobj = objtrack_handle_to_id(DNOBJ, handle);

	// disable all objects except for D/A
		
	config_disable_object(vpb_c, b, stobj+0);	// FIFO
	config_disable_object(vpb_c, b, stobj+1);	// UNPACK
	config_disable_object(vpb_c, b, stobj+2);	// TONEG
	config_disable_object(vpb_c, b, stobj+3);	// ALAW2LIN
	config_disable_object(vpb_c, b, stobj+4);	// MULAW2LIN
	config_disable_object(vpb_c, b, stobj+5);	// ADPCM2LIN
	config_disable_object(vpb_c, b, stobj+6);	// LIN2ALAW
	config_disable_object(vpb_c, b, stobj+8);	// DELAY

	// clear wires that may change
		
	config_clear_wire(vpb_c, b, stobj+6);
	config_clear_wire(vpb_c, b, stobj+8);

	// rewire and enable objects depending on compression mode
	// dont enable FIFO yet as we haven't filled it

	mode &= 0x7;
	switch (mode) {
		case (VPB_LINEAR):
			config_create_wire(vpb_c, b, stobj+0, stobj+6);
			config_create_wire(vpb_c, b, stobj+0, stobj+8);
			config_packrate(vpb_c, b, stobj, 1);
			play_comp[handle] = new LinearCompression;
		break;
		case (VPB_ALAW):
			config_create_wire(vpb_c, b, stobj+3, stobj+6);
			config_create_wire(vpb_c, b, stobj+3, stobj+8);
        	config_packrate(vpb_c, b, stobj, 2);
			config_packrate(vpb_c, b, stobj+1, 2);
			play_comp[handle] = new LinearCompression;
		break;
		case (VPB_MULAW):
			config_create_wire(vpb_c, b, stobj+4, stobj+6);
			config_create_wire(vpb_c, b, stobj+4, stobj+8);
        	config_packrate(vpb_c, b, stobj, 2);
			config_packrate(vpb_c, b, stobj+1, 2);
			play_comp[handle] = new LinearCompression;
		break;
		case (VPB_OKIADPCM):
			config_create_wire(vpb_c, b, stobj+5, stobj+6);
			config_create_wire(vpb_c, b, stobj+5, stobj+8);
        	config_packrate(vpb_c, b, stobj, 4);
			config_packrate(vpb_c, b, stobj+1, 4);
			config_adpcm_reset(vpb_c, b, stobj+5);
			play_comp[handle] = new LinearCompression;
		break;
		default:
			assert(0); // VPBAPI_PLAY_NOT_SUPPORTED
	} // switch

}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_buf_enable
	AUTHOR..: David Rowe
	DATE....: 8/9/98

	Enables the config manager objects to start playing.  Should
	only be called after valid data is in FIFO.
       
	Specifically, call only after there is at least LENGTH_FRAME
	bytes in the FIFO. This is not checked in the function.
        -- tjw 4/4/00

\*--------------------------------------------------------------------------*/

void play_buf_enable(int handle)
//	int		handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{
	ushort		b,ch;		// board and channel for this handle
	ushort		stobj;	        // first sig proc obj for this device
	ushort		objlist[CONFIG_MAX_LIST];	
	                                // list of config manager objects

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	stobj = objtrack_handle_to_id(DNOBJ, handle);
		
	int cnt = 0;
	objlist[cnt++] = stobj+0;		// FIFO
	objlist[cnt++] = stobj+1;		// UNPACK

	ushort mode = play[handle].mode;
	mode &= 0x7;
	switch (mode) {
		case (VPB_LINEAR):
		break;
		case (VPB_ALAW):
			objlist[cnt++] = stobj+3;		// ALAW
		break;
		case (VPB_MULAW):
			objlist[cnt++] = stobj+4;		// MULAW
		break;
		case (VPB_OKIADPCM):
			objlist[cnt++] = stobj+5;		// ADPCM
		break;
		default:
			assert(0); // VPBAPI_PLAY_NOT_SUPPORTED
	} // switch

	objlist[cnt++] = stobj+6;				// LIN2ALAW
	objlist[cnt++] = stobj+8;				// DELAY

	// all objects enabled at once to ensure one object doesnt start
	// processing before another (important for ADPCM)

	config_enable_object_list(vpb_c, b, objlist, cnt);

	// start checking for underflows

	play[handle].underflow_valid = 1;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_buf_sync
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Plays a buffer of speech samples to a VPB channel.  Returns when all of
	the buffer has been sent to the VPB's DSP FIFO, ie operates 
	synchronously.  Note that when this function exits the DSP is still 
	playing info, the finish function below must wait for the FIFO to empty
	before disabling the FIFO, and releasing the channel.

	However, if another buffer is to be sent, then when this function exits
	there is plenty of time to prepare the next buffer before the DSP fifo
	empties.

        The algorithm has been modified to permit a smaller FIFO size,
        for less voice delay. 
        It is a good idea to call this function with buffers larger
        than LENGTH_FRAME.
        -- tjw 4/4/00

\*--------------------------------------------------------------------------*/

int play_buf_sync(int handle, char *buf, unsigned short length)
//	int	       handle	 handle of destination device
//	char	       *buf	 ptr to buffer to send
//	unsigned short length	 length of buffer in bytes
{
	ushort	       b,ch;	 // board and channel for this handle
	Compression    *comp;
	word	       *wordbuf; // buffer of wrods to download
	ushort	       words;	 // number of samples (words) to download
	ushort	       bytes;	 // number of bytes to read from buf
	VPBREG	       *v;	 // ptr to vpbreg for this card

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	comp = play_comp[handle];
	v = vpb_c->vpbreg(b);
	vpb_c->CheckForAssert(b);
	
	// make sure device not idle

	if (play_ch[handle] == IDLE)
		assert(0);
		
	wordbuf = new word[v->sztxdf[ch]];
        words = v->txdf[ch]->HowEmpty();

	int f;
	while(length && arbch_keep_play_channel(handle)) {

	  if (play_ch[handle] == PLAYING) {

	    while ( (words = v->txdf[ch]->HowEmpty()) == 0)  	
	      GenericSleep(SLEEPMS);
	    
	    // determine how empty buf gets for diagnostic purposes

	    f = v->txdf[ch]->HowFull();
	    if (f < fplay_min) {
	      fplay_min = f;
	    }
	  } 

	  // obtain a buffer of unpacked words and send to DSP

	  bytes = comp->words2bytes(words);
	  if (bytes > length)  {
	    bytes = length;
	    words = comp->bytes2words(bytes);
	  }
	  comp->unpack(wordbuf, buf, bytes);
	  length -= bytes;
	  buf += bytes;

	  // send to DSP tx FIFO

	  if (v->txdf[ch]->Write(wordbuf, words) != 0)
	    assert(0);

	  // if this is the first buffer enable objects

	  if ( (play_ch[handle] == PREPARED) &&
	       ( v->txdf[ch]->HowFull() >= v->lsf) ) {
	    play_ch[handle] = PLAYING;
	    play_buf_enable(handle);
	  }
	}

	delete [] wordbuf;

	if (arbch_keep_play_channel(handle) == 0)
		return(VPB_FINISH);
	else
		return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_buf_finish
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Completes the playing of a channel.

\*--------------------------------------------------------------------------*/

void play_buf_finish(int handle)
//	int	   handle	handle of destination device
{
	ushort	   b,ch;	// board and channel for this handle
	VPBREG	   *v;		// ptr to vpbreg for this card
	ushort	   stobj;	// first sig proc obj for this device

	// validate

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	v = vpb_c->vpbreg(b);
		
	// stop checking for underflows !

	play[handle].underflow_valid = 0;

	// wait for DSP FIFO to empty
	
	int f;
	while ((f=v->txdf[ch]->DspHowFull()) >= v->lsf) {
		GenericSleep(SLEEPMS);
	}

	// disable all objects except D/A

	stobj = objtrack_handle_to_id(DNOBJ, handle);
	config_disable_object(vpb_c, b, stobj+0);	// FIFO
	config_disable_object(vpb_c, b, stobj+1);	// UNPACK
	config_disable_object(vpb_c, b, stobj+2);	// TONEG
	config_disable_object(vpb_c, b, stobj+3);	// ALAW2LIN
	config_disable_object(vpb_c, b, stobj+4);	// MULAW2LIN
	config_disable_object(vpb_c, b, stobj+5);	// ADPCM2LIN
	config_disable_object(vpb_c, b, stobj+6);	// DELAY
	config_disable_object(vpb_c, b, stobj+8);	// LIN2ALAW

	// flush remaining

	config_flush_fifo(vpb_c, b, stobj+0);
	// make sure fifo has been flushed
	while ((f=v->txdf[ch]->DspHowFull()) != 0) {
		GenericSleep(SLEEPMS);
	}

	// connect DELAY and LIN2ALAW to zero source

	config_clear_wire(vpb_c, b, stobj+6);
	config_clear_wire(vpb_c, b, stobj+8);
	config_connect_to_zerobuf(vpb_c, b, stobj+6);
	config_connect_to_zerobuf(vpb_c, b, stobj+8);
	config_enable_object(vpb_c, b, stobj+6);	
	config_enable_object(vpb_c, b, stobj+8);	
	
	// clean up!

	play_ch[handle] = IDLE;
	delete play_comp[handle];
	arbch_release_play_channel(handle);	
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_terminate
	AUTHOR..: David Rowe
	DATE....: 24/3/98

	Stops playing immediately. Does nothing if we are not playing.
	
\*--------------------------------------------------------------------------*/

int vpb_play_terminate(int handle)
//	int	handle	handle of channel 
{
       	arbch_grab_play_channel_async(handle);
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

			    RECORD FUNCTIONS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_buf_start
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	API callable version of record_buf_start.

\*--------------------------------------------------------------------------*/

int vpb_record_buf_start(int handle, unsigned short mode)
//	int		handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{
	record_buf_start(handle, mode);
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_buf_sync
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	API version of record_buf_sync().

\*--------------------------------------------------------------------------*/

int vpb_record_buf_sync(int handle, char *buf, unsigned short length)
//	int		handle		handle of destination device
//	char		*buf		ptr to buffer to receive
//	unsigned short	length		length of buffer in bytes
{
	int	        ret;

	ret = record_buf_sync(handle, buf, length);
	return(ret);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_buf_finish
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Completes the recording of a channel.

\*--------------------------------------------------------------------------*/

int vpb_record_buf_finish(int handle)
//	int	handle		handle of destination device
{
        record_buf_finish(handle);	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_buf_start
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Prepares a channel for recording a buffer by configuring the channel for
	the appropriate compression rate.  Internal API version.

\*--------------------------------------------------------------------------*/

void record_buf_start(int handle, unsigned short mode)
//	int	        handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{
	ushort		b,ch;		// board and channel for this handle

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	arbch_grab_record_channel(handle);
	assert(record_ch[handle] == IDLE);
	record_ch[handle] = PREPARED;
		
	// Set compression mode ------------------------------------------

	switch (vpb_c->vpbreg(b)->model) {
		case VPB4:
			record_configure_VPB4(handle, b,  mode);
		break;
		case VPB8L:
			record_configure_VPB8L(handle, b, mode);
		break;
		default:
			assert(0); //  VPBAPI_MODEL_NOT_SUPPORTED
		break;
	}

	record_ch[handle] = RECORDING;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_configure_VPB4
	AUTHOR..: David Rowe
	DATE....: 20/6/98

	Configures a VPB4 channel for recording.

\*--------------------------------------------------------------------------*/

void record_configure_VPB4(int handle, ushort b, ushort mode)
//	int		handle		handle of destination device
//	ushort	        b		board number
//	unsigned short	mode		specifies compression mode
{
	ushort		stobj;		// first sig proc obj for this device
		
	stobj = objtrack_handle_to_id(UPOBJ, handle);
	config_disable_object(vpb_c, b, stobj+5);	// ALAW
	config_disable_object(vpb_c, b, stobj+6);	// MULAW
	config_disable_object(vpb_c, b, stobj+7);	// ADPCM
	config_disable_object(vpb_c, b, stobj+9);	// PACK
	config_disable_object(vpb_c, b, stobj+10);	// PACKED_FIFO_UP

	// clear wiring

	config_clear_wire(vpb_c, b, stobj+9);		// PACKED_FIFO_UP
	config_clear_wire(vpb_c, b, stobj+10);		// PACK

	// clear FIFO

	// rewire connection to FIFO but dont enable just yet

	mode &= 0x7;
	switch (mode) {
		case (VPB_LINEAR):
			config_create_wire(vpb_c, b, stobj+2, stobj+10);
			config_packrate(vpb_c, b, stobj+10, 1);
			config_enable_object(vpb_c, b, stobj+10);   // FIFO
			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_ALAW):
			config_create_wire(vpb_c, b, stobj+5, stobj+9);
			config_create_wire(vpb_c, b, stobj+9, stobj+10);
			config_packrate(vpb_c, b, stobj+9, 2);
			config_packrate(vpb_c, b, stobj+10, 2);
			config_enable_object(vpb_c, b, stobj+5);
			config_enable_object(vpb_c, b, stobj+9);
			config_enable_object(vpb_c, b, stobj+10);   // FIFO
			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_MULAW):
			config_create_wire(vpb_c, b, stobj+6, stobj+9);
			config_create_wire(vpb_c, b, stobj+9, stobj+10);
			config_packrate(vpb_c, b, stobj+9, 2);
			config_packrate(vpb_c, b, stobj+10, 2);
			config_enable_object(vpb_c, b, stobj+6);
			config_enable_object(vpb_c, b, stobj+9);
			config_enable_object(vpb_c, b, stobj+10);  // FIFO
			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_OKIADPCM):
			config_create_wire(vpb_c, b, stobj+7, stobj+9);
			config_create_wire(vpb_c, b, stobj+9, stobj+10);
			config_packrate(vpb_c, b, stobj+9, 4);
			config_packrate(vpb_c, b, stobj+10, 4);
			config_adpcm_reset(vpb_c, b, stobj+7);
			config_enable_object(vpb_c, b, stobj+9);
			config_enable_two_objects(vpb_c, b, stobj+10,stobj+7);
			// FIFO and ADPCM
			record_comp[handle] = new LinearCompression;
		break;
		default:
			assert(0); // VPBAPI_REC_NOT_SUPPORTED
	} // switch

}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_configure_VPB8L
	AUTHOR..: David Rowe
	DATE....: 20/6/98

	Configures a VPB4 channel for recording.

\*--------------------------------------------------------------------------*/

void record_configure_VPB8L(int handle, ushort b, ushort mode)
//	int		handle		handle of destination device
//	ushort		b		board number
//	unsigned short	mode		specifies compression mode
{
	ushort		stobj;		// first sig proc obj for this device

	stobj = objtrack_handle_to_id(UPOBJ, handle);

	// Note, we assume objects are disabled....

	// clear wiring

	config_clear_wire(vpb_c, b, stobj+6);		// PACKED_FIFO_UP
	config_clear_wire(vpb_c, b, stobj+5);		// PACK

	// configure objects and make appropriate connections
	// enable objects to start sampling

	mode &= 0x7;
	switch (mode) {
		case (VPB_LINEAR):
			config_rate8k(vpb_c, b, stobj+0);
			config_rate8k(vpb_c, b, stobj+1);
			config_rate8k(vpb_c, b, stobj+2);
			config_rate8k(vpb_c, b, stobj+4);
			config_rate8k(vpb_c, b, stobj+5);
			config_rate8k(vpb_c, b, stobj+6);	

			// wiring: FIRD->PACKED_UP_FIFO

			config_create_wire(vpb_c, b, stobj+0, stobj+6);

			// packrate: FIFO.packrate = 1

			config_packrate(vpb_c, b, stobj+6, 1);

			// enable objects

			config_enable_object(vpb_c, b, stobj+6);
			
			// firmware already packs so use linear compression 
			// packer func at PC side

			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_ALAW):
			config_rate8k(vpb_c, b, stobj+0);
			config_rate8k(vpb_c, b, stobj+1);
			config_rate8k(vpb_c, b, stobj+2);
			config_rate8k(vpb_c, b, stobj+4);
			config_rate8k(vpb_c, b, stobj+5);
			config_rate8k(vpb_c, b, stobj+6);	

			// wiring: ALAW->PACK, PACK->PACKED_UP_FIFO

			config_create_wire(vpb_c, b, stobj+8, stobj+5);
			config_create_wire(vpb_c, b, stobj+5, stobj+6);

			// packrate: FIFO.packrate = 2, PACK.packrate = 2

			config_packrate(vpb_c, b, stobj+6, 2);
			config_packrate(vpb_c, b, stobj+5, 2);

			// enable objects

			config_enable_object(vpb_c, b, stobj+5);	// PACK
			config_enable_object(vpb_c, b, stobj+8);	// ALAW
			config_enable_object(vpb_c, b, stobj+6);	// FIFO
	
			// firmware already packs so use linear compression 
			// packer func at PC side

			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_MULAW):
			config_rate8k(vpb_c, b, stobj+0);
			config_rate8k(vpb_c, b, stobj+1);
			config_rate8k(vpb_c, b, stobj+2);
			config_rate8k(vpb_c, b, stobj+4);
			config_rate8k(vpb_c, b, stobj+5);
			config_rate8k(vpb_c, b, stobj+6);	

			// wiring: MULAW->PACK, PACK->PACKED_UP_FIFO

			config_create_wire(vpb_c, b, stobj+7, stobj+5);
			config_create_wire(vpb_c, b, stobj+5, stobj+6);

			// packrate: FIFO.packrate = 2, PACK.packrate = 2

			config_packrate(vpb_c, b, stobj+6, 2);
			config_packrate(vpb_c, b, stobj+5, 2);

			// enable objects

			config_enable_object(vpb_c, b, stobj+5);   // PACK
			config_enable_object(vpb_c, b, stobj+7);   // MULAW
			config_enable_object(vpb_c, b, stobj+6);   // FIFO

			// firmware already packs so use linear compression 
			// packer func at PC side

			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_OKIADPCM):
			config_rate8k(vpb_c, b, stobj+0);
			config_rate8k(vpb_c, b, stobj+1);
			config_rate8k(vpb_c, b, stobj+2);
			config_rate8k(vpb_c, b, stobj+4);
			config_rate8k(vpb_c, b, stobj+5);
			config_rate8k(vpb_c, b, stobj+6);	
			
			// wiring: ADPCM->PACK, PACK->PACKED_UP_FIFO

			config_create_wire(vpb_c, b, stobj+4, stobj+5);
			config_create_wire(vpb_c, b, stobj+5, stobj+6);
			config_adpcm_reset(vpb_c, b, stobj+4);

			// packrate: FIFO.packrate = 4, PACK.packrate = 4

			config_packrate(vpb_c, b, stobj+6, 4);
			config_packrate(vpb_c, b, stobj+5, 4);

			// enable objects

			config_enable_object(vpb_c, b, stobj+5);      // PACK
			config_enable_two_objects(vpb_c, b, stobj+4, stobj+6);
			                                     // ADPCM and FIFO

			// firmware already packs so use linear compression 
			// packer func at PC side

			record_comp[handle] = new LinearCompression;
		break;
		case (VPB_OKIADPCM24):

			config_rate6k(vpb_c, b, stobj+0);
			config_rate6k(vpb_c, b, stobj+1);
			config_rate6k(vpb_c, b, stobj+2);
			config_rate6k(vpb_c, b, stobj+4);
			config_rate6k(vpb_c, b, stobj+5);
			config_rate6k(vpb_c, b, stobj+6);	

			// wiring: ADPCM->PACK, PACK->PACKED_UP_FIFO

			config_create_wire(vpb_c, b, stobj+4, stobj+5);
			config_create_wire(vpb_c, b, stobj+5, stobj+6);
			config_adpcm_reset(vpb_c, b, stobj+4);

			// packrate: FIFO.packrate = 4, PACK.packrate = 4

			config_packrate(vpb_c, b, stobj+6, 4);
			config_packrate(vpb_c, b, stobj+5, 4);

			// enable objects

			config_enable_object(vpb_c, b, stobj+5);	// PACK
			config_enable_two_objects(vpb_c, b, stobj+4, stobj+6);
			                                     // ADPCM and FIFO

			// firmware already packs so use linear compression 
			// packer func at PC side

			record_comp[handle] = new LinearCompression;
		break;
		default:
			assert(0); //  VPBAPI_REC_NOT_SUPPORTED
	} // switch
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_buf_sync
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Records a buffer of speech samples from a VPB channel.  Returns when
	the buffer if filled from the VPB's DSP FIFO, ie operates 
	synchronously.  

	Note that when this function exits the DSP is still recording info, the
	finish function must be called to disable recording.

	If another buffer is to be obtained, then when this function exits
	there is plenty of time to prepare for the next buffer before the DSP
	fifo fills.

\*--------------------------------------------------------------------------*/

int record_buf_sync(int handle, char *buf, unsigned short length)
//	int		handle	  handle of destination device
//	unsigned short	mode	  specifies compression mode and async/sync
//	char		*buf	  ptr to buffer to receive
//	unsigned short	length	  length of buffer in bytes
{
	ushort		b,ch;	  // board and channel for this handle
	Compression	*comp;
	word		*wordbuf; // buffer of wrods to download
	ushort		words;	  // number of samples (words) to download
	ushort		bytes;	  // number of bytes to read from buf
	VPBREG	        *v;	  // ptr to vpbreg for this card

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	comp = record_comp[handle];
	v = vpb_c->vpbreg(b);
		
	// make sure device not idle

	if (record_ch[handle] == IDLE)
		assert(0); //  VPBAPI_DEVICE_NOT_PREPARED
		
	// upload half DSP FIFO size buffers at a time

	words = v->sztxdf[ch]/2;
	wordbuf = new word[words];

	ushort f;
	while(length && arbch_keep_record_channel(handle)) {

		// wait for buffer to half fill

		if (record_ch[handle] == RECORDING) {
			while ((f = v->rxdf[ch]->HowFull()) < words) {
				GenericSleep(SLEEPMS);
			}
			if (f > frec_max)
				frec_max = f;
		}

		// obtain a buffer of unpacked words from DSP
		
		bytes = comp->words2bytes(words);
		if (bytes > length)  {
			bytes = length;
			words = comp->bytes2words(bytes);
		}
		if (v->rxdf[ch]->Read(wordbuf, words) != 0)
			assert(0);		

		// pack words for PC

		comp->pack(buf, wordbuf, words);
		length -= bytes;
		buf += bytes;
	}

	delete [] wordbuf;

	if ((arbch_keep_record_channel(handle) == 0))
		return(VPB_FINISH);
	else
		return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_buf_finish
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Completes the recording of a channel.

\*--------------------------------------------------------------------------*/

void record_buf_finish(int handle)
//	int	handle		handle of source device
{
	// validate

	ValidHandleCheck(handle);
		
	// disable objects

	ushort b,ch,stobj;
	maphndletodev(handle, &b, &ch);

	switch (vpb_c->vpbreg(b)->model) {
		case VPB4:
			stobj = objtrack_handle_to_id(UPOBJ, handle);
			config_disable_object(vpb_c, b, stobj+5);   // ALAW
			config_disable_object(vpb_c, b, stobj+6);   // MULAW
			config_disable_object(vpb_c, b, stobj+7);   // ADPCM
			config_disable_object(vpb_c, b, stobj+9);   // PACK
			config_disable_object(vpb_c, b, stobj+10);  
			                                    // PACKED_FIFO_UP
		break;
		case VPB8L:
			stobj = objtrack_handle_to_id(UPOBJ, handle);
			config_disable_object(vpb_c, b, stobj+6);   
			                                   // PACKED_FIFO_UP
			config_disable_object(vpb_c, b, stobj+5);  // PACK
			config_disable_object(vpb_c, b, stobj+4);  // LIN2ADPCM
			config_disable_object(vpb_c, b, stobj+7);  // LIN2MULAW
			config_disable_object(vpb_c, b, stobj+8);  // LIN2ALAW
		break;
		default:
			assert(0); // VPBAPI_MODEL_NOT_SUPPORTED
		break;
	}

	// wait for FIFO to be disabled

	record_ch[handle] = WAITING_FOR_DISABLE_FIFO;
	int waiting = 0;
	while((record_ch[handle] == WAITING_FOR_DISABLE_FIFO) && 
	      (waiting < 1000)) {
		GenericSleep(SLEEPMS);
		waiting += SLEEPMS;
	}

	// Now we can safely flush the FIFO, knowing that there is no way
	// the DSP will be writing to it!

	// wait for PC to read DSP fifo into relay buf (relay buf enabled only)

	VPBREG *v = vpb_c->vpbreg(b);
	int f;
	waiting = 0;
	if (v->ddmodel == 1) {
		while(f = v->rxdf[ch]->DspHowFull())
			GenericSleep(SLEEPMS);
			waiting += SLEEPMS;
	}

	// now read (relay) fifo to flush

	f = v->rxdf[ch]->HowFull();
	word *wordbuf = new word[f];
	v->rxdf[ch]->Read(wordbuf, f);
	delete [] wordbuf;

	// clean up!

	delete record_comp[handle];
	record_ch[handle] = IDLE;
	arbch_release_record_channel(handle);	
}

// helper function
void playrec_fifo_disabled(int h) {
	if (record_ch[h] == WAITING_FOR_DISABLE_FIFO)
		record_ch[h] = FLUSH_FIFO;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_terminate
	AUTHOR..: David Rowe
	DATE....: 24/3/98

	Stops recording immediately. Does nothing if we are not recording.
	
\*--------------------------------------------------------------------------*/

int vpb_record_terminate(int handle)
//	int	handle	handle of channel 
{
	arbch_grab_record_channel_async(handle);

	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_set_gain
	AUTHOR..: David Rowe
	DATE....: 17/8/98

	Sets the gain of the play channel, using the hardware gain of the
	TS5070 codec on the VPB4.

\*--------------------------------------------------------------------------*/

int vpb_play_set_gain(int handle, float gain) 
{
	ushort	b,ch;	  // board and channel for this handle
	word	mess[PC_LCODEC_DNGAIN];

	// validate

	ValidHandleCheck(handle);

	// gain is in range of +/- 12dB for TS5070

	if ((gain > 12.0) || (gain < -12.0))
		assert(0); // VPBAPI_GAIN_OUT_OF_RANGE

	// now scale float gain to 8 bit unsigned int
	// each unit is 0.1 dB, 0dB is 0x80

	ushort scaled_gain = (ushort)(gain*10.0 + 0x80);

	maphndletodev(handle, &b, &ch);
	mess[0] = PC_LCODEC_DNGAIN;
	mess[1] = PC_CODEC_DNGAIN;
	mess[2] = ch;
	mess[3] = scaled_gain;
	vpb_c->PutMessageVPB(b,mess);
	play_gain[handle] = gain;

	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_get_gain
	AUTHOR..: David Rowe
	DATE....: 17/8/98

	Gets the gain of the play channel, using the hardware gain of the
	TS5070 codec on the VPB4.

\*--------------------------------------------------------------------------*/

int vpb_play_get_gain(int handle, float *gain) 
{
	// validate

	ValidHandleCheck(handle);

	*gain = play_gain[handle];	
			
	// gain is in range of +/- 12dB for TS5070

	if ((*gain > 12.0) || (*gain < -12.0))
		assert(0); // VPBAPI_GAIN_OUT_OF_RANGE

	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_set_gain
	AUTHOR..: David Rowe
	DATE....: 17/8/98

	Sets the gain of the record channel, using the hardware gain of the
	TS5070 codec on the VPB4.  I discovered by experiment that the
	gain values work backwards for the record side, ie the maximum
	positive value gives the miminum gain.  Hence I have negated the
	gain values to give the same gain convention as for play, +12 dB max,
	-12 dB min.

\*--------------------------------------------------------------------------*/

int vpb_record_set_gain(int handle, float gain) 
{
	ushort	b,ch;	 // board and channel for this handle
	word	mess[PC_LCODEC_UPGAIN];

	// validate

	ValidHandleCheck(handle);

	// gain is in range of +/- 12dB for TS5070

	if ((gain > 12.0) || (gain < -12.0))
		assert(0); //  VPBAPI_GAIN_OUT_OF_RANGE

	// now scale float gain to 8 bit unsigned int
	// each unit is 0.1 dB, 0dB is 0x80

	ushort scaled_gain = (ushort)(-gain*10.0 + 0x80);

	maphndletodev(handle, &b, &ch);
	mess[0] = PC_LCODEC_UPGAIN;
	mess[1] = PC_CODEC_UPGAIN;
	mess[2] = ch;
	mess[3] = scaled_gain;
	vpb_c->PutMessageVPB(b,mess);
	record_gain[handle] = gain;

	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_get_gain
	AUTHOR..: David Rowe
	DATE....: 17/8/98

	Gets the gain of the record channel, using the hardware gain of the
	TS5070 codec on the VPB4.

\*--------------------------------------------------------------------------*/

int vpb_record_get_gain(int handle, float *gain) 
{

	// validate

	ValidHandleCheck(handle);

	*gain = record_gain[handle];	
			
        // gain is in range of +/- 12dB for TS5070

	if ((*gain > 12.0) || (*gain < -12.0))
		assert(0); // VPBAPI_GAIN_OUT_OF_RANGE

	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_reset_record_fifo_alarm
	AUTHOR..: David Rowe
	DATE....: 23/9/98

	Resets the latch that detects record DSP buffer overflows.  After
	calling this function another event may be generated if the buffer
	overflows.

\*--------------------------------------------------------------------------*/

int vpb_reset_record_fifo_alarm(int handle)
{
	ushort	stobj,b,ch;

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	stobj = objtrack_handle_to_id(UPOBJ, handle);

	switch (vpb_c->vpbreg(b)->model) {
	case VPB4:
		config_reset_fifo_alarm(vpb_c, b, stobj+10); // PACKED_FIFO_UP
		break;
	case VPB8L:					
		config_reset_fifo_alarm(vpb_c, b, stobj+6);  // PACKED_FIFO_UP
		break;
	default:
		assert(0); //  VPBAPI_MODEL_NOT_SUPPORTED
		break;
	}

	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_reset_play_fifo_alarm
	AUTHOR..: David Rowe
	DATE....: 23/9/98

	Resets the latch that detects record DSP buffer underflows.  After
	calling this function another event may be generated if the buffer
	underflows.

\*--------------------------------------------------------------------------*/

int vpb_reset_play_fifo_alarm(int handle)
{
	ushort	stobj,b,ch;

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	stobj = objtrack_handle_to_id(DNOBJ, handle);

	assert(vpb_c->vpbreg(b)->model==VPB4);
	config_reset_fifo_alarm(vpb_c, b, stobj);	// PACKED_FIFO_UP
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_underflow_valid
	AUTHOR..: David Rowe
	DATE....: 19/10/98

	Returns non-zero if an underflow is valid.  At certain times (like
	when we are waiting for the FIFO to empty at the end of a play
	session), underflow events from the DSP are not valid.

\*--------------------------------------------------------------------------*/

int playrec_underflow_valid(int handle) {
	return(play[handle].underflow_valid);
}
