/*--------------------------------------------------------------------------*\

    FILE....: PORTUSER.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 24/8/00

    User mode low level port I/O module.  Used for O/S's where port I/O can
    be performed directly from user mode.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

				 INCLUDES

\*-------------------------------------------------------------------------*/

#include "config.h"

#include <assert.h>
#include <stdio.h>

#include "port.h"
#include "ioports.h"

static ushort base[MAX_VPB];
static ushort boards;

/*-------------------------------------------------------------------------*\

				 FUNCTIONS

\*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortUser::PortUser
    AUTHOR...: David Rowe
    DATE.....: 24/8/00

    Initialises PortUser object.

\*-------------------------------------------------------------------------*/

PortUser::PortUser() {
	boards = 0;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: ~PortUser::PortUser
    AUTHOR...: David Rowe
    DATE.....: 24/8/00

    Frees PortUser.

\*-------------------------------------------------------------------------*/

PortUser::~PortUser() {
	int i;
	for(i=0; i<boards; i++)
		iopriv(base[i], 0);
	boards = 0;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortUser::addBoard
    AUTHOR...: David Rowe
    DATE.....: 14/11/97

    Allocates the operating resources required to access a VPB.

\*-------------------------------------------------------------------------*/

void PortUser::addBoard(ushort new_base, ushort span)
//  ushort new_base;	base address of port block
//  ushort span;	number of ports in block
{
	assert(boards <MAX_VPB);

	// attempt to gain access to ports (must be root)
	assert(iopriv(new_base, 1) == 0);
	base[boards++] = new_base;
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortUser::write
    AUTHOR...: David Rowe
    DATE.....: 24/8/00

    Writes a single 16-bit word to a 16-bit I/O port.

\*-------------------------------------------------------------------------*/

void PortUser::write(ushort board, ushort offset, word value)
//      ushort	board;	VPB number
//      ushort  offset;	Offset of port from base address of board
//      word	value;	value to write
{
	assert(board < boards);

	ushort port = base[board] + offset;
	ioset(port, value);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortUser::blockWrite
    AUTHOR...: David Rowe
    DATE.....: 24/8/00

    Writes a block of 16-bit words to a 16-bit I/O port.  The length and
    value is specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void PortUser::blockWrite(ushort board, ushort offset, ushort length, 
			  word *buf)
//      ushort	board;	VPB number
//      ushort  offset;	Offset of port from base address of board
//      ushort  length;	number of words (not butes) in buf
//      word	buf[];	buffer to write to VPB
{
	ulong   i;

	/* validate arguments */

	assert(board < boards);
	assert(buf != NULL);
	assert(length < MAX_LENGTH);

	/* perform block write */

	ushort port = base[board] + offset;
	iowrite(port, buf, length);
}

/*-------------------------------------------------------------------------*\

    FUNCTION.: PortUser:blockRead
    AUTHOR...: David Rowe
    DATE.....: 24/8/00

    Reads a block of 16-bit words from a 16-bit I/O port.  The length and
    value is specified in words (not bytes).

\*-------------------------------------------------------------------------*/

void PortUser::blockRead(ushort board, ushort offset, ushort length, word *buf)
//      ushort	board;	VPB number
//      ushort  offset;	Offset of port from base address of board
//      ushort  length;	number of words (not butes) in buf
//      word	buf[];	buffer read from VPB
{
	ulong   i;

	/* validate arguments */

	assert(board < boards);
	assert(buf != NULL);
	assert(length < MAX_LENGTH);

	/* perform block read */

	ushort port = base[board] + offset;
	ioread(port, buf, length);	
}

