/*---------------------------------------------------------------------------*\

    FILE....: TLOAD.CPP
    TYPE....: C++ Program
    AUTHOR..: David Rowe
    DATE....: 23/8/00

    This program tests the VPB driver by loading it with 4 simultaneous
    channels of audio I/O.

    1. Dial the number of a port on the VPB4.
    2. Start this program: ./tload.
    3. The card should take the port off hook.
    4. It will then beep, record, playback, .............
    5. Try loading the PC (open a window etc), running top etc.  See if
       audio breaks up (sign of DSP buffer under/overflow). 
    6. Press return to end (will take up to 2 seconds to finish)

    Expected Results:  
    
    DR 24/8/00 - no evidence of breakup, top showed CPU load of only 
    0.1% for 4 channels (P133 running Red Hat 6.2).

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 2000 OST

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>

#include "vpbapi.h"
#include "verbose.h"

#define NUM_CH     4
#define BUF_LENGTH 16000

void *channel_thread(void *pv);
void *event_thread(void *pv);

int finito,event_finito;
int active_threads;
pthread_mutex_t mutex;
int ch[] = {1,2,3,4};

main() {
	pthread_t thread;
	pthread_t threads[NUM_CH];
        int i;
	int   iobase[] = {0x300};

	active_threads = 0;
	pthread_mutex_init(&mutex, NULL);

	event_finito = finito = 0;
	verbose(1);
	vpb_start(1, iobase, "vpbmain.out");
	pthread_create(&thread, NULL, event_thread, NULL);

	// start a thread for each channel

	for(i=0; i<NUM_CH; i++) {
		pthread_create(&threads[i], NULL, channel_thread, 
			       (void*)&ch[i]);
	}

	printf("Press return to finish.....\n");
	getchar();

	// signal all threads and wait for all to finish

	finito = 1;
	while(active_threads)
		vpb_sleep(10);

	// event thread must be stopped last

	event_finito = 1;
	while(event_finito)
		vpb_sleep(10);

	vpb_stop();

	pthread_mutex_destroy(&mutex);
}

void *channel_thread(void *pv)
{
	int   h;
	char  buf[BUF_LENGTH];
	int   ch = *(int*)pv;
	int   cycles = 0;

	// keep count of how many threads active

	pthread_mutex_lock(&mutex);
	active_threads++;
	pthread_mutex_unlock(&mutex);

	// init this port

	h = vpb_open(1,ch);
	vpb_sethook_sync(h,VPB_OFFHOOK);		 

	// beep, record, play,.........

	while(!finito) {
		vpb_dial_sync(h,"1");

		// record buffer
		vpb_record_buf_start(h, VPB_MULAW);
		vpb_record_buf_sync(h, buf, BUF_LENGTH);
		vpb_record_buf_finish(h);

		// play back buufer
		vpb_play_buf_start(h, VPB_MULAW);
		vpb_play_buf_sync(h, buf, BUF_LENGTH);
		vpb_play_buf_finish(h);
		printf("ch %d, cycles %d\n",ch,cycles++);
	}

	// hangup
	vpb_sethook_sync(h, VPB_ONHOOK);

	vpb_close(h);

	// decrement active thread count

	pthread_mutex_lock(&mutex);
	active_threads--;
	pthread_mutex_unlock(&mutex);
}

// IMPORTANT: vpb_get_event_sync must be called regularly, as driver uses
// this function to service time critical tasks in the driver.  This
// function must be polled regularly, even if no events are required for 
// the program.  Without it, other functions, such as play, record, will
// not work.

void *event_thread(void *pv) {
	VPB_EVENT e;
	char      s[VPB_MAX_STR];

	while(!event_finito) {
		vpb_get_event_async(1, &e);
		vpb_sleep(20);
	}

	event_finito = 0;
}

