/*--------------------------------------------------------------------------*\

    FILE....: VPB.C
    TYPE....: Linux device driver
    AUTHOR..: David Rowe
    DATE....: 22/9/99

    Linux kernel mode device driver for ISA & PCI VPB cards.  Interfaces user
    mode driver (VBPAPI).  This version compiles on 2.2.14 Linux Kernel. 
    
    NOTE: Not required when driver run in user-mode. 

    Compile with:

    gcc vpb.c -c -O -I/usr/include

    Install with:

    insmod vpb.o (driver wil print out major node number, eg 254)
    mknod /dev/vpb0 c 254 0 (subs 254 for your major node number)

    Use:

    cat /proc/msg

    to view debug info.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

			       DEFINES
							
\*---------------------------------------------------------------------------*/

#define	NAME	     "vpb"
#define SIZE_WD      0x10000    // size of DSP SRAM block in words
#define MAX_V4PCI    32         // max number of V4PCI's
#define BLOCK        2          // block size for read/writes
#define BLOCK_DELAY  1          // delay (us) between adjacent blocks

#define MODULE
#define __KERNEL__

/*---------------------------------------------------------------------------*\

			       INCLUDES
							
\*---------------------------------------------------------------------------*/

// Module version boilerplate
#include <linux/autoconf.h>
#if defined( CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS
#endif

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/ioport.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include "ioctllinux.h"
#include <asm/segment.h>
#include <asm/uaccess.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <asm/system.h>
#include <linux/vmalloc.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <asm/uaccess.h>

/*---------------------------------------------------------------------------*\

				GLOBALS
							
\*---------------------------------------------------------------------------*/

// ensure dynamic major number picked
unsigned int major = 0;

static int vpb_ioctl(struct inode *, struct file *, unsigned int, unsigned long);

static struct file_operations vpb_fops={

	NULL,
	NULL,
	NULL, 
	NULL, 
	NULL,
	vpb_ioctl,
};

// number of valid PCI devices detected
static int numPCI; 

// translated base address of PLX9050 regions
static unsigned char  *base0[MAX_V4PCI];
static unsigned short *base2[MAX_V4PCI];

// intermediate kernel space buffer from transfers between I/O memory (PCI
// card DSP memory) and user memory (application)
static short buf[SIZE_WD];

/*---------------------------------------------------------------------------*\

				   FUNCTIONS
							
\*---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: init_module()
	AUTHOR...: David Rowe
	DATE.....: 15/4/00

	Called by the O/S when the module is loaded. Searches for a 
	valid PCI device, no action for ISA cards.

\*--------------------------------------------------------------------------*/

int init_module(void) {
  char s[4];
  struct pci_dev *dev = NULL;

  major = register_chrdev(major, NAME, &vpb_fops);
  printk("<1>vpb: major = %d\n",major);

  // search for V4PCI devices on PCI bus

  numPCI = 0;
  while((dev = pci_find_device(PCI_VENDOR_ID_PLX, PCI_DEVICE_ID_PLX_9050, dev)) != NULL)
  {
    // check that subsytem ID & Subsytem Vendor matches
    pci_read_config_dword(dev, 0x2c, (u32*)s);
    if ((s[3] == 'V') && (s[2] == '4') && (s[1] == 'V') && (s[0] == 'T')) {
      
      // OK, V4PCI found, so map address regions
      base0[numPCI] = ioremap(dev->base_address[0],128);
      base2[numPCI] = ioremap(dev->base_address[2],sizeof(short)*SIZE_WD);

      numPCI++;
    }
  }

  printk("%d V4PCI's detected on PCI bus\n",numPCI);

  // this code was used for experimenting with wait-states
  #if 1 
  {
    int ctrl;

    ctrl=readl(base0[0]+0x28);
    printk("ctrl = 0x%x\n",ctrl);
    //writel(0x304e0222, base0[0]+0x28);
    writel(0x00440222, base0[0]+0x28);
    ctrl=readl(base0[0]+0x28);
    printk("ctrl = 0x%x\n",ctrl);
  }
  #endif
    
  return 0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: cleanup_module()
	AUTHOR...: David Rowe
	DATE.....: 15/4/00

	Called by the O/S when the module is unloaded. Frees any memory
	mapped areas for the PCI cards.

\*--------------------------------------------------------------------------*/

void cleanup_module(void) {
  int ret;
  int i;
  
  // unmap any PCI regions

  for(i=0; i<numPCI; i++) {
    vfree(base0[i]);    
    vfree(base2[i]);
  }
    
  ret = unregister_chrdev(major, NAME);
  if (ret < 0)
    printk(KERN_WARNING "unregister_chrdev() failed!\n");
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_iotcl()
	AUTHOR...: David Rowe
	DATE.....: 15/4/00

	Called when a user mode ioctl is issued, used to perform all ISA and
	PCI commands based on ioctl command.

\*--------------------------------------------------------------------------*/

static int vpb_ioctl(struct inode *inode,
			  struct file *filp,
			  unsigned int cmd,
			  unsigned long arg)
{
  int      ret;
  VPB_DATA vpb_data;  // ioctl parameters from user space
  int      port;      // I/O port (ISA)
  int      dsp_addr;  // addr in DSP memory (PCI)
  int      length;    // length of transfer in words
  short    *data;     // ptr to user space data
  int      pci_num;   // pci card number (PCI)
  int      ctrl;      // value of PLX9050 control reg
  int      i;
  int      l;         // remaining length
  short    *pbuf;     // source/dest for next block

  // copy parameter data from user address space

  ret = copy_from_user(&vpb_data, (void*)arg, sizeof(VPB_DATA));
  if (ret != 0) {
    printk(KERN_CRIT "verify area VPB_DATA failed\n");
    return -ENODEV;
  }

  // local copies of parameters

  port = vpb_data.port;
  dsp_addr = vpb_data.dsp_addr;
  length = vpb_data.length;
  data = vpb_data.data;
  pci_num = vpb_data.pci_num;

  switch(cmd) {

    // ISA commands ------------------------------------------------------

    case VPB_IOC_ADD_BOARD:
      ret = check_region(port,length);
      if (ret != 0) {
	return -ENODEV;
      }
    break;

    case VPB_IOC_REMOVE_BOARD:
      release_region(port,length);
    break;

    case VPB_IOC_BLOCK_WRITE:
      ret = access_ok(VERIFY_READ, data, length);

      if (ret != 1)
	return ret;
	
      if (port % 2) {
	// single byte transfers to odd addresses
	short w;
	__get_user(w,data);
	outb((char)w, port);
      }
      else {
        short w;
	// even address transfers
	for(i=0; i<length; i++) {
	  __get_user(w,&data[i]);
	  outw(w, port);
	}
      }
    break;

    case VPB_IOC_BLOCK_READ:	
      ret = access_ok(VERIFY_WRITE, data, length);
      if (ret != 1)
	return ret;
	
      for(i=0; i<length; i++)
	__put_user(inw(port),&data[i]);
    break;

    // PCI commands ----------------------------------------------------------

    case VPB_IOC_PCI_BLOCK_WRITE:
      copy_from_user(buf, data, length*sizeof(short));

      // divide write into blocks to avoid upsetting DSP interrupts

      l = length;
      pbuf = buf;
      while(l>0) {
        if (l > BLOCK) 
          memcpy_toio(base2[pci_num]+dsp_addr, pbuf, BLOCK*sizeof(short));
        else
          memcpy_toio(base2[pci_num]+dsp_addr, pbuf, l*sizeof(short));
        l -= BLOCK;
        pbuf += BLOCK;
        dsp_addr += BLOCK;
        udelay(BLOCK_DELAY);
      }
    break;
     
    case VPB_IOC_PCI_BLOCK_READ:
      // divide read into blocks to avoid upsetting DSP interrupts

      l = length;
      pbuf = buf;
      while(l>0) {
        if (l > BLOCK) 
          memcpy_fromio(pbuf, base2[pci_num]+dsp_addr, BLOCK*sizeof(short));
        else
          memcpy_fromio(pbuf, base2[pci_num]+dsp_addr, l*sizeof(short));
        l -= BLOCK;
        pbuf += BLOCK;
        dsp_addr += BLOCK;
        udelay(BLOCK_DELAY);
      }
      copy_to_user(data, buf, length*sizeof(short));
    break;
    
    case VPB_IOC_PCI_DSP_RESET:
      // reset PLX9050 pin user0, connected to DSP reset, placing DSP in reset
      ctrl = readl(base0[pci_num]+0x50);
      ctrl &= ~(1<<2);
      writew(ctrl,base0[pci_num]+0x50);            
    break;
    
    case VPB_IOC_PCI_DSP_RUN:

      // set PLX9050 pin user0, connected to DSP reest, placing DSP in run mode
      ctrl = readl(base0[pci_num]+0x50);
      ctrl |= (1<<2);
      writew(ctrl,base0[pci_num]+0x50);            
    break;
    
  }
    
   return 0;
}













