/*---------------------------------------------------------------------------*\

    FILE....: VPBAPI.H
    TYPE....: C Header File
    AUTHOR..: Voicetronix
    DATE....: 26/11/97

    Header file for:
  
    "Voice Processing Board Applications Programmer Interface" functions.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#ifndef __VPBAPI__
#define __VPBAPI__

/*-----------------------------------------------------------------------*\

			              GENERAL

\*-----------------------------------------------------------------------*/

// Return codes

#define	VPB_OK		 0		
#define	VPB_NO_EVENTS	-1		

// string length

#define	VPB_MAX_STR	256

// vpb_sethook() states

#define	VPB_ONHOOK	0
#define	VPB_OFFHOOK	1

/*-------------------------------------------------------------------------*\

			       MISC FUNCTIONS

\*-------------------------------------------------------------------------*/

int vpb_start(int num_cards, int *iobases, char *firmware_file);
int vpb_stop();

int vpb_open(unsigned int board, unsigned int channel);
int vpb_close(int handle);
void vpb_sleep(long time_ms);
int vpb_sethook_async(int handle, int hookstate);
int vpb_sethook_sync(int handle, int hookstate);
int vpb_get_model(char *s);
int vpb_reset_play_fifo_alarm(int handle);
int vpb_reset_record_fifo_alarm(int handle);

/*-----------------------------------------------------------------------*\

			       EVENT HANDLING	

\*-----------------------------------------------------------------------*/

typedef struct {
	int	type;		// event type (see below)
	int	handle;		// channel that generated event
	int	data;		// optional data
	unsigned long data1;
} VPB_EVENT;

// unsolicited events (maskable)

#define VPB_RING		0
#define VPB_DIGIT		1
#define	VPB_TONEDETECT		2
#define	VPB_TIMEREXP		3
#define	VPB_VOXON		4
#define	VPB_VOXOFF		5
#define	VPB_PLAY_UNDERFLOW	6
#define	VPB_RECORD_OVERFLOW	7
#define VPB_DTMF		8

// solicited events (not maskable)

#define	VPB_PLAYEND		100
#define	VPB_RECORDEND		101
#define	VPB_DIALEND		102
#define	VPB_TONE_DEBUG_END	103
#define	VPB_CALLEND		104

// Event mask values

#define VPB_MRING		(1<<VPB_RING)
#define VPB_MDIGIT		(1<<VPB_DIGIT)
#define VPB_MDTMF		(1<<VPB_DTMF)
#define	VPB_MTONEDETECT		(1<<VPB_TONEDETECT)
#define VPB_MTIMEREXP		(1<<VPB_TIMEREXP)
#define VPB_MVOXON		(1<<VPB_VOXON)
#define VPB_MVOXOFF		(1<<VPB_VOXOFF)
#define	VPB_MPLAY_UNDERFLOW	(1<<VPB_PLAY_UNDERFLOW)
#define	VPB_MRECORD_OVERFLOW	(1<<VPB_RECORD_OVERFLOW)

int vpb_enable_event(int handle, unsigned short mask);
int vpb_disable_event(int handle, unsigned short mask);
int vpb_get_event_mask(int handle);
int vpb_set_event_mask(int handle, unsigned short mask);

int vpb_get_event_async(int card, VPB_EVENT *event);
void vpb_translate_event(VPB_EVENT *e, char s[]);

/*-----------------------------------------------------------------------*\

			     PLAY AND RECORD

\*-----------------------------------------------------------------------*/

// return code for vpb_play_buf_sync and vpb_record_buf_sync

#define	VPB_FINISH      1	

// compression modes

#define	VPB_LINEAR      0	// 128 kbit/s 16 bit linear
#define	VPB_ALAW	1	// 64 kbit/s A-law companded
#define	VPB_MULAW	2	// 64 kbit/s mu-law companded
#define	VPB_OKIADPCM	3	// 32 kbit/s OKI ADPCM
#define	VPB_OKIADPCM24	4	// 24 kbit/s OKI ADPCM

// play functions

int vpb_play_buf_start(int handle, unsigned short mode);
int vpb_play_buf_sync(int handle, char *buf, unsigned short length);
int vpb_play_buf_finish(int handle);
int vpb_play_terminate(int handle);

int vpb_play_set_gain(int handle, float gain);
int vpb_play_get_gain(int handle, float *gain);

// record functions

int vpb_record_buf_start(int handle, unsigned short mode);
int vpb_record_buf_sync(int handle, char *buf, unsigned short length);
int vpb_record_buf_finish(int handle);
int vpb_record_terminate(int handle);

int vpb_record_set_gain(int handle, float gain);
int vpb_record_get_gain(int handle, float *gain);

/*-----------------------------------------------------------------------*\

			         DIALLING

\*-----------------------------------------------------------------------*/

// blind dialling functions

int vpb_dial_sync(int handle, char *dialstr);

// Call progress tone ids

#define	VPB_CALL_DISCONNECT	0	
#define	VPB_CALL_DIALTONE	1
#define	VPB_CALL_RINGBACK	2
#define	VPB_CALL_BUSY		3
#define	VPB_CALL_GRUNT		4

typedef struct {
	unsigned int	tone_id;      // prog tone detector tone id
	unsigned int	call_id;      // call progress tone id
	unsigned int	terminate;    // no zero to terminate list
} VPB_TONE_MAP;

#define	VPB_MAX_TONE_MAP        10    // maximum number of entries in tone map

/*-----------------------------------------------------------------------*\

			PROGRAMMABLE TONE GENERATOR

\*-----------------------------------------------------------------------*/

// Programmable tone generator structure ---------------------------------

typedef struct {
	unsigned short	freq1;	// frequency of first tone
	unsigned short	freq2;	// frequency of second tone
	unsigned short	freq3;	// frequency of third tone
	short		level1;	// first tone level in dB, 0dB maximum
	short		level2;	// second tone level in dB, 0dB maximum
	short	        level3;	// third tone level in dB, 0dB maximum
	unsigned long	ton;	// on time ms 
	unsigned long	toff;	// off time ms 
} VPB_TONE;

int vpb_settone(char ident, VPB_TONE *vpb_tone);
int vpb_gettone(char ident, VPB_TONE *vpb_tone);
int vpb_playtone_sync(int handle, VPB_TONE *vpb_tone);

/*-----------------------------------------------------------------------*\

			 PROGRAMMABLE TONE DETECTOR

\*-----------------------------------------------------------------------*/

// built in tone dectector IDs

#define	VPB_DIAL	0   // dial tone detected
#define	VPB_RINGBACK	1   // ringback detected
#define	VPB_BUSY	2   // busy tone detected
#define	VPB_GRUNT	3   // grunt detected

#define	VPB_MD	10	    // maximum number of tone detectors per device
#define	VPB_MS	10	    // maximum number of states per cadence state mach

// State transition table consists of one entry for each state transition.

#define	VPB_TIMER 	0
#define	VPB_RISING	1
#define	VPB_FALLING	2
#define	VPB_DELAY	3

typedef struct {
    unsigned short type;	// VPB_TIMER, VPB_RISING, or VPB_FALLING
    unsigned short tfire;	// timer mode only			
    unsigned short tmin;	// minimum tone on/off time (non timer)	in ms
    unsigned short tmax;	// maximum tone on/off time (non timer)	in ms
} VPB_STRAN;

typedef struct {
   unsigned short	nstates;     // number of cadence states
   unsigned short	tone_id;     // unique ID number for this tone	
   unsigned short	ntones;	     // number of tones (1 or 2)	
   unsigned short	freq1;	     // freq of first tone (Hz)
   unsigned short	bandwidth1;  // bandwidth of first tone (Hz)	
   unsigned short	freq2;	     // freq of first tone (Hz)
   unsigned short	bandwidth2;  // bandwidth of second tone (Hz)
   short		minlevel1;   // min amp of 1st tone ref 0dBm0
   short		minlevel2;   // min amp of 2nd tone ref 0dbm0
   short		twist;	     // allowable difference in tone powers
                                     // If (E1/E2 < twist) AND (E2/E1 < twist)
	                             // tone OK		
   short		snr;	     // min sig. to noise ratio to accept tone
   unsigned short	glitch;	     // short transitions of glitch ms ignored

   VPB_STRAN stran[VPB_MS];	     // cadence state transition table
} VPB_DETECT;

int vpb_settonedet(int handle, VPB_DETECT *d);
int vpb_gettonedet(int handle, int id, VPB_DETECT *d);
int vpb_debug_tonedet(int handle, int id, char file_name[], int sec);
int vpb_tonedet_make_default(VPB_DETECT *d);

/*-------------------------------------------------------------------------*\

			     RING FUNCTIONS

\*-------------------------------------------------------------------------*/

int vpb_set_ring(int handle, unsigned int rings_to_fire,
			unsigned int time_out);
int vpb_get_ring(int handle, unsigned int *rings_to_fire, 
			unsigned int *time_out);

#endif	// #ifndef __VPBAPI__	


