/*---------------------------------------------------------------------------*\

    FILE....: VPBREG.CPP
    TYPE....: C++ Functions
    AUTHOR..: David Rowe
    DATE....: 12/9/99

    Helper functions to set up the configuration database known as the DSP 
    registry.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

#include "config.h"

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "vpbreg.h"
#include "mess.h"
#include "vpbapi.h"
#include "apifunc.h"

// Compile-time constants --------------------------------------------

#define	NUM_CHANNELS	        4     // number of channels in VPB
#define	SIZE_MESS_Q		256   // size of message FIFOs (16-bit words)
#define	SIZE_RX_Q		800   // size of Rx (up) signal FIFOs
#define	SIZE_TX_Q		800   // size of Tx (down) signal FIFOs
#define	LENGTH_FRAME	        160   // samples in each processing frame
#define	MODEL_NUM		0     // 0: VPB4
				      // 1: VPB8L
#define	DD_MODEL		3     // 0: ISA port I/O only device driver
                                      // 1: ISA relay FIFO device driver
                                      // 2: Linux PCI driver
                                      // 3: User mode (no device driver)
     
// Run time Defaults (can be changed by user) -----------------------------

#define	NUM_CARDS		1                  // number of VPBs in PC
#define BASE                    0x300              // ISA card base address
#define FIRMWARE_FILE	        "/etc/vpbmain.out"	

/*-------------------------------------------------------------------------*\

			     GLOBALS

\*-------------------------------------------------------------------------*/

// defaults

static int default_base[NUM_CARDS] = {BASE};
static char *default_firmware_file = FIRMWARE_FILE;

// user-changable

static int vpb_num_cards = NUM_CARDS;
static int *vpb_base = default_base;
static char *vpb_firmware_file = default_firmware_file;

/*-------------------------------------------------------------------------*\

			      FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_config
	AUTHOR...: David Rowe
	DATE.....: 24/8/00

	Set installation specific config info.

\*--------------------------------------------------------------------------*/

int vpbreg_config(int num_cards, int *bases, char *firmware_file) {

	assert((num_cards > 0) && (num_cards <MAX_VPB));
	assert(bases != NULL);
	assert(firmware_file != NULL);

	vpb_num_cards = num_cards;
	vpb_base = bases;
	vpb_firmware_file = firmware_file;

	return VPB_OK;
}

/*-------------------------------------------------------------------------*\

			      CLASS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: ~VPBRegister
	AUTHOR...: John Kostogiannis
	DATE.....: 18/11/97

	Closes the VPB registry.

\*--------------------------------------------------------------------------*/

VPBRegister::~VPBRegister() {
	
	// free allocated VPB registry memory 

	assert(reg != NULL);
	free(reg);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: VPBRegister
	AUTHOR...: David Rowe
	DATE.....: 12/9/99

	Creates the VPB registry using compile-time constants.

\*--------------------------------------------------------------------------*/

VPBRegister::VPBRegister(ushort *num)
//      ushort *num;    number of VPBs
{
	*num = vpb_num_cards;
   
	VPBREG *preg; // ptr to current VPB registry (one per VPB card)
	ushort i,j;
	
	// validate arguments 

	assert(num != NULL);
	
	// hard code number of cards
	*num = vpb_num_cards ;
		
	// check if more than max devices have been detected 	
	assert(*num < MAX_VPB);

	// Allocate memory for VPB DSP registry
	reg = (VPBREG*)malloc(sizeof(VPBREG)*(*num));
	assert(reg != NULL);

	for(i=0; i<*num; i++) {
		// set up VPB Reg info
		preg = &reg[i];
		preg->base = (ushort)vpb_base[i];
		preg->numch = NUM_CHANNELS;
		preg->szmess = SIZE_MESS_Q;
		for(j=0; j<preg->numch; j++) {
			preg->szrxdf[j] = SIZE_RX_Q;
			preg->sztxdf[j] = SIZE_RX_Q;
		}
		preg->lsf = LENGTH_FRAME;
		preg->model = MODEL_NUM;
		preg->ddmodel = DD_MODEL;
		preg->szRelayBuf = 0; // unused

		// init ptrs to PC copy of FIFO state variables
		preg->dnmess = NULL;		
		preg->upmess = NULL;			
		for(j=0; j<MAXCH; j++) {
			preg->rxdf[j] = NULL;		
			preg->txdf[j] = NULL;		
		}
	}

	// set up firmware path and file name
	strcpy(preg->firm, vpb_firmware_file);
}

void vpbreg_load_default_tones(int numch)
{
	// defaults for tone detectors should be loaded here
}

