/*---------------------------------------------------------------------------*\

    FILE....: VPBTONED.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 26/2/98

    This file contains the implementation of the programmable tone
    detection API functions, vpb_settonedet and vpb_gettonedet.
	
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	Copyright (C) 1999 Voicetronix Pty Ltd

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

							INCLUDES
							
\*---------------------------------------------------------------------------*/

#include "config.h"

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <math.h>

#include "contypes.h"
#include "comm.h"
#include "vpbreg.h"
#include "vpbhandle.h"
#include "mapdev.h"
#include "vpbapi.h"
#include "apifunc.h"
#include "vpbtoned.h"
#include "vpbconfig.h"
#include "objtrack.h"
#include "mess.h"
#include "generic.h"
#include "vpbreg.h"

/*---------------------------------------------------------------------------*\

				 DEFINES
							
\*---------------------------------------------------------------------------*/

#define	FMIN		100	// minimum tone detection frequency
#define	FMAX		3900	// maximum tone detection frequency
#define	LMAX		0	// maximum tone level in dB
#define	SNRMIN		0	// maximum SNR in dB
#define	TWISTMIN	0	// minimum twist in dB
#define	TSAM		125E-6	// sampling period

#define	SIZEOFHEADERINWORDS	15
#define	STARTNSTATES		12
#define	SIZEOFVPB_STRANINWORDS	4

/*---------------------------------------------------------------------------*\

				  GLOBALS
							
\*---------------------------------------------------------------------------*/

static VPB_DETECT	*td;	// record of each tone detector for each ch
static ushort		*numtd;	// num detector for each ch

static	ushort		logging = 0;
static	ushort		logch;
static	long		logframes;
static	long		logcnt;
static	FILE*		logfile;

/*-------------------------------------------------------------------------*\

			     FUNCTION HEADERS

\*-------------------------------------------------------------------------*/

void check_freq(float f, float bw);
void check_level(float l);
void check_snr(float snr);
void check_twist(float twist);
void check_states(VPB_DETECT *d);

/*-------------------------------------------------------------------------*\

				  FUNCTIONS

\*-------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbtoned_open()
	AUTHOR...: David Rowe
	DATE.....: 26/2/98

	Initialises the programmable tone generator module.  This function 
	should be called when the API is initialised.  This function loads
	allocates memory for the tone detector params mirror store in PC
	memory.

\*--------------------------------------------------------------------------*/

void vpbtoned_open(int numdev)
//      int    numdev	total number of channel devices (and hence tone dets)
{
	CheckNew(td = new VPB_DETECT[numdev*VPB_MD]);
	CheckNew(numtd = new ushort[numdev*VPB_MD]);

	int i;

	for(i=0; i<numdev; i++)	
		numtd[i] = 0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbtoned_close()
	AUTHOR...: David Rowe
	DATE.....: 26/2/98

	Frees memory used for storing tone detector info.

\*--------------------------------------------------------------------------*/

void vpbtoned_close()
//      int    numdev	total number of channel devices (and hence tone dets)
{
	delete [] td;
	delete [] numtd;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: settonedet()
	AUTHOR...: David Rowe
	DATE.....: 26/2/98

	Enables user to set a user programmed tone, or replace an exisiting 
	tone.  Performs translation of user parameters to VPB message.  The
	tones are defined on a per channel basis.

\*--------------------------------------------------------------------------*/

void settonedet(int chdev, VPB_DETECT *d)
//      int		chdev	       channel device to add tone to
//	VPB_DETECT	*d	       structure containing detector parameters
{
	int		i;
	ushort		b,ch;

	int id = d->tone_id;
	
	// determine if a detector with this identifier already exists

	int exists = 0;
	int index = 0;

	for(i=0; i<numtd[chdev]; i++) {
		if (td[chdev*VPB_MD+i].tone_id == id) {
		    exists = 1;
			index = i;
		}
	}

	// if ident doesn't exist, make new tone

	if (!exists)
		index = numtd[chdev]++;
	if (numtd[chdev] >= VPB_MD)
		assert(0); //  VPBAPI_DETECT_TOO_MANY_DETECTORS

	// validate tone parameters
		
	if ((d->ntones != 1) && (d->ntones != 2))
		assert(0); //  VPBAPI_DETECT_NTONES_ERROR
	check_freq(d->freq1, d->bandwidth1);
	check_level(d->minlevel1);
	if (d->ntones == 2) {
		check_freq(d->freq2, d->bandwidth2);
	    check_level(d->minlevel2);
	    check_twist(d->twist);
	}
	check_snr(d->snr);
	check_states(d);

	// OK, translate params into message and send to VPB

	word	mess[COMM_MAX_MESS];
	assert(COMM_MAX_MESS > (SIZEOFHEADERINWORDS+VPB_MS*
				SIZEOFVPB_STRANINWORDS));

	mess[0] = (word)(SIZEOFHEADERINWORDS+d->nstates*
			 SIZEOFVPB_STRANINWORDS);
	mess[1] = (word)PC_TONED_ST;
	mess[2] = (word)objtrack_handle_to_id(CPOBJ,chdev);
	memcpy(&mess[3], d, sizeof(VPB_DETECT));
	maphndletodev(chdev, &b, &ch);
	vpb_c->PutMessageVPB(b, mess);	
	vpb_c->CheckForAssert(b);

	// keep record of tone detector in PC

	memcpy(&td[chdev*VPB_MD+id], d, sizeof(VPB_DETECT));
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_settonedet()
	AUTHOR...: David Rowe
	DATE.....: 26/2/98

	Enables user to set a user programmed tone, or replace an exisiting 
	tone.  Performs translation of user parameters to VPB message.  The
	tones are defined on a per channel basis.

	This version (for the API), stops the config mamanger before download-
	ing a tone to prevent computational overoad probs.

\*--------------------------------------------------------------------------*/

int vpb_settonedet(int chdev, VPB_DETECT *d)
//      int		chdev	 channel device to add tone to
//	VPB_DETECT	*d	 structure containing detector parameters
{
	ValidHandleCheck(chdev);
	settonedet(chdev, d);

	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_gettonedet()
	AUTHOR...: David Rowe
	DATE.....: 27/2/98

	Enables user to extract a tone detectors parameters.

\*--------------------------------------------------------------------------*/

int vpb_gettonedet(int chdev, int id, VPB_DETECT *d)
//      int	    chdev	channel device holding tone detector
//      int	    id		tone identifier
//	VPB_DETECT  *d		structure containing detector parameters
{
	int	    i;

	ValidHandleCheck(chdev);

	// determine if a detector with this identifier already exists

	int exists = 0;
	int index = 0;

	for(i=0; i<numtd[chdev]; i++) {
		if (td[chdev*VPB_MD+i].tone_id == id) {
			exists = 1;
			index = i;
		}
	}

	// if ident doesn't exist, make new tone

	if (!exists)
		assert(0); // VPBAPI_DETECT_INVALID_TONE

	// copy tone params from PC mirror

	memcpy(d, &td[chdev*VPB_MD+id], sizeof(VPB_DETECT));

	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: check_freq()
	AUTHOR...: David Rowe
	DATE.....: 27/2/98

	Frequency validation function.

\*--------------------------------------------------------------------------*/

void check_freq(float f, float bw)
{
	bw /= 2;
	if ( ((f-bw) < FMIN) || ((f+bw) > FMAX) )
		assert(0); // VPBAPI_DETECT_INVALID_FREQUENCY
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: check_level()
	AUTHOR...: David Rowe
	DATE.....: 27/2/98

	Level validation function.

\*--------------------------------------------------------------------------*/

void check_level(float l)
//	float	l	Level in dB with maximum of LMAX dB
{
	if (l > LMAX)
		assert(0); // VPBAPI_DETECT_INVALID_LEVEL
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: check_snr()
	AUTHOR...: David Rowe
	DATE.....: 27/2/98

	SNR validation function.

\*--------------------------------------------------------------------------*/

void check_snr(float snr)
//	float	snr		SNR in dB with minimum of SNRMIN dB
{
	if (snr < SNRMIN)
		assert(0); // VPBAPI_DETECT_INVALID_SNR
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: check_twist()
	AUTHOR...: David Rowe
	DATE.....: 27/2/98

	Twist validation function.

\*--------------------------------------------------------------------------*/

void check_twist(float twist)
//	float	twist		twist in dB with minimum of TWISTMIN dB
{
	if (twist < TWISTMIN)
		assert(0); // VPBAPI_DETECT_INVALID_TWIST
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: check_states()
	AUTHOR...: David Rowe
	DATE.....: 27/2/98

	Validation function for cadence states, also scales ton and toff for
	each state to 10ms increments rather than API level 1ms increments.

\*--------------------------------------------------------------------------*/

void check_states(VPB_DETECT *d)
//	VPB_DETECT *d		tone detector parameters
{
	int	   i,type;

	if (d->nstates > VPB_MS)
		assert(0); // VPBAPI_DETECT_TOO_MANY_STATES
	if (d->nstates == 0)
		assert(0); // VPBAPI_NOT_ENOUGH_STATES

	for(i=0; i<d->nstates; i++) {
		type = d->stran[i].type;
		if ((type != VPB_TIMER) && (type != VPB_RISING) && 
		    (type != VPB_FALLING) && (type != VPB_DELAY)) 
			assert(0); // VPBAPI_DETECT_INVALID_STATES
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpb_debug_tonedet()
	AUTHOR...: David Rowe
	DATE.....: 29/4/98

	Instructs API to produce a debug file of tone detector parameters for
	one channel. 

\*--------------------------------------------------------------------------*/

int pb_debug_tonedet(int chdev, int id, char file_name[], int sec)
//	int	chdev		channel to log debug info for
//	int	id		ID of tone detector to log
//	char	file_name[]	name of file where debug info is to be written
//	int	sec		number of seconds of info to log
{

	if (logging)
		assert(0); // VPBAPI_DETECT_ALREADY_LOGGING
	logging = 1;

	logfile = fopen(file_name, "wt");
	if (logfile == NULL)
		assert(0); // VPBAPI_DETECT_BAD_LOG_FILE_NAME
		
	// set up statics

	ushort b,ch;
	maphndletodev(chdev, &b, &ch);

	logch = chdev;
	VPBREG* pvpbreg = vpb_c->vpbreg(b);
	logframes = (long)sec*8000l/(pvpbreg->lsf);
	if (logframes > SHRT_MAX) logframes = SHRT_MAX;
	logcnt = 0;

	// tell DSP to switch debugging on

	word	mess[PC_LTONED_DEBUG];
	mess[0] = PC_LTONED_DEBUG;
	mess[1] = PC_TONED_DEBUG;
	mess[2] = objtrack_handle_to_id(CPOBJ, chdev);
	mess[3] = id;
	mess[4] = (word)logframes;
	vpb_c->PutMessageVPB(b, mess);	

	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: toned_debug_log
	AUTHOR...: David Rowe
	DATE.....: 29/4/98

	This function is called from the MMQ to iterate the debug logging
	data.

\*--------------------------------------------------------------------------*/

void toned_debug_log(ushort board, word mess[])
//	ushort	board
//	word	mess[]	messge from DSP
{
	ushort	chdev;

	// validate

	assert(logging);
	assert(mess[1] == DSP_TONED_LOG);
	assert(logfile != NULL);
	assert(logframes);
	assert(mess[4] == logcnt);

	// extract params from message and print to text file

	chdev = objtrack_id_to_handle(CPOBJ, mess[2], board);
	assert(chdev == logch);

	fprintf(logfile,"[%06d] ch:%02d id:%02d t1:%d t2:%d tw:%d snr:%d t:%d"
		" ton:%5d toff:%5d r:%d f:%d s:%02d ns:%02d\n", 
					logcnt,
					chdev,	
					mess[3],	// tone id
					mess[5],	// tone1
					mess[6],	// tone2
					mess[7],	// twist
					mess[8],	// snr
					mess[9],	// tone
					
					mess[10],	// tone_on
					mess[11],	// tone_off
					mess[12],	// rising
					mess[13],	// falling
					mess[14],	// state
					mess[15]	// next_state
			);

	// if we have finished logging close file

	logcnt++;
	if (logframes == logcnt) {
		logging = 0;
		fclose(logfile);

	}
}

